/*
   File: dcg_xput.c
   Defines basic transput routines

   Copyright (C) 2008 Marc Seutter

   This library is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published
   by the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: dcg_xput.c,v 1.15 2008/06/28 13:41:17 marcs Exp $"
*/

/* include config.h if autoconfigured */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* local includes */
#include "dcg.h"
#include "dcg_string.h"
#include "dcg_xput.h"

/*
   Estimate printing of standard types
*/
int est_int (int x)
	{ char buf[20];
	  sprintf (buf, "%d", x);
	  return (strlen (buf));
	};

int est_u_int (u_int x)
	{ char buf[20];
	  sprintf (buf, "%u", x);
	  return (strlen (buf));
	};

int est_real (real x)
	{ char buf[30];
	  sprintf (buf, "%g", x);
	  return (strlen (buf));
	};

int est_string (string x)
	{ if (x == string_nil) return (2);
	  return (strlen (x) + 2);
	};

void pppindent (FILE *f, int ind)
	{ int i;
	  for (i = 0; i < ind/8; i++) fputc ('\t', f);
	  for (i = 0; i < ind%8; i++) fputc (' ', f);
	};

void pppdelim (FILE *f, int horiz, int ind, char x)
	{ if ((x == ')') || (x == ']') || (x == '}') || (x == '>'))
	     { if (!horiz)
		  { fputc ('\n', f);
		    pppindent (f, ind);
		  }
	       else fputc (' ', f);
	       fputc (x, f);
	     }
	  else
	     { fputc (x, f);
	       if (!horiz)
		  { fputc ('\n', f);
		    pppindent (f, ind + 2);
		  }
	       else fputc (' ', f);
	     };
	};

/* Define support for saving */
static u_int savechecksum;
void initsave (FILE *f)
	{ savechecksum = 0;
	};

void finishsave (FILE *f)
	{ fputc ((0xff - savechecksum) & 0xff, f);
	};

void savechar (FILE *f, char x)
	{ savechecksum = (savechecksum + (u_int) x) & 0xff;
	  fputc (x, f);
	};

/*
   Ints are saved run length encoded according to the Dwarf 2 standard
   Chunks of 7 bit, beginning with the least significant bits are
   output until there are no more significant bits to output. The
   sign bit in each chunk is used to indicate if more chunks are
   following.
*/
void save_u_int (FILE *f, u_int x)
	{ u_int value = x;
	  do
	     { int byte = value & 0x7f;
	       value >>= 7;
	       if (value) byte |= 0x80;
	       savechar (f, (char) byte);
	     }
	  while (value);
	};

void save_int (FILE *f, int x)
	{ int value = x;
	  int more = 1;
	  do
	     { int byte = value & 0x7f;
	       value >>= 7;
	       if ((value == 0) && !(byte & 0x40)) more = 0;
	       if ((value == -1) && (byte & 0x40)) more = 0;
	       if (more) byte |= 0x80;
	       savechar (f, (char) byte);
	     }
	  while (more);
	};

void save_u_int64 (FILE *f, u_int64 x)
	{ u_int64 value = x;
	  do
	     { int byte = (int) (value & u_int64_const (0x7f));
	       value >>= 7;
	       if (value) byte |= 0x80;
	       savechar (f, (char) byte);
	     }
	  while (value);
	};

void save_int64 (FILE *f, int64 x)
	{ int64 value = x;
	  int more = 1;
	  do
	     { int byte = (int) (value & int64_const (0x7f));
	       value >>= 7;
	       if ((value == int64_const (0)) && !(byte & 0x40)) more = 0;
	       if ((value == int64_const (-1)) && (byte & 0x40)) more = 0;
	       if (more) byte |= 0x80;
	       savechar (f, (char) byte);
	     }
	  while (more);
	};

/*
   Save the size as an int
*/
void savesize (FILE *f, int sz)
	{ save_int (f, sz);
	};

/*
   Since all integer numbers are more or less saved in a little endian way,
   the real is saved as little endian
*/
void save_real (FILE *f, real x)
	{ int i;
	  char *dptr = (char *) &x;
#ifdef WORDS_BIGENDIAN
	  for (i = sizeof (real) - 1; 0 <= i; i--)
	     savechar (f, dptr[i]);
#else
	  for (i = 0; i < sizeof (real); i++)
	     savechar (f, dptr[i]);
#endif
	};

void save_string (FILE *f, string x)
	{ int i;
	  savesize (f, strlen (x));
	  for (i=0; i < (int) strlen (x); i++) savechar (f, x[i]);
	};

/* Define support for loading */
static int loadchecksum;
void initload (FILE *f)
	{ loadchecksum = 0;
	  /* version & comp date loading? */
	};

int finishload (FILE *f)
	{ char ch;
	  if (!loadchar (f, &ch)) return (0);
	  if (((int) loadchecksum & 0xff) != 0xff) return (0);
	  return (fgetc (f) == EOF);
	};

int loadchar (FILE *f, char *x)
	{ int ch = fgetc (f);
	  if (ch == EOF) return (0);
	  loadchecksum = (loadchecksum + ch) & 0xff;
	  *x = (char) ch;
	  return (1);
	};

int load_u_int (FILE *f, u_int *x)
	{ u_int value = 0;
	  int shift = 0;
	  char bb;
	  do
	     { if (!loadchar (f, &bb)) return (0);
	       value |= (((u_int) (bb & 0x7f)) << shift);
	       shift += 7;
	     }
	  while (bb & 0x80);
	  *x = value;
	  return (1);
	};

int load_int (FILE *f, int *x)
	{ int value = 0;
	  int shift = 0;
	  char bb;
	  do
	     { if (!loadchar (f, &bb)) return (0);
	       value |= (((u_int) (bb & 0x7f)) << shift);
	       shift += 7;
	     }
	  while (bb & 0x80);
	  if (shift > 31) shift = 31;
	  if (bb & 0x40) value |= -(1 << shift);
	  *x = value;
	  return (1);
	};

int load_u_int64 (FILE *f, u_int64 *x)
	{ u_int64 value = u_int64_const (0);
	  int shift = 0;
	  char bb;
	  do
	     { if (!loadchar (f, &bb)) return (0);
	       value |= (((u_int64) (bb & 0x7f)) << shift);
	       shift += 7;
	     }
	  while (bb & 0x80);
	  *x = value;
	  return (1);
	};

int load_int64 (FILE *f, int64 *x)
	{ int64 value = int64_const (0);
	  int shift = 0;
	  char bb;
	  do
	     { if (!loadchar (f, &bb)) return (0);
	       value |= (((u_int64) (bb & 0x7f)) << shift);
	       shift += 7;
	     }
	  while (bb & 0x80);
	  if (shift > 63) shift = 63;
	  if (bb & 0x40) value |= -(int64_const (1) << shift);
	  *x = value;
	  return (1);
	};

/*
   Load the size as an int
*/
int loadsize (FILE *f, int *x)
	{ return (load_int (f, x));
	};

int load_real (FILE *f, real *x)
	{ int i;
	  char ch;
	  char *dptr = (char *) x;
#ifdef WORDS_BIGENDIAN
	  for (i = sizeof (real) - 1; 0 <= i; i--)
	     { if (!loadchar (f, &ch)) return (0);
	       dptr[i] = ch;
	     };
#else
	  for (i = 0; i < sizeof (real); i++)
	     { if (!loadchar (f, &ch)) return (0);
	       dptr[i] = ch;
	     };
#endif
	  return (1);
	};

int load_string (FILE *f, string *x)
	{ int i, size;
	  char ch;
	  char buffer[MAXSTRLEN];
	  if (!loadsize (f, &size)) return (0);
	  for (i=0; i < size; i++)
	     { if (!loadchar (f, &ch)) return (0);
	       buffer[i] = ch;
	     };
	  buffer[size] = '\0';
	  *x = alloc_string (buffer);
	  return (1);
	};

int load_vptr (FILE *f, vptr *x)
	{ *x = vptr_nil;
	  return (1);
	};
