/*
   File: dcg_string.c
   Defines text operations
   Optionally provides routines to share string space

   Copyright (C) 2008 Marc Seutter

   This library is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published
   by the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: dcg_string.c,v 1.9 2008/06/28 13:41:17 marcs Exp $"
*/

/* standard includes */
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* local includes */
#include "dcg.h"
#include "dcg_error.h"
#include "dcg_alloc.h"
#include "dcg_string.h"

string new_string (char *t)
	{ string new;
	  if (t == NULL)
	     panic ("new_string called with NULL argument");
	  new = dcg_malloc (strlen (t) + 1);
	  strcpy (new, t);
	  return (new);
	};

string new_fmtd_string (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  return (new_string (buf));
	};

string concat_string (string t1, string t2)
	{ string new;
	  if ((t1 == NULL) || (t2 == NULL))
	     panic ("concat_string called with NULL argument");
	  new = dcg_malloc (strlen (t1) + strlen (t2) + 1);
	  strcpy (new, t1);
	  strcat (new, t2);
	  return (new);
	};

/*
   Do a case insensitive equality test
*/
int equal_string_nocase (string t1, string t2)
	{ char *t1ptr, *t2ptr;
	  if ((t1 == NULL) || (t2 == NULL))
	     panic ("equal_string_nocase called with NULL argument");
	  for (t1ptr = t1, t2ptr = t2; (*t1ptr) || (*t2ptr); t1ptr++, t2ptr++)
	     { char ch1, ch2;
	       if (!(*t1ptr) || !(*t2ptr)) return (0);
	       if (isupper (*t1ptr)) ch1 = tolower (*t1ptr); else ch1 = *t1ptr;
	       if (isupper (*t2ptr)) ch2 = tolower (*t2ptr); else ch2 = *t2ptr;
	       if (ch1 != ch2) return (0);
	     };
	  return (1);
	};

/*
   Provide data structure to share strings (with some overhead)
*/
typedef struct string_tree_rec *string_tree;
struct string_tree_rec
	{ string word;
	  string_tree left;
	  string_tree right;
	};
#define string_tree_nil ((string_tree) NULL)
static string_tree root;

string alloc_string (char *t)
	{ string_tree *ptr = &root; 
	  while (1)
	     { string_tree current = *ptr;
	       int cond;
	       if (current == string_tree_nil)
		  { string_tree new =
			(string_tree) dcg_malloc (sizeof (struct string_tree_rec));
		    *ptr = new;
		    new -> word = new_string (t);
		    new -> left = string_tree_nil;
		    new -> right = string_tree_nil;
		    return (attach_string (new -> word));
		  };
	       cond = strcmp (t, current -> word);
	       if (cond < 0) ptr = &current -> left;
	       else if (cond > 0) ptr = &current -> right;
	       else return (attach_string (current -> word));
	     }
	};

string alloc_fmtd_string (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);
	  return (alloc_string (buf));
	};

static void rfre_string_tree (string_tree *oroot)
	{ string_tree old = *oroot;
	  if (old == string_tree_nil) return;
	  rfre_string_tree (&old -> left);
	  rfre_string_tree (&old -> right);
	  detach_string (&old -> word);
	  dcg_detach ((char **) oroot);
	};

void init_string_adm ()
	{ root = string_tree_nil;
	};

void finish_string_adm ()
	{ rfre_string_tree (&root);
	};
