/*
   File: main.c
   Driver for mini compiler
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/wait.h>

/* options and or pragmats */
static int debug_allocation;
static int debug;
static int verbose;
static int optimize;
static int dump_lex;
static int dump_parser;
static int dump_checker;
static int dump_imcgen;
static int show_version;

static void init_options ()
	{ debug_allocation = 0;
	  dump_lex = 0;
	  dump_parser = 0;
	  dump_checker = 0;
	  dump_imcgen = 0;
	  show_version = 0;
	  verbose = 0;
	  debug = 0;
	};

static void print_usage ()
	{ fprintf (stderr, "usage: mini [flags] [filename] [more_flags]\n");
	  fprintf (stderr, "-h:  provide this help\n");
	  fprintf (stderr, "-v:  verbose\n");
	  fprintf (stderr, "-V:  report version\n");
	  fprintf (stderr, "-O:  optimize code\n");
	  fprintf (stderr, "-da: enable allocation debugging\n");
	  fprintf (stderr, "-dl: dump lexical analyzer output\n");
	  fprintf (stderr, "-dp: dump parser output\n");
	  fprintf (stderr, "-dc: dump checker output\n");
	  fprintf (stderr, "-di: dump intermediate code\n");
	  exit (4);
	};

static void syntax_error (char *syn_error)
	{ fprintf (stderr, "error on command line: %s\n", syn_error);
	  print_usage ();
	};

#define streq(s1,s2) (strcmp((s1),(s2)) == 0)
static void scan_option (char *ptr)
	{ if (streq (ptr, "dl"))      { dump_lex = 1; verbose = 1; }
	  else if (streq (ptr, "dp")) { dump_parser = 1; verbose = 1; }
	  else if (streq (ptr, "dc")) { dump_checker = 1; verbose = 1; }
	  else if (streq (ptr, "di")) { dump_imcgen = 1; verbose = 1; }
	  else if (streq (ptr, "da")) { debug_allocation = 1; verbose = 1; }
	  else if (streq (ptr, "d")) debug = 1;
	  else if (streq (ptr, "O")) optimize = 1;
	  else if (streq (ptr, "v")) { show_version = 1; verbose = 1; }
	  else if (streq (ptr, "V")) show_version = 1;
	  else if (streq (ptr, "h")) print_usage ();
	  else syntax_error ("illegal option specified");
	};

static char *parse_command_line (int argc, char **argv)
	{ char *input_name = NULL;
	  int ix;
	  for (ix = 1; ix < argc; ix++)
	     { char *arg = argv[ix];
	       if (arg[0] == '-') scan_option (arg+1);
	       else if (input_name == NULL) input_name = strdup (arg);
	       else syntax_error ("too many arguments");
	     };
	  if (input_name == NULL) syntax_error ("missing filename");
	  return (input_name);
	};

static char *basename_from_name (char *name)
	{ char basename[MAXPATHLEN+1];
	  char *sptr, *dptr;
	  for (sptr = name, dptr = basename;
	       (*sptr != '\0') && (*sptr != '.');
	       sptr++, dptr++) *dptr = *sptr;
	  *dptr = '\0';
	  return (strdup (basename));
	};

#if defined(sun)
#if !defined(__SVR4) && !defined(__svr4__)
extern char *sys_errlist[];
#define strerror(errnum) sys_errlist[errnum]
#endif
#endif
static int spawn_and_wait (char *argv[])
	{ pid_t pid;
	  int status;
	  pid = fork ();
	  if (pid < 0)
	     { /* could not fork, bad if so */
	       fprintf (stderr, "could not fork: %s", strerror (errno));
	       exit (4);
	     }
	  else if (pid == 0)
	     { /* child code */
	       execvp (argv[0], argv);

	       /* if we reach this point, we could not exec */
	       fprintf (stderr, "could not exec %s: %s",
			argv[0], strerror (errno));
	       exit (4);
	     };

	  /* parent code */
	  waitpid (pid, &status, 0);
	  return (status);
	};

static void do_mini_compile (char *input_name)
	{ char *comp_argv[13];
	  int status;
	  int ix = 0;

	  /* compile, pass arguments on */
	  comp_argv[ix++] = "mini-compile";
	  if (debug)		comp_argv[ix++] = "-d";
	  if (verbose)		comp_argv[ix++] = "-v";
	  if (show_version)     comp_argv[ix++] = "-V";
	  if (optimize)		comp_argv[ix++] = "-O";
	  if (debug_allocation) comp_argv[ix++] = "-da";
	  if (dump_lex)		comp_argv[ix++] = "-dl";
	  if (dump_parser)	comp_argv[ix++] = "-dp";
	  if (dump_checker)	comp_argv[ix++] = "-dc";
	  if (dump_imcgen)	comp_argv[ix++] = "-di";
	  comp_argv[ix++] = input_name;
	  comp_argv[ix] = NULL;
	  status = spawn_and_wait (comp_argv);
	  if (status) exit (4);
	};

static void do_assemble (char *basename)
	{ char ass_file[MAXPATHLEN + 1];
	  char *ass_argv[4];
	  int status;
	  int ix = 0;

	  sprintf (ass_file, "%s.s", basename);
	  if (verbose) fprintf (stderr,"   assembling %s...\n", ass_file);
	  ass_argv[ix++] = "gcc";
	  ass_argv[ix++] = "-c";
	  ass_argv[ix++] = ass_file;
	  ass_argv[ix] = NULL;
	  status = spawn_and_wait (ass_argv);
	  if (status) exit (4);
	};

static void do_link (char *basename)
	{ char ld_file[MAXPATHLEN + 1];
	  char lib_flag[MAXPATHLEN + 3];
	  char *ld_argv[8];
	  int status;
	  int ix = 0;

	  sprintf (ld_file, "%s.o", basename);
	  sprintf (lib_flag, "-L%s", LDIR);
	  if (verbose) fprintf (stderr, "   linking %s...\n", ld_file);
	  ld_argv[ix++] = "gcc";
	  ld_argv[ix++] = "-o";
	  ld_argv[ix++] = basename;
	  ld_argv[ix++] = ld_file;
	  ld_argv[ix++] = lib_flag;
	  ld_argv[ix++] = "-lmrts";
	  ld_argv[ix] = NULL;
	  status = spawn_and_wait (ld_argv);
	  if (status) exit (4);
	};

int main (int argc, char **argv)
	{ char *input_name;
	  char *basename;

	  /* parse command line options and arg */
	  init_options ();
	  input_name = parse_command_line (argc, argv);
	  basename = basename_from_name (input_name);

	  /* compile, assemble and link */
	  do_mini_compile (input_name);
	  do_assemble (basename);
	  do_link (basename);

	  return (0);
	};
