/*
   File: erts_trellis.c
   Defines the basic trellis operations for the EAG3 runtime system.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_trellis.c,v 1.20 2013/03/13 14:32:49 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_plist.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_utils.h>
#include <ebase_lexicon.h>
#include <ebase_lexicon_impl.h>
#include <ebase_voc_search.h>
#include <ebase_regexp_search.h>

/* local includes */
#include "erts_handle.h"
#include "erts_trellis.h"
#include "erts_trans.h"
#include "erts_handle_impl.h"
#include "erts_trellis_impl.h"

static State init_state (Trellis trel, int offset, int linenr, int colnr, uchar lex_state)
{ /* Determine index in state cache */
  int index = NR_LEX_STATES * offset + (lex_state - LEX_STATE_S); /* LEX_STATE_S != 0 */
  if (offset > trel -> length)
    dcg_abort ("init_state", "state offset %d is beyond end offset %d", offset, trel -> length);

  /* If we have not visited this state before, we create it */
  if (trel -> state_cache[index] == NULL)
    { /* Fresh state, initialize */
      State fresh = (State) dcg_malloc (sizeof (struct state_rec));
      fresh -> trans = transition_nil;
      fresh -> offset = offset;
      fresh -> linenr = linenr;
      fresh -> colnr = colnr;
      fresh -> lex_state = lex_state;
      fresh -> lex_flags = 0;
      fresh -> lex_phase = 0;
      if (trel -> neg_memo_size)
        fresh -> neg_memos = (int *) dcg_calloc (trel -> neg_memo_size, sizeof (int));
      else fresh -> neg_memos = NULL;
      trel -> state_cache[index] = fresh;
      return (fresh);
    }
  else
    { /* State has been reached earlier, do some consistency checks */
      State old = trel -> state_cache[index];
      if ((old -> offset != offset) || (old -> linenr != linenr) || (old -> colnr != colnr))
	dcg_abort ("init_state", "Mismatch between old (%d,%d,%d) and new locators (%d,%d,%d)",
		   old -> offset, old -> linenr, old -> colnr, offset, linenr, colnr);
      if (old -> lex_state != lex_state)
	dcg_abort ("init_state", "Mismatch between old state %d and new state %d",
		   old -> lex_state, lex_state);
      return (old);
    };
}

/* Determine the initial state */
static State make_initial_state (Trellis trel)
{ Lexicon lex = trel -> lexicon;
  char *input = trel -> input;

  /* The default is without initial white space */
  uchar lex_state = LEX_STATE_W;

  /* If we see white space or we can always insert epsilon, we start in state S */
  if (ebs_ahead_white_space (lex, input) || lex -> empty_white_space)
    lex_state = LEX_STATE_S;
  return (init_state (trel, 0, trel -> linenr, trel -> colnr, lex_state));
}

State erts_init_trellis (EagrtsHandle hnd, char *input, int length, int linenr, int colnr)
{ /* Allocate trellis and push */
  Trellis trel = (Trellis) dcg_malloc (sizeof (struct trellis_rec));
  trel -> lexicon = hnd -> lexicon;
  trel -> previous = hnd -> trellis;
  if (hnd -> negative_memoization && (hnd -> nr_neg_memos > 0))
    trel -> neg_memo_size = (hnd -> nr_neg_memos - 1)/ 32 + 1;
  else trel -> neg_memo_size = 0;
  hnd -> trellis = trel;

  /* Remember input text */
  trel -> input = input;
  trel -> length = length;
  trel -> linenr = linenr;
  trel -> colnr = colnr;

  /* Allocate state cache: we allocate a bit more than necessary */
  trel -> state_cache = dcg_calloc (NR_LEX_STATES * (length + 1), sizeof (State));
  trel -> curr_state = make_initial_state (trel);
  return (trel -> curr_state);
}

State erts_determine_next_start_state (EagrtsHandle hnd)
{ State final_state;
  if (!hnd -> partial_parse) return (state_nil);
  final_state = erts_get_final_parse_state (hnd);
  if (final_state == state_nil) return (state_nil);
  if (final_state -> offset != hnd -> trellis -> length)
    return (final_state);
  return (state_nil);
}

void erts_set_start_state (EagrtsHandle hnd, State start_state)
{ hnd -> trellis -> curr_state = start_state;
  hnd -> start_state = start_state;
  hnd -> curr_penalty = 0;
}

/* Initialize a transition */
static Transition init_transition (State target, char *from, char *to,
				   int class, int nr, int info, Penalty penalty)
{ Transition trans = (Transition) dcg_malloc (sizeof (struct transition_rec));
  trans -> next = transition_nil;
  trans -> target = target;
  trans -> from = from;
  trans -> to = to;
  trans -> class = class;
  trans -> flags = 0;
  trans -> nr = nr;
  trans -> info = info;
  trans -> penalty = penalty;
  return (trans);
}

/*
   Locate transitions that match our criterion
   Transition lists are kept sorted by class and number
*/
static Transition find_transition (Transition trans, int class, int nr)
{ while (trans != transition_nil)
    { if ((trans -> class == class) && (trans -> nr == nr))
	return (trans);
      if ((trans -> class > class) ||
          ((trans -> class == class) && (trans -> nr > nr)))
	return (transition_nil);
      trans = trans -> next;
    };
  return (transition_nil);
}

/* Add a new transition: if strict, abort if an identical transition is found */
static Transition add_transition (Transition *anchor, State target, char *from, char *to,
				  short class, int nr, int info, Penalty penalty, int strict)
{ Transition init = init_transition (target, from, to, class, nr, info, penalty);
  Transition *ptr = anchor;
  while (*ptr != transition_nil)
    { Transition curr = *ptr;
      if ((curr -> class > class) ||
	  ((curr -> class == class) && (curr -> nr > nr)))
	{ /* This is the right spot */
	  init -> next = curr;
	  *ptr = init;
	  return (init);
	};
      if (strict && (curr -> class == class) &&
	  (curr -> nr == nr) && (curr -> target == target))
	dcg_internal_error ("add_transition");
      ptr = &curr -> next;
    };
  *ptr = init;
  return (init);
}

/*
   Scan for white space or allowable epsilon transitions.
   We will execute this code only once for S states.
   It should not be possible to visit this code in other situations.
   It is also ensured that the result can only be a single transition.
*/
static Transition scan_white_space_transition (Trellis trel, State state)
{ Lexicon lex = trel -> lexicon;
  int linenr, colnr, offset;
  char *from, *to;
  State target;
  int ws_ok;

  /* Consistency check: we must be in state S */
  if (state -> lex_state != LEX_STATE_S)
    dcg_internal_error ("scan_white_space_transition");

  /* Check if this state points to the end */
  to = from = trel -> input + state -> offset;
  if (!(*from))
    return (transition_nil);

  /* Check if we have tried to match the white space before */
  if (state -> lex_flags & LEX_STATE_WS_TRIED)
    return (find_transition (state -> trans, TransWhiteSpace, 0));

  /* This is the first time we are here, check for white space */
  /* Note that the McCarthy OR of C is definitely intended here */
  state -> lex_flags |= LEX_STATE_WS_TRIED;
  ws_ok = ebs_is_white_spaces (lex, &to)  ||	/* White spaces are ok */
	  (lex -> empty_white_space)	  ||	/* We allow empty white space transitions */
	  (!lex -> white_spaces -> size)  ||	/* No white space: white space is in the grammar */
	  ebs_ahead_separator (lex, from) ||	/* A separator is ahead or */
	  ebs_past_separator (lex, from);	/* we just passed a separator */
  if (!ws_ok)
    return (transition_nil);
  
  /* Ok, we have parsed white space or matched empty: update position */
  offset = to - trel -> input;
  linenr = state -> linenr;
  colnr = state -> colnr;
  ebs_update_position (lex, from, to, &linenr, &colnr);

  /* Fetch the target state */
  target = init_state (trel, offset, linenr, colnr, LEX_STATE_W);
  return (add_transition (&state -> trans, target, from, to, TransWhiteSpace, 0, 0, 0, 1));
}

State erts_handle_epsilon_transition (Trellis trel, State state)
{ switch (state -> lex_state)
    { case LEX_STATE_S:
        { /* Only in state S, we may see white space ahead, maybe we have been here before */
	  Transition trans = scan_white_space_transition (trel, state);
	  if (trans != transition_nil)
	    return (trans -> target);
          return (state);
        };
      case LEX_STATE_I:
      case LEX_STATE_W: return (state);
      default: dcg_bad_tag (state -> lex_state, "erts_handle_epsilon_transition");
    };
  return (state_nil);
}

State erts_handle_glue (Trellis trel, State state)
{ if (state -> lex_state != LEX_STATE_S)
    /* Not in end of word state, ignore glue */
    return (state);
  return (init_state (trel, state -> offset, state -> linenr, state -> colnr, LEX_STATE_I));
}

/*
   Match a separator in the trellis ($SEP)
   Returns true iff the input state is at the 'end of a word' (not after
   a + or a glueing hyphen and there is a white space or separator ahead.
*/
int erts_state_matches_separator (Trellis trel, State state)
{ char *ptr = trel -> input + state -> offset;
  if (state -> lex_state != LEX_STATE_S)
    /* Not at end of word */
    return (0);
  if (ebs_ahead_white_space (trel -> lexicon, ptr))
    return (1);
  return (ebs_ahead_separator (trel -> lexicon, ptr));
}

/*
   Scan for EOS transitions
   EOS transitions are only possible for lexical state S or W
*/
Transition erts_scan_eos_transition (Trellis trel, State state)
{ State my_state = erts_handle_epsilon_transition (trel, state);
  char *from;
  if (my_state -> lex_state == LEX_STATE_I)
    { /* No good in the middle of a word */
      return (transition_nil);
    };

  /* If we are not at the end of input, done */
  if (my_state -> offset < trel -> length)
    return (transition_nil);

  /* Check if we have tried to match EOS before */
  if (my_state -> lex_flags & LEX_STATE_EOS_TRIED)
    return (find_transition (my_state -> trans, TransEndOfText, 0));

  /* Note that this is a transition to state nil */
  my_state -> lex_flags |= LEX_STATE_EOS_TRIED;
  from = trel -> input + trel -> length;
  return (add_transition (&my_state -> trans, my_state, from, from, TransEndOfText, 0, 0, 0, 1));
}

/*
   To scan $WORD, we must know whether a state is a proper
   end state for a word from the lexicon.
*/
static int is_proper_end_of_word (Trellis trel, State state)
{ return ((state != state_nil) &&
	  (state -> lex_state == LEX_STATE_S) &&
	  ((scan_white_space_transition (trel, state) != transition_nil) ||
	   (erts_scan_eos_transition (trel, state) != transition_nil)));
}

/*
   Determine whether a current lex state and a marker are allowed together

     ws+           word         empty
   S ---> W  ---------------> E ----->  S
           \                 /^\
            \               / | \
             \y-       -y- /  \ | -y
              \        /  /    \/
               \      V  /
                \       / ^
                 \     / /
                  V   / -y
                   \ / word
                   ^I
                  /  \
                 |   |
                 \---/
                y-  -y-
*/
static uchar determine_next_lex_state (uchar curr_lex_state, int marker)
{ switch (curr_lex_state)
    { case LEX_STATE_W:
	{ if (marker & LexemeSuffixBit)	/* Check Infix/Suffix */
	    return (LEX_STATE_ERROR);
	  if (marker & LexemePrefixBit)	/* Check Prefix */
	    return (LEX_STATE_I);
	  return (LEX_STATE_E);		/* Word or multi word */
	}
      case LEX_STATE_I:
	{ if (marker & LexemePrefixBit)	/* Check Prefix/Infix */
	    return (LEX_STATE_I);
	  return (LEX_STATE_E);		/* Suffix, Word or multi word */
	}
      case LEX_STATE_E:
	{ if ((marker & LexemeInfix) == LexemeInfix)	/* Check Prefix and Suffix set */
            return (LEX_STATE_I);
	  if (marker & LexemeSuffixBit)
	    return (LEX_STATE_E);
	  return (LEX_STATE_ERROR);
	};
      default: return (LEX_STATE_ERROR);
    }
}

/*
   Phase 1: recognition of lexicon vocabulary entries
   Note that this code runs without optimizing for full word/part word.
*/
static void scan_lexicon_entries (Trellis trel, State state)
{ Lexicon lex = trel -> lexicon;
  char *from = trel -> input + state -> offset;
  LexiconVocIterator iterator = ebs_init_lexicon_voc_match (lex, from);
  Penalty penalty;
  char *to;
  while ((to = ebs_next_lexicon_voc_match (iterator, &penalty)) != NULL)
    { int entry_nr, ix, marker, linenr, colnr, offset;
      uchar next_lex_state;
      int_list entries;
      int is_full_word;
      char *lexeme;
      State target;

      /* Get the information of the match */
      ebs_get_lexicon_voc_match_info (iterator, &entry_nr, &lexeme, &marker);

      /* Check if we allow this marker for this state */
      next_lex_state = determine_next_lex_state (state -> lex_state, marker);
      if (next_lex_state == LEX_STATE_ERROR)
	continue;

      /* We can make a valid transition here: first fetch/create the target state */
      /* We could do some optimization here at the expense of the extra position check */
      offset = to - trel -> input;
      linenr = state -> linenr;
      colnr = state -> colnr;
      ebs_update_position (lex, from, to, &linenr, &colnr);

      /* Fetch the target state */
      target = init_state (trel, offset, linenr, colnr, next_lex_state);

      /* Remember if this state is a proper end of a word */
      is_full_word = is_proper_end_of_word (trel, target) &&
		     ((marker & LexemeInfix) == 0);
 
      /* The entries are a list of pairs (call_id, frequency) */
      entries = ebs_get_entries_from_entry_nr (lex, entry_nr);
      for (ix = 0; ix < entries -> size; ix += 2)
        { int call_id = entries -> array[ix];
	  /* Penalty/Bonus calculation: int frequency = entries -> array[ix + 1]; */
	  int_list my_call = lex -> rt_lex_calls -> array[call_id];
	  int nont_id = my_call -> array[0];
	  Transition trans;
	  if (nont_id >= lex -> rt_lex_nonts -> size)
  	    { /* Call is terminal */
	      int term_nr = nont_id - lex -> rt_lex_nonts -> size;
  	      trans = add_transition (&state -> trans, target, from, to,
				      TransTerminal, term_nr, 0, penalty, 0);
	    }
	  else
	    { /* Call is lexicon nonterminal */
  	      trans = add_transition (&state -> trans, target, from, to,
				      TransLexNont, nont_id, call_id, penalty, 0);
	    }

	  /* If we have a full word, mark the transition */
	  if (is_full_word)
	    trans -> flags |= TRANS_FULL_WORD;
        }
    }
}

/*
   Scan word assumes that phase 1 has been done
*/
static void scan_word (Trellis trel, State state)
{ int from_offset = state -> offset;
  int min_length = trel -> length + 1;
  Transition selected = transition_nil;
  Transition trans;

  for (trans = state -> trans; trans != transition_nil; trans = trans -> next)
    { /* We are only interested in terminal or lexicon nonterminals */
      int trans_length;
      if (!(trans -> class & (TransTerminal | TransLexNont)))
	continue;
      if (!(trans -> flags & TRANS_FULL_WORD))
	continue;

      /* Calculate transition length */
      trans_length = trans -> target -> offset - from_offset;
      if (trans_length < min_length)
	{ selected = trans;
	  min_length = trans_length;
	}
    };

  /* If we have a valid shortest, make $WORD transition */
  if (selected == transition_nil)
    return;

  /* Make the transition: note that we do not apply transition penalties yet */
  (void) add_transition (&state -> trans, selected -> target, selected -> from, selected -> to,
			 TransWord, 0, 0, 0, 1);
}

/*
   Scan regular expressions
*/
static void scan_regexp (Trellis trel, State state, int kind)
{ Lexicon lex = trel -> lexicon;
  char *from = trel -> input + state -> offset;
  LexiconRegexpIterator iterator = ebs_init_lexicon_regexp_match (lex, from, kind);
  char *to;
  while ((to = ebs_next_lexicon_regexp_match (iterator)) != NULL)
    { int regexp_nr, regexp_kind, marker, offset, linenr, colnr, is_full_word;
      uchar next_lex_state;
      Transition trans;
      State target;

      /* Get the information of the match */
      ebs_get_lexicon_regexp_match_info (iterator, &regexp_nr, &regexp_kind, &marker);

      /* Check if we allow this marker for this state */
      next_lex_state = determine_next_lex_state (state -> lex_state, marker);
      if (next_lex_state == LEX_STATE_ERROR)
	continue;

      /* We could do some optimization here at the expense of the extra position check */
      offset = to - trel -> input;
      linenr = state -> linenr;
      colnr = state -> colnr;
      ebs_update_position (lex, from, to, &linenr, &colnr);

      /* We can make a valid transition here: first fetch/create the target state */
      target = init_state (trel, offset, linenr, colnr, next_lex_state);

      /* Remember if this state is a proper end of a word */
      is_full_word = is_proper_end_of_word (trel, target) &&
		     ((marker & LexemeInfix) == 0);
 
      /* Make the transition: no transition penalties yet */
      trans = add_transition (&state -> trans, target, from, to,
			      (kind == RegexpMatch)?TransMatch:TransSkip, regexp_nr, 0, 0, 1);

      /* If we have a full word, mark the transition */
      if (is_full_word)
	trans -> flags |= TRANS_FULL_WORD;
    }
}

/*
   Scan of an other transition i.e.
   a transition to the next white space or end of input.
*/
static void scan_other (Trellis trel, State state)
{ Lexicon lex = trel -> lexicon;
  char *from = trel -> input + state -> offset;
  int offset, linenr, colnr;
  Transition trans;
  char *to = from;
  State target;

  /* Skip until next end of sentence or white space */
  while (*to)
    { if (ebs_ahead_white_space (lex, to)) break;
      to = ebs_skip_one_char (lex, to);
    };

  /* We must have skipped at least one character */
  if (from == to) return;

  /* We could do some optimization here at the expense of the extra position check */
  offset = to - trel -> input;
  linenr = state -> linenr;
  colnr = state -> colnr;
  ebs_update_position (lex, from, to, &linenr, &colnr);
  
  /* We can make a valid transition here: first fetch/create the target state */
  target = init_state (trel, offset, linenr, colnr, LEX_STATE_S);

  /* Make the transition: note that we do not apply transition penalties yet */
  trans = add_transition (&state -> trans, target, from, to, TransOther, 0, 0, 0, 1);
  trans -> flags |= TRANS_FULL_WORD;	/* By definition */
}

/*
   Scan any assumes that phase 5 has been done
*/
void erts_scan_any_transition (Trellis trel, State state)
{ int from_offset = state -> offset;
  int min_length = trel -> length + 1;
  Transition selected = transition_nil;
  Transition trans;

  for (trans = state -> trans; trans != transition_nil; trans = trans -> next)
    { /* We are only interested in terminal or lexicon nonterminals */
      int trans_length;
      if (!(trans -> class & (TransTerminal | TransLexNont | TransWord |
		              TransMatch | TransSkip | TransOther)))
	continue;

      /* Calculate transition length */
      trans_length = trans -> target -> offset - from_offset;
      if (trans_length < min_length)
	{ selected = trans;
	  min_length = trans_length;
	}
    };

  /* If we have a valid shortest, make $ANY transition (no penalties yet) */
  if (selected == transition_nil) return;
  (void) add_transition (&state -> trans, selected -> target, selected -> from, selected -> to,
			 TransAny, 0, 0, 0, 1);
}

/*
   Check if there are valid transitions
*/
static int valid_transitions (Transition trans)
{ Transition ptr;
  for (ptr = trans; ptr != transition_nil; ptr = ptr -> next)
    if (ptr -> class & (TransTerminal | TransLexNont | TransWord |
		        TransMatch | TransSkip | TransOther))
      return (1);
  return (0);
}

/*
   We recognize 6 different phases in which a state may be:
   1: all entries in the lexicon vocabularies
   2: $WORD  (the shortest transition from the lexicon vocabularies)
   3: $MATCH (regular expression match)
   4: $SKIP  (should only be tried if there is no result from phase 1 to 3)
   5: $OTHER (should only be tried if there is no result from phase 1 to 4)
   6: $ANY   (the shortest transition from among all of the previous ones)

   The higher the phase, the more expensive the search action.
   Epsilon transition should be handled outside of erts_scan_phases
*/
void erts_scan_phases (Trellis trel, State state, uchar max_phase)
{
  /* If we have already scanned through all applicable phases */
  if (state -> lex_flags & LEX_STATE_FULLY_SCANNED)
    return;

  /* Iterate by phase: if more is not needed, do not try them */
  while (state -> lex_phase < max_phase)
    { /* Step to next state */
      state -> lex_phase++;
      switch (state -> lex_phase)
	{ case 1: scan_lexicon_entries (trel, state); break;
	  case 2: scan_word (trel, state); break;
	  case 3: scan_regexp (trel, state, RegexpMatch); break;
	  case 4: scan_regexp (trel, state, RegexpSkip); break;
	  case 5: scan_other (trel, state); break;
	  default: dcg_bad_tag (state -> lex_phase, "erts_scan_phases");
	};

      /*
	 Beyond phase 3, we break off further searches, if there is
	 a valid transition with data
      */
      if ((state -> lex_phase >= 3) && valid_transitions (state -> trans))
	{ state -> lex_flags |= LEX_STATE_FULLY_SCANNED;
	  return;
        };
    };
}

void erts_release_trellis (EagrtsHandle hnd)
{ /* Pop trellis and deallocate substructures */
  Trellis old = hnd -> trellis;
  int nr_states = (old -> length + 1) * NR_LEX_STATES;
  int ix;
  hnd -> trellis = old -> previous;

  /* Deallocate states & transitions */
  for (ix = 0; ix < nr_states; ix++)
    { State old_state = old -> state_cache[ix];
      Transition trans;
      if (old_state == state_nil) continue;

      /* Release transitions */
      trans = old_state -> trans;
      while (trans != transition_nil)
        { Transition old_trans = trans;
	  trans = trans -> next;
	  dcg_detach ((void **) &old_trans);
        };

      /* Release neg memos and self */
      dcg_detach ((void **) &old_state -> neg_memos);
      dcg_detach ((void **) &old_state);
    };

  /* Deallocate self */
  dcg_detach ((void **) &old -> state_cache);
  dcg_detach ((void **) &old);
}

/*
   Code to dump the trellis
*/
static char *string_from_lex_state (uchar lex_state)
{ switch (lex_state)
    { case LEX_STATE_ERROR: return ("ERROR");
      case LEX_STATE_S: return ("S");
      case LEX_STATE_W: return ("W");
      case LEX_STATE_I: return ("I");
      default: dcg_bad_tag (lex_state, "string_from_lex_state");
    };
  return (NULL);
}

void erts_write_trellis_state (State state)
{ dcg_eprint ("%2d.%s", state -> offset, string_from_lex_state (state -> lex_state));
}

void erts_write_transition (Lexicon lex, Transition trans)
{ int nr = trans -> nr;
  switch (trans -> class)
    { case TransError:
	dcg_eprint ("ERROR");
	break;
      case TransTerminal:
	dcg_eprint ("Terminal %d ('%s')", nr, lex -> rt_terminals -> array[nr] -> origin);
	break;
      case TransLexNont:
	ebs_dump_call (dcg_error_file (), lex, trans -> info);
	break;
      case TransWord:
	dcg_eprint ("$WORD");
	break;
      case TransMatch:
	{ char *origin = lex -> rt_regexp_nfas -> array[nr] -> origin;
	  dcg_eprint ("$MATCH %d ('%s')", trans -> nr, origin);
	}; break;
      case TransSkip:
	{ char *origin = lex -> rt_regexp_nfas -> array[nr] -> origin;
	  dcg_eprint ("$SKIP %d ('%s')", trans -> nr, origin);
	}; break;
      case TransOther:
	dcg_eprint ("$OTHER");
	break;
      case TransAny:
	dcg_eprint ("$ANY");
	break;
      case TransWhiteSpace:
	if (trans -> from == trans -> to) dcg_eprint ("<WS*>");
	else dcg_eprint ("<WS+>");
	break;
      case TransEndOfText: dcg_eprint ("<EOT>"); break;
      default: dcg_bad_tag (trans -> class, "dump_transition");
    };
}

static void dump_transition (Trellis trel, Transition trans)
{ Lexicon lex = trel -> lexicon;
  if (trans -> flags != 0)
    dcg_eprint ("     %c%c\t", (trans -> flags & TRANS_FULL_WORD)?'F':' ',
			      (trans -> flags & TRANS_USED)?'U':' ');
  else dcg_eprint ("\t");
  ebs_dump_text_from_to (dcg_error_file (), trans -> from, trans -> to);
  dcg_eprint ("\t");
  erts_write_transition (lex, trans);
  if (trans -> class != TransEndOfText)
    { State target = trans -> target;
      dcg_eprint (" => %d.%s", target -> offset, string_from_lex_state (target -> lex_state));
      if (trans -> penalty)
	dcg_eprint (" with penalty %d", trans -> penalty);
    };
  dcg_wlog ("");
}

static void dump_state (Trellis trel, State state)
{ Transition trans = state -> trans;
  dcg_wlog ("%5d.%s line %d, column %d, phase %d, flags=0x%02x", state -> offset,
	    string_from_lex_state (state -> lex_state), state -> linenr, state -> colnr,
	    state -> lex_phase, state -> lex_flags);
  while (trans != transition_nil)
    { dump_transition (trel, trans);
      trans = trans -> next;
    };
}

void erts_dump_trellis (Trellis trel)
{ int index;
  for (index = 0; index < (trel -> length + 1) * NR_LEX_STATES; index++)
    { State state = trel -> state_cache[index];
      if (state == state_nil)
	continue;
      dump_state (trel, state);
    }
}
