/*
   File: erts_tree_impl.h
   Defines the data structures necessary to store the parse tree,
   affix positions and propagation links.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_tree_impl.h,v 1.15 2012/12/22 16:21:15 marcs Exp $"
*/
#ifndef IncErtsTreeImpl
#define IncErtsTreeImpl

/* libeagbase includes */
#include <ebase_ds.h>

/* local includes */
#include "erts_tree.h"
#include "erts_handle.h"

/* Introduce the structure of parse trees */
struct tree_rec
{ tree_kind kind;		 	/* Kind of tree node */
  int number;				/* Nonterminal node nr, or terminal nr, or ... */
  int nr_sons;				/* Number of sons */
  int nr_pos;				/* Number of positions */
  Tree *sons;				/* Son trees */
  Position *positions;			/* Affix positions */
  AffixNode *frame;			/* Affix nodes */
  Transition trans;			/* Trellis transition for leaf cells */
					/* Doubles for free list admin */
};

/*
   Introduce the structure of Affix positions. Each affix position has two sides.
   in the case of calls, the lower side corresponds to the formal (lhs) side of
   the call, the upper side corresponds to the affix expression on the actual side
   of the call. For a confrontation, the left and right side correspond to the
   affix expressions on the left and right side of the confrontation.
   
   This consists of the sill and a pointer to the affix expression
*/
typedef struct position_side_rec
{ int sill;				/* The current number of unknown affixes */
  AffixExpr expr;			/* An affix expression */
} PositionSide;

/*
   Introduce the structure of positions
   We may want to add additional information on the intention of the
   affix flow so that confrontations are handled appropriately.
   Maybe we should add the position number as well, since this is
   enough information to derive the typing of the position
*/
#define lower_side 0
#define upper_side 1
#define left_side 0
#define right_side 1
struct position_rec
{ Tree tree_node;			/* The tree node of which I am a position */
					/* Doubles for free list admin */
  DelayedEvalFunc delayed_func;		/* The actual delayed function */
  PositionSide sides[2];		/* Lower and upper side */
  propagation_kind kind;		/* Direction of propagation */
};

/* Introduce the structure of dynamic affix positions */
struct affix_expr_rec
{ affix_expr_kind kind;			/* What kind of affix expression am I */
  union
    { AffixNode node;			/* Single affix node */
      struct
	{ operator op;			/* Arithmetic operation */
	  AffixExpr expr1;		/* One or two affix expressions */
	  AffixExpr expr2;		/* Doubles for free list adm */
	} arith;
      struct
	{ int nr;			/* A number of affix expressions */
	  AffixExpr *exprs;		/* to be concatenated */
	} text_concat;
      struct
	{ int typenr;			/* Resulting type */
	  int marker;			/* Marker of result */
	  int nr;			/* Number of parts */
	  AffixExpr *exprs;		/* Part expressions */
	} ast;
    } uni;
};

/* 
   An affix node corresponds with an affix in the original grammar.
   Via the links the affix value will be propagated, once the affix
   node gets its value. This may be upon creation or later through
   affix propagation. The node also holds a pointer to allow a meta
   grammar check for meta defined text affixes where a dynamic check
   may be necessary to see whether the propagated text value adheres
   to the meta grammar.
*/
struct affix_node_rec
{ string name;				/* Name of the affix (node): debugging only */
  affix_value value;			/* Its value (once defined) */
  void (*meta_check)(EagrtsHandle);	/* See above */
  AffixLink links;			/* Links for propagation, doubles for free list */
};

/* Macro to inspect critical affix positions */
#define critical_position_has_value(p) \
((p) -> sides[lower_side].expr -> uni.node -> value != affix_value_nil)

/*
   The list of affix links points from an affix node to all positions
   in which the affix occurs. Once an affix gets its value, this is
   the way in which affix values get propagated
*/
struct affix_link_rec
{ Position pos;				/* To which affix position */
  int side;				/* and which side do I propagate */
  AffixLink next;			/* Next link */
};

/* Define macros to push and pop from the tree stack */
#define tree_check(hnd) \
{ if ((hnd -> tree_sp + 2) > (hnd -> tree_stack + hnd -> tree_stack_size)) \
    dcg_abort ("tree_check", "Out of tree stack space"); \
}
#define push_tree(hnd,tree) { *hnd -> tree_sp++ = tree; }
#define pop_tree(hnd) (*--hnd -> tree_sp)
#define top_tree(hnd) (*(hnd -> tree_sp - 1))

#endif /* IncErtsTreeImpl */
