/*
   File: erts_tree.c
   Defines the data structures necessary to store the parse tree,
   affix positions and propagation links. Defines the functions
   necessary to build the parse tree.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_tree.c,v 1.19 2013/01/11 14:52:05 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_lexicon.h>
#include <ebase_lexicon_impl.h>

/* local includes */
#include "erts_handle.h"
#include "erts_handle_impl.h"
#include "erts_trellis.h"
#include "erts_trellis_impl.h"
#include "erts_tree.h"
#include "erts_tree_impl.h"
#include "erts_trace.h"
#include "erts_memory.h"
#include "erts_propagate.h"
#include "erts_cont.h"

/*
   Building and destroying of affix nodes.
   Note that the affix values from the lexicon or from the compiler are
   assumed to exist in the global memory pool for proper deallocation:
   they may be shared by many, many affix nodes. We do not attach
   initial affix value: it is the responsibility of the caller to
   deallocate. However, affix propagation will attach upon copying.
*/
AffixNode erts_make_affix_from_value (char *name, affix_value value)
{ AffixNode node = erts_make_affix_node (name);
  node -> name = name;
  node -> value = value;
  node -> meta_check = NULL;
  return (node);
}

AffixNode erts_make_affix_from_constant (char *name, EagrtsHandle hnd, int nr)
{ AffixNode node = erts_make_affix_node (name);
  node -> value = hnd -> rt_consts -> array[nr];
  node -> meta_check = NULL;
  return (node);
}

/*
   Building and destroying affix links
*/
static void add_link (AffixNode affix, Position pos, int side)
{ AffixLink link = erts_make_link_node ();
  link -> pos = pos;
  link -> side = side;
  link -> next = affix -> links;
  affix -> links = link;
  if (affix -> value != affix_value_nil)
    { if (pos -> sides[side].sill)
        pos -> sides[side].sill--;
      else dcg_internal_error ("add_link");
    };
}

static void delete_link (AffixNode affix, Position pos, int side)
{ AffixLink *links_ptr = &affix -> links;
  while ((*links_ptr) != affix_link_nil)
    { AffixLink curr_link = *links_ptr;
      if ((curr_link -> pos == pos) && (curr_link -> side == side))
	{ /* Found matching link, delete it */
	  if (affix -> value != affix_value_nil)
	    pos -> sides[side].sill++;
	  *links_ptr = curr_link -> next;
	  erts_free_link_node (curr_link);
	  return;
	};
      links_ptr = &curr_link -> next;
    };
  dcg_internal_error ("delete_link");
}

/*
   Building and destroying affix expressions
*/
static AffixExpr make_expr_from_affix_node (Position position, int side, AffixNode node)
{ AffixExpr expr = erts_make_affix_expr (single_affix_node);
  expr -> uni.node = node;
  position -> sides[side].sill += 1;
  add_link (node, position, side);
  return (expr);
}

static AffixExpr make_arithmetic_expr (affix_expr_kind kind, operator op,
				       AffixExpr expr1, AffixExpr expr2)
{ AffixExpr expr = erts_make_affix_expr (kind);
  expr -> uni.arith.op = op;
  expr -> uni.arith.expr1 = expr1;
  expr -> uni.arith.expr2 = expr2;
  return (expr);
}

static AffixExpr make_concat_expr (int nr, AffixExpr *exprs)
{ AffixExpr expr = erts_make_affix_expr (concat_node);
  expr -> uni.text_concat.nr = nr;
  expr -> uni.text_concat.exprs = exprs;
  return (expr);
}

static AffixExpr make_ast_expr (int typenr, int marker, int nr, AffixExpr *exprs)
{ AffixExpr expr = erts_make_affix_expr (ast_node);
  expr -> uni.ast.typenr = typenr;
  expr -> uni.ast.marker = marker;
  expr -> uni.ast.nr = nr;
  expr -> uni.ast.exprs = exprs;
  return (expr);
}

/*
   Creation and destruction of affix expressions from the continuation stack
*/
static AffixExpr construct_affix_expr (EagrtsHandle hnd, Position position, int side)
{ affix_expr_kind kind = cont_pop_int (hnd);
  switch (kind)
    { case single_affix_node:
	{ AffixNode node = cont_pop_affix_node (hnd);
	  return (make_expr_from_affix_node (position, side, node));
	};
      case ast_node:
	{ int typenr = cont_pop_int (hnd);
	  int marker = cont_pop_int (hnd);
	  int nr = cont_pop_int (hnd);
	  AffixExpr *exprs = erts_make_affix_expressions (nr);
	  int ix;
	  for (ix = 0; ix < nr; ix++)
	    exprs[ix] = construct_affix_expr (hnd, position, side);
	  return (make_ast_expr (typenr, marker, nr, exprs));
	};
      case concat_node:
	{ int nr = cont_pop_int (hnd);
	  AffixExpr *exprs = erts_make_affix_expressions (nr);
	  int ix;
	  for (ix = 0; ix < nr; ix++)
	    exprs[ix] = construct_affix_expr (hnd, position, side);
	  return (make_concat_expr (nr, exprs));
	};
      case dyop_arith_node:
	{ operator dop = cont_pop_int (hnd);
	  AffixExpr expr1 = construct_affix_expr (hnd, position, side);
	  AffixExpr expr2 = construct_affix_expr (hnd, position, side);
	  return (make_arithmetic_expr (kind, dop, expr1, expr2));
	};
      case monop_arith_node:
	{ operator mop = cont_pop_int (hnd);
	  AffixExpr arg = construct_affix_expr (hnd, position, side);
	  return (make_arithmetic_expr (kind, mop, arg, affix_expr_nil));
	};
      default: dcg_bad_tag (kind, "construct_affix_expr");
    };
  return (affix_expr_nil);
}

static void destruct_affix_expr (EagrtsHandle hnd, AffixExpr old, Position position, int side)
{ affix_expr_kind kind = old -> kind;
  switch (kind)
    { case single_affix_node:
	{ AffixNode node = old -> uni.node;
	  delete_link (node, position, side);
	  if (position -> sides[side].sill)
	    position -> sides[side].sill -= 1;
          else dcg_internal_error ("destruct_affix_expr");
	  cont_push_affix_node (hnd, node);
	}; break;
      case ast_node:
	{ int typenr = old -> uni.ast.typenr;
	  int marker = old -> uni.ast.marker;
	  int nr = old -> uni.ast.nr;
	  AffixExpr *exprs = old -> uni.ast.exprs;
	  int ix;
	  for (ix = nr - 1; 0 <= ix; ix--)
	    destruct_affix_expr (hnd, exprs[ix], position, side);
	  erts_free_affix_expressions (exprs, nr);
	  cont_push_int (hnd, nr);
	  cont_push_int (hnd, marker);
	  cont_push_int (hnd, typenr);
	}; break;
      case concat_node:
	{ int nr = old -> uni.text_concat.nr;
	  AffixExpr *exprs = old -> uni.text_concat.exprs;
	  int ix;
	  for (ix = nr - 1; 0 <= ix; ix--)
	    destruct_affix_expr (hnd, exprs[ix], position, side);
	  erts_free_affix_expressions (exprs, nr);
	  cont_push_int (hnd, nr);
	}; break;
      case dyop_arith_node:
	{ operator dop = old -> uni.arith.op;
	  destruct_affix_expr (hnd, old -> uni.arith.expr2, position, side);
	  destruct_affix_expr (hnd, old -> uni.arith.expr1, position, side);
	  cont_push_int (hnd, dop);
	}; break;
      case monop_arith_node:
	{ operator mop = old -> uni.arith.op;
	  destruct_affix_expr (hnd, old -> uni.arith.expr1, position, side);
	  cont_push_int (hnd, mop);
	}; break;
      default: dcg_bad_tag (kind, "destruct_affix_expr");
    }
  erts_free_affix_expr (old);
  cont_push_int (hnd, kind);
}

/*
   Create and destruction of affix position sides
*/
static void construct_position_sides (EagrtsHandle hnd, Position *positions, int nr_pos, int side)
{ int ix;
  for (ix = 0; ix < nr_pos; ix++)
    { Position pos = positions[ix];
      pos -> sides[side].expr = construct_affix_expr (hnd, pos, side);
    };
}

static void destruct_position_sides (EagrtsHandle hnd, Position *positions, int nr_pos, int side)
{ int ix;
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    { Position pos = positions[ix];
      destruct_affix_expr (hnd, pos -> sides[side].expr, pos, side);
      pos -> sides[side].expr = affix_expr_nil;		/* Remember the destruction */
    };
}

/*
   erts_make_affix_link is called to create the upper side of
   affix positions after a parsing action has create a proper
   subtree of the current one. This action is usually called
   before erts_link_son inserts the subtree as a son.
*/
void erts_make_affix_link (EagrtsHandle hnd)
{ /* Pick up reference to son tree positions */
  Tree son_tree = top_tree (hnd);
  Position *positions = son_tree -> positions;
  int nr_pos = son_tree -> nr_pos;
  cont_ptr old_sp;
  int ix;

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_enter_make_affix_link (hnd, son_tree);

#ifdef DEBUG
  /* Consistency check */
  if ((positions == positions_nil) || (!nr_pos))
    dcg_internal_error ("erts_make_affix_link");
#endif

  /* Construct the upper sides */
  construct_position_sides (hnd, positions, nr_pos, upper_side);

  /*
     Propagate every position, save the old sp for later discards
     Push the calls in reverse order, so the first position will be propagated first.
  */
  old_sp = hnd -> cont_sp;
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    { cont_push_position (hnd, positions[ix]);
      cont_push_continuation (hnd, erts_propagate_affix_value);
    };
  call_continuation (hnd);
  hnd -> cont_sp = old_sp;	/* Discard pushes */

  /* Undo everything */
  destruct_position_sides (hnd, positions, nr_pos, upper_side);

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_leave_make_affix_link (hnd, son_tree);

  cont_push_continuation (hnd, erts_make_affix_link);
}

/*
   Normal nodes have a node number, a specific number of sons
   and a specific number of affix positions (>0)
*/
void erts_make_normal_node (EagrtsHandle hnd, int node_nr, int nr_sons, int nr_pos,
			    AffixNode *frame)
{ Tree my_tree = erts_make_tree_node (normal_node, node_nr);
  Position *positions = erts_make_affix_positions (nr_pos);
  int_list formals = hnd -> rt_nonts -> array[node_nr] -> formals;
  int ix;
  my_tree -> nr_sons = nr_sons;
  my_tree -> nr_pos = nr_pos;
  my_tree -> sons = erts_make_tree_sons (nr_sons);
  my_tree -> positions = positions;
  my_tree -> frame = frame;
  my_tree -> trans = transition_nil;
  for (ix = 0; ix < nr_pos; ix++)
    { Position pos = erts_make_affix_position (my_tree);
      rt_type rt = hnd -> lexicon -> rt_types -> array[formals -> array[ix]];
      if (rt -> tag == TAGLattice_type) pos -> kind = p_restrict;
      else pos -> kind = p_bidirectional;
      positions[ix] = pos;
    };
  construct_position_sides (hnd, positions, nr_pos, lower_side);

  /* Call the continuation */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_normal_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  destruct_position_sides (hnd, positions, nr_pos, lower_side);
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    erts_free_affix_position (positions[ix]);
  erts_free_affix_positions (positions, nr_pos);
  erts_free_tree_sons (my_tree -> sons, nr_sons);
  erts_free_tree_node (my_tree);
}

/*
   Simple nodes have a node number and a specific number of sons
   There are no affix positions allocated for simple nodes.
   This is purely an optimization: in linguistic grammars there are
   quite a number of syntax rules that have no formal affix positions.
*/
void erts_make_simple_node (EagrtsHandle hnd, int node_nr, int nr_sons, AffixNode *frame)
{ Tree my_tree = erts_make_tree_node (simple_node, node_nr);
  my_tree -> sons = erts_make_tree_sons (nr_sons);
  my_tree -> nr_sons = nr_sons;
  my_tree -> frame = frame;
  
  /* Call the continuation */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_simple_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  erts_free_tree_sons (my_tree -> sons, nr_sons);
  erts_free_tree_node (my_tree);
}

/*
   Anonymous tree nodes have a node number, a specific number of sons
   Note that they inherit their affix frame from their father (topdown),
   allocate it themselves or negotiate it with the father (leftcorner)
*/
void erts_make_anonymous_node (EagrtsHandle hnd, int node_nr, int nr_sons, AffixNode *frame)
{ Tree my_tree = erts_make_tree_node (anonymous_node, node_nr);
  my_tree -> nr_sons = nr_sons;
  my_tree -> sons = erts_make_tree_sons (nr_sons);
  my_tree -> frame = frame;	/* Shared with others */

  /* Call the continuation */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_anonymous_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  erts_free_tree_sons (my_tree -> sons, nr_sons);
  erts_free_tree_node (my_tree);
}

/*
   Predicate nodes have a node number, no initial sons and initially a frame
   of affix nodes without values. However, a function is registered with each
   position which will be called whenever a value gets propagated towards its
   positions. This function must then check if all critical affix positions
   have received a value. If so, the actual predicate body will be called to
   evaluate the predicate.
*/
void erts_make_predicate_node (EagrtsHandle hnd, int node_nr, int nr_pos, AffixNode *frame,
                               void (*delayed_func)(EagrtsHandle, Position *))
{ Tree my_tree = erts_make_tree_node (predicate_node, node_nr);
  Position *positions = erts_make_affix_positions (nr_pos);
  int_list formals = hnd -> rt_nonts -> array[node_nr] -> formals;
  int ix;
  my_tree -> nr_sons = 0;
  my_tree -> sons = trees_nil;
  my_tree -> nr_pos = nr_pos;
  my_tree -> positions = positions;
  my_tree -> frame = frame;
  my_tree -> trans = transition_nil;
  for (ix = 0; ix < nr_pos; ix++)
    { AffixNode node = frame[ix];
      Position position = erts_make_affix_position (my_tree);
      position -> sides[lower_side].expr = make_expr_from_affix_node (position, lower_side, node);
      rt_type rt = hnd -> lexicon -> rt_types -> array[formals -> array[ix]];
      if (rt -> tag == TAGLattice_type) position -> kind = p_restrict;
      else position -> kind = p_bidirectional;
      position -> delayed_func = delayed_func;
      positions[ix] = position;
    };

  /* Call the continuation */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_predicate_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    { Position position = positions[ix];
      AffixExpr expr = position -> sides[lower_side].expr;
      AffixNode node = frame[ix];
      delete_link (node, position, lower_side);
      erts_free_affix_expr (expr);
      erts_free_affix_position (position);
    };
  erts_free_affix_positions (positions, nr_pos);
  erts_free_tree_node (my_tree);
}

/*
   Update predicate node must update the tree node with the proper
   number of sons for the current alternative of the predicate, save
   the original lower side expressions (including the sills and values),
   and substitute the alternative's lhs. Subsequently the body of the
   alternative can be executed.
*/
void erts_update_predicate_node (EagrtsHandle hnd, Tree my_tree, int nr_sons, AffixNode *frame)
{ AffixNode *old_frame = my_tree -> frame;
  int nr_pos = my_tree -> nr_pos;
  Position *save_positions = erts_make_affix_positions (nr_pos);
  Position *orig_positions = my_tree -> positions;
  cont_ptr old_sp;
  int ix;
  my_tree -> sons = erts_make_tree_sons (nr_sons);
  my_tree -> nr_sons = nr_sons;
  my_tree -> frame = frame;

  /* Copy lower side of the positions into save_pos */
  /* Note: we allocate a bit more memory than necessary */
  for (ix = 0; ix < nr_pos; ix++)
    { Position save_position = erts_make_affix_position (my_tree);
      Position orig_position = orig_positions[ix];
      save_positions[ix] = save_position;
      save_position -> sides[lower_side] = orig_position -> sides[lower_side];
      orig_position -> sides[lower_side].sill = 0;
      orig_position -> sides[lower_side].expr = affix_expr_nil;
    };

  /* Reconstruct lower side according to grammar */
  construct_position_sides (hnd, orig_positions, nr_pos, lower_side);

  /* Call the continuation */
  old_sp = hnd -> cont_sp;
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    { cont_push_position (hnd, orig_positions[ix]);
      cont_push_continuation (hnd, erts_propagate_affix_value);
    };
  call_continuation (hnd);
  hnd -> cont_sp = old_sp;	/* Discard pushes */

  /* Undo everything */
  destruct_position_sides (hnd, orig_positions, nr_pos, lower_side);
  for (ix = nr_pos - 1; 0 <= ix; ix--)
    { Position save_position = save_positions[ix];
      Position orig_position = orig_positions[ix];
      orig_position -> sides[lower_side] = save_position -> sides[lower_side];
      erts_free_affix_position (save_position);
    };
  erts_free_affix_positions (save_positions, nr_pos);
  erts_free_tree_sons (my_tree -> sons, nr_sons);
  my_tree -> frame = old_frame;
  my_tree -> nr_sons = 0;
}

/*
   Penalty tree nodes carry a penalty (for transduction), no sons, no positions
*/
void erts_make_penalty_node (EagrtsHandle hnd, int delta_penalty)
{ Tree my_tree = erts_make_tree_node (penalty_node, delta_penalty);
  int curr_penalty = hnd -> curr_penalty;

  /* Call the continuation */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  hnd -> curr_penalty += delta_penalty;
  call_continuation (hnd);
  hnd -> curr_penalty = curr_penalty;

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_penalty_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  erts_free_tree_node (my_tree);
}

/*
   Confrontation nodes have no sons and just one position
*/
void erts_make_confrontation_node (EagrtsHandle hnd, propagation_kind kind)
{ Tree my_tree = erts_make_tree_node (confrontation_node, -1);
  Position *positions = erts_make_affix_positions (1);
  Position pos = erts_make_affix_position (my_tree);
  my_tree -> nr_sons = 0;
  my_tree -> sons = trees_nil;
  my_tree -> nr_pos = 1;
  my_tree -> positions = positions;
  my_tree -> frame = affix_frame_nil;
  my_tree -> trans = transition_nil;
  positions[0] = pos;
  pos -> kind = kind;
  construct_position_sides (hnd, positions, 1, left_side);
  construct_position_sides (hnd, positions, 1, right_side);
  
  /* Push the tree */
  tree_check (hnd);
  push_tree (hnd, my_tree);

  /*
     Now we first call erts_propagate_affix_value before
     parsing may continue.
  */
  cont_push_position (hnd, pos);
  cont_push_continuation (hnd, erts_propagate_affix_value);
  call_continuation (hnd);
  cont_pop (hnd, 2);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_anonymous_node");
#else
  (void) pop_tree (hnd);
#endif

  /* Undo everything */
  destruct_position_sides (hnd, positions, 1, right_side);
  destruct_position_sides (hnd, positions, 1, left_side);
  erts_free_affix_position (pos);
  erts_free_affix_positions (positions, 1);
  erts_free_tree_node (my_tree);
}

/*
   For certain transitions we must convert the parsed text into
   a TEXT affix for subsequent use at the second level of the grammar.
   Make an explicit copy in a new value. After the continuation, we
   must release it again. For the moment we assume we may do it in
   the global memory pool. If not (for efficiency reasons), private
   pools should be allocated in the handle.
*/
static affix_value make_transition_value (EagrtsHandle hnd, Transition trans)
{ Lexicon lex = hnd -> lexicon;
  int length = trans -> to - trans -> from;
  char *text = (char *) dcg_malloc (length + 1);
  strncpy (text, trans -> from, length);
  return (new_Text_value (lex -> rt_text, text));
}

void erts_make_leaf_node (EagrtsHandle hnd, Transition trans, int nr_pos)
{ /* Create my tree node and administer my transition for transduction */
  Tree my_tree = erts_make_tree_node (leaf_node, -1);
  Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  my_tree -> trans = trans;

  /* If we carry a position, create its lower side from the parsed input */
  if (nr_pos)
    { affix_value value = make_transition_value (hnd, trans);
      AffixNode node = erts_make_affix_from_value ("_LEAF_", value);
      AffixNode *frame = erts_make_affix_frame (1);
      Position *positions = erts_make_affix_positions (1);
      Position position = erts_make_affix_position (my_tree);
      position -> sides[lower_side].expr = make_expr_from_affix_node (position, lower_side, node);
      position -> kind = p_lower_to_upper;
      positions[0] = position;
      frame[0] = node;
      my_tree -> frame = frame;
      my_tree -> positions = positions;
      my_tree -> nr_pos = 1;
    };

  /* Push the created tree and continue */
  trel -> curr_state = trans -> target;
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);
  trel -> curr_state = curr_state;

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_leaf_node");
#else
  (void) pop_tree (hnd);
#endif

  /* We had a position, destroy it */
  if (nr_pos)
    { Position *positions = my_tree -> positions;
      Position position = positions[0];
      AffixExpr expr = position -> sides[lower_side].expr;
      AffixNode *frame = my_tree -> frame;
      AffixNode node = frame[0];
      delete_link (node, position, lower_side);
      erts_free_affix_expr (expr);
      erts_free_affix_position (position);
      erts_free_affix_positions (positions, 1);
      erts_free_affix_frame (frame, 1);
      erts_free_affix_node (node);
    };

  erts_free_tree_node (my_tree);
}

/*
   For quasi terminals that are not linked to transitions in the trellis
   ($LINE, $COLUMN, $POS, $SEP), we will create an affix value containing
   the value that was recognized. An affix node is then created which is
   always added to a local frame, so that transduction is able to fetch
   it. Optionally, this node is also propagated when a position is present
*/
void erts_make_quasi_node (EagrtsHandle hnd, int quasi_nr, affix_value value, int nr_pos)
{ /* Create my tree node and administer my transition for transduction */
  Tree my_tree = erts_make_tree_node (quasi_node, quasi_nr);
  AffixNode node = erts_make_affix_from_value ("_QUASI_", value);
  AffixNode *frame = erts_make_affix_frame (1);
  my_tree -> frame = frame;
  frame[0] = node;

  /* If we carry a position, create its lower side from the parsed input */
  if (nr_pos)
    { Position *positions = erts_make_affix_positions (1);
      Position position = erts_make_affix_position (my_tree);
      position -> sides[lower_side].expr = make_expr_from_affix_node (position, lower_side, node);
      position -> kind = p_lower_to_upper;
      positions[0] = position;
      my_tree -> positions = positions;
      my_tree -> nr_pos = 1;
    };

  /* Push the created tree and continue */
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_quasi_node");
#else
  (void) pop_tree (hnd);
#endif

  /* We had a position, destroy it */
  if (nr_pos)
    { Position *positions = my_tree -> positions;
      Position position = positions[0];
      AffixExpr expr = position -> sides[lower_side].expr;
      delete_link (node, position, lower_side);
      erts_free_affix_expr (expr);
      erts_free_affix_position (position);
      erts_free_affix_positions (positions, 1);
    };

  erts_free_affix_frame (frame, 1);
  erts_free_affix_node (node);
  erts_free_tree_node (my_tree);
}

/*
   erts_make_lex_nont_node builds a lexicon nonterminal node (with
   grammar nonterminal number in the node, not the lexicon nonterminal
   number), together with the right number of positions for this call.
   The lower sides of these positions attach the affix values of the
   lexicon nonterminal call.
*/
void erts_make_lex_nont_node (EagrtsHandle hnd, Transition trans)
{ Lexicon lex = hnd -> lexicon;
  Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  int lex_nont_nr = trans -> nr;
  int call_id = trans -> info;
  lex_nont my_lex_nont = lex -> rt_lex_nonts -> array[lex_nont_nr];
  int node_nr = my_lex_nont -> rule_nr;
  int_list formals = hnd -> rt_nonts -> array[node_nr] -> formals;
  int_list my_call = lex -> rt_lex_calls -> array[call_id];
  int arity = my_lex_nont -> formals -> size;
  Position *positions;
  AffixNode *frame;
  Tree my_tree;
  int ix;

#ifdef DEBUG
  /* Consistency check */
  if (my_call -> size != arity + 1)
    dcg_internal_error ("erts_make_lex_nont_node");
#endif

  /* Create my tree node and administer my transition for transduction */
  positions = erts_make_affix_positions (arity);
  frame = erts_make_affix_frame (arity);
  my_tree = erts_make_tree_node (lex_nont_node, node_nr);
  my_tree -> nr_pos = arity;
  my_tree -> positions = positions;
  my_tree -> frame = frame;
  my_tree -> trans = trans;

  /* Create all positions */
  for (ix = 0; ix < arity; ix++)
    { char *name = dcg_new_fmtd_string ("_LEX_NONT%d", ix + 1);
      rt_type rt = hnd -> lexicon -> rt_types -> array[formals -> array[ix]];
      int actual = my_call -> array[ix + 1];
      affix_value value = lex -> rt_values -> array[actual];
      AffixNode node = erts_make_affix_from_value (name, value);	/* Attaches */
      Position position = erts_make_affix_position (my_tree);
      position -> sides[lower_side].expr = make_expr_from_affix_node (position, lower_side, node);
      positions[ix] = position;
      if (rt -> tag == TAGLattice_type) position -> kind = p_restrict;
      else position -> kind = p_lower_to_upper;
      frame[ix] = node;
    }

  /* Push the created tree, continue and undo */
  trel -> curr_state = trans -> target;
  tree_check (hnd);
  push_tree (hnd, my_tree);
  call_continuation (hnd);
  trel -> curr_state = curr_state;

#ifdef DEBUG
  /* Consistency check after recursion */
  if (pop_tree (hnd) != my_tree)
    dcg_internal_error ("erts_make_lex_nont_node");
#else
  (void) pop_tree (hnd);
#endif

  /*
     Destroy all positions
  */
  for (ix = arity - 1; 0 <= ix; ix--)
    { Position position = positions[ix];
      AffixExpr expr = position -> sides[lower_side].expr;
      AffixNode node = frame[ix];
      delete_link (node, position, lower_side);
      erts_free_affix_expr (expr);
      erts_free_affix_position (position);
      dcg_detach ((void **) &node -> name);
      erts_free_affix_node (node);
    };

  /* Undo tree build */
  erts_free_affix_frame (frame, arity);
  erts_free_affix_positions (positions, arity);
  erts_free_tree_node (my_tree);
}

/*
   erts_link_predicate_son will pop a son from the tree stack and insert
   him as son of the predicate node taken from the continuation stack
*/
void erts_link_predicate_son (EagrtsHandle hnd)
{ /* Fetch the actors */
  Tree father = cont_pop_tree (hnd);
  Tree son = pop_tree (hnd);
  int sonnr = cont_pop_int (hnd);

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_enter_link_son (hnd, son, sonnr + 1);

  /* consistency check */
  if (father -> sons[sonnr] != tree_nil)
    dcg_internal_error ("erts_link_predicate_son");

  /* insert and continue */
  father -> sons[sonnr] = son;
  call_continuation (hnd);
  father -> sons[sonnr] = tree_nil;

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_leave_link_son (hnd, son, sonnr + 1);

  /* undo */
  push_tree (hnd, son);
  cont_push_int (hnd, sonnr);
  cont_push_tree (hnd, father);
  cont_push_continuation (hnd, erts_link_predicate_son);
}

/*
   erts_link_son will pop a son from the tree stack and
   insert him as son of the top node of the tree stack
*/
void erts_link_son (EagrtsHandle hnd)
{ /* Fetch the actors */
  int sonnr = cont_pop_int (hnd);
  Tree son = pop_tree (hnd);
  Tree father = top_tree (hnd);

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_enter_link_son (hnd, son, sonnr + 1);

  /* consistency check */
  if (father -> sons[sonnr] != tree_nil)
    dcg_internal_error ("erts_link_son");

  /* insert and continue */
  father -> sons[sonnr] = son;
  call_continuation (hnd);
  father -> sons[sonnr] = tree_nil;

  /* Report what you are doing */
  if (hnd -> tracing) erts_trace_leave_link_son (hnd, son, sonnr + 1);

  /* undo */
  push_tree (hnd, son);
  cont_push_int (hnd, sonnr);
  cont_push_continuation (hnd, erts_link_son);
}

/*
   erts_exchange_top will pop the top two entries of the tree stack,
   insert the second node as son of the first and push this tree again
*/
void erts_exchange_top (EagrtsHandle hnd)
{ /* Fetch father and son */
  Tree father = pop_tree (hnd);
  Tree son = pop_tree (hnd);

  /* insert and continue */
  push_tree (hnd, father);
  push_tree (hnd, son);
  call_continuation (hnd);
  (void) pop_tree (hnd);
  (void) pop_tree (hnd);

  /* undo */
  push_tree (hnd, son);
  push_tree (hnd, father);
  cont_push_continuation (hnd, erts_exchange_top);
}

/*
   erts_locate_frame will search the tree stack for the frame with a specific nodenr
*/
AffixNode *erts_locate_frame (EagrtsHandle hnd, int nodenr)
{ Tree *tree_ptr;
  for (tree_ptr = hnd -> tree_sp - 1; tree_ptr != hnd -> tree_stack; tree_ptr --)
    { Tree my_tree = *tree_ptr;
      tree_kind my_kind = my_tree -> kind;
      if ((my_kind != normal_node) &&
	  (my_kind != simple_node) &&
	  (my_kind != anonymous_node)) continue;
      if (my_tree -> number == nodenr)
	return (my_tree -> frame);
    };
  dcg_abort ("erts_locate_frame", "Could not locate frame for nodenr %d", nodenr);
  return (affix_frame_nil);
}
