/*
   File: erts_trace.c
   Defines tracing support

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_trace.c,v 1.8 2013/01/11 14:52:04 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_ds.h>
#include <ebase_utils.h>
#include <ebase_lexicon.h>
#include <ebase_lexicon_impl.h>

/* local includes */
#include "erts_tree.h"
#include "erts_tree_impl.h"
#include "erts_trellis.h"
#include "erts_trellis_impl.h"
#include "erts_handle.h"
#include "erts_handle_impl.h"
#include "erts_predefs.h"
#include "erts_trace.h"

/*
   Code to trace the first level
*/
static void write_indent (EagrtsHandle hnd)
{ dcg_eprint ("%d:", hnd -> curr_depth);
  erts_write_trellis_state (hnd -> trellis -> curr_state);
  dcg_eprint (": ");
}

void erts_trace_enter_function (EagrtsHandle hnd, char *function_name, int nont_nr)
{ rt_nont rt = hnd -> rt_nonts -> array[nont_nr];
  write_indent (hnd);
  dcg_wlog (">%s, '%s'", function_name, rt -> canonic_name);
  hnd -> curr_depth++;
}

void erts_trace_alternative (EagrtsHandle hnd, int nont_nr, int alt_nr)
{ rt_nont rt = hnd -> rt_nonts -> array[nont_nr];
  write_indent (hnd);
  dcg_wlog ("-parsing alt %d of '%s'", alt_nr, rt -> canonic_name);
}

void erts_trace_leave_function (EagrtsHandle hnd, char *function_name, int nont_nr)
{ rt_nont rt = hnd -> rt_nonts -> array[nont_nr];
  hnd -> curr_depth--;
  write_indent (hnd);
  dcg_wlog ("<%s, '%s'", function_name, rt -> canonic_name);
}

void erts_trace_enter_transition (EagrtsHandle hnd, Transition trans)
{ write_indent (hnd);
  dcg_eprint (">");
  erts_write_transition (hnd -> lexicon, trans);
  dcg_wlog ("");
}

void erts_trace_leave_transition (EagrtsHandle hnd, Transition trans)
{ write_indent (hnd);
  dcg_eprint ("<");
  erts_write_transition (hnd -> lexicon, trans);
  dcg_wlog ("");
}

/*
   Code to trace affix values and affix propagation
*/
void erts_trace_tree_node (EagrtsHandle hnd, Tree tree)
{ switch (tree -> kind)
    { case normal_node:
      case simple_node:
      case lex_nont_node:
      case anonymous_node:
      case predicate_node:
	{ int node_nr = tree -> number;
	  rt_nont rt = hnd -> rt_nonts -> array[node_nr];
	  dcg_eprint ("node '%s'", rt -> canonic_name);
	}; break;
      case confrontation_node: dcg_eprint ("confrontation"); break;
      case leaf_node:
	{ Transition trans = tree -> trans;
	  dcg_eprint ("leafnode ");
	  erts_write_transition (hnd -> lexicon, trans);
	}; break;
      case penalty_node: dcg_eprint ("penalty %d", tree -> number); break;
      case quasi_node:
	{ int quasi_nr = tree -> number;
	  char *name = name_from_quasi (quasi_nr);
	  dcg_eprint ("quasi node '%s'", name);
	}; break;
      default: dcg_bad_tag (tree -> kind, "erts_trace_tree_node");
    };
}

/* Routines to trace the linking of affixes and sons */
void erts_trace_enter_make_affix_link (EagrtsHandle hnd, Tree son_tree)
{ write_indent (hnd);
  dcg_eprint (">linking positions of ");
  erts_trace_tree_node (hnd, son_tree);
  dcg_wlog ("");
}

void erts_trace_leave_make_affix_link (EagrtsHandle hnd, Tree son_tree)
{ write_indent (hnd);
  dcg_eprint ("<unlinking positions of ");
  erts_trace_tree_node (hnd, son_tree);
  dcg_wlog ("");
}

void erts_trace_enter_link_son (EagrtsHandle hnd, Tree son_tree, int sonnr)
{ write_indent (hnd);
  dcg_eprint (">inserting ");
  erts_trace_tree_node (hnd, son_tree);
  dcg_wlog (" as son %d", sonnr);
}

void erts_trace_leave_link_son (EagrtsHandle hnd, Tree son_tree, int sonnr)
{ write_indent (hnd);
  dcg_eprint ("<removing ");
  erts_trace_tree_node (hnd, son_tree);
  dcg_wlog (" as son %d", sonnr);
}

static void trace_affix_node (EagrtsHandle hnd, AffixNode node)
{ dcg_eprint ("%s", node -> name);
  if (node -> value != affix_value_nil)
    { dcg_eprint ("(");
      ebs_dump_affix_value (dcg_error_file (), node -> value, hnd -> lexicon -> rt_domains);
      dcg_eprint (")");
    };
}

static void trace_affix_expr (EagrtsHandle hnd, AffixExpr exp)
{ /* We may be traced when the expression has not yet be filled in */
  if (exp == affix_expr_nil)
    { dcg_eprint ("___");
      return;
    };

  /* Otherwise pick the right tag and proceed accordingly */
  switch (exp -> kind)
    { case single_affix_node:
	trace_affix_node (hnd, exp -> uni.node);
	break;
      case ast_node:
	{ AffixExpr *exprs = exp -> uni.ast.exprs;
	  int nr = exp -> uni.ast.nr;
	  int ix;
	  dcg_eprint ("<AST %d,%d: ", exp -> uni.ast.typenr, exp -> uni.ast.marker);
	  for (ix = 0; ix < nr; ix++)
	    { if (ix) dcg_eprint (",");
	      trace_affix_expr (hnd, exprs[ix]);
	    };
	  dcg_eprint (">");
	}; break;
      case concat_node:
	{ AffixExpr *exprs = exp -> uni.text_concat.exprs;
	  int nr = exp -> uni.text_concat.nr;
	  int ix;
	  for (ix = 0; ix < nr; ix++)
	    { if (ix) dcg_eprint (" + ");
	      trace_affix_expr (hnd, exprs[ix]);
	    };
	}; break;
      case dyop_arith_node:
        trace_affix_expr (hnd, exp -> uni.arith.expr1);
	dcg_eprint (" %s ", ebs_string_from_operator (exp -> uni.arith.op));
        trace_affix_expr (hnd, exp -> uni.arith.expr2);
	break;
      case monop_arith_node:
	dcg_eprint (" %s ", ebs_string_from_operator (exp -> uni.arith.op));
        trace_affix_expr (hnd, exp -> uni.arith.expr1);
	break;
      default: dcg_bad_tag (exp -> kind, "trace_affix_expr");
    };
}

static void trace_pos_side (EagrtsHandle hnd, Position pos, int side)
{ write_indent (hnd);
  dcg_eprint ("side %d, sill = %d: ", side, pos -> sides[side].sill);
  trace_affix_expr (hnd, pos -> sides[side].expr);
  dcg_wlog ("");
}

static int find_position_nr (Tree tree, Position pos)
{ int ix;
  for (ix = 0; ix < tree -> nr_pos; ix++)
    if (tree -> positions[ix] == pos)
      return (ix);
  return (-1);
}

void erts_trace_position (EagrtsHandle hnd, Position pos)
{ Tree tree = pos -> tree_node;
  int pos_nr = find_position_nr (tree, pos);
  write_indent (hnd);
  dcg_eprint ("tracing %s position %d of ",
	      ebs_string_from_propagation_kind (pos -> kind), pos_nr + 1);
  erts_trace_tree_node (hnd, tree);
  dcg_wlog ("");
  trace_pos_side (hnd, pos, 0);
  trace_pos_side (hnd, pos, 1);
}

void erts_trace_restriction (EagrtsHandle hnd, Position pos, affix_value meet_value)
{ Tree tree = pos -> tree_node;
  int pos_nr = find_position_nr (tree, pos);
  write_indent (hnd);
  dcg_eprint ("restricting position %d of ", pos_nr);
  erts_trace_tree_node (hnd, tree);
  dcg_eprint (" with value: ");
  ebs_dump_affix_value (dcg_error_file (), meet_value, hnd -> lexicon -> rt_domains);
  dcg_wlog ("");
  trace_pos_side (hnd, pos, 0);
  trace_pos_side (hnd, pos, 1);
}
