/*
   File: erts_predefs.c
   Defines the predefined predicates for the EAG3 runtime system.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_predefs.c,v 1.5 2013/03/13 10:06:07 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_lexutil.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_ds.h>
#include <ebase_lexicon.h>

/* local includes */
#include "erts_tree.h"
#include "erts_tree_impl.h"
#include "erts_trellis.h"
#include "erts_trellis_impl.h"
#include "erts_handle.h"
#include "erts_handle_impl.h"
#include "erts_propagate.h"
#include "erts_memory.h"
#include "erts_cont.h"
#include "erts_predefs.h"

/*
   Currently, the set of quasi terminals that are not linked to
   trellis transitions is the set $LINE, $COLUMN, $POS and $SEP
*/
#define quasi_line 0
#define quasi_column 1
#define quasi_pos 2
#define quasi_sep 3
char *name_from_quasi (int quasi_nr)
{ switch (quasi_nr)
    { case quasi_line:	 return ("$LINE");
      case quasi_column: return ("$COLUMN");
      case quasi_pos:	 return ("$POS");
      case quasi_sep:	 return ("$SEP");
      default: dcg_bad_tag (quasi_nr, "name_from_quasi");
    };
  return ((char *) NULL);
}

void erts_yield_line (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  char *tval = dcg_new_fmtd_string ("%d", curr_state -> linenr);
  affix_value value = new_Text_value (-1, tval);
  int nr_pos = cont_pop_int (hnd);
  erts_make_quasi_node (hnd, quasi_line, value, nr_pos);
  detach_affix_value (&value);
  cont_push_int (hnd, nr_pos);
  cont_push_continuation (hnd, erts_yield_line);
}

void erts_yield_column (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  char *tval = dcg_new_fmtd_string ("%d", curr_state -> colnr);
  affix_value value = new_Text_value (-1, tval);
  int nr_pos = cont_pop_int (hnd);
  erts_make_quasi_node (hnd, quasi_column, value, nr_pos);
  detach_affix_value (&value);
  cont_push_int (hnd, nr_pos);
  cont_push_continuation (hnd, erts_yield_column);
}

void erts_yield_pos (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  char *tval = dcg_new_fmtd_string ("%d", curr_state -> offset);
  affix_value value = new_Text_value (-1, tval);
  int nr_pos = cont_pop_int (hnd);
  erts_make_quasi_node (hnd, quasi_pos, value, nr_pos);
  detach_affix_value (&value);
  cont_push_int (hnd, nr_pos);
  cont_push_continuation (hnd, erts_yield_pos);
}

void erts_match_separator (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  if (erts_state_matches_separator (trel, curr_state))
    { char *from = trel -> input + curr_state -> offset;
      char *to = ebs_skip_one_char (trel -> lexicon, from);
      char *tval = (char *) dcg_malloc (to - from + 1);
      int nr_pos = cont_pop_int (hnd);
      affix_value value;
      strncpy (tval, from, to - from);
      value = new_Text_value (-1, tval);
      erts_make_quasi_node (hnd, quasi_pos, value, nr_pos);
      detach_affix_value (&value);
      cont_push_int (hnd, nr_pos);
    }
  cont_push_continuation (hnd, erts_match_separator);
}

/*
   Current list of predefined predicates
   We should think of a better way to recognize the predefined nodenrs
*/
static struct {
int nodenr;
char *canonic_name;
} my_predefs[] =
{{ -1, "convert (INT) to (TEXT)" },
 { -1, "convert (TEXT) to (INT)" },
 { -1, "convert (REAL) to (TEXT)" },
 { -1, "convert (TEXT) to (REAL)" }
};

#define entry_convert_int_to_text 0
#define entry_convert_text_to_int 1
#define entry_convert_real_to_text 2
#define entry_convert_text_to_real 3

void erts_resolve_predefs (EagrtsHandle hnd)
{ ext_nont_list exts = hnd -> ext_nonts;
  int ix;
  for (ix = 0; ix < exts -> size; ix++)
    { ext_nont et = exts -> array[ix];
      int iy;
      for (iy = 0; iy < 4; iy++)
	if (streq (my_predefs[iy].canonic_name, et -> canonic_name))
	  my_predefs[iy].nodenr = et -> rule_nr;
    };
}

/*
   Predicate convert (>INT) to (TEXT>)
*/
static void actual_convert_int_to_text (EagrtsHandle hnd, Position *args)
{ affix_value val = (args[0]) -> sides[lower_side].expr -> uni.node -> value;
  if (val -> tag == TAGInt_value)
    { char *text = dcg_new_fmtd_string ("%d", val -> Int_value.ival);
      affix_value new_tval = new_Text_value (-1, text);
      cont_push_position (hnd, args[1]);
      cont_push_affix_value (hnd, new_tval);
      cont_push_continuation (hnd, erts_propagate_predicate_value);
      call_continuation (hnd);
      cont_pop (hnd, 3);
      detach_affix_value (&new_tval);	/* will also detach text */
    };
}

static void delayed_convert_int_to_text (EagrtsHandle hnd, Position *args)
{ if (critical_position_has_value (args[0]))
    { args[0] -> delayed_func = delayed_eval_func_nil;
      args[1] -> delayed_func = delayed_eval_func_nil;
      actual_convert_int_to_text (hnd, args);
      args[0] -> delayed_func = delayed_convert_int_to_text;
      args[1] -> delayed_func = delayed_convert_int_to_text;
    }
  else call_continuation (hnd);
}

void erts_convert_int_to_text (EagrtsHandle hnd)
{ /* Pick my resolved node nr */
  int my_nodenr = my_predefs[entry_convert_int_to_text].nodenr;

  /* Allocate delayed affix frame */
  AffixNode *frame = erts_make_affix_frame (2);
  frame[0] = erts_make_affix_node ("erts_convert_int_to_text_pafx1");
  frame[1] = erts_make_affix_node ("erts_convert_int_to_text_pafx2");

  /* Make predicate node with delayed affix positions */
  erts_make_predicate_node (hnd, my_nodenr, 2, frame, delayed_convert_int_to_text);
    
  /* Free delayed affix frame */
  erts_free_affix_node (frame[0]);
  erts_free_affix_node (frame[1]);
  erts_free_affix_frame (frame, 2);
    
  /* Push self reference */
  cont_push_continuation (hnd, erts_convert_int_to_text);
}

/*
   Predicate convert (>TEXT) to (INT>)
*/
static void actual_convert_text_to_int (EagrtsHandle hnd, Position *args)
{ affix_value val = (args[0]) -> sides[lower_side].expr -> uni.node -> value;
  if (val -> tag == TAGText_value)
    { int ival = dcg_convert_integer (val -> Text_value.text, 10);
      affix_value new_ival = new_Int_value (-1, ival);
      cont_push_position (hnd, args[1]);
      cont_push_affix_value (hnd, new_ival);
      cont_push_continuation (hnd, erts_propagate_predicate_value);
      call_continuation (hnd);
      cont_pop (hnd, 3);
      detach_affix_value (&new_ival);
    };
}

static void delayed_convert_text_to_int (EagrtsHandle hnd, Position *args)
{ if (critical_position_has_value (args[0]))
    { args[0] -> delayed_func = delayed_eval_func_nil;
      args[1] -> delayed_func = delayed_eval_func_nil;
      actual_convert_text_to_int (hnd, args);
      args[0] -> delayed_func = delayed_convert_text_to_int;
      args[1] -> delayed_func = delayed_convert_text_to_int;
    }
  else call_continuation (hnd);
}

void erts_convert_text_to_int (EagrtsHandle hnd)
{ /* Pick my resolved node nr */
  int my_nodenr = my_predefs[entry_convert_text_to_int].nodenr;

  /* Allocate delayed affix frame */
  AffixNode *frame = erts_make_affix_frame (2);
  frame[0] = erts_make_affix_node ("erts_convert_text_to_int_pafx1");
  frame[1] = erts_make_affix_node ("erts_convert_text_to_int_pafx2");

  /* Make predicate node with delayed affix positions */
  erts_make_predicate_node (hnd, my_nodenr, 2, frame, delayed_convert_text_to_int);
    
  /* Free delayed affix frame */
  erts_free_affix_node (frame[0]);
  erts_free_affix_node (frame[1]);
  erts_free_affix_frame (frame, 2);

  /* Push self reference */
  cont_push_continuation (hnd, erts_convert_text_to_int);
}

/*
   Predicate convert (>REAL) to (TEXT>)
*/
static void actual_convert_real_to_text (EagrtsHandle hnd, Position *args)
{ affix_value val = (args[0]) -> sides[lower_side].expr -> uni.node -> value;
  if (val -> tag == TAGReal_value)
    { char *text = dcg_new_fmtd_string ("%g", val -> Real_value.rval);
      affix_value new_tval = new_Text_value (-1, text);
      cont_push_position (hnd, args[1]);
      cont_push_affix_value (hnd, new_tval);
      cont_push_continuation (hnd, erts_propagate_predicate_value);
      call_continuation (hnd);
      cont_pop (hnd, 3);
      detach_affix_value (&new_tval);	/* will also detach text */
    };
}

static void delayed_convert_real_to_text (EagrtsHandle hnd, Position *args)
{ if (critical_position_has_value (args[0]))
    { args[0] -> delayed_func = delayed_eval_func_nil;
      args[1] -> delayed_func = delayed_eval_func_nil;
      actual_convert_real_to_text (hnd, args);
      args[0] -> delayed_func = delayed_convert_real_to_text;
      args[1] -> delayed_func = delayed_convert_real_to_text;
    }
  else call_continuation (hnd);
}

void erts_convert_real_to_text (EagrtsHandle hnd)
{ /* Pick my resolved node nr */
  int my_nodenr = my_predefs[entry_convert_real_to_text].nodenr;

  /* Allocate delayed affix frame */
  AffixNode *frame = erts_make_affix_frame (2);
  frame[0] = erts_make_affix_node ("erts_convert_real_to_text_pafx1");
  frame[1] = erts_make_affix_node ("erts_convert_real_to_text_pafx2");

  /* Make predicate node with delayed affix positions */
  erts_make_predicate_node (hnd, my_nodenr, 2, frame, delayed_convert_real_to_text);
    
  /* Free delayed affix frame */
  erts_free_affix_node (frame[0]);
  erts_free_affix_node (frame[1]);
  erts_free_affix_frame (frame, 2);
    
  /* Push self reference */
  cont_push_continuation (hnd, erts_convert_real_to_text);
}

/*
   Predicate convert (>TEXT) to (REAL>)
*/
static void actual_convert_text_to_real (EagrtsHandle hnd, Position *args)
{ affix_value val = (args[0]) -> sides[lower_side].expr -> uni.node -> value;
  if (val -> tag == TAGText_value)
    { real rval = dcg_convert_real (val -> Text_value.text);
      affix_value new_rval = new_Real_value (-1, rval);
      cont_push_position (hnd, args[1]);
      cont_push_affix_value (hnd, new_rval);
      cont_push_continuation (hnd, erts_propagate_predicate_value);
      call_continuation (hnd);
      cont_pop (hnd, 3);
      detach_affix_value (&new_rval);
    };
}

static void delayed_convert_text_to_real (EagrtsHandle hnd, Position *args)
{ if (critical_position_has_value (args[0]))
    { args[0] -> delayed_func = delayed_eval_func_nil;
      args[1] -> delayed_func = delayed_eval_func_nil;
      actual_convert_text_to_real (hnd, args);
      args[0] -> delayed_func = delayed_convert_text_to_real;
      args[1] -> delayed_func = delayed_convert_text_to_real;
    }
  else call_continuation (hnd);
}

void erts_convert_text_to_real (EagrtsHandle hnd)
{ /* Pick my resolved node nr */
  int my_nodenr = my_predefs[entry_convert_text_to_real].nodenr;

  /* Allocate delayed affix frame */
  AffixNode *frame = erts_make_affix_frame (2);
  frame[0] = erts_make_affix_node ("erts_convert_text_to_real_pafx1");
  frame[1] = erts_make_affix_node ("erts_convert_text_to_real_pafx2");

  /* Make predicate node with delayed affix positions */
  erts_make_predicate_node (hnd, my_nodenr, 2, frame, delayed_convert_text_to_real);

  /* Free delayed affix frame */
  erts_free_affix_node (frame[0]);
  erts_free_affix_node (frame[1]);
  erts_free_affix_frame (frame, 2);

  /* Push self reference */
  cont_push_continuation (hnd, erts_convert_text_to_real);
}
