/*
   File: erts_memory.c
   Maintains free lists for all really dynamic objects in the runtime system

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_memory.c,v 1.3 2012/12/05 14:11:57 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* local includes */
#include "erts_tree.h"
#include "erts_tree_impl.h"

/*
   Tree administration
*/
static Tree free_trees;
static int allocated_trees;
static int in_free_trees;
Tree erts_make_tree_node (tree_kind kind, int number)
{ Tree my_tree = tree_nil;
  if ((in_free_trees > 0) && (free_trees != tree_nil))
    { my_tree = free_trees;
      free_trees = (Tree) free_trees -> trans;
      in_free_trees--;
    }
  else if ((in_free_trees == 0) && (free_trees == tree_nil))
    { my_tree = (Tree) dcg_malloc (sizeof (struct tree_rec));
      allocated_trees++;
    }
  else dcg_internal_error ("erts_make_tree_node");
  memset ((void *) my_tree, 0, sizeof (struct tree_rec));
  my_tree -> kind = kind;
  my_tree -> number = number;
  return (my_tree);
}

void erts_free_tree_node (Tree old_tree)
{ if (old_tree == tree_nil)
    dcg_internal_error ("erts_free_tree_node");
  memset ((void *) old_tree, 0, sizeof (struct tree_rec));
  old_tree -> trans = (Transition) free_trees;
  free_trees = old_tree;
  in_free_trees++;
}

/*
   Tree space administration
*/
static int max_son_space;
static Tree **free_son_space;
static int *allocated_sons;
static int *in_free_sons;
Tree *erts_make_tree_sons (int nr_sons)
{ int idx = nr_sons - 1;
  Tree *my_sons = trees_nil;
  int ix;

  /* No sons needed, return nil */
  if (nr_sons == 0) return (trees_nil);

  /* Check the size of our admin */
  if (nr_sons > max_son_space)
    { dcg_recalloc ((void **) &free_son_space, nr_sons, sizeof (Tree *));
      dcg_recalloc ((void **) &allocated_sons, nr_sons, sizeof (int));
      dcg_recalloc ((void **) &in_free_sons, nr_sons, sizeof (int));
      for (ix = max_son_space; ix < nr_sons; ix++)
        { free_son_space[ix] = trees_nil;
	  allocated_sons[ix] = 0;
	  in_free_sons[ix] = 0;
	};
      max_son_space = nr_sons;
    };

  /* Check the appropriate free list */
  if ((in_free_sons[idx] > 0) && (free_son_space[idx] != trees_nil))
    { my_sons = free_son_space[idx];
      free_son_space[idx] = (Tree *) my_sons[0];
      in_free_sons[idx]--;
    }
  else if ((in_free_sons[idx] == 0) && (free_son_space[idx] == trees_nil))
    { my_sons = (Tree *) dcg_calloc (nr_sons, sizeof (Tree));
      allocated_sons[idx]++;
    }
  else dcg_internal_error ("erts_make_tree_sons");

  /* Clear everything and yield */
  for (ix = 0; ix < nr_sons; ix++)
    my_sons[ix] = tree_nil;
  return (my_sons);
}

void erts_free_tree_sons (Tree *old_sons, int nr_sons)
{ int idx = nr_sons - 1;
  if (nr_sons == 0) return;
  old_sons[0] = (Tree) free_son_space[idx];
  free_son_space[idx] = old_sons;
  in_free_sons[idx]++;
}

/*
   Position administration
*/
static Position free_positions;
static int allocated_positions;
static int in_free_positions;
Position erts_make_affix_position (Tree my_tree)
{ Position my_position = position_nil;
  if ((in_free_positions > 0) && (free_positions != position_nil))
    { my_position = free_positions;
      free_positions = (Position) free_positions -> tree_node;
      in_free_positions--;
    }
  else if ((in_free_positions == 0) && (free_positions == position_nil))
    { my_position = (Position) dcg_malloc (sizeof (struct position_rec));
      allocated_positions++;
    }
  else dcg_internal_error ("erts_make_affix_position");
  memset ((void *) my_position, 0, sizeof (struct position_rec));
  my_position -> tree_node = my_tree;
  return (my_position);
}

void erts_free_affix_position (Position old_position)
{ if (old_position == position_nil)
    dcg_internal_error ("erts_free_affix_position");
  memset ((void *) old_position, 0, sizeof (struct position_rec));
  old_position -> tree_node = (Tree) free_positions;
  free_positions = old_position;
  in_free_positions++;
}

/*
   Position space administration
*/
static int max_position_space;
static Position **free_position_space;
static int *allocated_position_space;
static int *in_free_position_space;
Position *erts_make_affix_positions (int nr_pos)
{ int idx = nr_pos - 1;
  Position *my_positions = positions_nil;
  int ix;

  /* No positions needed, return nil */
  if (nr_pos == 0) return (positions_nil);

  /* Check the size of our admin */
  if (nr_pos > max_position_space)
    { dcg_recalloc ((void **) &free_position_space, nr_pos, sizeof (Position *));
      dcg_recalloc ((void **) &allocated_position_space, nr_pos, sizeof (int));
      dcg_recalloc ((void **) &in_free_position_space, nr_pos, sizeof (int));
      for (ix = max_position_space; ix < nr_pos; ix++)
        { free_position_space[ix] = positions_nil;
	  allocated_position_space[ix] = 0;
	  in_free_position_space[ix] = 0;
	};
      max_position_space = nr_pos;
    };

  /* Check the appropriate free list */
  if ((in_free_position_space[idx] > 0) && (free_position_space[idx] != positions_nil))
    { my_positions = free_position_space[idx];
      free_position_space[idx] = (Position *) my_positions[0];
      in_free_position_space[idx]--;
    }
  else if ((in_free_position_space[idx] == 0) && (free_position_space[idx] == positions_nil))
    { my_positions = (Position *) dcg_calloc (nr_pos, sizeof (Position));
      allocated_position_space[idx]++;
    }
  else dcg_internal_error ("erts_make_affix_positions");

  /* Clear everything and yield */
  for (ix = 0; ix < nr_pos; ix++)
    my_positions[ix] = position_nil;
  return (my_positions);
}

void erts_free_affix_positions (Position *old_pos, int nr_pos)
{ int idx = nr_pos - 1;
  if (nr_pos == 0) return;
  old_pos[0] = (Position) free_position_space[idx];
  free_position_space[idx] = old_pos;
  in_free_position_space[idx]++;
}

/*
   Affix expression administration
*/
static AffixExpr free_exprs;
static int allocated_exprs;
static int in_free_exprs;
AffixExpr erts_make_affix_expr (affix_expr_kind kind)
{ AffixExpr my_expr = affix_expr_nil;
  if ((in_free_exprs > 0) && (free_exprs != affix_expr_nil))
    { my_expr = free_exprs;
      free_exprs = free_exprs -> uni.arith.expr2;
      in_free_exprs--;
    }
  else if ((in_free_exprs == 0) && (free_exprs == affix_expr_nil))
    { my_expr = (AffixExpr) dcg_malloc (sizeof (struct affix_expr_rec));
      allocated_exprs++;
    }
  else dcg_internal_error ("erts_make_affix_expr");
  memset ((void *) my_expr, 0, sizeof (struct affix_expr_rec));
  my_expr -> kind = kind;
  return (my_expr);
}

void erts_free_affix_expr (AffixExpr old_expr)
{ if (old_expr == affix_expr_nil)
    dcg_internal_error ("erts_free_affix_expr");
  memset ((void *) old_expr, 0, sizeof (struct affix_expr_rec));
  old_expr -> uni.arith.expr2 = free_exprs;
  free_exprs = old_expr;
  in_free_exprs++;
}

/*
   Affix expr space administration
*/
static int max_expr_space;
static AffixExpr **free_expr_space;
static int *allocated_expr_space;
static int *in_free_expr_space;
AffixExpr *erts_make_affix_expressions (int nr_exp)
{ int idx = nr_exp - 1;
  AffixExpr *my_exprs = affix_exprs_nil;
  int ix;

  /* No exprs needed, return nil */
  if (nr_exp == 0) return (affix_exprs_nil);

  /* Check the size of our admin */
  if (nr_exp > max_expr_space)
    { dcg_recalloc ((void **) &free_expr_space, nr_exp, sizeof (AffixExpr *));
      dcg_recalloc ((void **) &allocated_expr_space, nr_exp, sizeof (int));
      dcg_recalloc ((void **) &in_free_expr_space, nr_exp, sizeof (int));
      for (ix = max_expr_space; ix < nr_exp; ix++)
        { free_expr_space[ix] = affix_exprs_nil;
	  allocated_expr_space[ix] = 0;
	  in_free_expr_space[ix] = 0;
	};
      max_expr_space = nr_exp;
    };

  /* Check the appropriate free list */
  if ((in_free_expr_space[idx] > 0) && (free_expr_space[idx] != affix_exprs_nil))
    { my_exprs = free_expr_space[idx];
      free_expr_space[idx] = (AffixExpr *) my_exprs[0];
      in_free_expr_space[idx]--;
    }
  else if ((in_free_expr_space[idx] == 0) && (free_expr_space[idx] == affix_exprs_nil))
    { my_exprs = (AffixExpr *) dcg_calloc (nr_exp, sizeof (AffixExpr));
      allocated_expr_space[idx]++;
    }
  else dcg_internal_error ("erts_make_affix_expressions");

  /* Clear everything and yield */
  for (ix = 0; ix < nr_exp; ix++)
    my_exprs[ix] = affix_expr_nil;
  return (my_exprs);
}

void erts_free_affix_expressions (AffixExpr *old_exprs, int nr_exp)
{ int idx = nr_exp - 1;
  if (nr_exp == 0) return;
  old_exprs[0] = (AffixExpr) free_expr_space[idx];
  free_expr_space[idx] = old_exprs;
  in_free_expr_space[idx]++;
}

/*
   Affix node administration
*/
static AffixNode free_nodes;
static int allocated_nodes;
static int in_free_nodes;
AffixNode erts_make_affix_node (char *name)
{ AffixNode my_node = affix_node_nil;
  if ((in_free_nodes > 0) && (free_nodes != affix_node_nil))
    { my_node = free_nodes;
      free_nodes = (AffixNode) free_nodes -> links;
      in_free_nodes--;
    }
  else if ((in_free_nodes == 0) && (free_nodes == affix_node_nil))
    { my_node = (AffixNode) dcg_malloc (sizeof (struct affix_node_rec));
      allocated_nodes++;
    }
  else dcg_internal_error ("erts_make_affix_node");
  memset ((void *) my_node, 0, sizeof (struct affix_node_rec));
  my_node -> name = name;
  return (my_node);
}

void erts_free_affix_node (AffixNode old_node)
{ if (old_node == affix_node_nil)
    dcg_internal_error ("erts_free_affix_node");
  memset ((void *) old_node, 0, sizeof (struct affix_node_rec));
  old_node -> links = (AffixLink) free_nodes;
  free_nodes = old_node;
  in_free_nodes++;
}

/*
   Affix frame administration
*/
static int max_frame_size;
static AffixNode **free_frames;
static int *allocated_frames;
static int *in_free_frames;
AffixNode *erts_make_affix_frame (int nr)
{ int idx = nr - 1;
  AffixNode *my_frame = affix_frame_nil;
  int ix;

  /* No sons needed, return nil */
  if (nr == 0) return (affix_frame_nil);

  /* Check the size of our admin */
  if (nr > max_frame_size)
    { dcg_recalloc ((void **) &free_frames, nr, sizeof (AffixNode *));
      dcg_recalloc ((void **) &allocated_frames, nr, sizeof (int));
      dcg_recalloc ((void **) &in_free_frames, nr, sizeof (int));
      for (ix = max_frame_size; ix < nr; ix++)
        { free_frames[ix] = affix_frame_nil;
	  allocated_frames[ix] = 0;
	  in_free_frames[ix] = 0;
	};
      max_frame_size = nr;
    };

  /* Check the appropriate free list */
  if ((in_free_frames[idx] > 0) && (free_frames[idx] != affix_frame_nil))
    { my_frame = free_frames[idx];
      free_frames[idx] = (AffixNode *) my_frame[0];
      in_free_frames[idx]--;
    }
  else if ((in_free_frames[idx] == 0) && (free_frames[idx] == affix_frame_nil))
    { my_frame = (AffixNode *) dcg_calloc (nr, sizeof (AffixNode));
      allocated_frames[idx]++;
    }
  else dcg_internal_error ("erts_make_tree_sons");

  /* Clear everything and yield */
  for (ix = 0; ix < nr; ix++)
    my_frame[ix] = affix_node_nil;
  return (my_frame);
}

void erts_free_affix_frame (AffixNode *old_frame, int nr)
{ int idx = nr - 1;
  if (nr == 0) return;
  old_frame[0] = (AffixNode) free_frames[idx];
  free_frames[idx] = old_frame;
  in_free_frames[idx]++;
}

/*
   Affix link administration
*/
static AffixLink free_links;
static int allocated_links;
static int in_free_links;
AffixLink erts_make_link_node ()
{ AffixLink my_link = affix_link_nil;
  if ((in_free_links > 0) && (free_links != affix_link_nil))
    { my_link = free_links;
      free_links = free_links -> next;
      in_free_links--;
    }
  else if ((in_free_links == 0) && (free_links == affix_link_nil))
    { my_link = (AffixLink) dcg_malloc (sizeof (struct affix_link_rec));
      allocated_links++;
    }
  else dcg_internal_error ("erts_make_link_node");
  memset ((void *) my_link, 0, sizeof (struct affix_link_rec));
  return (my_link);
}

void erts_free_link_node (AffixLink old_link)
{ if (old_link == affix_link_nil)
    dcg_internal_error ("erts_free_link_node");
  memset ((void *) old_link, 0, sizeof (struct affix_link_rec));
  old_link -> next = free_links;
  free_links = old_link;
  in_free_links++;
}

/*
   Dump for debugging
*/
void erts_dump_memory ()
{ int ix;
  dcg_wlog ("Allocated tree nodes: %d, in free list: %d", allocated_trees, in_free_trees);
  dcg_wlog ("Allocated affix positions: %d, in free list: %d",
	    allocated_positions, in_free_positions);
  dcg_wlog ("Allocated affix exprs: %d, in free list: %d", allocated_exprs, in_free_exprs);
  dcg_wlog ("Allocated affix nodes: %d, in free list: %d", allocated_nodes, in_free_nodes);
  dcg_wlog ("Allocated affix links: %d, in free list: %d", allocated_links, in_free_links);
  for (ix = 0; ix < max_son_space; ix++)
    if (allocated_sons[ix])
      dcg_wlog ("Son space %d: allocated %d, in free list %d",
		ix + 1, allocated_sons[ix], in_free_sons[ix]);
  for (ix = 0; ix < max_position_space; ix++)
    if (allocated_position_space[ix])
      dcg_wlog ("Position space %d: allocated %d, in free list %d",
		ix + 1, allocated_position_space[ix], in_free_position_space[ix]);
  for (ix = 0; ix < max_expr_space; ix++)
    if (allocated_expr_space[ix])
      dcg_wlog ("Affix expr space %d: allocated %d, in free list %d",
		ix + 1, allocated_expr_space[ix], in_free_expr_space[ix]);
  for (ix = 0; ix < max_frame_size; ix++)
    if (allocated_frames[ix])
      dcg_wlog ("Frame with %d affixes: allocated %d, in free list %d",
		ix + 1, allocated_frames[ix], in_free_frames[ix]);
}

void erts_init_memory ()
{ int ix;
  free_trees = tree_nil;
  allocated_trees = 0;
  in_free_trees = 0;
  free_positions = position_nil;
  allocated_positions = 0;
  in_free_positions = 0;
  free_exprs = affix_expr_nil;
  allocated_exprs = 0;
  in_free_exprs = 0;
  free_nodes = affix_node_nil;
  allocated_nodes = 0;
  in_free_nodes = 0;
  free_links = affix_link_nil;
  allocated_links = 0;
  in_free_links = 0;

  /* Initialize the son space */
  max_son_space = 32;
  free_son_space = (Tree **) dcg_calloc (max_son_space, sizeof (Tree *));
  allocated_sons = (int *) dcg_calloc (max_son_space, sizeof (int));
  in_free_sons = (int *) dcg_calloc (max_son_space, sizeof (int));
  for (ix = 0; ix < max_son_space; ix++)
    { free_son_space[ix] = trees_nil;
      allocated_sons[ix] = 0;
      in_free_sons[ix] = 0;
    };

  /* Initialize the position space */
  max_position_space = 32;
  free_position_space = (Position **) dcg_calloc (max_position_space, sizeof (Position *));
  allocated_position_space = (int *) dcg_calloc (max_position_space, sizeof (int));
  in_free_position_space = (int *) dcg_calloc (max_position_space, sizeof (int));
  for (ix = 0; ix < max_position_space; ix++)
    { free_position_space[ix] = positions_nil;
      allocated_position_space[ix] = 0;
      in_free_position_space[ix] = 0;
    };

  /* Initialize the expr space */
  max_expr_space = 32;
  free_expr_space = (AffixExpr **) dcg_calloc (max_expr_space, sizeof (AffixExpr *));
  allocated_expr_space = (int *) dcg_calloc (max_expr_space, sizeof (int));
  in_free_expr_space = (int *) dcg_calloc (max_expr_space, sizeof (int));
  for (ix = 0; ix < max_expr_space; ix++)
    { free_expr_space[ix] = affix_exprs_nil;
      allocated_expr_space[ix] = 0;
      in_free_expr_space[ix] = 0;
    };

  /* Initialize the affix frame space */
  max_frame_size = 64;
  free_frames = (AffixNode **) dcg_calloc (max_frame_size, sizeof (AffixNode *));
  allocated_frames = (int *) dcg_calloc (max_frame_size, sizeof (int));
  in_free_frames = (int *) dcg_calloc (max_frame_size, sizeof (int));
  for (ix = 0; ix < max_frame_size; ix++)
    { free_frames[ix] = affix_frame_nil;
      allocated_frames[ix] = 0;
      in_free_frames[ix] = 0;
    };
}
