/*
   File: erts_handle.c
   Defines the data structure containing the runtime parametrization
   of the EAG3 runtime system, the EAG3 tree and continuation stack.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_handle.c,v 1.11 2012/12/26 15:43:17 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <limits.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_lexicon.h>

/* local includes */
#include "erts_trellis.h"
#include "erts_handle.h"
#include "erts_handle_impl.h"

EagrtsHandle erts_init_handle ()
{
  /* Allocate handle and set sensible defaults */
  EagrtsHandle hnd = (EagrtsHandle) dcg_malloc (sizeof (struct eagrts_handle_rec));

  /* File names */
  hnd -> dir_name = NULL;
  hnd -> grammar_name = NULL;
  hnd -> input_fname = NULL;
  hnd -> output_fname = NULL;
  hnd -> input_pos_fname = NULL;

  /* Lexicon */
  hnd -> lexicon = NULL;

  /* options */
  hnd -> hybrid_parsing = 0;
  hnd -> partial_parse = 0;
  hnd -> input_mode = 0;
  hnd -> tracing = 0;
  hnd -> lcsdoc_sync_option = 0;
  hnd -> transduce_option = tr_affix; /* For the moment */
  hnd -> show_trellis = 0;
  hnd -> max_parses = INT_MAX;
  hnd -> best_parses = 0;

  /* Input/Output handling */
  hnd -> input_file = NULL;
  hnd -> interactive = 0;
  hnd -> linenr = 1;
  hnd -> colnr = 1;
  hnd -> seen_eof = 0;
  hnd -> seen_doc_sync = 0;
  hnd -> output_file = NULL;

  /* Input per chunk (line/paragraph/document) */
  hnd -> input_buffer = NULL;
  hnd -> input_bufptr = NULL;
  hnd -> input_size = 0;

  hnd -> input_length = 0;
  hnd -> input_linenr = 1;
  hnd -> input_colnr = 1;

  /* Dynamic data structures */
  hnd -> trellis = trellis_nil;
  hnd -> start_state = state_nil;
  hnd -> curr_penalty = 0;
  hnd -> curr_depth = 0;

  /* Tree stack */
  hnd -> tree_stack = NULL;
  hnd -> tree_stack_size = 8192;
  hnd -> tree_sp = NULL;

  /* Continuation stack */
  hnd -> cont_stack = NULL;
  hnd -> cont_stack_size = 65536;
  hnd -> cont_sp = NULL;

  /* Parse results */
  hnd -> all_results = parse_result_nil;
  hnd -> curr_result = parse_result_nil;
  hnd -> nr_of_results = 0;
  hnd -> max_penalty = INT_MAX;
  return (hnd);
}

#define RTI_SUFFIX "rti"
static BinFile open_runtime_information_file (EagrtsHandle hnd)
{ char *fname = dcg_new_fmtd_string ("%s.%s", hnd -> grammar_name, RTI_SUFFIX);
  BinFile bf = dcg_bin_fopen_path (hnd -> dir_name, fname, "r", 1);
  detach_string (&fname);
  return (bf);
}

static void load_pragmats (BinFile bf, EagrtsHandle hnd)
{ load_int (bf, &hnd -> input_mode);
  load_int (bf, &hnd -> partial_parse);
  load_int (bf, &hnd -> hybrid_parsing);
  load_int (bf, &hnd -> tracing);
  load_int (bf, &hnd -> negative_memoization);
  load_int (bf, &hnd -> nr_neg_memos);
  load_int (bf, &hnd -> positive_memoization);
}

#define RTI_KIND "Runtime Information"
void erts_read_runtime_information (EagrtsHandle hnd)
{ BinFile bf = open_runtime_information_file (hnd);
  dcg_bin_verify_header (bf, "EAG3", RTI_KIND, EAG3_VERSION);
  load_pragmats (bf, hnd);
  load_rt_nont_list (bf, &hnd -> rt_nonts);
  load_ext_nont_list (bf, &hnd -> ext_nonts);
  load_affix_value_list (bf, &hnd -> rt_consts);

  /* Finalize reading */
  dcg_bin_verify_eof (bf);
  dcg_bin_fclose (&bf);
}

void erts_init_stacks (EagrtsHandle hnd)
{ hnd -> tree_stack = (Tree *) dcg_calloc (hnd -> tree_stack_size, sizeof (Tree));
  hnd -> cont_stack = (cont_ptr) dcg_calloc (hnd -> cont_stack_size, sizeof (cont_cell));
  hnd -> tree_sp = hnd -> tree_stack;
  hnd -> cont_sp = hnd -> cont_stack;
}

void erts_release_handle (EagrtsHandle *hnd)
{ EagrtsHandle old_hnd;
  if (hnd == NULL)
    dcg_internal_error ("erts_release_handle");
  old_hnd = *hnd;

  /* Recursive free of all parameters */
  detach_string (&old_hnd -> dir_name);
  detach_string (&old_hnd -> grammar_name);
  detach_string (&old_hnd -> input_fname);
  detach_string (&old_hnd -> output_fname);
  detach_string (&old_hnd -> input_pos_fname);
  ebs_detach_lexicon (&old_hnd -> lexicon);
  dcg_detach ((void **) hnd);
}
