/*
   File: bli_parser.c
   Parses the binary lexicon generator interface file

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: bli_parser.c,v 1.8 2012/07/28 20:02:48 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_binfile.h>

/* libeagbase includes */
#include <ebase_version.h>
#include <ebase_ds.h>

/* Local includes */
#include "options.h"
#include "globals.h"
#include "bli_parser.h"

/*
   The lexicon interface file is a binary encoded file consisting of the following parts

   1) A list of lexicon names
   2) A list of fact table names
   3) A list of triples file names
   4) All affix super domains
   5) The type table
   6) Those pragmat settings that influence the lexicalization process
   7) The number of facts
   8) The lexicon/fact nonterminal table (defining all nonterminals with their typing)
   9) The terminal table (defining all grammar terminals)
   10) The character set table (defining all characters sets)
   11) The regexp nfa table (defining all nfas for the regexps).
*/

static BinFile open_lexgen_interface_file ()
{ char *fname = dcg_new_fmtd_string ("%s.%s", base_gname, BLI_SUFFIX);
  BinFile bf = dcg_bin_fopen_path (dir_name, fname, "r", 1);
  dcg_detach ((void **) &fname);
  return (bf);
}

static void load_pragmats (BinFile bf)
{ dcg_bin_load_int (bf, &hyphen_convention_active);
  dcg_bin_load_int (bf, &hybrid_parsing);
  dcg_bin_load_int (bf, &utf8_processing);
  dcg_bin_load_int (bf, &translate_option);
  dcg_bin_load_int (bf, &empty_white_space_option);
  load_string (bf, &translate_fname);
  load_int_list (bf, &white_spaces);
  load_int_list (bf, &separators);
}

void parse_lexgen_interface_file ()
{ BinFile bf = open_lexgen_interface_file ();
  dcg_bin_verify_header (bf, "EAG3", "Lexicon Interface", EAG3_VERSION);
  load_string_list (bf, &all_lexica);
  load_string_list (bf, &all_fact_tables);
  load_string_list (bf, &all_triple_databases);
  load_rt_domain_list (bf, &all_rt_domains);
  load_rt_type_list (bf, &all_rt_types);
  load_pragmats (bf);
  dcg_bin_load_int (bf, &nr_facts);
  load_lex_nont_list (bf, &all_lex_nonts);
  load_terminal_list (bf, &all_terminals);
  load_cset_list (bf, &all_csets);
  load_nfa_list (bf, &all_regexp_nfas);
  dcg_bin_verify_eof (bf);
  dcg_bin_fclose (&bf);
}
