/*
   File: transform.c
   Transforms anonymous rules into rules with a frame to generate parsers
   other than the topdown recursive backup parsers.

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: transform.c,v 1.3 2013/01/03 16:38:49 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>
#include <dcg_plist_ops.h>

/* libeagbase includes */
#include <ebase_ds.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "ast_utils.h"
#include "transform.h"

/*
   First we have to collect the set of free locals in the alternatives
*/
static void collect_free_locals_in_affix_terms (affix_term_list terms, int_list free_locals);
static void collect_free_locals_in_affix_term (affix_term term, int_list free_locals)
{ switch (term -> tag)
    { case TAGDyop:
	collect_free_locals_in_affix_term (term -> Dyop.arg1, free_locals);
	collect_free_locals_in_affix_term (term -> Dyop.arg2, free_locals);
	break;
      case TAGMonop:
	collect_free_locals_in_affix_term (term -> Monop.arg, free_locals);
	break;
      case TAGAst:
	collect_free_locals_in_affix_terms (term -> Ast.terms, free_locals);
	break;
      case TAGVar:
	{ variable vdef = term -> Var.vdef;
	  int vnr = vdef -> vnr;
	  add_uniquely_to_int_list (free_locals, vnr);
	}; 
      case TAGTerminal:
      case TAGInt:
      case TAGReal:
      case TAGText:
      case TAGRegexp: break;
      default: dcg_bad_tag (term -> tag, "collect_free_locals_in_affix_term");
    };
}

static void collect_free_locals_in_affix_terms (affix_term_list terms, int_list free_locals)
{ int ix;
  for (ix = 0; ix < terms -> size; ix++)
    collect_free_locals_in_affix_term (terms -> array[ix], free_locals);
}

static void collect_free_locals_in_member (member mem, int_list free_locals)
{ int ix;
  switch (mem -> tag)
    { case TAGRes_guard:
        { res_conf_list rconfs = mem -> Res_guard.rconfs;
	  for (ix = 0; ix < rconfs -> size; ix++)
	    { res_conf rconf = rconfs -> array[ix];
	      collect_free_locals_in_affix_term (rconf -> lhs, free_locals);
	      collect_free_locals_in_affix_term (rconf -> rhs, free_locals);
	    };
	}; break;
      case TAGRes_call:
	{ affix_term_list args = mem -> Res_call.args;
	  for (ix = 0; ix < args -> size; ix++)
	    collect_free_locals_in_affix_term (args -> array[ix], free_locals);
	}; break;
      case TAGRes_term:
	if (mem -> Res_term.arg != affix_term_nil)
	  collect_free_locals_in_affix_term (mem -> Res_term.arg, free_locals);
      case TAGOp: break;
      default: dcg_bad_tag (mem -> tag, "collect_free_locals_in_member");
    };
}

static void collect_free_locals_in_fwo (fwo_group fwo, int_list free_locals)
{ switch (fwo -> tag)
    { case TAGSingle:
	collect_free_locals_in_member (fwo -> Single.mem, free_locals);
	break;
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    collect_free_locals_in_member (mems -> array[ix], free_locals);
	}; break;
      default: dcg_bad_tag (fwo -> tag, "collect_free_locals_in_fwo");
    };
}

static void collect_free_locals_in_members (fwo_group_list fwos, int_list free_locals)
{ int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    collect_free_locals_in_fwo (fwos -> array[ix], free_locals);
}

static void collect_free_locals_in_alt (alternative alt, int_list free_locals)
{ collect_free_locals_in_members (alt -> members, free_locals);
  /* collect_free_locals_in_transduction (alt -> trans, free_locals); */
}

static void collect_free_locals (group grp, int_list free_locals)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    collect_free_locals_in_alt (alts -> array[ix], free_locals);
}

/*
   We now know, which of the variables in the ancestor def must become
   part of the signature of the new rule. Update it therefore
*/
static void update_signature (spec rspec, variable_list old_locals, int_list mapping)
{ int ix;
  for (ix = 0; ix < mapping -> size; ix++)
    { variable old_local = old_locals -> array[mapping -> array[ix]];
      app_affix_rule_list (rspec -> rsig, attach_affix_rule (old_local -> adef));
      app_dir_list (rspec -> dirs, d_unknown);
    };
}

static variable_list make_new_locals (variable_list old_locals, int_list mapping)
{ variable_list new_locals = init_variable_list (mapping -> size);
  int ix;
  for (ix = 0; ix < mapping -> size; ix++)
    { variable old_local = old_locals -> array[mapping -> array[ix]];
      variable new_local = new_variable (attach_string (old_local -> vname));
      new_local -> vnr = ix;	/* New frame position */
      new_local -> adef = old_local -> adef;
      new_local -> cnr = -1;
      app_variable_list (new_locals, new_local);
    };
  return (new_locals);
}

static fpar_list make_new_formal_parameters (int line, int col, variable_list new_locals)
{ fpar_list new_formals = init_fpar_list (new_locals -> size);
  int ix; 
  for (ix = 0; ix < new_locals -> size; ix++)
    { affix_term_list new_fexp = new_affix_term_list ();
      variable new_local = new_locals -> array[ix];
      fpar new_formal;

      affix_term new_term = new_Var (line, col, attach_string (new_local -> vname));
      new_term -> adef = new_local -> adef;
      new_term -> Var.vdef = new_local;
      app_affix_term_list (new_fexp, new_term);
      new_formal = new_fpar (line, col, d_unknown, new_fexp);
      new_formal -> adef = new_local -> adef;
      app_fpar_list (new_formals, new_formal);
    };
  return (new_formals);
}

/*
   Remap the set of free locals in the local alternative
*/
static variable remap_variable (variable old, variable_list new_locals, int_list mapping)
{ int ix;
  for (ix = 0; ix < mapping -> size; ix++)
    if (old -> vnr == mapping -> array[ix])
      return (new_locals -> array[ix]);
  return (variable_nil);
}

static void remap_free_locals_in_affix_terms (affix_term_list terms,
					      variable_list new_locals, int_list mapping);
static void remap_free_locals_in_affix_term (affix_term term,
					     variable_list new_locals, int_list mapping)
{ switch (term -> tag)
    { case TAGDyop:
	remap_free_locals_in_affix_term (term -> Dyop.arg1, new_locals, mapping);
	remap_free_locals_in_affix_term (term -> Dyop.arg2, new_locals, mapping);
	break;
      case TAGMonop:
	remap_free_locals_in_affix_term (term -> Monop.arg, new_locals, mapping);
	break;
      case TAGAst:
	remap_free_locals_in_affix_terms (term -> Ast.terms, new_locals, mapping);
	break;
      case TAGVar:
	{ variable old_vdef = term -> Var.vdef;
	  variable new_vdef = remap_variable (old_vdef, new_locals, mapping);
	  if (new_vdef == variable_nil)
	    dcg_internal_error ("remap_free_locals_in_affix_term");
	  term -> Var.vdef = new_vdef;
	};
      case TAGTerminal:
      case TAGInt:
      case TAGReal:
      case TAGText:
      case TAGRegexp: break;
      default: dcg_bad_tag (term -> tag, "remap_free_locals_in_affix_term");
    };
}

static void remap_free_locals_in_affix_terms (affix_term_list terms,
					      variable_list new_locals, int_list mapping)
{ int ix;
  for (ix = 0; ix < terms -> size; ix++)
    remap_free_locals_in_affix_term (terms -> array[ix], new_locals, mapping);
}

static void remap_free_locals_in_member (member mem, variable_list new_locals, int_list mapping)
{ int ix;
  switch (mem -> tag)
    { case TAGRes_guard:
        { res_conf_list rconfs = mem -> Res_guard.rconfs;
	  for (ix = 0; ix < rconfs -> size; ix++)
	    { res_conf rconf = rconfs -> array[ix];
	      remap_free_locals_in_affix_term (rconf -> lhs, new_locals, mapping);
	      remap_free_locals_in_affix_term (rconf -> rhs, new_locals, mapping);
	    };
	}; break;
      case TAGRes_call:
	{ affix_term_list args = mem -> Res_call.args;
	  for (ix = 0; ix < args -> size; ix++)
	    remap_free_locals_in_affix_term (args -> array[ix], new_locals, mapping);
	}; break;
      case TAGRes_term:
	if (mem -> Res_term.arg != affix_term_nil)
	  remap_free_locals_in_affix_term (mem -> Res_term.arg, new_locals, mapping);
      case TAGOp: break;
      default: dcg_bad_tag (mem -> tag, "remap_free_locals_in_member");
    };
}

static void remap_free_locals_in_fwo (fwo_group fwo, variable_list new_locals, int_list mapping)
{ switch (fwo -> tag)
    { case TAGSingle:
	remap_free_locals_in_member (fwo -> Single.mem, new_locals, mapping);
	break;
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    remap_free_locals_in_member (mems -> array[ix], new_locals, mapping);
	}; break;
      default: dcg_bad_tag (fwo -> tag, "remap_free_locals_in_fwo");
    };
}

static void remap_free_locals_in_members (fwo_group_list fwos,
					  variable_list new_locals, int_list mapping)
{ int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    remap_free_locals_in_fwo (fwos -> array[ix], new_locals, mapping);
}

/*
   Transform each alternative of the group into a new definition with its own frame
*/
static void make_new_definitions (group grp, definition_list new_defs,
				  definition anc_def, int_list mapping)
{ variable_list old_locals = anc_def -> locals;
  alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    { alternative alt = alts -> array[ix];
      alternative_list new_alts = new_alternative_list ();
      variable_list new_locals = make_new_locals (old_locals, mapping);
      fpar_list new_fpars = make_new_formal_parameters (alt -> line, alt -> col, new_locals);
      group new_grp = new_group (new_alts);
      definition new_def;

      app_alternative_list (new_alts, attach_alternative (alt));
      remap_free_locals_in_members (alt -> members, new_locals, mapping);
      /* remap_free_locals_in_transduction (alt -> trans, new_locals, mapping); */
      new_def = new_definition (new_fpars, new_locals, new_grp);
      new_def -> pr = anc_def -> pr;
      app_definition_list (new_defs, new_def);
    };
}

static void update_anonymous_call (rule srule, variable_list old_locals, int_list mapping)
{ member call;
  int ix;
  if (srule -> tag == TAGAnonymous_option)
    call = srule -> Anonymous_option.call;
  else call = srule -> Anonymous_group.call;
  for (ix = 0; ix < mapping -> size; ix++)
    { variable old_arg = old_locals -> array[mapping -> array[ix]];
      affix_term new_arg = new_Var (call -> line, call -> col, attach_string (old_arg -> vname));
      new_arg -> Var.vdef = old_arg;
      new_arg -> adef = old_arg -> adef;
      app_affix_term_list (call -> Res_call.args, new_arg);
      app_dir_list (call -> Res_call.adirs, d_unknown);
    };
}

static void update_syntax_rule (rule srule, definition_list defs)
{ if (srule -> tag == TAGAnonymous_option)
    detach_group (&srule -> Anonymous_option.grp);
  else detach_group (&srule -> Anonymous_group.grp);
  srule -> tag = TAGDefs;
  srule -> Defs.defs = defs;
}

/*
   To transform an anonymous rule, we first collect the set of free locals in the
   group, so that we now know the interface of the transformed anonymous rule.
*/
static void transform_anonymous_rule (rule srule, group grp, spec anc_spec, definition anc_def)
{ int_list free_locals = init_int_list (anc_def -> locals -> size);
  definition_list new_defs = init_definition_list (grp -> alts -> size);
  collect_free_locals (grp, free_locals);
  update_signature (srule -> rspec, anc_def -> locals, free_locals);
  make_new_definitions (grp, new_defs, anc_def, free_locals);
  update_anonymous_call (srule, anc_def -> locals, free_locals);
  update_syntax_rule (srule, new_defs);
  detach_int_list (&free_locals);
}

void try_transform_anonymous_rules ()
{ definition anc_def;
  spec anc_spec;
  group grp;
  int ix;
  if (td_parser) return;
  dcg_hint ("      transforming anonymous rules for non topdown parsing");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule srule = all_syntax_rules -> array[ix];
      if (srule -> empty == e_always_produces_empty) continue;
      if (try_pick_anonymous_group_spec_and_def (srule, &grp, &anc_spec, &anc_def))
	transform_anonymous_rule (srule, grp, anc_spec, anc_def);
    };
}
