/*
   File: reachability.c
   Determines which syntax and affix rules are reachable in the start rule
   This is a simple depth first marking, which propagates through to affix rules.

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: reachability.c,v 1.1 2012/01/11 15:43:49 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* libeagbase includes */
#include <ebase_ds.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "reachability.h"

static void mark_in_affix_rule (affix_rule arule);
static void mark_in_affix_element (affix_element elem)
{ if (elem -> tag != TAGAffix_var) return;
  mark_in_affix_rule (elem -> Affix_var.vdef);
}

static void mark_in_affix_alternative (affix_alternative alt)
{ affix_element_list elems = alt -> elems;
  int ix;
  for (ix = 0; ix < elems -> size; ix++)
    mark_in_affix_element (elems -> array[ix]);
}

static void mark_in_affix_rule (affix_rule arule)
{ if (arule -> reachable) return;
  arule -> reachable = 1;
  switch (arule -> tag)
    { case TAGAffix_alts:
	{ affix_alternative_list alts = arule -> Affix_alts.alts;
	  int ix;
	  for (ix = 0; ix < alts -> size; ix++)
	    mark_in_affix_alternative (alts -> array[ix]);
	}; break;
      case TAGAffix_synonym:
	mark_in_affix_rule (arule -> Affix_synonym.syndef);
      case TAGAffix_prim: break;
      default: dcg_bad_tag (arule -> tag, "mark_in_affix_rule");
    };
}

static void mark_in_affix_term (affix_term term);
static void mark_in_affix_terms (affix_term_list terms)
{ int ix;
  for (ix = 0; ix < terms -> size; ix++)
    mark_in_affix_term (terms -> array[ix]);
}

static void mark_in_affix_term (affix_term term)
{ if (term == affix_term_nil) return;
  mark_in_affix_rule (term -> adef);
  switch (term -> tag)
    { case TAGDyop:
	mark_in_affix_term (term -> Dyop.arg1);
	mark_in_affix_term (term -> Dyop.arg2);
	break;
      case TAGMonop:
	mark_in_affix_term (term -> Monop.arg);
	break;
      case TAGAst:
	mark_in_affix_terms (term -> Ast.terms);
	break;
      case TAGVar:
	mark_in_affix_rule (term -> Var.vdef -> adef);
      case TAGTerminal:
      case TAGInt:
      case TAGReal:
      case TAGText:
      case TAGRegexp:
	break;
      default: dcg_bad_tag (term -> tag, "mark_in_affix_term");
    };
}

static void mark_in_rule (rule srule);
static void mark_in_member (member m)
{ int ix;
  switch (m -> tag)
    { case TAGRes_guard:
	{ res_conf_list rconfs = m -> Res_guard.rconfs;
	  for (ix = 0; ix < rconfs -> size; ix++)
	    { res_conf rconf = rconfs -> array[ix];
	      mark_in_affix_term (rconf -> lhs);
	      mark_in_affix_term (rconf -> rhs);
	    };
	}; break;
      case TAGRes_call:
	{ affix_term_list args = m -> Res_call.args;
	  mark_in_rule (m -> Res_call.rdef);
	  for (ix = 0; ix < args -> size; ix++)
	    mark_in_affix_term (args -> array[ix]);
	}; break;
      case TAGRes_term:
	mark_in_affix_term (m -> Res_term.arg);
      case TAGOp: break;
      default: dcg_bad_tag (m -> tag, "mark_in_member");
    }
}

static void mark_in_alternative (alternative alt)
{ /* Check when we introduce transduction */
  fwo_group_list fwos = alt -> members;
  int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    { fwo_group fwo = fwos -> array[ix];
      switch (fwo -> tag)
	{ case TAGSingle:
	    mark_in_member (fwo -> Single.mem);
	    break;
	  case TAGFwo:
	    { member_list mems = fwo -> Fwo.mems;
	      int iy;
	      for (iy = 0; iy < mems -> size; iy++)
		mark_in_member (mems -> array[iy]);
	    };
	    break;
	  default: dcg_bad_tag (fwo -> tag, "mark_in_alternative");
	};
    };
}

static void mark_in_group (group grp)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    mark_in_alternative (alts -> array[ix]);
}

static void mark_in_fpar_list (fpar_list fpars)
{ int ix;
  for (ix = 0; ix < fpars -> size; ix++)
    { mark_in_affix_rule (fpars -> array[ix] -> adef);
      mark_in_affix_term (fpars -> array[ix] -> fexp -> array[0]);
    };
}

static void mark_in_definition (definition def)
{ mark_in_fpar_list (def -> lhs_pars);
  mark_in_group (def -> grp);
}

static void mark_in_spec (spec rspec)
{ affix_rule_list rsig = rspec -> rsig;
  int ix;
  for (ix = 0; ix < rsig -> size; ix++)
    mark_in_affix_rule (rsig -> array[ix]);
}

static void mark_in_rule (rule srule)
{ if (srule == rule_nil)
    dcg_abort ("mark_in_rule", "null rule");
  if (srule -> reachable) return;
  srule -> reachable = 1;
  mark_in_spec (srule -> rspec);
  switch (srule -> tag)
    { case TAGDefs:
	{ definition_list defs = srule -> Defs.defs;
	  int ix;
	  for (ix = 0; ix < defs -> size; ix++)
	    mark_in_definition (defs -> array[ix]);
	}; 
      case TAGExt_rule:
      case TAGQuasi_rule: break;
      case TAGAnonymous_option: mark_in_group (srule -> Anonymous_option.grp); break;
      case TAGAnonymous_group:  mark_in_group (srule -> Anonymous_group.grp); break;
      default: dcg_bad_tag (srule -> tag, "mark_in_rule");
    };
}

static void mark_affix_rule_synonyms ()
{ int ix;
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      if (arule -> tag != TAGAffix_synonym) continue;
      arule -> reachable = arule -> Affix_synonym.syndef -> reachable;
    };
}

static void dump_reachability ()
{ int ix;
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      dcg_wlog ("Affix rule %s is %sreachable", arule -> aname,
		(arule -> reachable)?"":"not ");
    };

  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule srule = all_syntax_rules -> array[ix];
      spec rspec = srule -> rspec;
      dcg_wlog ("Rule %s is %sreachable", rspec -> canonic_name,
		(srule -> reachable)?"":"not ");
    };
}

void determine_reachability ()
{ dcg_hint ("      determining reachability");
  mark_in_rule (root_rule);
  mark_affix_rule_synonyms ();
  if (dump_properties)
    dump_reachability ();
}
