/*
   File: parser.c
   Defines a parser for EAG3

   Copyright (C) 2009-2010 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: parser.c,v 1.24 2012/09/26 13:09:41 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>
#include <dcg_plist_ops.h>

/* local includes */
#include "eag_ds.h"
#include "globals.h"
#include "options.h"
#include "lexer.h"
#include "parser.h"

/* error administration */
static void parser_error (char *format, ...)
{ string fname;
  int line, col;
  char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  get_current_filename (&fname);
  get_current_position (&line, &col);
  dcg_error (0, "File %s, line %d, col %d: %s", fname, line, col, buf);
};

static void parser_error_at (int line, int col, char *format, ...)
{ string fname;
  char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  get_current_filename (&fname);
  dcg_error (0, "File %s, line %d, col %d: %s", fname, line, col, buf);
};

/*
   Forward declarations to build parser CDL3 style
*/
static int is_grammar (string gname, int main, grammar *gra);
static int is_grammar_head (string gname);
static void should_be_grammar_body (grammar gra);
static int is_grammar_body (grammar gra);
static int is_statement (grammar gra);
static int is_lexicon_interface (grammar gra);
static int is_fact_table_interface (grammar gra);
static int is_triples_interface (grammar gra);
static int is_defines_part (grammar gra);
static int is_uses_part (grammar gra);
static void should_be_module_name_list (string_list sl);
static int is_module_name_list (string_list sl);
static void should_be_lex_defines_part (prule_list sl, rule_kind rkind, rule_type rtype);
static int is_lex_defines_part (prule_list sl, rule_kind rkind, rule_type rtype);
static void should_be_typed_specification_head_list (grammar gra);
static int is_typed_specification_head_list (grammar gra);
static void should_be_typed_specification_head (grammar gra);
static int is_typed_specification_head (grammar gra);
static void should_be_specification_head_list (prule_list sl, rule_layer rlayer,
					       rule_kind rkind, rule_type rtype);
static int is_specification_head_list (prule_list sl, rule_layer rlayer,
				       rule_kind rkind, rule_type rtype);
static void should_be_specification_head (prule_list sl, rule_layer rlayer,
					  rule_kind rkind, rule_type rtype);
static int is_specification_head (prule_list sl, rule_layer rlayer,
				  rule_kind rkind, rule_type rtype);
static int is_layer_aspect (rule_layer *rlayer);
static void may_be_rule_type (rule_type *rrtype);
static int is_rule_type (rule_type *rrtype);
static int is_root_call (grammar gra);
static int is_rule_or_specification (grammar gra);
static int is_name_list (name_list *ids);
static void should_be_name (int *line, int *col, string_list *id);
static int is_name (int *line, int *col, string_list *id);
static void should_be_rest_syntax_rule_or_specification (grammar gra,
							 rule_layer rlayer, rule_type rtype);
static void should_be_rest_quasi_definition (grammar gra, rule_layer rlayer, rule_type rtype);
static int is_rest_syntax_rule_or_specification (grammar gra, rule_layer rlayer, rule_type rtype);
static void promote_name_list_to_lhs_list (name_list ids, plhs_list *rlhss);
static int is_rule_lhs_list (plhs_list *rlhss, rule_type rtype);
static void may_be_rest_rule_lhs_list (plhs_list rlhss, rule_type rtype);
static void should_be_rule_lhs (plhs *rlhs, rule_type rtype);
static int is_rule_lhs (plhs *rlhs, rule_type rtype);
static void may_be_rest_rule_lhs (plhs lhs);
static void may_be_formal_parameter_list_pack (plhs lhs);
static int is_formal_parameter_list_pack (plhs lhs);
static void should_be_formal_parameter_list (plhs lhs);
static int is_formal_parameter_list (plhs lhs);
static void should_be_formal_parameter (fpar *rpar);
static int is_formal_parameter (fpar *rpar);
static void should_be_syntax_rule_or_specification_rhs (grammar gra, plhs_list lhss);
static int is_syntax_rule_or_specification_rhs (grammar gra, plhs_list lhss);
static void expand_specifications (grammar gra, plhs_list lhss);
static void expand_external_definitions (grammar gra, plhs_list lhss, string ename);
static void expand_rule_definitions (grammar gra, plhs_list lhss, alternative_list alts);
static void may_be_alternatives (int in_option, alternative_list *alts);
static void may_be_alternative (int in_option, alternative *alt);
static int may_be_frequency_or_option (real *freq, member *mem);
static void may_be_rest_alternative_syntax_part (int line, int col, real frequency,
						 member first_mem, alternative *alt);
static void should_be_tail_free_word_order_groups (glue gl, fwo_group_list fwos);
static int is_glue (glue *gl);
static int should_be_fwo_group_or_control_operator (glue gl, fwo_group *fwo);
static int is_fwo_group (glue gl, fwo_group *fwo);
static void may_be_rest_fwo_group (glue gl, member first_mem, fwo_group *fwo);
static void may_be_control_operator (member *mem);
static int is_control_operator (member *mem);
static void should_be_member (member *mem);
static int is_member (member *mem);
static int is_option (member *mem);
static int is_group (member *mem);
static void should_be_call (member *mem);
static int is_call (member *mem);
static void may_be_rest_call (member mem);
static int is_call_actual_parameter_list_pack (member mem);
static void should_be_call_actual_parameter_list (member mem);
static int is_call_actual_parameter_list (member mem);
static void may_be_actual_parameter_list_pack (apar_list *rpars);
static void should_be_actual_parameter_list (apar_list *rpars);
static int is_actual_parameter_list (apar_list *rpars);
static void may_be_actual_parameter_pack (apar *rpar);
static void should_be_actual_parameter (apar *rpar);
static int is_actual_parameter (apar *rpar);
static int is_guard (member *mem);
static void should_be_confrontation (confrontation *conf);
static int is_confrontation (confrontation *conf);
static int is_terminal (member *mem);
static int is_quasi_terminal (member *mem);
static void should_be_affix_expression (affix_term_list *terms);
static int is_affix_expression (affix_term_list *terms);
static int is_affix_term (affix_term *term);
static int is_affix_formula_of_prio (int prio, affix_term *term);
static int is_arithmetic_prio_operator (int prio, operator *rop);
static void should_be_affix_monadic_formula (affix_term *term);
static int is_affix_monadic_formula (affix_term *term);
static int is_monadic_operator (operator *rop);
static int is_affix_primary (affix_term *term);
static void may_be_alternative_transduction_part (int in_option, alternative alt);
static void should_be_transduction_members (int in_option, trans_member_list tr_mems, glue gl);
static int is_transduction_members (int in_option, trans_member_list tr_mems, glue gl);
static void may_be_rest_transduction_members (int in_option, trans_member_list tr_mems);
static int is_transduction_member_or_output_operator (int in_option, glue gl, trans_member *tmem);
static int is_call_placeholder (glue gl, trans_member *tmem);
static int is_quasi_terminal_placeholder (glue gl, trans_member *tmem);
static int is_group_placeholder (glue gl, trans_member *tmem);
static int is_option_placeholder_or_hybrid_square_open (glue gl, trans_member *tmem);
static int is_text_transduction (glue gl, trans_member *tmem);
static int is_other_hybrid_operator (int in_option, glue gl, trans_member *tmem);
static void may_be_dot_number (int *nr);
static void should_be_rest_affix_rule (grammar gra, name_list ids, int is_affix_definition);
static void verify_affix_name_list (name_list ids);
static void expand_lhs_synonyms (affix_rule_list arules, name_list anames);
static void may_be_affix_alternatives (affix_alternative_list *alts);
static void may_be_affix_alternative (affix_alternative *alt);
static void should_be_rest_affix_union (affix_alternative *alt, affix_element_list elems);
static int is_rest_affix_union (affix_alternative *alt, affix_element_list elems);
static void should_be_rest_affix_partition (affix_alternative *alt, affix_element_list elems);
static int is_rest_affix_partition (affix_alternative *alt, affix_element_list elems);
static void should_be_rest_affix_concatenation (affix_alternative *alt, affix_element_list elems);
static int is_rest_affix_concatenation (affix_alternative *alt, affix_element_list elems);
static void may_be_rest_affix_sequence (affix_alternative *alt, affix_element_list elems);
static void should_be_affix_element (affix_element *elem);
static int is_affix_element (affix_element *elem);
static void should_be_invisible_nonterminal ();
static int is_invisible_nonterminal ();

/*
   A.2 Grammars

   An EAG3 grammar consists of a main grammar which may contain the root of the grammar
   together with a number (possibly 0) of subgrammars and lexicons. If the root rule
   is absent, an anonymous one is derived from the first syntax rule in the main grammar.

   The parser is commented by giving the relevant grammar rules
   In these rules the following abbreviations are used:
  
   NOTION option:   NOTION; .
   NOTION sequence: NOTION, NOTION sequence option.
   NOTION list:     NOTION, comma, NOTION list.
   NOTION pack:     left parenthesis, NOTION, right parenthesis.

   a) main grammar:
        grammar head, grammar body.

   b) sub grammar:
        grammar head option, grammar body.
    
   c) grammar head:
        "GRAMMAR", grammar name, ".".
*/
void should_be_grammar (string gname, int main, grammar *gra)
{ if (is_grammar (gname, main, gra))
    { if (!end_of_file ())
	parser_error ("garbage at or after end of main grammar");
      return;
    };
  parser_error ("grammar expected");
};

static int is_grammar (string gname, int main, grammar *gra)
{ string fname;
  if (!main) (void) is_grammar_head (gname);
  else if (!is_grammar_head (gname)) return (0);
  get_current_filename (&fname);
  *gra = new_grammar (attach_string (fname), attach_string (gname), new_string_list (),
		      new_string_list (), new_string_list (), new_string_list (),
		      new_prule_list (), new_prule_list (), new_affix_rule_list (), member_nil);
  should_be_grammar_body (*gra);
  return (1);
}

static int is_grammar_head (string gname)
{ int line, col;
  string gname2;
  if (!is_keyword ("GRAMMAR")) return (0);
  get_current_position (&line, &col);
  should_be_module_name (&gname2);
  if (!streq (gname, gname2))
    parser_error_at (line, col, "grammar name %s differs from file name %s", gname2, gname);
  should_be_symbol (".");
  return (1);
}

/* 
   c) grammar body:
        statement sequence.

   d) statement:
        pragmat;
        lexicon interface;
        fact table interface;
        triples interface;
        defines part;
        uses part;
        root statement;
        specification;
        syntax rule;
        affix rule.
*/ 
static void should_be_grammar_body (grammar gra)
{ if (is_grammar_body (gra)) return;
  parser_error ("grammar body expected");
}

static int is_grammar_body (grammar gra)
{ if (!is_statement (gra)) return (0);
  while (is_statement (gra)) ;
  return (1);
}

static int is_statement (grammar gra)
{ if (is_pragmat ()) return (1);
  if (is_lexicon_interface (gra)) return (1);
  if (is_fact_table_interface (gra)) return (1);
  if (is_triples_interface (gra)) return (1);
  if (is_defines_part (gra)) return (1);
  if (is_uses_part (gra)) return (1);
  if (is_root_call (gra)) return (1);
  if (is_rule_or_specification (gra)) return (1);
  return (0);
}

/*
   A.2.2 Interfaces

   a) lexicon interface:
        "LEXICON", lexicon name list, lex defines part, ".".
       
   b) fact table interface:
        "TABLE", fact table name list, lex defines part, ".". 

   c) triples interface:
        "TRIPLES", triples database name list, ".".

   d) defines part:
        "DEFINES", typed specification head list, ".".
      
   e) uses part:
        "USES", grammar name list, ".";
        "INCLUDES", grammar name list, ".".

   e) lex defines part:
        "DEFINES", specification head list.
*/
static int is_lexicon_interface (grammar gra)
{ if (!is_keyword ("LEXICON")) return (0);
  should_be_module_name_list (gra -> lexica);
  should_be_lex_defines_part (gra -> prules, r_lexicon, r_rule);	/* never empty */
  should_be_symbol (".");
  return (1);
}

static int is_fact_table_interface (grammar gra)
{ if (!is_keyword ("TABLE")) return (0);
  should_be_module_name_list (gra -> fact_tables);
  should_be_lex_defines_part (gra -> prules, r_fact, r_predicate);	/* always empty */
  should_be_symbol (".");
  return (1);
}

static int is_triples_interface (grammar gra)
{ if (!is_keyword ("TRIPLES")) return (0);
  should_be_module_name_list (gra -> triple_databases);
  should_be_symbol (".");
  return (1);
}

static int is_defines_part (grammar gra)
{ if (!is_keyword ("DEFINES")) return (0);
  should_be_typed_specification_head_list (gra);        /* Both affix rule and rule exports */
  should_be_symbol (".");
  return (1);
}

static int is_uses_part (grammar gra)
{ if (is_keyword ("USES") || is_keyword ("INCLUDES"))	/* AGFL compatibility */
    { should_be_module_name_list (gra -> uses);
      should_be_symbol (".");
      return (1);
    };
  return (0);
}

static void should_be_module_name_list (string_list sl)
{ if (is_module_name_list (sl)) return;
  parser_error ("module name expected");
}

static int is_module_name_list (string_list sl)
{ string mname;
  if (!is_module_name (&mname)) return (0);
  add_uniquely_to_string_list (sl, mname);
  while (is_symbol (","))
    { should_be_module_name (&mname);
      add_uniquely_to_string_list (sl, mname);
    };
  return (1);
}

static void should_be_lex_defines_part (prule_list sl, rule_kind rkind, rule_type rtype)
{ if (is_lex_defines_part (sl, rkind, rtype)) return;
  parser_error ("DEFINES expected");
}

static int is_lex_defines_part (prule_list sl, rule_kind rkind, rule_type rtype)
{ if (!is_keyword ("DEFINES")) return (0);
  should_be_specification_head_list (sl, r_lex, rkind, rtype);
  return (1);
}

/*
   A.2.3 Specifications

   Note: When parsing a DEFINES part we know to expect a typed specification list.
	 When parsing a lex DEFINES part we know to expect a specification head list.
         When parsing the body of the grammar or a defines part we still cannot
	 recognize from a name whether it is a rule name or an affix rule name except
	 by circumstantial evidence.

   a) specification:
        layer aspect option, rule type option, specification head list, ".".

   b) typed specification head:
        affix name;
        layer aspect option, rule type option, specification head.

   c) specification head:
        name part, rest specification option.

   d) rest specification:
        name part, rest specification option;
        formal parameter list pack, rest strict specification option.

   e) rest strict specification:
        name part, rest specification option.

   f) layer aspect:
        "TERMINAL";
        "LEX".

   g) rule type:
        "RULE";			-- always generating
	"OPT";			-- sometimes generating
        "OPTION";		-- sometimes generating
        "COND";			-- never generating (AGFL compatibility)
        "CONDITION";		-- never generating (AGFL compatibility)
        "PREDICATE";		-- never generating (EAG/2vWG extended)
*/
static void should_be_typed_specification_head_list (grammar gra)
{ if (is_typed_specification_head_list (gra)) return;
  parser_error ("affix rule or rule specification head expected");
}

static int is_typed_specification_head_list (grammar gra)
{ if (!is_typed_specification_head (gra)) return (0);
  while (is_symbol (","))
    should_be_typed_specification_head (gra);
  return (1);
}

static void should_be_typed_specification_head (grammar gra)
{ if (is_typed_specification_head (gra)) return;
  parser_error ("affix rule or rule specification head expected");
}

static int is_typed_specification_head (grammar gra)
{ rule_layer rlayer;
  rule_type rtype;
  if (is_layer_aspect (&rlayer))
    { may_be_rule_type (&rtype);
      should_be_specification_head (gra -> exp_prules, rlayer, r_normal, rtype);
    }
  else if (is_rule_type (&rtype))
    should_be_specification_head (gra -> exp_prules, r_syntax, r_normal, rtype);
  else if (!is_specification_head (gra -> exp_prules, r_syntax, r_normal, r_unknown))
    return (0);
  return (1);
}

static void should_be_specification_head_list (prule_list sl, rule_layer rlayer,
					       rule_kind rkind, rule_type rtype)
{ if (is_specification_head_list (sl, rlayer, rkind, rtype)) return;
  parser_error ("specification head expected");
}

static int is_specification_head_list (prule_list sl, rule_layer rlayer,
				       rule_kind rkind, rule_type rtype)
{ if (!is_specification_head (sl, rlayer, rkind, rtype)) return (0);
  while (is_symbol (","))
    should_be_specification_head (sl, rlayer, rkind, rtype);
  return (1);
};

static void should_be_specification_head (prule_list sl, rule_layer rlayer,
					  rule_kind rkind, rule_type rtype)
{ if (is_specification_head (sl, rlayer, rkind, rtype)) return;
  parser_error ("specification head expected");
}

static int is_specification_head (prule_list sl, rule_layer rlayer,
				  rule_kind rkind, rule_type rtype)
{ string_list rname_parts;
  int_list rname_chars;
  int line, col;
  string rname;
  prule nspec;
  plhs lhs;

  get_current_position (&line, &col);
  if (!is_name_part (&rname)) return (0);
  rname_parts = new_string_list ();
  rname_chars = new_int_list ();
  app_string_list (rname_parts, rname);
  app_int_list (rname_chars, 1);
  lhs = new_plhs (line, col, rlayer, rkind, rtype, rname_parts, rname_chars, new_fpar_list ());
  nspec = new_Pspec (lhs);
  app_prule_list (sl, nspec);
  may_be_rest_rule_lhs (lhs);
  return (1);
}

static int is_layer_aspect (rule_layer *rlayer) 
{ if (is_keyword ("TERMINAL"))	*rlayer = r_terminal;
  else if (is_keyword ("LEX"))	*rlayer = r_lex;
  else return (0);
  return (1);
}

static void may_be_rule_type (rule_type *rrtype)
{ if (is_rule_type (rrtype)) return;
  *rrtype = r_unknown;
}

static int is_rule_type (rule_type *rrtype)
{ if (is_keyword ("RULE"))           *rrtype = r_rule; 
  else if (is_keyword ("OPT"))       *rrtype = r_option;
  else if (is_keyword ("OPTION"))    *rrtype = r_option;
  else if (is_keyword ("COND"))      *rrtype = r_predicate;
  else if (is_keyword ("CONDITION")) *rrtype = r_predicate;
  else if (is_keyword ("PREDICATE")) *rrtype = r_predicate;
  else return (0);
  return (1);
}

/*
   A.2.4 The Root call

   a) root call:
        "ROOT", call, "."
*/
static int is_root_call (grammar gra)
{ member mem;
  if (!is_keyword ("ROOT")) return (0);
  should_be_call (&mem);
  should_be_symbol (".");
  if (gra -> root_call == member_nil) gra -> root_call = mem;
  else parser_error ("Multiple root call found");
  return (1);
}

/* 
   A.3 Rules
  
   a) syntax rule:
        rule type option, rule lhs list, ":", "EXTERNAL", text constant, ".".
        rule type option, rule lhs list, ":", group, ".".

   b) specification:
        rule type option, specification head list, ".".

   c) rule lhs:
        name part, rest rule lhs option.

   d) rest rule lhs:
        name part, rest rule lhs option;
        formal parameter list pack, rest strict rule lhs option.

   e) rest strict rule lhs:
        name part, rest rule lhs option.

   f) formal parameter:
        direction, affix expression;
        affix expression, direction option.

   g) direction:
        ">".

   Since we have adopted the AGFL lexical conventions for affix nonterminals
   and rules, we cannot directly determine if our current symbol is the start
   of an affix rule, a syntax rule or a specification. We use the same strategy
   as in the AGFL parser to resolve these ambiguities.
*/ 
static int is_rule_or_specification (grammar gra)
{ rule_layer rlayer = r_syntax;
  rule_type rtype = r_unknown;
  int is_affix_definition;
  plhs_list lhss;
  name_list ids;
  if (is_layer_aspect (&rlayer))
    { may_be_rule_type (&rtype);
      should_be_rest_syntax_rule_or_specification (gra, rlayer, rtype);
      return (1);
    };
  if (is_rule_type (&rtype))
    { should_be_rest_syntax_rule_or_specification (gra, rlayer, rtype);
      return (1);
    };

  if (!is_name_list (&ids)) return (0);
  is_affix_definition = is_symbol ("::");
  if (is_affix_definition || is_symbol ("="))
    { should_be_rest_affix_rule (gra, ids, is_affix_definition);
      detach_name_list (&ids);
      return (1);
    };

  /* Promote name list to lhs list and continue */
  promote_name_list_to_lhs_list (ids, &lhss);
  may_be_rest_rule_lhs (lhss -> array[lhss -> size - 1]);
  may_be_rest_rule_lhs_list (lhss, rtype);
  should_be_syntax_rule_or_specification_rhs (gra, lhss);
  nonrec_detach_plhs_list (&lhss);
  detach_name_list (&ids);
  return (1);
}

static int is_name_list (name_list *ids)
{ int line, col;
  string_list id;
  if (!is_name (&line, &col, &id)) return (0);
  *ids = new_name_list ();
  app_name_list (*ids, new_name (line, col, id));
  while (is_symbol (","))
    { should_be_name (&line, &col, &id);
      app_name_list (*ids, new_name (line, col, id));
    };
  return (1);
}

static void should_be_name (int *line, int *col, string_list *id)
{ if (is_name (line, col, id)) return;
  *id = new_string_list ();
  parser_error ("name expected");
}

static int is_name (int *line, int *col, string_list *id)
{ string part;
  get_current_position (line, col);
  if (!is_name_part (&part)) return (0);
  *id = new_string_list ();
  app_string_list (*id, part);
  while (is_name_part (&part))
    app_string_list (*id, part);
  return (1);
}

static void should_be_rest_syntax_rule_or_specification (grammar gra,
							 rule_layer rlayer, rule_type rtype)
{ if (is_predefs_spec () && is_symbol ("$"))
    { should_be_rest_quasi_definition (gra, rlayer, rtype);
      return;
    };
  if (is_rest_syntax_rule_or_specification (gra, rlayer, rtype)) return;
  parser_error ("rule name expected");
}

static void should_be_rest_quasi_definition (grammar gra, rule_layer rlayer, rule_type rtype)
{ string_list rname_parts = new_string_list ();
  int_list rname_chars = new_int_list ();
  string bname, dbname;
  plhs quasi_lhs;
  int line, col;
  string qual;
  prule qrule;

  get_current_position (&line, &col);		/* position of MATCH, etc */
  should_be_bold_name (&bname);
  dbname = dcg_new_fmtd_string ("$%s", bname);
  detach_string (&bname);
  app_string_list (rname_parts, dbname);	/* Make sure $ heads name part */
  app_int_list (rname_chars, 1);
  quasi_lhs = new_plhs (line, col, rlayer, r_external, rtype,
		        rname_parts, rname_chars, new_fpar_list ());
  may_be_formal_parameter_list_pack (quasi_lhs);
  should_be_symbol (":");
  should_be_keyword ("EXTERNAL");
  should_be_text_constant (&qual);
  should_be_symbol (".");
  qrule = new_Quasi_prule (quasi_lhs, qual);
  app_prule_list (gra -> prules, qrule);
}

static int is_rest_syntax_rule_or_specification (grammar gra, rule_layer rlayer, rule_type rtype)
{ plhs_list lhss;
  if (!is_rule_lhs_list (&lhss, rtype)) return (0);
  should_be_syntax_rule_or_specification_rhs (gra, lhss);
  nonrec_detach_plhs_list (&lhss);
  return (1);
}

static void promote_name_list_to_lhs_list (name_list ids, plhs_list *rlhss)
{ int ix;
  *rlhss = init_plhs_list (ids -> size);
  for (ix = 0; ix < ids -> size; ix++)
    { int line = ids -> array[ix] -> line;
      int col = ids -> array[ix] -> col;
      string_list id = ids -> array[ix] -> id;
      int nparts = id -> size;
      string_list rname_parts = init_string_list (nparts);
      int_list rname_chars = init_int_list (nparts);
      fpar_list fpars = new_fpar_list ();
      plhs lhs = new_plhs (line, col, r_syntax, r_normal, r_unknown,
			   rname_parts, rname_chars, fpars);
      int iy;
      for (iy = 0; iy < nparts; iy++)
	{ app_string_list (rname_parts, attach_string (id -> array[iy]));
	  app_int_list (rname_chars, 1);
	};
      app_plhs_list (*rlhss, lhs);
    }
}

static int is_rule_lhs_list (plhs_list *rlhss, rule_type rtype)
{ plhs lhs;
  if (!is_rule_lhs (&lhs, rtype)) return (0);
  *rlhss = new_plhs_list ();
  app_plhs_list (*rlhss, lhs);
  may_be_rest_rule_lhs_list (*rlhss, rtype);
  return (1);
}

static void may_be_rest_rule_lhs_list (plhs_list rlhss, rule_type rtype)
{ plhs lhs;
  while (is_symbol (","))
    { should_be_rule_lhs (&lhs, rtype);
      app_plhs_list (rlhss, lhs);
    };
}

static void should_be_rule_lhs (plhs *rlhs, rule_type rtype)
{ if (is_rule_lhs (rlhs, rtype)) return;
  parser_error ("rule left hand side expected");
  *rlhs = plhs_nil;
}

static int is_rule_lhs (plhs *rlhs, rule_type rtype)
{ string_list rname_parts;
  int_list rname_chars;
  int line, col;
  string rname;
  get_current_position (&line, &col);
  if (!is_name_part (&rname)) return (0);
  rname_parts = new_string_list ();
  rname_chars = new_int_list ();
  app_string_list (rname_parts, rname);
  app_int_list (rname_chars, 1);
  *rlhs = new_plhs (line, col, r_syntax, r_normal, rtype,
		    rname_parts, rname_chars, new_fpar_list ());
  may_be_rest_rule_lhs (*rlhs);
  return (1);
}

static void may_be_rest_rule_lhs (plhs lhs)
{ while (1)
    { int had_name_part = lhs -> rname_chars -> array[lhs -> rname_chars -> size - 1];
      string pname;
      if (is_name_part (&pname))
        { app_string_list (lhs -> rname_parts, pname);
	  app_int_list (lhs -> rname_chars, 1);
    	}
      else if (had_name_part && is_formal_parameter_list_pack (lhs)) ;
      else break;
    };
}

static void may_be_formal_parameter_list_pack (plhs lhs)
{ (void) is_formal_parameter_list_pack (lhs);
}

static int is_formal_parameter_list_pack (plhs lhs)
{ if (!is_symbol ("(")) return (0); 
  should_be_formal_parameter_list (lhs);
  should_be_symbol (")");
  return (1);
}

static void should_be_formal_parameter_list (plhs lhs)
{ if (is_formal_parameter_list (lhs)) return;
  parser_error ("formal parameter expected");
}

static int is_formal_parameter_list (plhs lhs)
{ fpar par;
  if (!is_formal_parameter (&par)) return (0);
  app_fpar_list (lhs -> lhs_pars, par);
  app_int_list (lhs -> rname_chars, 0);
  while (is_symbol (","))
    { should_be_formal_parameter (&par);
      app_fpar_list (lhs -> lhs_pars, par);
      app_int_list (lhs -> rname_chars, 0);
    };
  return (1);
}

static void should_be_formal_parameter (fpar *rpar)
{ if (is_formal_parameter (rpar)) return;
  parser_error ("formal parameter expected");
  *rpar = fpar_nil;
}

static int is_formal_parameter (fpar *rpar)
{ int predef = is_predefs_spec ();
  dir fdir = d_unknown;
  affix_term_list fexp;
  int line, col;

  get_current_position (&line, &col);
  if (predef && is_keyword ("STATIC"))
    { should_be_affix_expression (&fexp);
      fdir = d_static;
    }
  else if (predef && is_keyword ("REGEXP"))
    { should_be_affix_expression (&fexp);
      fdir = d_regexp;
    }
  else if (is_symbol (">"))
    { should_be_affix_expression (&fexp);
      fdir = d_inherited;
    }
  else if (is_affix_expression (&fexp))
    { if (is_symbol (">"))
	fdir = d_derived;
    }
  else return (0);
  *rpar = new_fpar (line, col, fdir, fexp);
  return (1);
}

static void should_be_syntax_rule_or_specification_rhs (grammar gra, plhs_list lhss)
{ if (is_syntax_rule_or_specification_rhs (gra, lhss)) return;
  parser_error ("':' or '.' expected");
}

static int is_syntax_rule_or_specification_rhs (grammar gra, plhs_list lhss)
{ if (is_symbol ("."))
    { expand_specifications (gra, lhss);
      return (1);
    }
  else if (!is_symbol (":")) return (0);
  if (is_keyword ("EXTERNAL"))
    { string ename;
      should_be_text_constant (&ename);
      should_be_symbol (".");
      expand_external_definitions (gra, lhss, ename);
    }
  else
    { alternative_list alts;   
      may_be_alternatives (0, &alts);
      should_be_symbol (".");
      expand_rule_definitions (gra, lhss, alts);
    };
  return (1);
}

static void expand_specifications (grammar gra, plhs_list lhss)
{ int ix;
  for (ix = 0; ix < lhss -> size; ix++)
    { plhs this_lhs = attach_plhs (lhss -> array[ix]);
      prule nrule = new_Pspec (this_lhs);
      app_prule_list (gra -> prules, nrule);
    };
}

static void expand_external_definitions (grammar gra, plhs_list lhss, string ename)
{ int ix;
  for (ix = 0; ix < lhss -> size; ix++)
    { plhs this_lhs = attach_plhs (lhss -> array[ix]);
      prule nrule = new_Ext_prule (this_lhs, attach_string (ename));
      this_lhs -> rkind = r_external;
      app_prule_list (gra -> prules, nrule);
    };
}

static void expand_rule_definitions (grammar gra, plhs_list lhss, alternative_list rhs_alts)
{ int ix;
  for (ix = 0; ix < lhss -> size; ix++)
    { plhs this_lhs = attach_plhs (lhss -> array[ix]);
      group rhs_grp = new_group (rdup_alternative_list (rhs_alts));
      prule nrule = new_Proto_rule (this_lhs, rhs_grp);
      app_prule_list (gra -> prules, nrule);
    };
}

/* 
   A.3.2 Alternatives
  
   a) alternatives:
	alternative, ";", alternatives
	alternative, "!", alternatives
        alternative.

   b) alternative:
	frequency option, alternative syntax part, alternative transduction part.

   c) frequency part:	
        "[", number, "]".

   d) alternative syntax part:
	free word order group, ",", tail free word order groups;
	free word order group, "+", tail free word order groups;
	free word order group;
        control operator option.

   e) tail free word order groups:
        free word order group, ",", tail free word order groups;
	free word order group, "+", tail free word order groups;
	free word order group;
	control operator.

   f) free word order group:
        member, "&", free word order group;
	member.

   Note1: We need some left factorisation to distinguish a frequency part from an option

   Note2: We accept commits to be compatible with AGFL, store them in the previous alternative
          but we are going to ignore them, except in predicates, since they interfere with
	  the leftcorner parsing process.

   Note3: Since the ']' output operator for hybrid parsing interferes with the parsing of
          options, we must use the same heuristic as applied in AGFL namely by remembering
          that we are currently parsing an option
*/
static void may_be_alternatives (int in_option, alternative_list *alts)
{ alternative alt;
  int commit;
  may_be_alternative (in_option, &alt);
  *alts = new_alternative_list ();
  app_alternative_list (*alts, alt);
  while ((commit = is_symbol ("!")) || is_symbol (";"))
    { if (commit) alt -> commited = 1;
      may_be_alternative (in_option, &alt);
      app_alternative_list (*alts, alt);
    };
}

static void may_be_alternative (int in_option, alternative *alt)
{ int line, col, is_option;
  real frequency;
  member mem;

  /* recognize folded alternative syntax part */
  get_current_position (&line, &col);
  is_option = may_be_frequency_or_option (&frequency, &mem);

  if (is_option)
    may_be_rest_alternative_syntax_part (line, col, frequency, mem, alt);
  else if (is_member (&mem))
    may_be_rest_alternative_syntax_part (line, col, frequency, mem, alt);
  else
    { fwo_group_list fwo_gl = new_fwo_group_list ();
      may_be_control_operator (&mem);
      app_fwo_group_list (fwo_gl, new_Single (g_none, mem));
      *alt = new_alternative (line, col, frequency, fwo_gl, 0, trans_member_list_nil);
    };
  may_be_alternative_transduction_part (in_option, *alt);
}

/*
   We have to do some left factorisation here
*/
static int may_be_frequency_or_option (real *freq, member *mem)
{ int inum, number_kind;
  alternative_list alts;
  int line, col;
  real rnum;
  *freq = -1.0;					/* Unspecified frequency */
  if (!is_symbol ("[")) return (0);

  /* Option or frequency: the number will tell */
  get_current_position (&line, &col);
  if (is_number (&number_kind, &inum, &rnum))
    { if (!number_kind) rnum = (real) inum;
      *freq = rnum;
      should_be_symbol ("]");
      return (0);
    };

  /* We saw the opening of an option */
  may_be_alternatives (1, &alts);
  should_be_symbol ("]");
  *mem = new_Option (line, col, new_group (alts));
  return (1);
}

static void may_be_rest_alternative_syntax_part (int line, int col, real frequency,
						 member first_mem, alternative *alt)
{ fwo_group_list fwos = new_fwo_group_list ();
  fwo_group first_fwo;
  glue gl;

  /*
     We have seen a first member, check for the rest of an fwo group
  */
  may_be_rest_fwo_group (g_none, first_mem, &first_fwo);
  app_fwo_group_list (fwos, first_fwo);
  if (is_glue (&gl))
    should_be_tail_free_word_order_groups (gl, fwos);
  *alt = new_alternative (line, col, frequency, fwos, 0, trans_member_list_nil);
}

static void should_be_tail_free_word_order_groups (glue gl, fwo_group_list fwos)
{ fwo_group fwo;

  do
    { int was_operator = should_be_fwo_group_or_control_operator (gl, &fwo);
      app_fwo_group_list (fwos, fwo);
      if (was_operator) return;		/* syntax part ends at control operator */
    }
  while (is_glue (&gl));
}

static int is_glue (glue *gl)
{ if (is_symbol (","))
    { *gl = g_comma;
      return (1);
    };
  if (is_symbol ("+"))
    { *gl = g_plus;
      return (1);
    };
  return (0);
}

static int should_be_fwo_group_or_control_operator (glue gl, fwo_group *fwo)
{ member mem;
  if (is_control_operator (&mem))
    { *fwo = new_Single (gl, mem);
      return (1);
    };

  if (is_fwo_group (gl, fwo)) return (0);
  *fwo = fwo_group_nil;
  parser_error ("member or control operator expected");
  return (0);
}

static int is_fwo_group (glue gl, fwo_group *fwo)
{ member mem;
  if (!is_member (&mem)) return (0);
  may_be_rest_fwo_group (gl, mem, fwo);
  return (1);
}

static void may_be_rest_fwo_group (glue gl, member first_mem, fwo_group *fwo)
{ member_list mems;
  member second_mem;

  if (!is_symbol ("&"))
    { *fwo = new_Single (gl, first_mem);
      return;
    };

  mems = new_member_list ();
  *fwo = new_Fwo (gl, mems);
  app_member_list (mems, first_mem);
  do
     { should_be_member (&second_mem);
       app_member_list (mems, first_mem);
     }
  while (is_symbol ("&"));
}

/* 
   h) control operator:
        failure operator;
        success operator;
        abort operator;
        exit operator.
  
   i) failure operator: "-".
      success operator: "+".
      abort operator:   "?".
      exit operator:    "!".
*/ 
static void may_be_control_operator (member *mem)
{ int line, col;
  if (is_control_operator (mem)) return;
  get_current_position (&line, &col);
  *mem = new_Op (line, col, c_success);
}

static int is_control_operator (member *mem)
{ int line, col;
  control_op op;
  get_current_position (&line, &col);
  if (is_symbol ("+")) op = c_success;
  else if (is_symbol ("-")) op = c_failure;
  else if (is_symbol ("?")) op = c_abort;
  else if (is_symbol ("!")) op = c_exit;
  else return (0);
  *mem = new_Op (line, col, op);
  return (1);
}

/* 
   A.3.3 Members

   a) member:
        call;
        group;
        option;
        guard;
        terminal;
        quasi terminal.
*/
static void should_be_member (member *mem)
{ if (is_member (mem)) return;
  parser_error ("member expected");
  *mem = member_nil;
}

static int is_member (member *mem)
{ if (is_call (mem)) return (1);
  if (is_group (mem)) return (1);
  if (is_option (mem)) return (1);
  if (is_guard (mem)) return (1);
  if (is_terminal (mem)) return (1);
  if (is_quasi_terminal (mem)) return (1);
  return (0);
}

/*
   A.3.4 Calls
   
   a) call:
        name part, rest call option
  
   b) rest call:
        name part, rest call option;
        actual parameter list pack, rest strict call option.
 
   c) rest strict call:
        name part, rest call option.
 
   d) actual parameter:
        direction, affix expression;
        affix expression, direction option.
*/
static void should_be_call (member *mem)
{ if (is_call (mem)) return;
  parser_error ("call expected");
  *mem = member_nil;
}

static int is_call (member *mem)
{ string_list rname_parts;
  int_list rname_chars;
  int line, col;
  string rname;

  get_current_position (&line, &col);
  if (!is_name_part (&rname)) return (0);
  rname_parts = new_string_list ();
  rname_chars = new_int_list ();
  app_string_list (rname_parts, rname);
  app_int_list (rname_chars, 1);
  *mem = new_Call (line, col, rname_parts, rname_chars, new_apar_list ());
  may_be_rest_call (*mem);
  return (1);
}

static void may_be_rest_call (member mem)
{ while (1)
    { int_list chars = mem -> Call.rname_chars;
      int had_name_part = chars -> array[chars -> size - 1];
      string pname;
      if (is_name_part (&pname))
        { app_string_list (mem -> Call.rname_parts, pname);
	  app_int_list (mem -> Call.rname_chars, 1);
    	}
      else if (had_name_part && is_call_actual_parameter_list_pack (mem)) ;
      else break;
    };
}

static int is_call_actual_parameter_list_pack (member mem)
{ if (!is_symbol ("(")) return (0);
  should_be_call_actual_parameter_list (mem);
  should_be_symbol (")");
  return (1);
}

static void should_be_call_actual_parameter_list (member mem)
{ if (is_call_actual_parameter_list (mem)) return;
  parser_error ("actual parameter expected");
}

static int is_call_actual_parameter_list (member mem)
{ apar par;
  if (!is_actual_parameter (&par)) return (0);
  app_apar_list (mem -> Call.apars, par);
  app_int_list (mem -> Call.rname_chars, 0);
  while (is_symbol (","))
    { should_be_actual_parameter (&par);
      app_apar_list (mem -> Call.apars, par);
      app_int_list (mem -> Call.rname_chars, 0);
    };
  return (1);
}

static void may_be_actual_parameter_list_pack (apar_list *rpars)
{ if (is_symbol ("("))
    { should_be_actual_parameter_list (rpars);
      should_be_symbol (")");
    }
  else *rpars = new_apar_list ();
}

static void should_be_actual_parameter_list (apar_list *rpars)
{ if (is_actual_parameter_list (rpars)) return;
  *rpars = apar_list_nil;
  parser_error ("actual parameter expected");
}

static int is_actual_parameter_list (apar_list *rpars)
{ apar par;
  if (!is_actual_parameter (&par)) return (0);
  *rpars = new_apar_list ();
  app_apar_list (*rpars, par);
  while (is_symbol (","))
    { should_be_actual_parameter (&par);
      app_apar_list (*rpars, par);
    };
  return (1);
}

static void may_be_actual_parameter_pack (apar *rpar)
{ if (is_symbol ("("))
    { should_be_actual_parameter (rpar);
      should_be_symbol (")");
    }
  else *rpar = apar_nil;
}

static void should_be_actual_parameter (apar *rpar)
{ if (is_actual_parameter (rpar)) return;
  parser_error ("actual parameter expected");
  *rpar = apar_nil;
}

static int is_actual_parameter (apar *rpar)
{ affix_term_list aexp;
  dir adir = d_unknown;
  if (is_symbol (">"))
    { should_be_affix_expression (&aexp);
      adir = d_inherited;
    }
  else if (is_affix_expression (&aexp))
    { if (is_symbol (">")) adir = d_derived;
    }
  else return (0);
  *rpar = new_apar (adir, aexp);
  return (1);
}

/*
   A.3.5 Options and groups

   a) option:
 	"[", alternatives, "]".

   b) group:
 	"(", alternatives, ")".
*/
static int is_group (member *mem)
{ int line, col;
  alternative_list alts;
  get_current_position (&line, &col);
  if (!is_symbol ("(")) return (0);
  may_be_alternatives (0, &alts);
  should_be_symbol (")");
  *mem = new_Group (line, col, new_group (alts));
  return (1);
}

static int is_option (member *mem)
{ int line, col;
  alternative_list alts;
  get_current_position (&line, &col);
  if (!is_symbol ("[")) return (0);
  may_be_alternatives (1, &alts);
  should_be_symbol ("]");
  *mem = new_Option (line, col, new_group (alts));
  return (1);
}


/* 
   A.3.6 Guards
  
   a) guard:
        "{", confrontation list, "}".
  
   b) confrontation:
        join; split; assign; bidir; equal; not equal; restrict.
  
   c) join:	affix expression, "->", affix variable.
      split:	affix variable, "->", affix expression.
      assign:	affix variable, "->", affix variable.
      bidir:	affix variable, "<->", affix expression;
		affix expression, "<->", affix variable.
      equal:	affix variable, "=", affix variable.
      unequal:	affix variable, "<>", affix variable.
      restrict:	affix variable, "::", affix expression.
  
   Since it is absolutely impossible to distinguish the first 3 alternatives
   while parsing they are recognized in one syntactic parse and analyzed later.
*/ 
static int is_guard (member *mem)
{ confrontation_list confs;
  confrontation conf;
  int line, col;
  get_current_position (&line, &col);
  if (!is_symbol ("{")) return (0);
  confs = new_confrontation_list ();
  do
    { should_be_confrontation (&conf);
      app_confrontation_list (confs, conf);
    }
  while (is_symbol (","));
  should_be_symbol ("}");
  *mem = new_Guard (line, col, confs);
  return (1);
}

static void should_be_confrontation (confrontation *conf)
{ if (is_confrontation (conf)) return;
  parser_error ("confrontation expected");
  *conf = confrontation_nil;
}

static int is_confrontation (confrontation *conf)
{ guard_type typ = undetermined;
  affix_term_list lhs, rhs;
  int line, col;
  get_current_position (&line, &col);
  if (!is_affix_expression (&lhs)) return (0);
  if (is_symbol ("=")) typ = equal;
  else if (is_symbol ("<>")) typ = unequal;
  else if (is_symbol ("::")) typ = restrict;
  else if (is_symbol ("<->")) typ = bidir_assign;
  else if (!is_symbol ("->"))
    parser_error ("confrontation type '->', '<->', '<>', '::' or '=' expected");
  should_be_affix_expression (&rhs);
  *conf = new_confrontation (line, col, typ, lhs, rhs);
  return (1);
}

/*
   A.3.6 Terminals and quasi terminals
 
   A terminal is written as a regular expression between quotes. The intention
   of such a terminal is to match a longest matching string. We choose this
   interpretation to ease the description of compilers in EAG. For other
   interpretatons, the AGFL quasi terminals $MATCH, $SKIP, $WORD, $ANY,
   $OTHER should be used.
 
   a) terminal:
        "\"[^\n]*\"", actual parameter pack option.
 
   b) quasi terminal:
        "$", bold name, actual parameter pack option
*/
static int is_terminal (member *mem)
{ int line, col;
  string text;
  apar par;
  get_current_position (&line, &col);
  if (!is_text_constant (&text)) return (0);
  may_be_actual_parameter_pack (&par);
  *mem = new_Term (line, col, text, par);
  return (1);
}

static int is_quasi_terminal (member *mem)
{ string bname, dbname;
  string_list dname_parts;
  int_list dname_chars;
  apar_list parms;
  int line, col, ix;
  get_current_position (&line, &col);
  if (!is_symbol ("$")) return (0);
  should_be_bold_name (&bname);

  /* Create artificial name parts and chars */
  dbname = dcg_new_fmtd_string ("$%s", bname);
  detach_string (&bname);
  dname_parts = new_string_list ();
  dname_chars = new_int_list ();
  app_string_list (dname_parts, dbname);
  app_int_list (dname_chars, 1);

  may_be_actual_parameter_list_pack (&parms);
  for (ix = 0; ix < parms -> size; ix++)
    app_int_list (dname_chars, 0);
  *mem = new_Quasi (line, col, dname_parts, dname_chars, parms);
  return (1);
}

/*
   A.4 Affix expressions

   PRIO :: "i"; "ii"; "iii"; "iiii"; "iiiii"; "iiiiii".
  
   a) affix expression:
        affix term sequence.
  
   b) affix term:
        affix priority i formula.
*/
static void should_be_affix_expression (affix_term_list *terms)
{ if (is_affix_expression (terms)) return;
  parser_error ("affix expression expected");
  *terms = affix_term_list_nil;
}

static int is_affix_expression (affix_term_list *terms)
{ affix_term term;
  if (!is_affix_term (&term)) return (0);
  *terms = new_affix_term_list ();
  app_affix_term_list (*terms, term);
  while (is_affix_term (&term))
    app_affix_term_list (*terms, term);
  return (1);
}

static int is_affix_term (affix_term *term)
{ return (is_affix_formula_of_prio (1, term));
}

/*
   c) affix priority PRIO formula:
	{ PRIO :: "i"; "ii"; "iii"; "iiii"; "iiiii" },
	  ( affix priority PRIO formula, arithmetic PRIO operator, affix priority PRIOi formula;
            affix priority PRIOi formula
	  ).

      affix priority iiiiii formula:
	monadic affix formula.

   d) arithmetic i operator:		"|"; "^".
      arithmetic ii operator:		"&".
      arithmetic iii operator:		"<<"; ">>".
      arithmetic iiii operator:		"+"; "-".
      arithmetic iiiii operator:	"*"; "%"; "%%".
*/
static void should_be_affix_formula_of_prio (int prio, affix_term *term)
{ if (is_affix_formula_of_prio (prio, term)) return;
  parser_error ("affix_formula of priority %d expected", prio);
  *term = affix_term_nil;
}

static int is_affix_formula_of_prio (int prio, affix_term *term)
{ int line, col;
  operator dop;
  if (prio >= 6) return (is_affix_monadic_formula (term));
  get_current_position (&line, &col);
  if (!is_affix_formula_of_prio (prio + 1, term)) return (0);
  while (is_arithmetic_prio_operator (prio, &dop))
    { affix_term arg2;
      should_be_affix_formula_of_prio (prio + 1, &arg2);
      *term = new_Dyop (line, col, dop, *term, arg2);
    };
  return (1);
}

static int is_arithmetic_prio_operator (int prio, operator *rop)
{ if (ahead_symbol ("->")) return (0);
  if (ahead_symbol ("=")) return (0);
  switch (prio)
    { case 1:
	/* To be resolved later as bitwise_and or a_part */
	if (is_symbol ("|"))	   { *rop = a_union; return (1); }
	else if (is_symbol ("^"))  { *rop = bitwise_xor; return (1); }
	break;
      case 2:
	/* To be resolved later as bitwise_and or a_part */
	if (is_symbol ("&"))	   { *rop = bitwise_and; return (1); }
	break;
      case 3:
	/* MS: check if this matches C priorities */
	if (is_symbol ("<<"))	   { *rop = shift_left;  return (1); }
	else if (is_symbol (">>")) { *rop = shift_right; return (1); }
	break;
      case 4:
	if (is_symbol ("+")) 	   { *rop = plus;  return (1); }
	else if (is_symbol ("-"))  { *rop = minus; return (1); }
	break;
      case 5:
	if (is_symbol ("*")) 	   { *rop = times; return (1); }
	else if (is_symbol ("%%")) { *rop = modulo;  return (1); }
	else if (is_symbol ("%"))  { *rop = divides; return (1); }
      default: break;
    };
  return (0);
}

/*
   e) monadic affix formula:
	monadic operator, monadic affix formula;
	affix primary.

   f) monadic operator:
	"+"; "-"; "~".
  
*/
static void should_be_affix_monadic_formula (affix_term *term)
{ if (is_affix_monadic_formula (term)) return;
  parser_error ("affix primary expected");
  *term = affix_term_nil;
}

static int is_affix_monadic_formula (affix_term *term)
{ int line, col;
  operator mop;
  get_current_position (&line, &col);
  if (is_monadic_operator (&mop))
    { affix_term arg;
      should_be_affix_monadic_formula (&arg);
      *term = new_Monop (line, col, mop, arg);
      return (1);
    };
  return (is_affix_primary (term));
}

static int is_monadic_operator (operator *rop)
{ if (ahead_symbol ("->")) return (0);
  if (ahead_special_affix_terminal ()) return (0);
  if (is_symbol ("+")) { *rop = plus; return (1); }
  if (is_symbol ("-")) { *rop = minus; return (1); }
  if (is_symbol ("~")) { *rop = bitwise_not; return (1); }
  return (0);
}

/*
   g) affix primary:
	affix terminal;
	affix variable;
	number;
	text constant;
	"(", affix expression, ")".

   h) affix variable:
	affix variable name.
*/
static int is_affix_primary (affix_term *term)
{ string marker, tconst, aname;
  int inum, kind, line, col;
  real rnum;
  get_current_position (&line, &col);
  if (is_affix_terminal (&marker))
    { *term = new_Terminal (line, col, marker);
      return (1);
    };
  if (is_number (&kind, &inum, &rnum))
    { if (kind) *term = new_Real (line, col, rnum);
      else *term = new_Int (line, col, inum);
      return (1);
    };
  if (is_text_constant (&tconst))
    { *term = new_Text (line, col, tconst);
      return (1);
    };
  if (is_symbol ("("))
    { affix_term_list terms;
      should_be_affix_expression (&terms);
      should_be_symbol (")");
      *term = new_Enclosed (line, col, terms);
      return (1);
    };
  if (is_affix_variable_name (&aname))
    { *term = new_Var (line, col, aname);
      return (1);
    };
  return (0);
}

/*
   A.5 Transduction

   a) alternative transduction part:
        "/", transduction members option.

   b) transduction members:
        transduction member, rest transduction members;
        transduction output operator, rest transduction members.

   c) rest transduction members:
        ",", transduction members;
        "+", transduction members;
        transduction member, rest transduction members;
        transduction output operator, rest transduction members;
        .
*/
static void may_be_alternative_transduction_part (int in_option, alternative alt)
{ if (!is_symbol ("/")) return;
  alt -> trans = new_trans_member_list ();
  (void) is_transduction_members (in_option, alt -> trans, g_none);
}

static void should_be_transduction_members (int in_option, trans_member_list tr_mems, glue gl)
{ if (is_transduction_members (in_option, tr_mems, gl)) return;
  parser_error ("transduction member or output operator expected");
}

static int is_transduction_members (int in_option, trans_member_list tr_mems, glue gl)
{ trans_member tmem;
  if (!is_transduction_member_or_output_operator (in_option, gl, &tmem)) return (0);
  app_trans_member_list (tr_mems, tmem);
  may_be_rest_transduction_members (in_option, tr_mems);
  return (1);
}

static void may_be_rest_transduction_members (int in_option, trans_member_list tr_mems)
{ trans_member tmem;
  glue gl;
  
  if (is_glue (&gl)) should_be_transduction_members (in_option, tr_mems, gl);
  else if (is_transduction_member_or_output_operator (in_option, g_none, &tmem))
    { app_trans_member_list (tr_mems, tmem);
      may_be_rest_transduction_members (in_option, tr_mems);
    };
}

/*
   d) transduction member:
        affix variable;  
        call placeholder;
        quasi terminal placeholder;
        group placeholder;
        option placeholder;
        text transduction.

   e) transduction output operator:
        "["; "]";
        "{"; "}";
        "|";
        "<", relation name;
        ">", relation name.

   f) quasi terminal placeholder:
        "$", bold name part, dot number option.

   g) call placeholder:   name part sequence, dot number option.
   h) group placeholder:  "(", ")", dot number option.
   i) option placeholder: "[", "]", dot number option.
   j) text transduction: text constant.

   k) dot number:
        ".", number.

   Note that we cannot distinguish an affix variable yet from
   a call placeholder, so we parse as a call placeholder and
   use identification to determine the difference.
*/
static int is_transduction_member_or_output_operator (int in_option, glue gl, trans_member *tmem)
{ if (is_call_placeholder (gl, tmem)) return (1);
  if (is_quasi_terminal_placeholder (gl, tmem)) return (1);
  if (is_group_placeholder (gl, tmem)) return (1);
  if (is_option_placeholder_or_hybrid_square_open (gl, tmem)) return (1);
  if (is_text_transduction (gl, tmem)) return (1);
  if (is_other_hybrid_operator (in_option, gl, tmem)) return (1);
  return (0);
}

static int is_call_placeholder (glue gl, trans_member *tmem)
{ int line, col, nr;
  string_list cname;
  string pname;
  get_current_position (&line, &col);
  if (!is_name_part (&pname)) return (0);
  cname = new_string_list ();
  do
    { app_string_list (cname, pname);
    }
  while (is_name_part (&pname));
  may_be_dot_number (&nr);
  *tmem = new_Tcall (line, col, gl, cname, nr);
  return (1);
}

static int is_quasi_terminal_placeholder (glue gl, trans_member *tmem)
{ int line, col, nr;
  string bname;
  get_current_position (&line, &col);
  if (!is_symbol ("$")) return (0);
  should_be_bold_name (&bname);
  may_be_dot_number (&nr);
  *tmem = new_Tquasi (line, col, gl, bname, nr);
  return (1);
}

static int is_group_placeholder (glue gl, trans_member *tmem)
{ int line, col, nr;
  get_current_position (&line, &col);
  if (!is_symbol ("(")) return (0);
  should_be_symbol (")");
  may_be_dot_number (&nr);
  *tmem = new_Tgroup (line, col, gl, nr);
  return (1);
}

static int is_option_placeholder_or_hybrid_square_open (glue gl, trans_member *tmem)
{ int line, col;
  get_current_position (&line, &col);
  if (!is_symbol ("[")) return (0);

  /* Two possibilities: either a option place holder or an operator */
  if (is_symbol ("]"))
    { int nr;
      may_be_dot_number (&nr);
      *tmem = new_Toption (line, col, gl, nr);
      return (1);
    };

  /* It must be a hybrid [ */
  *tmem = new_Thybrid (line, col, gl, new_Hyb_sqr_open ());
  return (1);
}

static void may_be_dot_number (int *nr)
{ if (!ahead_dot_number ()) return;
  if (is_symbol ("."))
    should_be_integer_number (nr);
}

static int is_text_transduction (glue gl, trans_member *tmem)
{ int line, col;
  string text;
  get_current_position (&line, &col);
  if (!is_text_constant (&text)) return (0);
  *tmem = new_Ttext (line, col, gl, text);
  return (1);
}

static int is_other_hybrid_operator (int in_option, glue gl, trans_member *tmem)
{ int line, col;
  string relname;
  get_current_position (&line, &col);
  if (!in_option && is_symbol ("]")) *tmem = new_Thybrid (line, col, gl, new_Hyb_sqr_close ());
  else if (is_symbol ("{")) *tmem = new_Thybrid (line, col, gl, new_Hyb_curl_open ());
  else if (is_symbol ("}")) *tmem = new_Thybrid (line, col, gl, new_Hyb_curl_close ());
  else if (is_symbol ("|")) *tmem = new_Thybrid (line, col, gl, new_Hyb_bar ());
  else if (is_symbol ("<"))
    { should_be_relation_name (&relname);
      *tmem = new_Thybrid (line, col, gl, new_Hyb_rel_left (relname));
    }
  else if (is_symbol (">"))
    { should_be_relation_name (&relname);
      *tmem = new_Thybrid (line, col, gl, new_Hyb_rel_right (relname));
    }
  else return (0);
  return (1);
}

/* 
   A.6 Affix rules
 
   a) affix rule:
	affix rule head, "::", EXTERNAL, ".".
	affix rule head, "::", affix alternatives, "."
	affix rule head, "=", affix name, "."

   b) affix rule head:
        affix name list.
*/
static void should_be_rest_affix_rule (grammar gra, name_list ids, int is_affix_definition)
{ int line = ids -> array[0] -> line;
  int col = ids -> array[0] -> col;
  string aname = attach_string (ids -> array[0] -> id -> array[0]);
  affix_rule arule;

  /* Check validity of ids: we do not allow spaces in affix names */
  verify_affix_name_list (ids);

  if (!is_affix_definition)
    { /* Affix rule is a synonym definition */
      string syn_name;
      should_be_affix_name (&syn_name);
      should_be_symbol (".");
      arule = new_Affix_synonym (line, col, aname, syn_name);
      app_affix_rule_list (gra -> arules, arule);
    }
  else
    { if (is_keyword ("EXTERNAL"))
        { should_be_symbol (".");
	  arule = new_Affix_prim (line, col, aname);
	}
      else
	{ affix_alternative_list alts;
	  may_be_affix_alternatives (&alts);
          should_be_symbol (".");
	  arule = new_Affix_alts (line, col, aname, alts);
        };
      app_affix_rule_list (gra -> arules, arule);
    }
  expand_lhs_synonyms (gra -> arules, ids);
}

static void verify_affix_name_list (name_list ids)
{ int ix;
  for (ix = 0; ix < ids -> size; ix++)
    if (!valid_affix_nonterminal_name (ids -> array[ix] -> id))
      parser_error_at (ids -> array[ix] -> line, ids -> array[ix] -> col,
		       "Invalid affix nonterminal name");
}

static void expand_lhs_synonyms (affix_rule_list arules, name_list anames)
{ int ix;
  for (ix = 1; ix < anames -> size; ix++)
    { string lhs_name = attach_string (anames -> array[ix] -> id -> array[0]);
      string syn_name = attach_string (anames -> array[0] -> id -> array[0]);
      int line = anames -> array[ix] -> line;
      int col = anames -> array[ix] -> col;
      affix_rule syn_rule = new_Affix_synonym (line, col, lhs_name, syn_name);
      app_affix_rule_list (arules, syn_rule);
    };
}

/*
   c) affix alternatives:
        affix alternative, ";", affix alternatives;
        affix alternative.
*/
static void may_be_affix_alternatives (affix_alternative_list *alts)
{ affix_alternative alt;
  *alts = new_affix_alternative_list ();
  may_be_affix_alternative (&alt);
  app_affix_alternative_list (*alts, alt);
  while (is_symbol (";"))
    { may_be_affix_alternative (&alt);
      app_affix_alternative_list (*alts, alt);
    };
}

/*
   d) affix alternative:
	affix element, "|", formal affix union;
	affix element, "&", formal affix partition;
	affix element, ",", formal affix concatenation;
        affix element sequence;
	.

   e) formal affix union:
	affix element, "|", formal affix union;
        affix element.

   f) formal affix partition:
	affix element, "&", formal affix partition;
        affix element.

   g) formal affix concatenation:
	affix element, ",", formal affix concatenation;
	affix element.
 
   h) affix element:
        affix variable name;
        affix terminal;
        text constant;
	number.

   We have to fold the alternatives a bit and we need to introduce some
   notation for regular expressions on affix level.
*/ 
static void may_be_affix_alternative (affix_alternative *alt)
{ affix_element_list elems;
  affix_element elem;
  elems = new_affix_element_list ();
  if (!is_affix_element (&elem))
    { *alt = new_Affix_empty (elems);
      return;
    };
  app_affix_element_list (elems, elem);
  if (is_symbol ("|"))
    should_be_rest_affix_union (alt, elems);
  else if (is_symbol ("&"))
    should_be_rest_affix_partition (alt, elems);
  else if (is_symbol (","))
    should_be_rest_affix_concatenation (alt, elems);
  else if (is_affix_element (&elem))
    { app_affix_element_list (elems, elem);
      may_be_rest_affix_sequence (alt, elems);
    }
  else
    /* Single affix element in alternative */
    *alt = new_Affix_single (elems);
  return;
}

static void should_be_rest_affix_union (affix_alternative *alt, affix_element_list elems)
{ if (is_rest_affix_union (alt, elems)) return;
  parser_error ("affix element expected");
  *alt = affix_alternative_nil;
}

static int is_rest_affix_union (affix_alternative *alt, affix_element_list elems)
{ affix_element elem;
  if (!is_affix_element (&elem)) return (0);
  app_affix_element_list (elems, elem);
  while (is_symbol ("|"))
    { should_be_affix_element (&elem);
      app_affix_element_list (elems, elem);
    };
  *alt = new_Affix_union (elems);
  return (1);
}

static void should_be_rest_affix_partition (affix_alternative *alt, affix_element_list elems)
{ if (is_rest_affix_partition (alt, elems)) return;
  parser_error ("affix element expected");
  *alt = affix_alternative_nil;
}

static int is_rest_affix_partition (affix_alternative *alt, affix_element_list elems)
{ affix_element elem;
  if (!is_affix_element (&elem)) return (0);
  app_affix_element_list (elems, elem);
  while (is_symbol ("&"))
    { should_be_affix_element (&elem);
      app_affix_element_list (elems, elem);
    };
  *alt = new_Affix_partition (elems);
  return (1);
}

static void should_be_rest_affix_concatenation (affix_alternative *alt, affix_element_list elems)
{ if (is_rest_affix_concatenation (alt, elems)) return;
  parser_error ("affix element expected");
  *alt = affix_alternative_nil;
}

static int is_rest_affix_concatenation (affix_alternative *alt, affix_element_list elems)
{ affix_element elem;
  if (!is_affix_element (&elem)) return (0);
  app_affix_element_list (elems, elem);
  while (is_symbol (","))
    { should_be_affix_element (&elem);
      app_affix_element_list (elems, elem);
    };
  *alt = new_Affix_concat (elems);
  return (1);
}

static void may_be_rest_affix_sequence (affix_alternative *alt, affix_element_list elems)
{ affix_element elem;
  while (is_affix_element (&elem))
    app_affix_element_list (elems, elem);
  *alt = new_Affix_sequence (elems);
}

static void should_be_affix_element (affix_element *elem)
{ if (is_affix_element (elem)) return;
  parser_error ("affix element expected");
  *elem = affix_element_nil;
}

static int is_affix_element (affix_element *elem)
{ int line, col, ival, kind;
  string affix, text;
  real rval;
  get_current_position (&line, &col);
  if (is_affix_variable_name (&affix))
    { *elem = new_Affix_var (line, col, affix);
      return (1);
    }
  else if (is_affix_terminal (&affix))
    { *elem = new_Affix_term (line, col, affix);
      return (1);
    }
  else if (is_text_constant (&text))
    { *elem = new_Affix_text (line, col, text);
      return (1);
    }
  else if (is_number (&kind, &ival, &rval))
    { if (kind) *elem = new_Affix_rnum (line, col, rval);
      else *elem = new_Affix_inum (line, col, ival);
      return (1);
    };
  return (0);
}

/*
   Needed to parse the invisible pragmat:
   invisible nonterminal:
     name part list, "/", number.
*/
void should_be_invisible_nonterminal_list ()
{ should_be_invisible_nonterminal ();
  while (is_symbol (","))
    should_be_invisible_nonterminal ();
}

void should_be_invisible_nonterminal ()
{ if (is_invisible_nonterminal ()) return;
  parser_error ("invisible nonterminal expected");
}

int is_invisible_nonterminal ()
{ int line, col;
  int nr = -1;
  string_list cname;
  string fname, pname;
  invisible_nonterminal invis;
  get_current_filename (&fname);
  get_current_position (&line, &col);
  if (!is_name_part (&pname)) return (0);
  cname = new_string_list ();
  do
    { app_string_list (cname, pname);
    }
  while (is_name_part (&pname));
  should_be_symbol ("/");
  should_be_integer_number (&nr);
  invis = new_invisible_nonterminal (attach_string (fname), line, col, cname, nr);
  app_invisible_nonterminal_list (all_invisible_nonterminals, invis);
  return (1);
}
