/*
   File: match.c
   Tries to match an affix expression against an affix nonterminal
   This code is for checking definitions, guards and resolving calls

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: match.c,v 1.1 2012/10/05 19:56:17 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* support lib includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "globals.h"
#include "affix_rules.h"
#include "contsens.h"
#include "options.h"
#include "lattices.h"
#include "match.h"

/*
   Our backup parsing administration is managed through a continuation stack
   on which member administration alternates with rule goals. Parse_arule is
   responsible for parsing a rule sentential form to check whether it matches
   an affix nonterminal using recursive backup leftcorner parsing. It is
   guaranteed to be only called for affix rules of the tree kind.

   For this purpose, it pushes a rule goal and then iterates through all rules
   which are in a (reflexive transitive) left corner relation to try and parse
   a left prefix of its rule in a bottom up fashion. Upon a succesfull partial
   parse of a left prefix, the reduce_arule will continue the parse.

   The parameter complete specifies that the parse must be complete or not
   (end of sentence match): 0 (no eos required), 1 (total eos required) or
   2 (local (Enclosed terms) eos required).

   Upon a succesfull parse, we return an attributed affix expression. Optionally,
   the parse may have introduced implicit Enclosed terms if there was a local
   match. We assume that the expression has already been submitted to check_expr.
   Thus, terms whose adef is not null, is already correctly typed and may therefore
   be shared.

   Note: all parsing routines return 0 (no parse), 1 (one success), >1 (definitely ambiguous)
*/
static int parse_leftcorner (affix_rule lc_rule, affix_term_list terms, int iptr,
			     madm_list cont_stack, affix_term_list tree_stack, affix_term *result);
static int parse_alternative (affix_rule lc_rule, int alt_nr, int from_mem, 
			      affix_term_list terms, int iptr, madm_list cont_stack,
			      affix_term_list tree_stack, affix_term *result);
static int parse_members (affix_term_list terms, int iptr, madm_list cont_stack,
			  affix_term_list tree_stack, affix_term *result);
static int reduce_arule (affix_term_list terms, int iptr, madm_list cont_stack,
			 affix_term_list tree_stack, affix_term *result);
static int parse_non_tree_term (affix_rule arule, affix_term term, affix_term *result);

#define NO_EOS_REQUIRED 0
#define EOS_REQUIRED 1
static int parse_arule (affix_rule arule, int complete, affix_term_list terms, int iptr,
			madm_list cont_stack, affix_term_list tree_stack, affix_term *result)
{ /* Create new rule admin and goal */
  affix_rule goal = get_prime_affix_rule (arule);
  int_list lc_nrs = arule -> lc_nrs;
  madm new_top = new_Rule_adm ();
  int stat = 0;
  int ix;

  /* Register the new top of the continuation stack */
  new_top -> Rule_adm.complete = complete;
  new_top -> Rule_adm.terms = terms;
  new_top -> Rule_adm.iptr = iptr;
  new_top -> Rule_adm.goal = goal;
  app_madm_list (cont_stack, new_top);

  /* Add an extra internal consistency check */
  if (lc_nrs == int_list_nil)
    { dcg_wlog ("Goal %s, no left corners available", goal -> aname);
      dcg_internal_error ("parse_arule");
    };

  /*
     Iterate through the list of left corner nonterminals to get a left corner
     Note that we have to include ourselves for getting a leftcorner, but not twice
  */
  for (ix = 0; ix < lc_nrs -> size; ix++)
    { int lc_nr = lc_nrs -> array[ix];
      affix_rule lc_rule = get_prime_affix_rule (all_affix_rules -> array[lc_nr]);
      if (lc_rule != goal)
        stat += parse_leftcorner (lc_rule, terms, iptr, cont_stack, tree_stack, result);
    };
  stat += parse_leftcorner (goal, terms, iptr, cont_stack, tree_stack, result);

  /* Remove what we have pushed */
  del_madm_list (cont_stack, cont_stack -> size - 1);
  det_madm (&new_top);
  return (stat);
}

/*
   We have selected a left corner of the goal on the continuation stack.
   Parse according to all possible alternatives of that left corner
*/
static int parse_leftcorner (affix_rule lc_rule, affix_term_list terms, int iptr,
			     madm_list cont_stack, affix_term_list tree_stack, affix_term *result)
{ affix_alternative_list alts = pick_rhs_affix_alts (lc_rule);
  int stat = 0;
  int alt_nr;
  if (alts == affix_alternative_list_nil)
    { affix_rule goal = cont_stack -> array[cont_stack -> size - 1] -> Rule_adm.goal;
      dcg_wlog ("Goal %s, LC %s", goal -> aname, lc_rule -> aname);
      dcg_internal_error ("parse_leftcorner: 2");
    };

  /* Iterate over the alternatives of this left corner */
  for (alt_nr = 0; alt_nr < alts -> size; alt_nr++)
    stat += parse_alternative (lc_rule, alt_nr, 0, terms, iptr,
			       cont_stack, tree_stack, result);
  return (stat);
};

/*
   We have selected an alternative of a left corner of the goal on the continuation stack
   Now try and match the members of this alternative against tht terms in the input
*/
static int parse_alternative (affix_rule lc_rule, int alt_nr, int from_mem,
			      affix_term_list terms, int iptr, madm_list cont_stack,
			      affix_term_list tree_stack, affix_term *result)
{ /* Create member administration for this alternative and start member */
  madm new_top = new_Member_adm (alt_nr, from_mem);
  int stat;
  new_top -> Member_adm.arule = lc_rule;
  app_madm_list (cont_stack, new_top);

  /* Call the essential parse_members */
  stat = parse_members (terms, iptr, cont_stack, tree_stack, result);

  /* Remove what we have pushed */
  del_madm_list (cont_stack, cont_stack -> size - 1);
  det_madm (&new_top);
  return (stat);
}

/*
   The following function is the essential parsing algorithm, matching a sequence
   of terms against the members of the current alternative on the top of the
   continuation stack
*/
static int parse_members (affix_term_list terms, int iptr, madm_list cont_stack,
			  affix_term_list tree_stack, affix_term *result)
{ int mem_nr, alt_nr, cur_line, cur_col;
  affix_alternative_list alts;
  affix_rule lc_rule, vrule;
  madm top_madm, goal_madm; 
  affix_element_list elems;
  affix_element curr_elem;
  affix_term curr_term;
  int rstat = 0;
 
  /* Consistency check */
  if (cont_stack -> size < 2) dcg_internal_error ("parse_members: cont_stack_size");
  top_madm = cont_stack -> array[cont_stack -> size - 1];
  goal_madm = cont_stack -> array[cont_stack -> size - 2];
  if (top_madm -> tag != TAGMember_adm) dcg_internal_error ("parse_members: top_madm");
  if (goal_madm -> tag != TAGRule_adm) dcg_internal_error ("parse_members: goal_adm");

  /* Pick the necessary data from the continuation stack */
  alt_nr = top_madm -> Member_adm.alt_nr;
  mem_nr = top_madm -> Member_adm.mem_nr;
  lc_rule = top_madm -> Member_adm.arule;
  alts = pick_rhs_affix_alts (lc_rule);
  elems = alts -> array[alt_nr] -> elems;

  /* Check for a complete partial parse, if so, reduce and continue */
  if (mem_nr == elems -> size)
    { /*
	 We have a successful partial parse, construct local affix tree
	 and reduce according to the tree grammar
      */
      int ast_line, ast_col, tidx, ix;
      affix_term ast;

      /* 
	 Pop parts from tree stack and push as AST parts.
         Note that we do not push the affix terminals and that we must attach
         the parts. If we find a final result, we want to have a separate result AST.
      */
      affix_term_list ast_parts = new_affix_term_list ();
      for (ix = elems -> size - 1; 0 <= ix; ix--)
	if (elems -> array[ix] -> tag == TAGAffix_var)
	  { affix_term part = tree_stack -> array [tree_stack -> size - 1];
	    del_affix_term_list (tree_stack, tree_stack -> size - 1);
	    ins_affix_term_list (ast_parts, 0, attach_affix_term (part));
	  };

      /* Locate position of first element to store in AST */
      tidx = iptr - elems -> size;
      if (tidx < 0) dcg_internal_error ("parse_members: pos");
      ast_line = terms -> array[tidx] -> line;
      ast_col = terms -> array[tidx] -> col;

      /* construct Ast and push on tree stack */
      ast = new_Ast (ast_line, ast_col, ast_parts, alt_nr);
      ast -> adef = lc_rule;
      app_affix_term_list (tree_stack, ast);

      if (dump_affix_rules)
        { dcg_hint ("Constructed local ast for lc rule %s, alt %d", lc_rule -> aname, alt_nr);
	  if (full_verbose) pp_affix_term (dcg_error_file (), ast);
	  dcg_hint ("");
	};

      /* On the tree stack, the reduction is done, now do the continuation */
      del_madm_list (cont_stack, cont_stack -> size - 1);
      rstat = reduce_arule (terms, iptr, cont_stack, tree_stack, result);
      app_madm_list (cont_stack, top_madm);

      /* Pop new ast and push all parts back on tree stack to undo */
      del_affix_term_list (tree_stack, tree_stack -> size - 1);
      for (ix = 0; ix < ast_parts -> size; ix++)
	app_affix_term_list (tree_stack, ast_parts -> array[ix]);
      detach_affix_term (&ast);
      return (rstat);
    };

  /*
     There is still a member to recognize, so we cannot succeed
     if we are already at the end of input
  */
  if (iptr >= terms -> size) return (0);

  /*
     If the current element to be parsed is an affix terminal, try to match it
  */
  curr_term = terms -> array[iptr];
  curr_elem = elems -> array[mem_nr];
  if (curr_elem -> tag == TAGAffix_term)
    { /*
	 Try and match with the current element at the iptr
      */
      if ((curr_term -> tag == TAGTerminal) &&
          (streq (curr_elem -> Affix_term.tname, curr_term -> Terminal.marker)))
	{ /* We match the affix terminal */
	  top_madm -> Member_adm.mem_nr++;
	  rstat = parse_members (terms, iptr + 1, cont_stack, tree_stack, result);
	  top_madm -> Member_adm.mem_nr--;
          return (rstat);
	};

      /* No match for this affix terminal */
      return (0);
    };

  /*
     The current element must be a nonterminal
     The primitive ones should match a term or variable
  */
  if (curr_elem -> tag != TAGAffix_var) dcg_internal_error ("parse_members:2");
  vrule = get_prime_affix_rule (curr_elem -> Affix_var.vdef);
  if (curr_term -> adef == AFFIX_ERROR) return (0);
  cur_line = curr_term -> line;
  cur_col = curr_term -> col;

  if (similar_affix_rule (vrule, curr_term -> adef))
    { /*
	 If the type of the term matches with the type of vrule,
	 we have a (partial) match, push the term onto the tree
	 stack and continue
      */
      app_affix_term_list (tree_stack, curr_term);
      top_madm -> Member_adm.mem_nr++;
      rstat += parse_members (terms, iptr + 1, cont_stack, tree_stack, result);
      top_madm -> Member_adm.mem_nr--;
      del_affix_term_list (tree_stack, tree_stack -> size - 1);
      return (rstat);
    };

  /*
     We only arrive here when the current element to match is a nonterminal and the
     current term has an unknown (maybe a marker) or a non equivalent type

     Check for an enclosed list of terms to match the current tree type nonterminal
  */
  if ((curr_term -> adef == affix_rule_nil) && (curr_term -> tag == TAGEnclosed) &&
      (vrule -> kind == arule_tree))
    { affix_term_list lterms = curr_term -> Enclosed.terms;
      affix_term lresult = affix_term_nil;
      affix_term_list ltree_stack = new_affix_term_list ();
      madm_list lcont_stack = new_madm_list ();
      int lstatus = parse_arule (vrule, EOS_REQUIRED, lterms, 0,
			         lcont_stack, ltree_stack, &lresult);
      detach_madm_list (&lcont_stack);
      detach_affix_term_list (&ltree_stack);
      if (lstatus)
	{ /* We had a local parse with a result, see if we can parse the whole */
	  int cstatus;
	  app_affix_term_list (tree_stack, lresult);
	  top_madm -> Member_adm.mem_nr++;
	  cstatus = parse_members (terms, iptr + 1, cont_stack, tree_stack, result);
	  top_madm -> Member_adm.mem_nr--;
          del_affix_term_list (tree_stack, tree_stack -> size - 1);
	  rstat += lstatus * cstatus;
	}
    };

  /*
     Do an explicit check for the expecting vrule to be of the lattice kind
  */
  if ((vrule -> kind == arule_lattice) && (curr_term -> adef == affix_rule_nil))
    { affix_term local;
      if (parse_non_tree_term (vrule, curr_term, &local))
	{ app_affix_term_list (tree_stack, local);
	  top_madm -> Member_adm.mem_nr++;
	  rstat += parse_members (terms, iptr + 1, cont_stack, tree_stack, result);
	  top_madm -> Member_adm.mem_nr--;
	  del_affix_term_list (tree_stack, tree_stack -> size - 1);
	};
      return (rstat);
    }

  /*
     If we are not the first member to match (we do leftcorner parsing) and the type
     of the current term is unknown or not equivalent we try and recognize the
     member topdown by calling parse_rule
  */
  if (!equivalent_affix_rule (vrule, curr_term -> adef) &&
      (vrule -> kind == arule_tree) && (mem_nr != 0))
    { /*
	 We call parse_rule recursively: when we succeed we recognized this member
	 Note that we do not require this parse to match the whole sentential form
      */ 
      top_madm -> Member_adm.mem_nr++;
      rstat += parse_arule (vrule, NO_EOS_REQUIRED, terms, iptr,
			    cont_stack, tree_stack, result);
      top_madm -> Member_adm.mem_nr--;
    };
  return (rstat);
}

static int reduce_arule (affix_term_list terms, int iptr, madm_list cont_stack,
			 affix_term_list tree_stack, affix_term *result)
{ /* Consistency check */
  madm top_madm = cont_stack -> array[cont_stack -> size - 1];
  affix_term top_tree = tree_stack -> array[tree_stack -> size - 1];
  affix_rule top_trule = top_tree -> adef;
  affix_rule goal = top_madm -> Rule_adm.goal;
  int_list lc_nrs = goal -> lc_nrs;
  int lc_nr;
  int stat = 0;

  if (top_madm -> tag != TAGRule_adm) dcg_internal_error ("reduce_arule");
  if (top_trule == affix_rule_nil) dcg_internal_error ("reduce_arule");
  if (dump_affix_rules)
    dcg_hint ("In reduce_arule, cont_stack_size: %d, tree_stacksize: %d",
	      cont_stack -> size, tree_stack -> size);

  /*
     If our goal and top tree are the same (equivalent),
     we have a match. If it is also a complete match, we
     have a succesfull parse.
  */
  if (equivalent_affix_rule (goal, top_trule))
    { /* Either we have a complete parse or we have to continue */
      if ((cont_stack -> size == 1) && (iptr == terms -> size) &&
	  (top_madm -> Rule_adm.complete == EOS_REQUIRED))
	{ /* Our top_tree is a complete parse tree */
	  if (*result == affix_term_nil)
	    { if (dump_affix_rules) dcg_hint ("Full parse succeeded");
	      *result = attach_affix_term (top_tree);
	      return (1);
	    }
	  else return (2);	/* Ambiguous resolution */
	}
      else if ((cont_stack -> size > 1) || (top_madm -> Rule_adm.complete == NO_EOS_REQUIRED))
	{ /* Incomplete parse, pop goal and continue */
	  del_madm_list (cont_stack, cont_stack -> size - 1);
          stat += parse_members (terms, iptr, cont_stack, tree_stack, result);
	  app_madm_list (cont_stack, top_madm);
	};
    };

  /*
     There is another possibility: we have only partly recognized the syntactic structure
     but the nonterminal that we just recognized is a leftcorner of our goal, but either
     not our goal or some leftcorner child/grandchild/....
     So we do not pop our goal, but push a member admin on the stack towards the goal
     (Kind of meet in the middle attack).
  */

  /* Iterate thru list of left corner nonterminals to get a left corner */
  for (lc_nr = 0; lc_nr < lc_nrs -> size; lc_nr++)
    { affix_rule lc_rule = all_affix_rules -> array[lc_nrs -> array[lc_nr]];
      affix_alternative_list alts = pick_rhs_affix_alts (lc_rule);
      int alt_nr;
      if (alts == affix_alternative_list_nil)
        { dcg_wlog ("Goal rule %s, LC %s", goal -> aname, lc_rule -> aname);
          dcg_internal_error ("reduce_arule");
        };
      for (alt_nr = 0; alt_nr < alts -> size; alt_nr++)
	{ /* Select member list */
	  affix_element_list elems = alts -> array[alt_nr] -> elems;
	  affix_element first_member = elems -> array[0];
	  affix_rule mrule;

	  /* Check that we have a leftcorner towards the goal, if not deplore */
	  if (first_member -> tag != TAGAffix_var) continue;
	  mrule = first_member -> Affix_var.vdef;
	  if (!equivalent_affix_rule (top_trule, mrule)) continue;

	  /* We have determined that our current top tree is really on its way towards the goal */
	  stat += parse_alternative (lc_rule, alt_nr, 1, terms, iptr,
				     cont_stack, tree_stack, result);
	};
    };
  return (stat);
}

/*
   Recursively match an affix term against a lattice affix rule
*/
static affix_term match_lattice_term (affix_rule arule, affix_term term)
{ switch (term -> tag)
    { case TAGTerminal:
	{ element elt = identify_lattice_element (term -> Terminal.marker);
	  affix_term new_term;
	  if (elt == element_nil) return (affix_term_nil);
	  if (!element_in_affix_rule (elt, arule)) return (affix_term_nil);
	  new_term = new_Terminal (term -> line, term -> col,
				   attach_string (term -> Terminal.marker));
	  new_term -> adef = arule;
	  new_term -> Terminal.edef = elt;
	  return (new_term);
	};
      case TAGDyop: 
	{ operator dop = term -> Dyop.dop;
	  affix_term narg1, narg2, new_term;
	  if ((dop != a_union) && (dop != a_part)) return (affix_term_nil);
	  narg1 = match_lattice_term (arule, term -> Dyop.arg1);
	  if (narg1 == affix_term_nil) return (affix_term_nil);
	  narg2 = match_lattice_term (arule, term -> Dyop.arg2);
	  if (narg2 == affix_term_nil) return (affix_term_nil);
	  new_term = new_Dyop (term -> line, term -> col, dop, narg1, narg2);
	  new_term -> adef = arule;
	  return (new_term);
	};
      case TAGEnclosed:
	{ affix_term_list terms = term -> Enclosed.terms;
	  if (terms -> size != 1) return (affix_term_nil);
	  return (match_lattice_term (arule, terms -> array[0]));
	};
      case TAGVar:
	{ affix_rule vrule = term -> adef;
	  if (vrule == affix_rule_nil) return (affix_term_nil);
	  if (!subset_of_affix_rule (vrule, arule)) return (affix_term_nil);
	  return (attach_affix_term (term));		/* No change, so share */
	};
      default: break;
    };
  return (affix_term_nil);
}

static void collect_interfering_elts (affix_term term, element_list elts, affix_rule_list arules)
{ switch (term -> tag)
    { case TAGTerminal:
	app_element_list (elts, term -> Terminal.edef);
	break;
      case TAGDyop:
	collect_interfering_elts (term -> Dyop.arg1, elts, arules);
	collect_interfering_elts (term -> Dyop.arg2, elts, arules);
	break;
      case TAGVar:
	app_affix_rule_list (arules, term -> adef);
	break;
      default: dcg_bad_tag (term -> tag, "collect_interfering_elts");
    };
}

static void add_term_elts_to_interference_matrix (affix_term term)
{ element_list elements = new_element_list ();
  affix_rule_list arules = new_affix_rule_list ();
  int ix, iy;
  collect_interfering_elts (term, elements, arules);
  for (ix = 0; ix < arules -> size; ix++)
    { for (iy = 0; iy < elements -> size; iy++)
        add_element_to_interference_matrix (elements -> array[iy], arules -> array[ix]);
      add_arule_to_interference_matrix (arules -> array[ix], term -> adef);
      add_arule_to_interference_matrix (term -> adef, arules -> array[ix]);
    };
  nonrec_detach_element_list (&elements);
  nonrec_detach_affix_rule_list (&arules);
}

/*
   Try and match a single affix term against an affix rule of a non tree type kind
*/
static int parse_non_tree_term (affix_rule arule, affix_term term, affix_term *result)
{ if (arule -> kind == arule_tree)
    return (0);
  if (term -> adef != affix_rule_nil)
    { /*
	 We get here if the term has been given a type during the precomputation.
         If the affix rule is the ANY affix type, we match.
         If the affix rules are similar, we match.
      */
      if ((arule -> kind == arule_any) || similar_affix_rule (arule, term -> adef))
	{ /* Ok */
	  *result = attach_affix_term (term);
	  return (1);
	}
      else return (0);
    }

  /* We have not got a type yet, try confront against lattice affix rules */
  if (arule -> kind == arule_lattice)
    { affix_term resolved = match_lattice_term (arule, term);
      if (resolved != affix_term_nil)
        { *result = resolved;
	  add_term_elts_to_interference_matrix (resolved);
	  return (1);
	};
    };

  /* No resolution found */
  return (0);
}

/*
   Check if an affix expression conforms to the given affix rule.
*/
int match_expr (affix_rule arule, affix_term_list expr, affix_term *result)
{ affix_term local = affix_term_nil;
  int status;
  if (dump_affix_rules)
    { dcg_hint ("Matching term list with affix rule %s", arule -> aname);
      if (full_verbose) pp_affix_term_list (dcg_error_file (), expr);
      dcg_hint ("");
    };

  /*
     Take one shortcut: if the affix expression is a single affix variable
     whose type is equivalent to the type of affix rule, we always have
     a match (Note: this also matches ANY against ANY)
  */
  if (is_a_single_affix_variable (expr) && similar_affix_rule (expr -> array[0] -> adef, arule))
    { local = attach_affix_term (expr -> array[0]);
      status = 1;
    }
  else if (arule -> kind == arule_tree)
    { /* For a tree type, we have to parse, resulting in a single affix term */
      affix_term_list tree_stack = new_affix_term_list ();
      madm_list cont_stack = new_madm_list ();
      status = parse_arule (arule, EOS_REQUIRED, expr, 0, cont_stack, tree_stack, &local);
      detach_madm_list (&cont_stack);
      detach_affix_term_list (&tree_stack);
    }
  else if (expr -> size == 1)
    { /* Only if the expression is a single term can it match a non tree type */
      status = parse_non_tree_term (arule, expr -> array[0], &local);
    }
  else status = 0;	/* A tree structure can never match a non tree affix rule */

  /* create the result */
  if (status == 1) *result = local;

  /* Return the result if it is there */
  return (status);
}
