/*
   File: lexicon_info.c
   Collects the runtime domains, type system and lexicon nonterminals.

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: lexicon_info.c,v 1.2 2012/08/11 20:19:13 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>


/* libeagbase includes */
#include <ebase_version.h>

/* Local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "contsens.h"
#include "affix_rules.h"
#include "lexicon_info.h"

/*
   Construct all runtime lattice super domains
*/
static void construct_runtime_domains ()
{ int ix;
  dcg_hint ("      constructing runtime lattice domains");
  for (ix = 0; ix < all_domains -> size; ix++)
    { domain dom = all_domains -> array[ix];
      int_list elts = dom -> elts;
      rt_element_list rt_elts = init_rt_element_list (elts -> size);
      int iy;
      for (iy = 0; iy < elts -> size; iy++)
	{ element elt = all_elements -> array[elts -> array[iy]];
	  rt_element rte = new_rt_element (attach_string (elt -> name),
					   attach_affix_value (elt -> value));
          app_rt_element_list (rt_elts, rte);
	};
      app_rt_domain_list (all_rt_domains, new_rt_domain (dom -> width, rt_elts));
    };
}

/*
   Construct the runtime typing system and determine which
   types can safely be used in the lexicon generator.
*/
static rt_alt construct_runtime_alt (affix_alternative alt, int marker)
{ affix_element_list elems;
  rt_elem_list rt_elems;
  int ix;
  if (alt -> tag != TAGAffix_sequence)
    dcg_bad_tag (alt -> tag, "construct_runtime_alt");
  elems = alt -> elems;
  rt_elems = init_rt_elem_list (elems -> size);
  for (ix = 0; ix < elems -> size; ix++)
    { affix_element elem = elems -> array[ix];
      rt_elem elt = rt_elem_nil;
      switch (elem -> tag)
	{ case TAGAffix_var:
	    elt = new_Ref (elem -> Affix_var.vdef -> anr);
	    break;
	  case TAGAffix_term:
	    elt = new_Marker (attach_string (elem -> Affix_term.tname));
	    break;
	  default: dcg_bad_tag (elem -> tag, "construct_runtime_alt");
	};
      app_rt_elem_list (rt_elems, elt);
    };
  return (new_rt_alt (marker, rt_elems));
}

static rt_type construct_runtime_type (affix_rule arule, int lexgen)
{ affix_value value;
  if (arule -> value != affix_value_nil) value = attach_affix_value (arule -> value);
  else value = new_Null_value (arule -> anr);
  if (arule -> tag == TAGAffix_synonym)
    return (new_Synonym_type (arule -> anr, arule -> aname, lexgen, value,
			      arule -> Affix_synonym.syndef -> anr));
  switch (arule -> kind)
    { case arule_any:  return (new_Any_type (arule -> anr, arule -> aname, lexgen, value));
      case arule_int:  return (new_Int_type (arule -> anr, arule -> aname, lexgen, value));
      case arule_real: return (new_Real_type (arule -> anr, arule -> aname, lexgen, value));
      case arule_text: return (new_Text_type (arule -> anr, arule -> aname, lexgen, value));
      case arule_lattice:
        return (new_Lattice_type (arule -> anr, arule -> aname, lexgen, value,
				  arule -> dom -> dnr));
      case arule_tree:
	{ affix_alternative_list alts;
	  rt_alt_list rt_alts;
	  int ix;
	  if (arule -> tag != TAGAffix_alts)
	    dcg_bad_tag (arule -> tag, "construct_runtime_type");
	  alts = arule -> Affix_alts.alts;
	  rt_alts = init_rt_alt_list (alts -> size);
	  for (ix = 0; ix < alts -> size; ix++)
	    app_rt_alt_list (rt_alts, construct_runtime_alt (alts -> array[ix], ix));
	  return (new_Tree_type (arule -> anr, arule -> aname, lexgen, value, rt_alts));
	};
      default: dcg_bad_tag (arule -> kind, "construct_runtime_type");
    };

  /* Default escape: keep gcc happy */
  return (rt_type_nil);
}

static int determine_lexgen_usability (affix_rule arule)
{ if (arule -> tag == TAGAffix_synonym)
    return (determine_lexgen_usability (arule -> Affix_synonym.syndef));
  switch (arule -> kind)
    { case arule_any: return (0);
      case arule_int:
      case arule_real:
      case arule_text:
	/*
	   We can use a scalar type if it is primitive or is a constant value
	   In all other cases, it has a meta definition, which we are not
	   going to check in the lexicon generator.
	*/
	if (arule -> tag == TAGAffix_prim) return (1);
	else return (arule -> value != affix_value_nil);
      case arule_lattice: return (1);
      case arule_tree:
	/* Only enumerable tree types are acceptable in the lexicon generator */
	if (arule -> tag != TAGAffix_alts)
	  dcg_bad_tag (arule -> tag, "determine_lexgen_usability");
	return (arule -> Affix_alts.enumerable);
      default: dcg_bad_tag (arule -> kind, "determine_lexgen_usability");
    };

  /* Default escape: keep gcc happy */
  return (0);
}

static void construct_runtime_typing_info ()
{ int ix;
  dcg_hint ("      constructing runtime type system");
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      int lexgen = determine_lexgen_usability (arule);
      rt_type rt = construct_runtime_type (arule, lexgen);
      app_rt_type_list (all_rt_types, rt);
    };
}

/*
   Validate that all lexicon rules and facts have parameters
   the lexicon generator can cope with: we do not want rules
   with a second level grammar (or tree type rules). For facts,
   we can also check that the critical parameters are of an
   appropriate type. While checking we create the lex_nont
   structure which will be written to the lexicon generator.
*/
static void check_usable_signature (int rnr, spec rspec)
{ prule pr = rspec -> pr;
  signature sig = rspec -> rsig;
  dir_list dirs = rspec -> dirs;
  int fact = (rspec -> rkind == r_fact);
  lex_nont lnont = new_lex_nont (rnr, -1, rspec -> rname_parts, rspec -> rname_chars,
				 0, new_int_list (), new_int_list ());
  int ix;
  for (ix = 0; ix < sig -> size; ix++)
    { affix_rule arule = sig -> array[ix];
      dir pdir = dirs -> array[ix];
      rt_type rt = all_rt_types -> array[arule -> anr];
      int crit = (pdir == d_inherited);
      app_int_list (lnont -> formals, arule -> anr);
      app_int_list (lnont -> crits, crit);
      if (crit) lnont -> ncrits++;
      if (!rt -> lexgen)
	contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
			       "Affix rule '%s' cannot be used in the lexicon generator",
			       arule -> aname);
      if (fact && crit && (arule -> kind != arule_text))
	contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
			       "The critical parameters of facts must have type TEXT");
    };
  if (lnont -> ncrits && !fact)
    contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
			   "Lexicon rules may not have inherited parameters");
  if (!lnont -> ncrits && fact)
    contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
			   "Facts must have at least one critical parameter");
  if (fact)
    { lnont -> fact_nr = nr_facts;
      nr_facts++;
    };

  /* Register lexicon nonterminal */
  rspec -> lnr = all_lex_nonts -> size;
  app_lex_nont_list (all_lex_nonts, lnont);
}

static void collect_lexicon_rules_and_facts ()
{ int ix;
  dcg_hint ("      checking lexicon rules and facts");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule srule = all_syntax_rules -> array[ix];
      spec rspec = srule -> rspec;
      if ((rspec -> rkind == r_lexicon) || (rspec -> rkind == r_fact))
        check_usable_signature (srule -> rnr, rspec);
    };
}

void collect_lexicon_info ()
{ dcg_warning (0, "   collecting runtime domains and rules...");
  construct_runtime_domains ();
  construct_runtime_typing_info ();
  collect_lexicon_rules_and_facts ();
  dcg_panic_if_errors ();
}
