/*
   File: groups.c
   Checks groups and assigns the initial member marking for nullability

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: groups.c,v 1.17 2013/01/03 15:21:21 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* support lib includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_dstring.h>

/* libeagbase includes */
#include <ebase_input.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "globals.h"
#include "options.h"
#include "affix_rules.h"
#include "contsens.h"
#include "lexemes.h"
#include "match.h"
#include "expr.h"
#include "match.h"
#include "rules.h"
#include "groups.h"

/*
   While checking alternatives, the checking of the members will introduce a
   restructuring of the alternative, as local groups and options will be rewritten
   into calls of anonymous syntax rules and all other kinds of members transformed
   into their resolved equivalent. Subsequently, the transduction of the
   alternative must be checked against the transformed structure.
*/

/*
   Rule and quasi rule identification and resolution
*/
static string make_error_rname (string_list rname_parts, int_list rname_chars)
{ dstring ds = dcg_init_dstring(64);
  int in_args = 0;
  int ix, cidx;

  for (ix = 0, cidx = 0; ix < rname_chars -> size; ix++)
    if (rname_chars -> array[ix])
      { /* Add a new name part */
        dcg_sprintfa_dstring (ds, "%s %s", (in_args)?")":"", rname_parts -> array[cidx]);
	in_args = 0;
        cidx++;
      }
    else
      { /* Add a new formal placeholder */
	if (!in_args) dcg_append_dstring (ds, " (_");
	else dcg_append_dstring (ds, ",_");
	in_args = 1;
      };

  if (in_args) dcg_append_dstring_c (ds, ')');
  return (dcg_finish_dstring (&ds));
}

static void check_apar (definition d, apar ap)
{ (void) check_expr (d, ap -> aexp);
}

static void check_apar_list (definition d, apar_list apars)
{ int ix;
  for (ix = 0; ix < apars -> size; ix++)
    check_apar (d, apars -> array[ix]);
}

static int signature_matches_actuals (signature sig, apar_list apars, affix_term_list *res_apars)
{ int status = 1;
  int ix;
  *res_apars = init_affix_term_list (sig -> size);
  for (ix = 0; ix < sig -> size; ix++)
    { affix_term local_term = affix_term_nil;
      status *= match_expr (sig -> array[ix], apars -> array[ix] -> aexp, &local_term);
      if (status == 0)
	{ detach_affix_term_list (res_apars);
	  return (0);
	};
      app_affix_term_list (*res_apars, local_term);
    };
  return (status);
}

static void try_resolve_call (definition d, member m, rdecl_list rdecls, int_list rname_chars,
			      apar_list apars, string ename, int quasi)
{ /* Collect all possible call resolutions */
  call_solution_list solutions = new_call_solution_list ();
  call_solution the_solution;
  dir_list adirs;
  spec rspec;
  int ix;
  for (ix = 0; ix < rdecls -> size; ix++)
    { rdecl rd = rdecls -> array[ix];
      affix_term_list res_pars;
      int nr_matches = signature_matches_actuals (rd -> rsig, apars, &res_pars);
      if (nr_matches)
        { call_solution solution = new_call_solution (res_pars, nr_matches);
	  solution -> rdef = rd -> rdef;
	  app_call_solution_list (solutions, solution);
	}
    };

  /* No solutions, complain */
  if (solutions -> size == 0)
    { contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
		      	     "cannot identify call to '%s'", ename);
      detach_call_solution_list (&solutions);
      return;
    };

  /* More than 1 solution: we have a multiple identification of rules */
  if (solutions -> size > 1)
    { contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
		      	     "the call to '%s' is ambiguous", ename);
      dcg_wlog ("the candidates are:");
      for (ix = 0; ix < solutions -> size; ix++)
	{ spec sp = solutions -> array[ix] -> rdef -> rspec;
	  dcg_wlog (sp -> canonic_name);
	};
      detach_call_solution_list (&solutions);
      return;
    };

  /* Only one solution */
  the_solution = solutions -> array[0];
  rspec = the_solution -> rdef -> rspec;
  if (the_solution -> nr_matches > 1)
    { /* Still bad, we know what to call, but there is ambiguity in the arguments */
      contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
			     "the call to '%s' is ambiguous in its actual arguments",
			     rspec -> canonic_name);
      detach_call_solution_list (&solutions);
      return;
    };

  /* We have a single solution, check the writing to warn */
  if (!equal_int_list (rspec -> rname_chars, rname_chars))
    contsens_warning_by_gnr (d -> pr -> gnr, m -> line, m -> col,
			"the call to '%s' has its arguments and parts in a different order",
			rspec -> canonic_name);

  /* All checks done, transform the call */
  adirs = init_dir_list (apars -> size);
  for (ix = 0; ix < apars -> size; ix++)
    app_dir_list (adirs, apars -> array[ix] -> adir);
  if (quasi)
    { detach_string_list (&m -> Quasi.qname_parts);
      detach_int_list (&m -> Quasi.qname_chars);
      detach_apar_list (&m -> Quasi.args);
    }
  else
    { detach_string_list (&m -> Call.rname_parts);
      detach_int_list (&m -> Call.rname_chars);
      detach_apar_list (&m -> Call.apars);
    }
  m -> tag = TAGRes_call;
  m -> Res_call.rdef = the_solution -> rdef;
  m -> Res_call.args = attach_affix_term_list (the_solution -> apars);
  m -> Res_call.adirs = adirs;
  detach_call_solution_list (&solutions);
}

static void transform_and_check_call (definition d, member m)
{ string_list rname_parts = m -> Call.rname_parts;
  int_list rname_chars = m -> Call.rname_chars;
  apar_list apars = m -> Call.apars;
  rdecl_list rdecls;

  /* Create the error rule name from parts and check the apars as precomputation */
  string ename = make_error_rname (rname_parts, m -> Call.rname_chars);
  check_apar_list (d, apars);
  rdecls = pre_identify_rule (rname_parts, apars -> size);
  if (rdecls == rdecl_list_nil)
    contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
			   "cannot identify call to '%s'", ename);
  else if (rdecls -> size == 1)
    { string cname = rdecls -> array[0] -> rdef -> rspec -> canonic_name;
      try_resolve_call (d, m, rdecls, rname_chars, apars, cname, 1);
    }
  else try_resolve_call (d, m, rdecls, rname_chars, apars, ename, 0);
  detach_string (&ename);

  /* If the tag stays TAGCall, there was an error */
}

/*
   Check for arguments with a REGEXP formal direction
*/
static void resolve_regexp_actual (int gnr, affix_term act, dir adir, int kind)
{ int termnr, is_regexp;
  nullability empty;
  char *text;
  if ((adir != d_unknown) && (adir != d_inherited))
    contsens_error_by_gnr (gnr, act -> line, act -> col, "Illegal direction for regexp argument");

  if (act -> tag != TAGText)
    { contsens_error_by_gnr (gnr, act -> line, act -> col,
			     "An actual regexp argument should be a TEXT constant");
      return;
    };
  text = act -> Text.txt;
  analyze_lexeme (gnr, act -> line, act -> col, 1, text, kind, &termnr, &is_regexp, &empty);
  if (!is_regexp)
    { contsens_error_by_gnr (gnr, act -> line, act -> col,
 			     "This regexp is actually a simple terminal");
      return;
    }
  if (empty != e_never_produces_empty)
    { contsens_error_by_gnr (gnr, act -> line, act -> col,
 			     "This regexp may match empty");
      return;
    }

  act -> tag = TAGRegexp;
  act -> Regexp.text = text;
  act -> Regexp.termnr = termnr;
}

static int regexp_kind_from_name (char *name)
{ if (streq (name, "$MATCH")) return (RegexpMatch);
  else if (streq (name, "$SKIP")) return (RegexpSkip);
  else return (0);
}

static void try_resolve_regexps_in_quasi (definition d, member m, char *name)
{ affix_term_list args = m -> Res_call.args;
  dir_list adirs = m -> Res_call.adirs;
  rule qrule = m -> Res_call.rdef;
  dir_list fdirs = qrule -> rspec -> dirs;
  int ix;
  for (ix = 0; ix < fdirs -> size; ix++)
    if (fdirs -> array[ix] == d_regexp)
      { int kind = regexp_kind_from_name (name);
	if (!kind)
	  dcg_abort ("try_resolve_regexps_in_quasi", "part is %s", name);

        resolve_regexp_actual (d -> pr -> gnr, args -> array[ix],
			       adirs -> array[ix], kind);
	m -> empty = e_never_produces_empty;
      };
}

static void transform_and_check_quasi_call (definition d, member m)
{ string_list qname_parts = m -> Quasi.qname_parts;
  int_list qname_chars = m -> Quasi.qname_chars;
  char *name = qname_parts -> array[0];
  apar_list apars = m -> Quasi.args;
  rdecl_list rdecls;
   
  /* Create the error rule name from parts and check the apars as precomputation */
  string ename = make_error_rname (qname_parts, qname_chars);
  check_apar_list (d, apars);
  rdecls = pre_identify_quasi_rule (qname_parts, apars -> size);
  if (rdecls == rdecl_list_nil)
    contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
                           "cannot identify call to '%s'", ename);
  else if (rdecls -> size == 1)
    { string cname = rdecls -> array[0] -> rdef -> rspec -> canonic_name;
      try_resolve_call (d, m, rdecls, qname_chars, apars, cname, 1);
    }
  else try_resolve_call (d, m, rdecls, qname_chars, apars, ename, 1);

  /* If the tag stays TAGQuasi, there was an error */
  if (m -> tag == TAGRes_call)
    try_resolve_regexps_in_quasi (d, m, name);
}

/*
   Resolution of guards
*/
static void resolve_comparison (definition d, confrontation conf, res_conf_list rconfs)
{ /* Precheck the sides of the guard */
  affix_term_list lhs = conf -> lhs;
  affix_term_list rhs = conf -> rhs;
  affix_term lhs_term, rhs_term;
  affix_rule lhs_rule, rhs_rule;
  (void) check_expr (d, lhs);
  (void) check_expr (d, rhs);

  /* Check the form of the lhs and rhs */
  if (!is_a_single_affix_variable (lhs) || !is_a_single_affix_variable (rhs))
    { contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "a comparison can only be applied to two variables");
      return;
    };
  lhs_term = attach_affix_term (lhs -> array[0]);
  rhs_term = attach_affix_term (rhs -> array[0]);
  lhs_rule = lhs_term -> adef;
  rhs_rule = rhs_term -> adef;

  /* Type check */
  if (lhs_rule == AFFIX_ERROR) return;
  if (rhs_rule == AFFIX_ERROR) return;

  if (!similar_affix_rule (lhs_rule, rhs_rule))
    { /* Typing error */
      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "the two sides of this comparison have different types");
    }
  else if ((lhs_rule -> kind == arule_lattice) && (rhs_rule -> kind == arule_lattice))
    { /*
	 Comparison for two affix variables of the lattice kind,
	 that may get restricted at a later time is a bad thing
      */
      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "the two sides of this comparison have a lattice type");
    };

  /* Comparison resolved */
  app_res_conf_list (rconfs, new_res_conf (conf -> typ, lhs_term, rhs_term));
}

static void resolve_restrict (definition d, confrontation conf, res_conf_list rconfs)
{ /* Precheck the sides of the guard */
  affix_term_list lhs = conf -> lhs;
  affix_term_list rhs = conf -> rhs;
  affix_term lhs_term, rhs_term;
  affix_rule lhs_rule;
  int status;
  (void) check_expr (d, lhs);
  (void) check_expr (d, rhs);

  /* Check the form of the lhs */
  if (!is_a_single_affix_variable (lhs))
    { contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "a restrict must have a variable at its left hand side");
      return;
    };
  lhs_term = attach_affix_term (lhs -> array[0]);
  lhs_rule = lhs_term -> adef;

  /* The right hand side may be an affix expression that must match */
  /* If there is no match then the intersection of both domains is empty */
  status = match_expr (lhs_rule, rhs, &rhs_term);
  if (status == 0)
    { /* Rhs does not conform to lhs rule */
      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
          "the two sides of this restrict have different types or an empty intersection");
      return;
    };
  app_res_conf_list (rconfs, new_res_conf (restrict, lhs_term, rhs_term));
}

static void resolve_assignment (definition d, confrontation conf, res_conf_list rconfs)
{ /* Check the form of the lhs and rhs */
  affix_term_list lhs = conf -> lhs;
  affix_term_list rhs = conf -> rhs;
  affix_term lhs_term, rhs_term;
  affix_rule lhs_rule, rhs_rule;
  int status = 0;
  (void) check_expr (d, lhs);
  (void) check_expr (d, rhs);

  /* Determine which side is a single affix variable, so that we know the type */
  if (is_a_single_affix_variable (lhs))
    { /* Lhs determines the type */
      lhs_term = attach_affix_term (lhs -> array[0]);
      lhs_rule = lhs_term -> adef;
      if (is_a_single_affix_variable (rhs))
	{ /* confrontation has the form A -> B or A <-> B */
	  rhs_term = attach_affix_term (rhs -> array[0]);
          rhs_rule = rhs_term -> adef;
	  if (lhs_rule == AFFIX_ERROR) return;
	  if (rhs_rule == AFFIX_ERROR) return;
      	  if (!similar_affix_rule (lhs_rule, rhs_rule))
	    { /* Typing error */
	      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "the two sides of this assignment have different types");
	    }
	  else app_res_conf_list (rconfs, new_res_conf (conf -> typ, lhs_term, rhs_term));
	}
      else
	{ /* confrontation has the form A -> expression or A <-> expression */
	  if (lhs_rule == AFFIX_ERROR) return;
	  status = match_expr (lhs_rule, rhs, &rhs_term);
	  if (status == 0)
	    { /* Rhs does not conform to lhs rule */
	      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "the two sides of this assignment have different types");
	    }
	  else if (status == 1)
	    { /* Rhs conforms to lhs rule */
	      switch (lhs_rule -> kind)
		{ case arule_int:
		  case arule_real:
		  case arule_text:
		    if (conf -> typ == bidir_assign)
		      { contsens_warning_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
			"bidirectional assignment replaced by reverse unidirectional assignment");
			app_res_conf_list (rconfs, new_res_conf (assign, rhs_term, lhs_term));
		      }
		    else app_res_conf_list (rconfs, new_res_conf (arith_equal, lhs_term, rhs_term));
		    break;
		  case arule_tree:
		    app_res_conf_list (rconfs, new_res_conf (assign, lhs_term, rhs_term));
		    break;
		  default:
		    /* Neither, lattice nor any can be assigned */
		    contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
					   "neither lattice nor any types can be split");
		}
	    }
	  else if (status > 1)
	    { /* Ambiguous split */
	      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
				     "this split is ambiguous");
	    }
	}
    }
  else if (is_a_single_affix_variable (rhs))
    { /* Rhs determines the type, confrontation has the form expr -> A or expr <-> A */
      affix_term rhs_term = attach_affix_term (rhs -> array[0]);
      affix_rule rhs_rule = rhs_term -> adef;
      if (rhs_rule == AFFIX_ERROR) return;
      status = match_expr (rhs_rule, lhs, &lhs_term);
      if (status == 0)
        { /* Lhs does not conform to rhs rule */
          contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                                 "the two sides of this assignment have different types");
        }
      else if (status == 1)
        { /* Lhs conforms to rhs rule */
	  switch (rhs_rule -> kind)
	    { case arule_int:
              case arule_real:
              case arule_text:
                { if (conf -> typ == bidir_assign)
		    contsens_warning_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
			"bidirectional assignment replaced by unidirectional assignment");
		  app_res_conf_list (rconfs, new_res_conf (assign, lhs_term, rhs_term));
		}; break;
	      case arule_tree:
                app_res_conf_list (rconfs, new_res_conf (assign, lhs_term, rhs_term));
		break;
	      default:
                /* Neither, lattice nor any can be assigned */
                contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                                       "neither lattice or any types can be split");
	    }
        }
      else if (status > 1)
        { /* Ambiguous join */
          contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                                 "this join is ambiguous");
        }
    }
  else
    { /* Nothing to determine the type from */
      contsens_error_by_gnr (d -> pr -> gnr, conf -> line, conf -> col,
                             "this guard is neither a split, join or (bidirectional) assign");
    }
}

static void resolve_confrontation (definition d, confrontation conf, res_conf_list rconfs)
{ switch (conf -> typ)
    { case equal:
      case unequal: resolve_comparison (d, conf, rconfs); break;
      case restrict: resolve_restrict (d, conf, rconfs); break;
      case bidir_assign:
      case undetermined: resolve_assignment (d, conf, rconfs); break;
      default: dcg_bad_tag (conf -> typ, "resolve_confrontation");
    };
}

static void transform_and_check_guard (definition d, member m)
{ confrontation_list confs = m -> Guard.confs;
  res_conf_list rconfs = init_res_conf_list (confs -> size);
  int ix;
  for (ix = 0; ix < confs -> size; ix++)
    resolve_confrontation (d, confs -> array[ix], rconfs);
  detach_confrontation_list (&m -> Guard.confs);
  m -> tag = TAGRes_guard;
  m -> Res_guard.rconfs = rconfs;
  m -> empty = e_always_produces_empty;
}

static void transform_and_check_terminal (definition d, member m)
{ affix_term local_term = affix_term_nil;
  int termnr, is_regexp;
  nullability empty;
  char *text = m -> Term.text;
  apar par = m -> Term.par;
  dir adir = d_unknown;
  analyze_lexeme (d -> pr -> gnr, m -> line, m -> col, !agfl_compatible, text, RegexpMatch,
		  &termnr, &is_regexp, &empty);
  if (empty != e_never_produces_empty)
    contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
			   "This %s may match empty", (is_regexp)?"regular expression":"terminal"); 
  if (par != apar_nil)
    { affix_rule text_arule = get_primitive_type (arule_text);
      check_apar (d, par);
      if (!match_expr (text_arule, par -> aexp, &local_term))
	{ contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
				 "The argument of a %s must be of type TEXT",
				 (is_regexp)?"regular expression":"terminal");
	  local_term = affix_term_nil;
	}
      else adir = par -> adir;
      detach_apar (&par);
    };
  m -> tag = TAGRes_term;
  m -> Res_term.text = text;
  m -> Res_term.termnr = termnr;
  m -> Res_term.is_regexp = is_regexp;
  m -> Res_term.adir = adir;
  m -> Res_term.arg = local_term;
  m -> empty = e_never_produces_empty;
}

static void transform_and_check_group (definition d, member m)
{ group grp = m -> Group.grp;
  rule child_rule = make_anonymous_rule (d, grp, 0);
  m -> tag = TAGRes_call;
  m -> Res_call.rdef = child_rule;
  m -> Res_call.args = new_affix_term_list ();
  m -> Res_call.adirs = new_dir_list ();
  child_rule -> Anonymous_group.call = m;
}

static void transform_and_check_option (definition d, member m)
{ group grp = m -> Option.grp;
  rule child_rule = make_anonymous_rule (d, grp, 1);
  m -> tag = TAGRes_call;
  m -> Res_call.rdef = child_rule;
  m -> Res_call.args = new_affix_term_list ();
  m -> Res_call.adirs = new_dir_list ();
  child_rule -> Anonymous_option.call = m;
}

static void transform_and_check_operator (definition d, member m)
{ m -> Op.gnr = d -> pr -> gnr;
  m -> empty = e_always_produces_empty;
}

static void check_member (definition d, member m)
{ switch (m -> tag)
    { case TAGCall:	transform_and_check_call (d, m); break;
      case TAGGuard:	transform_and_check_guard (d, m); break;
      case TAGTerm:	transform_and_check_terminal (d, m); break;
      case TAGQuasi:	transform_and_check_quasi_call (d, m); break;
      case TAGGroup:	transform_and_check_group (d, m); break;
      case TAGOption:	transform_and_check_option (d, m); break;
      case TAGOp:	transform_and_check_operator (d, m); break;
      default:	dcg_bad_tag (m -> tag, "check_member");
    };
}

/*
   This check is purely for identification and the like.
   It does not concern itself yet with the glue between the members
*/
static void check_fwo_group (definition d, fwo_group fwo)
{ switch (fwo -> tag)
    { case TAGSingle:
	check_member (d, fwo -> Single.mem);
	break;
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    { member m = mems -> array[ix];
	      if (m -> tag == TAGOp)
	        contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
				 "A control operator cannot be a part of a free word order group");
	      else if (m -> tag == TAGGuard)
		contsens_error_by_gnr (d -> pr -> gnr, m -> line, m -> col,
                                 "A guard cannot be a part of a free word order group");
	      else check_member (d, mems -> array[ix]);
	    };
	};
    };
}

static void check_fwo_groups (definition d, fwo_group_list fl)
{ int ix;
  for (ix = 0; ix < fl -> size; ix++)
    check_fwo_group (d, fl -> array[ix]);
}

static void check_alternative (definition d, alternative alt)
{ check_fwo_groups (d, alt -> members);
  /* check_alt_transduction (alt -> trans); */
}

void check_group (definition d, group grp)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    check_alternative (d, alts -> array[ix]);
}
