/*
   File: grammars.c
   Maintains and checks the list of all grammars.

   Copyright (C) 2008-2010 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: grammars.c,v 1.8 2012/02/01 12:29:06 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>
#include <dcg_plist_ops.h>

/* local includes */
#include "eag_ds.h"
#include "globals.h"
#include "grammars.h"

/* Grammar addition and lookup */
void register_grammar (grammar gra)
{ gra -> gnr = all_grammars -> size;
  app_grammar_list (all_grammars, gra);
}

grammar identify_grammar (string gname)
{ int ix;
  for (ix = 0; ix < all_grammars -> size; ix++)
    if (streq (gname, all_grammars -> array[ix] -> gname))
      return (all_grammars -> array[ix]);
  return (grammar_nil);
}

static void add_lexicon_inputs (grammar gra)
{ int ix;
  for (ix = 0; ix < gra -> lexica -> size; ix++)
    add_uniquely_to_string_list (all_lexica, gra -> lexica -> array[ix]);
  for (ix = 0; ix < gra -> fact_tables -> size; ix++)
    add_uniquely_to_string_list (all_fact_tables, gra -> fact_tables -> array[ix]);
  for (ix = 0; ix < gra -> triple_databases -> size; ix++)
    add_uniquely_to_string_list (all_triple_databases, gra -> triple_databases -> array[ix]);
}

/* Checks on grammars */
void pre_analyze_grammars ()
{ int ix;
  dcg_warning (0, "   checking grammars...");

  /* Check that we have no multiple root rules */
  for (ix = 0; ix < all_grammars -> size; ix++)
    { grammar gra = all_grammars -> array[ix];
      if (gra -> root_call != member_nil)
        { if (root_grammar == grammar_nil)
	    root_grammar = gra;
	  else dcg_error (0, "Multiple root rules found");
	};

      add_lexicon_inputs (gra);
    };

  /* Check that we have a root rule (and grammar) */
  if (root_grammar == grammar_nil)
    { root_grammar = all_grammars -> array[0];
      dcg_warning (0, "      No root rule found, taking first rule of grammar %s as root",
	           root_grammar -> gname);
    };

  /* Finish up */
  dcg_panic_if_errors ();
  dcg_hint ("      collected %d grammars to compile", all_grammars -> size);
}

void dump_grammars ()
{ dcg_wlog ("List of all grammars:");
  pp_grammar_list (dcg_error_file (), all_grammars);
  dcg_wlog ("");
}
