/*
   File: constant.c
   Tries to evaluate constant affix rules and affix terms to prepare
   for code generation and for static expressions like PENALTY's.
   As a side effect affix terms may be flattened.

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: constant.c,v 1.12 2012/12/16 12:57:51 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* libeagbase includes */
#include <ebase_ds.h>
#include <ebase_utils.h>
#include <ebase_hash.h>
#include <ebase_affix_value_utils.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "contsens.h"
#include "affix_rules.h"
#include "lexemes.h"
#include "constant.h"

/*
   The following code tries to evaluate constant affix rules
*/
static affix_value try_evaluate_affix_element (affix_element elem)
{ switch (elem -> tag)
    { case TAGAffix_var:
	{ affix_rule vdef = elem -> Affix_var.vdef;
	  if (vdef -> value != affix_value_nil)
	    return (attach_affix_value (vdef -> value));
	}; break;
      case TAGAffix_text:
	{ int termnr = elem -> Affix_text.termnr;
	  terminal term = all_terminals -> array[termnr];
	  if (elem -> Affix_text.is_regexp) break;
	  if (term -> marker) break;
	  if (utf8_processing)
	    return (new_Text_value (-1, ebs_convert_utf8_string (term -> text)));
	  else return (new_Text_value (-1, ebs_convert_non_utf8_string (term -> text)));
	}; break;
      case TAGAffix_inum:
	return (new_Int_value (-1, elem -> Affix_inum.ival));
      case TAGAffix_rnum:
	return (new_Real_value (-1, elem -> Affix_rnum.rval));
      default: dcg_bad_tag (elem -> tag, "try_evaluate_affix_element");
    };
  return (affix_value_nil);
}

static void try_evaluate_affix_rule (affix_rule arule, int *change)
{ affix_value val = affix_value_nil;
  affix_alternative_list alts;
  affix_alternative alt;
  affix_value_list vals;
  int ix;

  /* already evaluated */
  if (arule -> value != affix_value_nil) return;

  /* primitive or synonym */
  if (arule -> tag == TAGAffix_prim) return;
  if (arule -> tag == TAGAffix_synonym)
    { affix_rule syndef = arule -> Affix_synonym.syndef;
      if (syndef -> value != affix_value_nil)
	{ arule -> value = attach_affix_value (syndef -> value);
	  *change = 1;
	};
      return;
    };

  /* more than one alternative */
  alts = arule -> Affix_alts.alts;
  if (alts -> size > 1) return;
  alt = alts -> array[0];

  /* Evaluate the elements */
  vals = init_affix_value_list (alt -> elems -> size);
  for (ix = 0; ix < alt -> elems -> size; ix++)
    { affix_element elem = alt -> elems -> array[ix];
      affix_value val;
      if (elem -> tag == TAGAffix_term) /* Do not evaluate markers */
	continue;
      val = try_evaluate_affix_element (alt -> elems -> array[ix]);
      if (val == affix_value_nil)
	{ /* Could not evaluate this element */
	  detach_affix_value_list (&vals);
	  return;
	}
      else app_affix_value_list (vals, val);
    };

  /* All elements could be evaluated, try construct value */
  switch (alt -> tag)
    { case TAGAffix_sequence:
	val = new_Composed_value (arule -> anr, 0, vals);
	break;
      case TAGAffix_concat:
	val = ebs_concatenate_text_values (vals);
	detach_affix_value_list (&vals);
	break;
      case TAGAffix_single:
	val = vals -> array[0];
	nonrec_detach_affix_value_list (&vals);
	break;
      default: dcg_bad_tag (alt -> tag, "try_evaluate_affix_rule");
    };
  val -> rule_nr = arule -> anr;
  arule -> value = val;
}

/*
   The following code checks if an affix rule of the tree kind is enumerable
*/
static int enumerable_elements (affix_element_list elems)
{ int ix;
  for (ix = 0; ix < elems -> size; ix++)
    if (elems -> array[ix] -> tag != TAGAffix_term)
      return (0);
  return (1);
}

static int enumerable_alts (affix_alternative_list alts)
{ int ix;
  for (ix = 0; ix < alts -> size; ix++)
    if (!enumerable_elements (alts -> array[ix] -> elems))
      return (0);
  return (1);
}

static void check_enumerable_affix_rules ()
{ int ix;
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      if (arule -> kind != arule_tree) continue;
      if (arule -> tag != TAGAffix_alts) continue;
      arule -> Affix_alts.enumerable = enumerable_alts (arule -> Affix_alts.alts);
    };
}

static void try_dump_evaluated_affix_rules ()
{ int ix;
  if (!dump_properties) return;
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      affix_value value = arule -> value;
      if ((arule -> tag == TAGAffix_alts) && arule -> Affix_alts.enumerable)
	dcg_wlog ("Affix rule %s is enumerable", arule -> aname);
      if (value == affix_value_nil) continue;
      dcg_eprint ("Affix rule %s has value: ", arule -> aname);
      pp_affix_value (dcg_error_file (), value);
      dcg_wlog ("");
    };
}

/*
   Constant evaluation within syntax rules actually does more
   than just the evaluation. All constants (either by evaluation
   or by themselves) are collected in a global list of affix constants.

   In the syntax rules, constants are then replaced by anonymous
   affix variables which are added to the list of local (definition)
   variables. At runtime they are then immediately created and
   initialized. This may introduce a bit overhead for local groups
   and options. Note that this procedure also applies to lattice
   affixes.

   Before this replacement, affix expressions are simplified.
*/
static void collect_terms (affix_term_list concs, affix_term term)
{ affix_term_list srcs;
  int ix;
  if (term -> tag != TAGConcat)
    { app_affix_term_list (concs, term);
      return;
    };

  /* Copy the constituents */
  srcs = term -> Concat.terms;
  for (ix = 0; ix < srcs -> size; ix++)
    app_affix_term_list (concs, srcs -> array[ix]);

  nonrec_detach_affix_term_list (&srcs);
  dcg_detach ((void **) &term);
}

static void simplify_affix_term (affix_term *term, definition def)
{ affix_term dterm = *term;
  switch (dterm -> tag)
    { case TAGTerminal: break;
      case TAGDyop:
        { simplify_affix_term (&dterm -> Dyop.arg1, def);
          simplify_affix_term (&dterm -> Dyop.arg2, def);
	  if (dterm -> Dyop.dop == text_plus_text)
            { affix_term_list concs = new_affix_term_list ();
              collect_terms (concs, dterm -> Dyop.arg1);
              collect_terms (concs, dterm -> Dyop.arg2);
              *term = new_Concat (dterm -> line, dterm -> col, concs);
              dcg_detach ((void **) &dterm);
            };
	}; break;
      case TAGMonop:
        simplify_affix_term (&dterm -> Monop.arg, def);
	break;
      case TAGAst:
	{ affix_term_list terms = dterm -> Ast.terms;
	  int ix;
	  for (ix = 0; ix < terms -> size; ix++)
	    simplify_affix_term (&terms -> array[ix], def);
	}; break;
      case TAGText:
	{ char *text = dterm -> Text.txt;
	  int warn;
	  dterm -> Text.txt = normalize_text_constant (text, &warn);
	  dterm -> adef = get_primitive_type (arule_text);
	  detach_string ((void **) &text);
	  if (warn)
	    contsens_warning_by_gnr (def -> pr -> gnr, dterm -> line, dterm -> col,
				     "Superfluous escape sequence");
	}; /* Fall through */
      case TAGConcat: /* simplification of shared terms */
      case TAGInt:
      case TAGReal:
      case TAGRegexp:
      case TAGVar: break;
      default: dcg_bad_tag (dterm -> tag, "simplify_affix_term");
    };
}

static affix_value evaluate_dyadic_operator (affix_term term, affix_value value1,
					     affix_value value2, definition def)
{ tags_affix_value tag1 = TAGInt_value;
  tags_affix_value tag2 = TAGInt_value;
  switch (term -> Dyop.dop)
    { case a_union:
	{ affix_value_list vals;
	  affix_value result;
	  int dom1, dom2;
	  switch (value1 -> tag)
	    { case TAGSmall_lattice: dom1 = value1 -> Small_lattice.dom; break;
	      case TAGLarge_lattice: dom1 = value1 -> Large_lattice.dom; break;
	      default:
                contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				       "Typing error during constant evaluation");
                return (affix_value_nil);
            };
	  switch (value2 -> tag)
	    { case TAGSmall_lattice: dom2 = value2 -> Small_lattice.dom; break;
	      case TAGLarge_lattice: dom2 = value2 -> Large_lattice.dom; break;
	      default:
                contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				       "Typing error during constant evaluation");
                return (affix_value_nil);
            };
	  if ((value1 -> tag != value2 -> tag) || (dom1 != dom2))
            { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				     "Typing error during constant evaluation");
              return (affix_value_nil);
            };
	  vals = new_affix_value_list ();
	  app_affix_value_list (vals, value1);
	  app_affix_value_list (vals, value2);
	  result = ebs_join_lattice_values (vals);
	  nonrec_detach_affix_value_list (&vals);
	  return (result);
	};
      case a_part:
	contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
			       "Partition operator not yet implemented");
	return (affix_value_nil);
      case real_times_real:
      case real_plus_real:
      case real_minus_real:
	tag1 = TAGReal_value;
	tag2 = TAGReal_value;
        break;
      case text_plus_text: tag1 = TAGText_value;	/* Fall through */
      case int_times_text: tag2 = TAGText_value;	/* Fall through */
      case modulo:
      case divides:
      case shift_left:
      case shift_right:
      case bitwise_xor:
      case bitwise_or:
      case bitwise_and:
      case int_times_int:
      case int_minus_int:
      case int_plus_int: break;
      default: dcg_bad_tag (term -> Dyop.dop, "evaluate_dyadic_operator");
    };

  /* Typecheck value1 and value2 */
  if (value1 -> tag != tag1)
    { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
			     "Typing error during constant evaluation");
      return (affix_value_nil);
    };
  if (value2 -> tag != tag2)
    { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
			     "Typing error during constant evaluation");
      return (affix_value_nil);
    };

  /* Evaluate */
  switch (term -> Dyop.dop)
    { case real_times_real:
	return (new_Real_value (-1, value1 -> Real_value.rval * value2 -> Real_value.rval));
      case real_plus_real:
	return (new_Real_value (-1, value1 -> Real_value.rval + value2 -> Real_value.rval));
      case real_minus_real:
	return (new_Real_value (-1, value1 -> Real_value.rval - value2 -> Real_value.rval));
      case int_times_text:
	return (ebs_replicate_text_value (value1, value2));
      case modulo:
	if (!value2 -> Int_value.ival)
	  { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				   "Division by 0 during constant evaluation");
	    return (affix_value_nil);
	  }
	else return (new_Int_value (-1, value1 -> Int_value.ival % value2 -> Int_value.ival));
      case divides:
	if (!value2 -> Int_value.ival)
	  { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				   "Division by 0 during constant evaluation");
	    return (affix_value_nil);
	  }
	else return (new_Int_value (-1, value1 -> Int_value.ival / value2 -> Int_value.ival));
      case shift_left:
	return (new_Int_value (-1, value1 -> Int_value.ival << value2 -> Int_value.ival));
      case shift_right:
	return (new_Int_value (-1, value1 -> Int_value.ival >> value2 -> Int_value.ival));
      case bitwise_xor:
	return (new_Int_value (-1, value1 -> Int_value.ival ^ value2 -> Int_value.ival));
      case bitwise_or:
	return (new_Int_value (-1, value1 -> Int_value.ival | value2 -> Int_value.ival));
      case bitwise_and:
	return (new_Int_value (-1, value1 -> Int_value.ival & value2 -> Int_value.ival));
      case int_times_int:
	return (new_Int_value (-1, value1 -> Int_value.ival * value2 -> Int_value.ival));
      case int_minus_int:
	return (new_Int_value (-1, value1 -> Int_value.ival - value2 -> Int_value.ival));
      case int_plus_int:
	return (new_Int_value (-1, value1 -> Int_value.ival + value2 -> Int_value.ival));
      default: dcg_bad_tag (term -> Dyop.dop, "evaluate_dyadic_operator");
    };
  return (affix_value_nil);
}

static affix_value evaluate_monadic_operator (affix_term term, affix_value value1, definition def)
{ tags_affix_value tag = TAGText_value;
  switch (term -> Monop.mop)
    { case int_plus_int:
      case int_minus_int:
      case bitwise_not:
	tag = TAGInt_value;
	break;
      case real_plus_real:
      case real_minus_real:
	tag = TAGReal_value;
	break;
      default: dcg_bad_tag (term -> Monop.mop, "evaluate_monadic_operator");
    };

  if (value1 -> tag != tag)
    { contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
			     "Typing error during constant evaluation");
      return (affix_value_nil);
    };

  switch (term -> Monop.mop)
    { case int_plus_int:    return (attach_affix_value (value1));
      case int_minus_int:   return (new_Int_value (-1, -value1 -> Int_value.ival));
      case bitwise_not:	    return (new_Int_value (-1, ~value1 -> Int_value.ival));
      case real_plus_real:  return (attach_affix_value (value1));
      case real_minus_real: return (new_Real_value (-1, -value1 -> Real_value.rval));
      default: dcg_bad_tag (term -> Monop.mop, "evaluate_monadic_operator");
    };
  return (affix_value_nil);
}

static affix_value try_evaluate_affix_term (affix_term term, definition def);
static affix_value evaluate_ast (affix_term term, definition def)
{ /* Evaluate the constituents */
  affix_term_list terms = term -> Ast.terms;
  affix_value_list vals = init_affix_value_list (terms -> size);
  affix_rule adef = term -> adef;
  int anr = adef -> anr;
  int ix;
  for (ix = 0; ix < terms -> size; ix++)
    (void) try_evaluate_affix_term (terms -> array[ix], def);

  /* Check if this is an enumerable type */
  if ((adef -> tag == TAGAffix_alts) && (adef -> Affix_alts.enumerable))
    { if (terms -> size != 0)
	dcg_internal_error ("evaluate_ast");
      return (new_Int_value (anr, term -> Ast.altnr));
    };

  /* If any of subterms is not evaluated, we cannot evaluate ourselves */
  vals = init_affix_value_list (terms -> size);
  for (ix = 0; ix < terms -> size; ix++)
    { affix_value value = terms -> array[ix] -> value;
      if (value != affix_value_nil)
	app_affix_value_list (vals, attach_affix_value (value));
      else
	{ detach_affix_value_list (&vals);
	  return (affix_value_nil);
	};
    };

  return (new_Composed_value (anr, term -> Ast.altnr, vals));
}

static affix_value evaluate_concatenation (affix_term term, definition def)
{ /* Evaluate the constituents */
  affix_term_list terms = term -> Concat.terms;
  affix_value_list vals = init_affix_value_list (terms -> size);
  affix_value result;
  int ix;
  for (ix = 0; ix < terms -> size; ix++)
    (void) try_evaluate_affix_term (terms -> array[ix], def);

  /* Put everything into a value list */
  for (ix = 0; ix < terms -> size; ix++)
    { affix_value value = terms -> array[ix] -> value;
      if (value != affix_value_nil)
	app_affix_value_list (vals, attach_affix_value (value));
      else
	{ detach_affix_value_list (&vals);
	  return (affix_value_nil);
	};
    };

  result = ebs_concatenate_text_values (vals);
  detach_affix_value_list (&vals);
  return (result);
}

static affix_value try_evaluate_affix_term (affix_term term, definition def)
{ affix_value value = affix_value_nil;
  switch (term -> tag)
    { case TAGTerminal:
	{ element edef = term -> Terminal.edef;
	  if (edef == element_nil)
	    dcg_internal_error ("try_evaluate_affix_term");
	  value = attach_affix_value (edef -> value);
	}; break;
      case TAGDyop:
	{ affix_value value1 = try_evaluate_affix_term (term -> Dyop.arg1, def);
	  affix_value value2 = try_evaluate_affix_term (term -> Dyop.arg2, def);
	  if ((value1 != affix_value_nil) && (value2 != affix_value_nil))
	    value = evaluate_dyadic_operator (term, value1, value2, def);
	}; break;
      case TAGMonop:
	{ affix_value value1 = try_evaluate_affix_term (term -> Monop.arg, def);
	  if (value1 != affix_value_nil)
	    value = evaluate_monadic_operator (term, value1, def);
	}; break;
      case TAGAst: value = evaluate_ast (term, def); break;
      case TAGConcat: evaluate_concatenation (term, def); break;
      case TAGInt:  value = new_Int_value (-1, term -> Int.num); break;
      case TAGReal: value = new_Real_value (-1, term -> Real.num); break;
      case TAGText: value = new_Text_value (-1, attach_string (term -> Text.txt)); break;
      case TAGRegexp: return (affix_value_nil);
      case TAGVar:
        { variable vdef = term -> Var.vdef;
	  affix_rule adef = vdef -> adef;
	  if (adef -> value != affix_value_nil)
	    value = attach_affix_value (adef -> value);
	  else return (affix_value_nil);
	}; break;
      default: return (affix_value_nil);
    };

  /* Check if we could evaluate the operators */
  if (value == affix_value_nil)
    return (affix_value_nil);

  /* Store value and return it as well */
  term -> value = value;
  return (value);
}

/*
   Affix value collection
*/
static int_list_list affix_hash_container;
static void allocate_affix_hash_container (int affix_hash_size)
{ affix_hash_container = init_int_list_list (affix_hash_size);
  int ix;
  for (ix = 0; ix < affix_hash_size; ix++)
    app_int_list_list (affix_hash_container, new_int_list ());
}

#define SOME_PRIME 5693
static int calculate_affix_hash (affix_value value)
{ int affix_hash_size = affix_hash_container -> size;
  switch (value -> tag)
    { case TAGText_value: return (ebs_hash_text (value -> Text_value.text, affix_hash_size));
      case TAGInt_value:  return (ebs_hash_int  (value -> Int_value.ival,  affix_hash_size));
      case TAGReal_value: return (ebs_hash_real (value -> Real_value.rval, affix_hash_size));
      case TAGSmall_lattice:
	{ unsigned int low = (unsigned int) value -> Small_lattice.slat;
	  unsigned int high = (unsigned int) (value -> Small_lattice.slat >> 32);
	  /* Ignore overflow */
	  unsigned int total = high + low;
	  total += (unsigned int) (SOME_PRIME * value -> Small_lattice.dom);
	  return (ebs_hash_int (total, affix_hash_size));
	};
      case TAGLarge_lattice:
	{ u_int64_list ul = value -> Large_lattice.llat;
	  unsigned int total = 0;
	  int ix;
	  /* Ignore overflow */
	  for (ix = 0; ix < ul -> size; ix++)
	    { u_int64 uv = ul -> array[ix];
	      total += (unsigned int) uv;
	      total += (unsigned int) (uv >> 32);
	    };
	  total += (unsigned int) (SOME_PRIME * value -> Large_lattice.dom);
	  return (ebs_hash_int (total, affix_hash_size));
	};
      case TAGComposed_value:
	{ affix_value_list parts = value -> Composed_value.parts;
	  unsigned int total = 0;
	  int ix;
	  for (ix = 0; ix < parts -> size; ix++)
	    total += (unsigned int) (SOME_PRIME * calculate_affix_hash (parts -> array[ix]));
	  total += (unsigned int) value -> Composed_value.marker;
	  return (ebs_hash_int (total, affix_hash_size));
	};
      default: dcg_bad_tag (value -> tag, "calculate_affix_hash");
    };
  return (0);
}

static int collect_constant_value (affix_value value)
{ int hash = calculate_affix_hash (value);
  int_list bucket = affix_hash_container -> array[hash];
  int new_idx;
  int ix;

  /* Check if affix is already present */
  for (ix = 0; ix < bucket -> size; ix++)
    { int his_idx = bucket -> array[ix];
      affix_value his_value = all_affix_constants -> array[his_idx];
      if (equal_affix_value (his_value, value))
        return (his_idx);
    };

  /* We have a new one */
  new_idx = all_affix_constants -> size;
  app_int_list (bucket, new_idx);
  app_affix_value_list (all_affix_constants, attach_affix_value (value));
  return (new_idx);
}

static void collect_constants_in_locals (variable_list locals)
{ int ix;
  for (ix = 0; ix < locals -> size; ix++)
    { variable local = locals -> array[ix];
      affix_rule adef = local -> adef;
      if (adef -> value == affix_value_nil)
	continue;
      local -> cnr = collect_constant_value (adef -> value);
    }
}

static int unique_affix_nr;
static void replace_by_affix_constant (affix_term *term, definition def, int valnr)
{ affix_term old_term = *term;
  char *vname = dcg_new_fmtd_string ("_%s_%d", old_term -> adef -> aname, unique_affix_nr);
  variable new_var = new_variable (vname);
  affix_term new_term;
  new_var -> vnr = def -> locals -> size;
  new_var -> adef = old_term -> adef;
  new_var -> cnr = valnr;
  app_variable_list (def -> locals, new_var);
  new_term = new_Var (old_term -> line, old_term -> col, attach_string (vname));
  new_term -> adef = old_term -> adef;
  new_term -> value = old_term -> value;
  new_term -> Var.vdef = new_var;
  *term = new_term;
  detach_affix_term (&old_term);
  unique_affix_nr++;
}

static void collect_nested_constant_affix_terms (affix_term_list terms, definition def);
static void collect_constant_affix_terms (affix_term *term, definition def)
{
  /* Never replace an affix variable by an anonymous constant */
  affix_term dterm = *term;
  if (dterm -> tag == TAGVar)
    return;
      
  if (dterm -> value != affix_value_nil)
    { int valnr = collect_constant_value (dterm -> value);
      replace_by_affix_constant (term, def, valnr);
      return;
    };

  switch (dterm -> tag)
    { case TAGDyop:
	collect_constant_affix_terms (&dterm -> Dyop.arg1, def);
        collect_constant_affix_terms (&dterm -> Dyop.arg2, def);
	break;
      case TAGMonop:
	collect_constant_affix_terms (&dterm -> Monop.arg, def);
	break;
      case TAGAst:
	collect_nested_constant_affix_terms (dterm -> Ast.terms, def);
	break;
      case TAGConcat:
	collect_nested_constant_affix_terms (dterm -> Concat.terms, def);
        break;
      default: dcg_bad_tag (dterm -> tag, "collect_constant_affix_terms");
    };
}

static void collect_nested_constant_affix_terms (affix_term_list terms, definition def)
{ int ix;
  for (ix = 0; ix < terms -> size; ix++)
    collect_constant_affix_terms (&terms -> array[ix], def);
}

static void evaluate_constants_in_affix_term (affix_term *term, definition def)
{ (void) try_evaluate_affix_term (*term, def);
  collect_constant_affix_terms (term, def);
}

static void evaluate_constants_in_formals (fpar_list fpars, definition def)
{ int ix;
  for (ix = 0; ix < fpars -> size; ix++)
    { fpar fp = fpars -> array[ix];
      if (fp -> fexp -> size != 1)	/* Consistency check */
	dcg_internal_error ("evaluate_constants_in_formals");
      simplify_affix_term (&fp -> fexp -> array[0], def);
      evaluate_constants_in_affix_term (&fp -> fexp -> array[0], def);
    };
}

static void evaluate_constants_in_confrontations (res_conf_list rconfs, definition def)
{ int ix;
  for (ix = 0; ix < rconfs -> size; ix++)
    { res_conf rconf = rconfs -> array[ix];
      simplify_affix_term (&rconf -> lhs, def);
      simplify_affix_term (&rconf -> rhs, def);
      evaluate_constants_in_affix_term (&rconf -> lhs, def);
      evaluate_constants_in_affix_term (&rconf -> rhs, def);
    };
}

static void evaluate_constants_in_call (member m, definition def)
{ affix_term_list terms = m -> Res_call.args;
  rule rdef = m -> Res_call.rdef;
  dir_list fdirs = rdef -> rspec -> dirs;
  int ix;
  for (ix = 0; ix < terms -> size; ix++)
    { dir fdir = fdirs -> array[ix];
      simplify_affix_term (&terms -> array[ix], def);
      switch (fdir)
	{ case d_static:
	    { affix_term term = terms -> array[ix];
	      affix_value value = try_evaluate_affix_term (term, def);
	      if (value == affix_value_nil)
		contsens_error_by_gnr (def -> pr -> gnr, term -> line, term -> col,
				       "Could not evaluate static expression");
	      else term -> value = value;
	    };
	  case d_regexp: break;		/* Allready prepared */
	  case d_inherited:
	  case d_derived:
	  case d_unknown:
	    evaluate_constants_in_affix_term (&terms -> array[ix], def);
	    break;
	  default: dcg_bad_tag (fdir, "evaluate_constants_in_call");
	};
    };
}

static void evaluate_constants_in_member (member m, definition def)
{ switch (m -> tag)
    { case TAGRes_call: evaluate_constants_in_call (m, def); break;
      case TAGRes_guard: evaluate_constants_in_confrontations (m -> Res_guard.rconfs, def); break;
      case TAGRes_term:
	if (m -> Res_term.arg != affix_term_nil)
	  evaluate_constants_in_affix_term (&m -> Res_term.arg, def);
	break;
      case TAGOp: break;
      default: dcg_bad_tag (m -> tag, "evaluate_constants_in_member");
    };
}

static void evaluate_constants_in_fwo (fwo_group fwo, definition def)
{ switch (fwo -> tag)
    { case TAGSingle: evaluate_constants_in_member (fwo -> Single.mem, def); break;
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    evaluate_constants_in_member (mems -> array[ix], def);
	}; break;
      default: dcg_bad_tag (fwo -> tag, "evaluate_constants_in_member");
    };
}

static void evaluate_constants_in_alternative (alternative alt, definition def)
{ fwo_group_list members = alt -> members;
  int ix;
  for (ix = 0; ix < members -> size; ix++)
    evaluate_constants_in_fwo (members -> array[ix], def);
}

static void evaluate_constants_in_group (group grp, definition def)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    evaluate_constants_in_alternative (alts -> array[ix], def);
}

static void evaluate_constants_in_defs (definition_list defs)
{ int ix;
  for (ix = 0; ix < defs -> size; ix++)
    { definition def = defs -> array[ix];
      collect_constants_in_locals (def -> locals);
      evaluate_constants_in_formals (def -> lhs_pars, def);
      evaluate_constants_in_group (def -> grp, def);
    };
}

static void evaluate_constants_in_rule (rule srule)
{ if (srule -> rspec -> rkind != r_normal) return;
  switch (srule -> tag)
    { case TAGDefs:
	evaluate_constants_in_defs (srule -> Defs.defs);
	break;
      case TAGAnonymous_option:
	evaluate_constants_in_group (srule -> Anonymous_option.grp, srule -> Anonymous_option.def);
        break;
      case TAGAnonymous_group:
	evaluate_constants_in_group (srule -> Anonymous_group.grp, srule -> Anonymous_group.def);
      case TAGExt_rule: break;
      default: dcg_bad_tag (srule -> tag, "evaluate_constants_in_rule");
    };
}

static void try_dump_constants_table ()
{ int ix;
  if (!dump_properties) return;
  dcg_wlog ("Dump of all affix constants:");
  for (ix = 0; ix < all_affix_constants -> size; ix++)
    { dcg_eprint ("Constant %d: ", ix);
      pp_affix_value (dcg_error_file (), all_affix_constants -> array[ix]);
      dcg_wlog ("");
    };
}


void try_evaluate_constants ()
{ int change, ix;
  int nr = 0;
  dcg_hint ("      evaluating constant affix rules");
  do
     { change = 0;
       for (ix = 0; ix < all_affix_rules -> size; ix++)
         try_evaluate_affix_rule (all_affix_rules -> array[ix], &change);
       nr++;
     }
  while (change);
  dcg_hint ("      needed %d pass%s to evaluate constant affix rules", nr, (nr == 1)?"":"es");
  check_enumerable_affix_rules ();
  try_dump_evaluated_affix_rules ();
  dcg_hint ("      evaluating constant affixes");
  unique_affix_nr = 0;
  allocate_affix_hash_container (8192);		/* Or nr of syntax rules * 20 */
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    evaluate_constants_in_rule (all_syntax_rules -> array[ix]);
  try_dump_constants_table ();
}
