/*
   File: code_topdown.c
   Defines the topdown recursive backup parser generator
   This code also generates the code for those rules that must
   be recognized in a topdown fashion (predicates, semipredicates)

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: code_topdown.c,v 1.19 2013/01/11 11:05:44 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <stdlib.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_fileutil.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "options.h"
#include "globals.h"
#include "code_common.h"
#include "code_topdown.h"

/*
   Forward declare all rules
*/
static void code_rule_declaration (rule srule)
{ if (!srule -> codable) return;
  code_string ("static void ");
  code_rule_name (srule, 0);
  code_line (" (EagrtsHandle hnd);");
}

static void code_rule_declarations ()
{ int ix;
  code_line ("/*");
  code_line ("   Forward declare all reachable syntax rules and predicates");
  code_line ("*/");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    code_rule_declaration (all_syntax_rules -> array[ix]);
  code_line ("");
}

/*
   Topdown coding of alternatives
*/
static int code_fwo (fwo_group fwo, int nr, int pred, int count_only)
{ int sum = 0;
  if (fwo -> tag == TAGSingle)
    sum += code_member (fwo -> Single.mem, nr, 0, pred, count_only);
  else /* Very difficult */
    dcg_abort ("code_fwo", "Free word order groups are still difficult");
  sum += code_glue (fwo -> gl, count_only);
  return (sum);
}

static int code_alternative (alternative alt, int pred, int count_only)
{ fwo_group_list fwos = alt -> members;
  int sum = 0;
  int ix;
  for (ix = fwos -> size - 1; 0 <= ix; ix--)
    sum += code_fwo (fwos -> array[ix], ix, pred, count_only);
  return (sum);
}

static void code_definition (rule srule, definition def, int nr, int memo_ok)
{ alternative alt = def -> grp -> alts -> array[0];
  int nr_sons = count_sons (alt -> members);
  int nr_pushes = 0;
  int nmemo = -1;
  code_line ("  /* %s, definition %d */", srule -> rspec -> canonic_name, nr);
  if (code_negative_memoization && memo_ok)
    { nmemo = register_neg_memo ();
      code_line ("  /* Negative memo check %d */", nmemo);
      code_line ("  if (erts_test_and_set_negative_memo (hnd, %d))", nmemo);
    }
  else code_line ("  /* Lookahead check */");
  code_line ("    {");
  if (code_negative_memoization && memo_ok)
    code_line ("      State entry_state = hnd -> trellis -> curr_state;");
  if (code_stack_checks)
    code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
  code_make_locals (def -> locals);
  try_generate_trace_alternative (srule, nr);
  if (code_negative_memoization && memo_ok)
    { nr_pushes += 3;
      code_line ("      /* Clear negative memo %d */", nmemo);
      code_line ("      cont_push_int (hnd, %d);", nmemo);
      code_line ("      cont_push_state (hnd, entry_state);");
      code_line ("      cont_push_continuation (hnd, erts_clear_negative_memo);\n");
    };
  nr_pushes += code_alternative (alt, 0, 0);
  nr_pushes += code_rule_lhs (srule -> rnr, nr_sons, def -> lhs_pars, 0); /* To add transduction */
  code_line ("      /* Alt trailer */");
  code_line ("      cont_pop (hnd, %d);", nr_pushes);
  if (code_stack_checks)
    { code_line ("      if (save_sp != hnd -> cont_sp)");
      code_line ("        dcg_internal_error (\"%s, definition %d\");",
	  	 srule -> rspec -> canonic_name, nr);
    };
  code_free_locals (def -> locals);
  code_line ("    }");
}

static void code_definitions (rule srule, int memo_ok)
{ definition_list defs = srule -> Defs.defs;
  int ix;
  for (ix = 0; ix < defs -> size; ix++)
    code_definition (srule, defs -> array[ix], ix + 1, memo_ok);
}

/*
   Anonymous rules derive their frame from their definition ancestor (in case
   of topdown coding), either from the stack (because the parent is a predicate)
   or they can take it from a direct ancestor on the tree stack.
*/
static void try_inherit_frame_for_anonymous_rules (rule srule)
{ spec parent_spec;
  switch (srule -> tag)
    { case TAGAnonymous_option:
	parent_spec = srule -> Anonymous_option.anc_spec;
	break;
      case TAGAnonymous_group:
	parent_spec = srule -> Anonymous_group.anc_spec;
	break;
      default: return;
    };

  /* Locate the spec of the parent */
  code_line ("  /* Inherit frame from father */");
  if (parent_spec -> rtype == r_predicate)
    code_line ("  AffixNode *frame = cont_pop_affix_frame (hnd);");
  else code_line ("  AffixNode *frame = top_tree (hnd) -> frame;");
}

static void try_push_frame_for_anonymous_rules (rule srule)
{ spec parent_spec;
  switch (srule -> tag)
    { case TAGAnonymous_option:
	parent_spec = srule -> Anonymous_option.anc_spec;
	break;
      case TAGAnonymous_group:
	parent_spec = srule -> Anonymous_group.anc_spec;
	break;
      default: return;
    };

  /* Locate the spec of the parent */
  if (parent_spec -> rtype == r_predicate)
    code_line ("  cont_push_affix_frame (hnd, frame);");
}

static void code_anonymous_rule (rule srule, group grp, int memo_ok)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    { alternative alt = alts -> array[ix];
      int nr_sons = count_sons (alt -> members);
      int nr_pushes = 0;
      int nmemo = -1;
      code_line ("  /* %s, alternative %d */", srule -> rspec -> canonic_name, ix + 1);
      if (code_negative_memoization && memo_ok)
	{ nmemo = register_neg_memo ();
	  code_line ("  /* Negative memo check %d */", nmemo);
	}
      else code_line ("  /* Lookahead check */");
      code_line ("    {");
      if (code_stack_checks)
	code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
      if (code_negative_memoization && memo_ok)
	code_line ("      State entry_state = hnd -> trellis -> curr_state;");
      try_generate_trace_alternative (srule, ix + 1);
      if (code_negative_memoization && memo_ok)
	{ nr_pushes += 3;
	  code_line ("      /* Clear negative memo %d */", nmemo);
	  code_line ("      cont_push_int (hnd, %d);", nmemo);
	  code_line ("      cont_push_state (hnd, entry_state);");
	  code_line ("      cont_push_continuation (hnd, erts_clear_negative_memo);\n");
	};
      nr_pushes += code_alternative (alt, 0, 0);
      code_anonymous_rule_lhs (srule -> rnr, nr_sons); /* To add transduction */
      code_line ("      /* Alt trailer */");
      code_line ("      cont_pop (hnd, %d);", nr_pushes);
      if (code_stack_checks)
	{ code_line ("      if (save_sp != hnd -> cont_sp)");
	  code_line ("        dcg_internal_error (\"%s, definition %d\");",
	  	     srule -> rspec -> canonic_name, ix + 1);
	}
      code_line ("    }");
    };
}

static void code_syntax_rule (rule srule, int memo_ok)
{ code_rule_header (srule);
  try_inherit_frame_for_anonymous_rules (srule);
  try_generate_trace_enter (srule);
  switch (srule -> tag)
    { case TAGDefs:
	code_definitions (srule, memo_ok);
	break;
      case TAGAnonymous_option:
	code_anonymous_rule (srule, srule -> Anonymous_option.grp, memo_ok);
	break;
      case TAGAnonymous_group:
	code_anonymous_rule (srule, srule -> Anonymous_group.grp, memo_ok);
	break;
      default: dcg_bad_tag (srule -> tag, "code_syntax_rule");
    }
  try_generate_trace_leave (srule);
  try_push_frame_for_anonymous_rules (srule);
  code_rule_trailer (srule);
}

/*
   Coding of predicates
*/
static void code_predicate_header (rule prule, char *part)
{ code_line ("/*");
  code_line ("   PREDICATE %s: %s part", prule -> rspec -> canonic_name, part);
  code_line ("*/");
  code_line ("static void %s_%d (EagrtsHandle hnd, Position *args)", part, prule -> rnr);
  code_line ("{");
}

static int code_predicate_lhs (fpar_list lhs_pars, int count_only)
{ int sum = 0;
  int ix;
  for (ix = lhs_pars -> size - 1; 0 <= ix; ix--)
    sum += code_affix_expr (lhs_pars -> array[ix] -> fexp -> array[0], count_only);
  return (sum);
}

static void code_actual_predicate_alternative (rule prule, definition pdef, int defnr)
{ alternative alt = pdef -> grp -> alts -> array[0];
  int nr_sons = count_sons (alt -> members);
  int nr_pushes = 0;
  code_line ("    { /* Definition %d */", defnr);
  if (code_stack_checks)
    code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
  code_make_locals (pdef -> locals);
  try_generate_trace_alternative (prule, defnr);
  nr_pushes += code_alternative (alt, 1, 0);
  code_line ("      /* Update the predicate node for this alternative */");
  nr_pushes += code_predicate_lhs (pdef -> lhs_pars, 0);
  code_line ("      erts_update_predicate_node (hnd, pnode, %d, frame);\n", nr_sons);
  code_line ("      /* Alt trailer */");
  code_line ("      cont_pop (hnd, %d);", nr_pushes);
  if (code_stack_checks)
    { code_line ("  if (save_sp != hnd -> cont_sp)");
      code_line ("    dcg_internal_error (\"%s, definition %d\");",
	  	 prule -> rspec -> canonic_name, defnr);
    };
  code_line ("");
  code_free_locals (pdef -> locals);
  code_line ("    }");
}

static void code_actual_predicate (rule prule)
{ int ix;
  definition_list pdefs = prule -> Defs.defs;
  code_predicate_header (prule, "actual");
  code_line ("  Tree pnode = args[0] -> tree_node;");
  for (ix = 0; ix < pdefs -> size; ix++)
    code_actual_predicate_alternative (prule, pdefs -> array[ix], ix + 1);
  code_line ("}\n");
}

static void code_check_inherited_affixes (rule prule)
{ dir_list dirs = prule -> rspec -> dirs;
  int first = 1;
  int ix;
  for (ix = 0; ix < dirs -> size; ix++)
    if (dirs -> array[ix] == d_inherited)
      { if (!first)
	  code_string (" &&\n      ");
	code_string ("critical_position_has_value (args[%d])", ix);
	first = 0;
      };
}

static void code_delayed_predicate (rule prule)
{ int fsize = prule -> rspec -> dirs -> size;
  int ix;
  code_predicate_header (prule, "delayed");
  code_string ("  if (");
  code_check_inherited_affixes (prule);
  code_line (")");
  code_line ("    {");
  for (ix = 0; ix < fsize; ix++)
    code_line ("      args[%d] -> delayed_func = delayed_eval_func_nil;", ix);
  code_line ("      actual_%d (hnd, args);", prule -> rnr);
  for (ix = 0; ix < fsize; ix++)
    code_line ("      args[%d] -> delayed_func = delayed_%d;", ix, prule -> rnr);
  code_line ("    }");
  code_line ("  else call_continuation (hnd);");
  code_line ("}\n");
}

static void code_make_predicate_locals (rule prule, int fsize)
{ int ix;
  code_line ("  /* Allocate delayed affix frame */");
  code_line ("  AffixNode *frame = erts_make_affix_frame (%d);", fsize);
  for (ix = 0; ix < fsize; ix++)
    code_line ("  frame[%d] = erts_make_affix_node (\"pred_%d_pafx%d\");",
	       ix, prule -> rnr, ix + 1);
  code_line ("");
}

static void code_free_predicate_locals (int fsize)
{ int ix;
  code_line ("  /* Free delayed affix frame */");
  for (ix = 0; ix < fsize; ix++)
    code_line ("  erts_free_affix_node (frame[%d]);", ix);
  code_line ("  erts_free_affix_frame (frame, %d);\n", fsize);
}

static void code_enter_predicate (rule prule)
{ int fsize = prule -> rspec -> dirs -> size;
  if (prule -> tag != TAGDefs)
    dcg_internal_error ("code_enter_predicate:1");
  if (!fsize)
    dcg_internal_error ("code_enter_predicate:2");
  code_rule_header (prule);
  if (code_stack_checks)
    code_line ("  cont_ptr save_sp = hnd -> cont_sp;");
  code_make_predicate_locals (prule, fsize);
  try_generate_trace_enter (prule);
  code_line ("  /* Make predicate node with delayed affix positions */");
  code_line ("  erts_make_predicate_node (hnd, %d, %d, frame, delayed_%d);\n",
	     prule -> rnr, fsize, prule -> rnr);
  if (code_stack_checks)
    { code_line ("  if (save_sp != hnd -> cont_sp)");
      code_line ("    dcg_internal_error (\"pred_%d\");", prule -> rnr);
    };
  code_free_predicate_locals (fsize);
  try_generate_trace_leave (prule);
  code_rule_trailer (prule);
}

void code_predicate (rule prule)
{ code_actual_predicate (prule);
  code_delayed_predicate (prule);
  code_enter_predicate (prule);
}

/*
   A semi predicate is coded as if it were a normal topdown syntax rule,
   because its evaluation does not depend on its arguments nor is it
   necessary that the input pointer advances during its execution.
*/
void code_semipredicate (rule srule)
{ code_syntax_rule (srule, 0);
}

static void code_rule (rule srule)
{ if (!srule -> codable) return;
  switch (srule -> rspec -> rtype)
    { case r_rule:
      case r_option: code_syntax_rule (srule, 1); break;
      case r_predicate: code_predicate (srule); break;
      case r_semipredicate: code_semipredicate (srule); break;
      default: dcg_bad_tag (srule -> rspec -> rtype, "code_rule");
    };
}

static void code_rules ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    code_rule (all_syntax_rules -> array[ix]);
}

void code_topdown_parser ()
{ code_rule_declarations ();
  code_rules ();
}
