/*
   File: code_leftcorner.c
   Defines the leftcorner recursive backup parser generator

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: code_leftcorner.c,v 1.13 2013/03/13 11:58:37 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <stdlib.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_fileutil.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "options.h"
#include "globals.h"
#include "lc_rel.h"
#include "code_common.h"
#include "code_topdown.h"
#include "code_leftcorner.h"

/*
   We must allocate goals for all codable rules that are not (semi-)predicates
*/
static void try_allocate_goal (rule srule, int *all_goals)
{ if (!srule -> codable) return;
  if (srule -> empty != e_always_produces_empty)
    { srule -> goalnr = *all_goals;
      (*all_goals)++;
      code_line ("  Goal %d: %s", srule -> goalnr, srule -> rspec -> canonic_name);
    };
}

static int allocate_and_code_goals ()
{ int all_goals = 0;
  int ix;
  code_line ("/*");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    try_allocate_goal (all_syntax_rules -> array[ix], &all_goals);
  code_line ("*/\n");
  return (all_goals);
}

/*
   Fill and code the leftcorner relation between the goals
*/
static char *goal_lc_relation;
static void fill_goal_relation_for_rules (int ix, int iy, int all_goals)
{ rule srule_ix = all_syntax_rules -> array[ix];
  rule srule_iy = all_syntax_rules -> array[iy];
  int size = all_syntax_rules -> size;
  if (!srule_ix -> codable) return;
  if (!srule_iy -> codable) return;
  if (srule_ix -> empty == e_always_produces_empty) return;
  if (srule_iy -> empty == e_always_produces_empty) return;
  if ((rules_lc_rel[srule_ix -> rnr * size + srule_iy -> rnr]) ||
      (srule_ix == srule_iy))		/* Include reflexivity */
    goal_lc_relation[srule_ix -> goalnr * all_goals + srule_iy -> goalnr] = 1;
}

static void fill_goal_lc_relation (int all_goals)
{ int ix, iy;
  goal_lc_relation = (char *) dcg_calloc (all_goals * all_goals, sizeof (char));
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    for (iy = 0; iy < all_syntax_rules -> size; iy++)
      fill_goal_relation_for_rules (ix, iy, all_goals);
}

static void code_goal_lc_relation (int all_goals)
{ int ix, iy;
  int eltnr = 0;
  int mincol = (all_goals < 24)?all_goals:24;
  code_line ("/* Coded left corner relation between goals */");
  code_line ("static char goal_lc_relation[] = {");
  for (ix = 0; ix < all_goals; ix++)
    for (iy = 0; iy < all_goals; iy++)
      { code_string ("%d, ", goal_lc_relation [ix * all_goals + iy]);
	eltnr++;
	if (eltnr == mincol)
	  { code_string ("\n");
	    eltnr = 0;
	  };
      };
  code_line ("%s};", (eltnr)?"\n":"");
  code_line ("#define on_spine_to_goal(x) goal_lc_relation[cont_top_int(hnd) * %d + (x)]\n",
	     all_goals);

}

static void code_goal_administration ()
{ int all_goals = allocate_and_code_goals ();
  fill_goal_lc_relation (all_goals);
  code_goal_lc_relation (all_goals);
}

/*
   Forward declare all rule_N, get_N, emp_N (if necessary) and red_N rules
*/
static void code_rule_declaration (rule srule)
{ spec rspec = srule -> rspec;
  if (!srule -> codable) return;
  code_string ("static void ");
  code_rule_name (srule, 0);
  code_line (" (EagrtsHandle hnd);");
  switch (rspec -> rtype)
    { case r_option:
	code_string ("static void ");
        code_rule_name (srule, 1);
	code_line (" (EagrtsHandle hnd);");
      case r_rule:
	code_string ("static void ");
	code_string ("get_%d", srule -> rnr);
	code_line (" (EagrtsHandle hnd);");
	code_string ("static void ");
	code_string ("red_%d", srule -> rnr);
	code_line (" (EagrtsHandle hnd);");
      case r_predicate:
      case r_semipredicate: break;
      default: dcg_bad_tag (rspec -> rtype, "code_rule_declaration");
    };
}

static void code_rule_declarations ()
{ int ix;
  code_line ("/*");
  code_line ("   Forward declare all reachable syntax rules and predicates");
  code_line ("*/");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    code_rule_declaration (all_syntax_rules -> array[ix]);
  code_line ("");
}

/*
   Tracing support
*/
static void try_generate_lc_trace_enter (rule srule, char *routine)
{ if (!code_tracing) return;
  code_line ("  erts_trace_enter_function (hnd, \"%s_%d\", %d);",
	     routine, srule -> rnr, srule -> rnr);
}

static void try_generate_lc_trace_leave (rule srule, char *routine)
{ if (!code_tracing) return;
  code_line ("  erts_trace_leave_function (hnd, \"%s_%d\", %d);",
	     routine, srule -> rnr, srule -> rnr);
}

/*
   Code rule_N parser routine
*/
static void code_rule_routine (rule srule)
{ code_rule_header (srule);
  try_generate_trace_enter (srule);
  code_line ("  /* Lookahead check */");
  code_line ("    {");
  if (code_stack_checks)
    code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
  code_line ("      cont_push_int (hnd, %d);", srule -> goalnr);
  code_line ("      cont_push_continuation (hnd, get_%d);", srule -> rnr);
  code_line ("      call_continuation (hnd);");
  code_line ("      cont_pop (hnd, 2);");
  if (code_stack_checks)
    { code_line ("      if (save_sp != hnd -> cont_sp)");
      code_line ("        dcg_internal_error (\"rule_%d, part 1\");", srule -> rnr);
    };
  code_line ("    };");
  if (srule -> rspec -> rtype == r_option)
    { code_line ("  /* Lookahead check: follow set? */");
      code_line ("    {");
      if (code_stack_checks)
	code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
      code_line ("      cont_push_continuation (hnd, emp_%d);", srule -> rnr);
      code_line ("      call_continuation (hnd);");
      code_line ("      cont_pop (hnd, 1);");
      if (code_stack_checks)
	{ code_line ("      if (save_sp != hnd -> cont_sp)");
	  code_line ("        dcg_internal_error (\"rule_%d, part 2\");", srule -> rnr);
	};
      code_line ("    };");
    };
  try_generate_trace_leave (srule);
  code_rule_trailer (srule);
}

/*
   Generation of rule members 
*/
static int code_rule_members_in_fwo (fwo_group fwo, int my_sonnr, int count_only)
{ switch (fwo -> tag)
    { case TAGSingle:
	return (code_member (fwo -> Single.mem, my_sonnr, 0, 0, count_only));
      case TAGFwo:
	{ dcg_abort ("code_rule_members_in_fwo", "Free word order groups are still difficult");
	  /* member_list mems = fwo -> Fwo.mems;
	  int nr_pushes = 0;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    nr_pushes += code_member (mems -> array[ix], my_sonnr + ix, 0, 0, count_only);
	  return (nr_pushes);
	  */
	};
      default: dcg_bad_tag (fwo -> tag, "code_rule_members_in_fwo");
    };
  return (0);
}

static int code_rule_members_after (alternative alt, int my_sonnr, int count_only)
{ fwo_group_list fwos = alt -> members;
  int sonnr = count_sons (fwos); 
  int nr_pushes = 0;
  int ix;
  for (ix = (fwos -> size - 1); 0 <= ix && (my_sonnr < sonnr); ix--)
    { fwo_group fwo = fwos -> array[ix];
      sonnr -= count_sons_in_fwo (fwo);
      if (my_sonnr < sonnr)
        nr_pushes += code_rule_members_in_fwo (fwo, sonnr, count_only);
      nr_pushes += code_glue (fwo -> gl, count_only);
    };
  return (nr_pushes);
}

/*
   Generation of empty members (and empty right hand sides)
*/
static int code_emp_member (member m, int my_sonnr, int no_qc, int count_only)
{ if (m -> tag == TAGRes_call)
    { rule rdef = m -> Res_call.rdef;
      if ((rdef -> tag == TAGQuasi_rule) && no_qc && (rdef -> empty == e_always_produces_empty))
	/* Do not code calls to quasi terminals that are conditions */
	return (0);
    };
  return (code_member (m, my_sonnr, 1, 0, count_only));
}

static int code_emp_members_in_fwo (fwo_group fwo, int my_sonnr, int no_qc, int count_only)
{ switch (fwo -> tag)
    { case TAGSingle:
	return (code_emp_member (fwo -> Single.mem, my_sonnr, no_qc, count_only));
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int nr_pushes = 0;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    nr_pushes += code_emp_member (mems -> array[ix], my_sonnr + ix, no_qc, count_only);
	  return (nr_pushes);
	};
      default: dcg_bad_tag (fwo -> tag, "code_emp_members_in_fwo");
    };
  return (0);
}

static int code_emp_members_before (alternative alt, int my_sonnr, int no_qc, int count_only)
{ fwo_group_list fwos = alt -> members;
  int nr_pushes = 0;
  int sonnr = 0;
  int ix;
  for (ix = 0; (ix < fwos -> size) && (sonnr < my_sonnr); ix++)
    { fwo_group fwo = fwos -> array[ix];
      if (fwo -> gl == g_plus)
	dcg_wlog ("coding glue before emp members");
      nr_pushes += code_emp_members_in_fwo (fwo, sonnr, no_qc, count_only);
      nr_pushes += code_glue (fwo -> gl, count_only);
      sonnr += count_sons_in_fwo (fwo);
    };
  return (nr_pushes);
}

static int code_emp_rhs (alternative alt, int count_only)
{ fwo_group_list fwos = alt -> members;
  int sonnr = count_sons (fwos);
  int nr_pushes = 0;
  int ix;
  for (ix = fwos -> size - 1; 0 <= ix; ix--)
    { fwo_group fwo = fwos -> array[ix];
      sonnr -= count_sons_in_fwo (fwo);
      nr_pushes += code_emp_members_in_fwo (fwo, sonnr, 0, count_only);
    };
  return (nr_pushes);
}

/*
   Calls to quasi terminals that are conditions should be coded
   before the left corner in a lcit alternative so that the affixes
   of $LINE, $COLUMN, etc. get evaluated at the right point of the
   trellis.
*/
static int code_cond_quasi_call (member m, int my_sonnr, int count_only)
{ rule rdef;
  if (m -> tag != TAGRes_call) return (0);
  rdef = m -> Res_call.rdef;
  if (rdef -> tag != TAGQuasi_rule) return (0);
  if (rdef -> empty != e_always_produces_empty) return (0);
  return (code_member (m, my_sonnr, 1, 0, count_only));
}

static int code_cond_quasi_calls_in_fwo (fwo_group fwo, int my_sonnr, int count_only)
{ switch (fwo -> tag)
    { case TAGSingle:
	return (code_cond_quasi_call (fwo -> Single.mem, my_sonnr, count_only));
      case TAGFwo:
	{ member_list mems = fwo -> Fwo.mems;
	  int nr_pushes = 0;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    nr_pushes += code_cond_quasi_call (mems -> array[ix], my_sonnr + ix, count_only);
	  return (nr_pushes);
	};
      default: dcg_bad_tag (fwo -> tag, "code_quasi_calls_in_fwo");
    };
  return (0);
}

static int code_cond_quasi_calls_before (alternative alt, int my_sonnr, int count_only)
{ fwo_group_list fwos = alt -> members;
  int nr_pushes = 0;
  int sonnr = 0;
  int ix;
  for (ix = 0; (ix < fwos -> size) && (sonnr < my_sonnr); ix++)
    { fwo_group fwo = fwos -> array[ix];
      nr_pushes += code_cond_quasi_calls_in_fwo (fwo, sonnr, count_only);
      sonnr += count_sons_in_fwo (fwo);
    };
  return (nr_pushes);
}

/*
   Code get_N parser routine
   We generate an LCiT alternative for rule_N (i.e. in get_N)
   if we have an alternative N0 => m1,..., mi, .... where mi
   is a terminal production (i.e. a terminal, lexicon call, $MATCH, ..., etc)
   and N0 is a leftcorner of N (reflexivity included)
*/
static int code_lc_alternative_header (rule lrule, definition def, int defnr, int get_code)
{ int nr_pushes = 1;
  int nmemo = -1;
  if (code_negative_memoization && get_code)
    { nmemo = register_neg_memo ();
      code_line ("  /* Negative memo check %d */", nmemo);
      code_line ("  if (erts_test_and_set_negative_memo (hnd, %d))", nmemo);
    }
  else code_line ("  /* Lookahead check for lc alternative */");
  code_line ("    {");
  code_line ("      /* %s, definition %d */", lrule -> rspec -> canonic_name, defnr);
  if (code_negative_memoization && get_code)
    code_line ("      State entry_state = hnd -> trellis -> curr_state;");
  if (code_stack_checks)
    code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
  code_make_locals (def -> locals);
  code_line ("      /* reduce to %s */", lrule -> rspec -> canonic_name);
  try_generate_trace_alternative (lrule, defnr);
  code_line ("      cont_push_continuation (hnd, red_%d);\n", lrule -> rnr);
  if (code_negative_memoization && get_code)
    { nr_pushes += 3;
      code_line ("      /* Clear negative memo %d */", nmemo);
      code_line ("      cont_push_int (hnd, %d);", nmemo);
      code_line ("      cont_push_state (hnd, entry_state);");
      code_line ("      cont_push_continuation (hnd, erts_clear_negative_memo);\n");
    };
  return (nr_pushes);
}

static void code_lc_alternative_trailer (rule srule, rule lrule, definition def, int defnr,
					 int nr_sons, int nr_pushes)
{ if (lrule -> tag == TAGDefs)
    /* To add transduction */
    nr_pushes += code_rule_lhs (lrule -> rnr, nr_sons, def -> lhs_pars, 0);
  else code_anonymous_rule_lhs (lrule -> rnr, nr_sons);
  code_line ("      /* Alt trailer */");
  code_line ("      cont_pop (hnd, %d);", nr_pushes);
  if (code_stack_checks)
    { char *routine_name;
      if (srule == rule_nil) routine_name = dcg_new_fmtd_string ("red_%d", lrule -> rnr);
      else routine_name = dcg_new_fmtd_string ("get_%d", srule -> rnr);
      code_line ("      if (save_sp != hnd -> cont_sp)");
      code_line ("        dcg_internal_error (\"%s, %s, definition %d\");\n",
		 routine_name, lrule -> rspec -> canonic_name, defnr);
      detach_string (&routine_name);
    };
  code_free_locals (def -> locals);
  code_line ("    }");
}

static void code_lcit_alternative_for_call (rule srule, rule lrule, member m, alternative alt,
					    definition def, int defnr, int sonnr)
{ int nr_sons = count_sons (alt -> members);
  int nr_pushes = code_lc_alternative_header (lrule, def, defnr, 1);
  nr_pushes += code_rule_members_after (alt, sonnr, 0);
  nr_pushes += code_emp_members_before (alt, sonnr, 1, 0);
  nr_pushes += code_call (m, sonnr, 0, 0, 0);
  nr_pushes += code_cond_quasi_calls_before (alt, sonnr, 0);
  code_lc_alternative_trailer (srule, lrule, def, defnr, nr_sons, nr_pushes);
}

static void code_lcit_alternative_for_terminal (rule srule, rule lrule, member m, alternative alt,
						definition def, int defnr, int sonnr)
{ int nr_sons = count_sons (alt -> members);
  int nr_pushes = code_lc_alternative_header (lrule, def, defnr, 1);
  nr_pushes += code_rule_members_after (alt, sonnr, 0);
  nr_pushes += code_emp_members_before (alt, sonnr, 1, 0);
  nr_pushes += code_terminal (m, sonnr, 0);
  nr_pushes += code_cond_quasi_calls_before (alt, sonnr, 0);
  code_lc_alternative_trailer (srule, lrule, def, defnr, nr_sons, nr_pushes);
}

static void try_generate_lcit_alternative_for_call (rule srule, rule lrule, member m,
						    alternative alt, definition def,
						    int defnr, int sonnr)
{ rule rdef = m -> Res_call.rdef;
  if (m -> empty == e_always_produces_empty) return;
  if (rdef -> tag == TAGQuasi_rule)
    code_lcit_alternative_for_call (srule, lrule, m, alt, def, defnr, sonnr);
  else
    switch (rdef -> rspec -> rkind)
      { case r_lexicon:
	case r_external:
	  code_lcit_alternative_for_call (srule, lrule, m, alt, def, defnr, sonnr);
	case r_normal:
	case r_fact: break;
	default: dcg_bad_tag (rdef -> rspec -> rkind, "try_generate_lcit_alternative_for_call");
      };
}

static void try_generate_lcit_alternative_for_member (rule srule, rule lrule, member m,
						      alternative alt, definition def,
						      int defnr, int sonnr)
{ switch (m -> tag)
    { case TAGRes_call:
	try_generate_lcit_alternative_for_call (srule, lrule, m, alt, def, defnr, sonnr);
	break;
      case TAGRes_term:
	code_lcit_alternative_for_terminal (srule, lrule, m, alt, def, defnr, sonnr);
	break;
      default: break;
    };
}

/*
   MS: Note that to generate an lcit alternative for a free word
       order group, that it is sufficient to generate one for all
       of its constituents using the remainders as followers (in
       any order).
      
       Also the empty constituents of a free word order group before
       the lcit alternative only need to be generated in fixed order.
*/
static void try_generate_lcit_alternative_for_fwo (rule srule, rule lrule, fwo_group fwo,
						   alternative alt, definition def,
						   int defnr, int sonnr)
{ if (fwo -> tag == TAGFwo)
    dcg_abort ("try_generate_lcit_alternative_for_fwo", "Still very difficult");
  try_generate_lcit_alternative_for_member (srule, lrule, fwo -> Single.mem,
					    alt, def, defnr, sonnr);
}

static void try_generate_lcit_alternative_for_alt (rule srule, rule lrule, alternative alt,
						   definition def, int defnr)
{ fwo_group_list fwos = alt -> members;
  int sonnr = 0; 
  int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    { fwo_group fwo = fwos -> array[ix];
      try_generate_lcit_alternative_for_fwo (srule, lrule, fwo, alt, def, defnr, sonnr);
      sonnr += count_sons_in_fwo (fwo);

      /* If we do not generate empty ourselves we're done */
      if (!fwo -> empty) return;
    };
}

static void try_generate_lcit_alternatives_for_defs (rule srule, rule lrule)
{ definition_list defs = lrule -> Defs.defs;
  int ix;
  for (ix = 0; ix < defs -> size; ix++)
    { definition def = defs -> array[ix];
      alternative alt = def -> grp -> alts -> array[0];
      try_generate_lcit_alternative_for_alt (srule, lrule, alt, def, ix + 1);
    };
}

static void try_generate_anonymous_lcit_alternatives (rule srule, rule lrule,
						      group grp, definition def)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    try_generate_lcit_alternative_for_alt (srule, lrule, alts -> array[ix], def, ix + 1);
}

static void try_generate_lcit_alternatives_for_rule (rule srule, rule lrule)
{ switch (lrule -> tag)
    { case TAGDefs:
	try_generate_lcit_alternatives_for_defs (srule, lrule);
	break;
      case TAGAnonymous_option:
	try_generate_anonymous_lcit_alternatives (srule, lrule, lrule -> Anonymous_option.grp,
						  lrule -> Anonymous_option.def);
	break;
      case TAGAnonymous_group:
	try_generate_anonymous_lcit_alternatives (srule, lrule, lrule -> Anonymous_group.grp,
						  lrule -> Anonymous_group.def);
	break;
      default: dcg_bad_tag (lrule -> tag, "try_generate_lcit_alternatives_for_rule");
    };
}

static void try_generate_lcit_alternatives (rule srule)
{ int size = all_syntax_rules -> size;
  int ix;
  for (ix = 0; ix < size; ix++)
    { rule lrule = all_syntax_rules -> array[ix];
      if (!lrule -> codable) continue;
      if (lrule -> empty == e_always_produces_empty) continue;
      if (!rules_lc_rel[srule -> rnr * size + lrule -> rnr] && (lrule != srule)) continue;
      try_generate_lcit_alternatives_for_rule (srule, lrule);
    };
}

static void code_get_routine (rule srule)
{ code_line ("/*");
  code_line ("   GET %s", srule -> rspec -> canonic_name);
  code_line ("*/");
  code_line ("static void get_%d (EagrtsHandle hnd)", srule -> rnr);
  code_line ("{");
  try_generate_lc_trace_enter (srule, "get");
  try_generate_lcit_alternatives (srule);
  try_generate_lc_trace_leave (srule, "get");
  code_line ("  /* Push self reference */");
  code_line ("  cont_push_continuation (hnd, get_%d);", srule -> rnr);
  code_line ("}\n");
}

/*
   Code emp_N parser routine
*/
static void try_generate_emp_alternative (rule srule, alternative alt, definition def, int defnr)
{ int nr_pushes, nr_sons;
  if (!alt -> empty) return;
  nr_sons = count_sons (alt -> members);
  nr_pushes = 0;
  code_line ("  /* %s, definition %d */", srule -> rspec -> canonic_name, defnr);
  code_line ("    {");
  if (code_stack_checks)
    code_line ("      cont_ptr save_sp = hnd -> cont_sp;");
  if (srule -> tag == TAGDefs)
    code_make_locals (def -> locals);
  else code_line ("      AffixNode *frame = top_tree (hnd) -> frame;");
  try_generate_trace_alternative (srule, defnr);
  nr_pushes += code_emp_rhs (alt, 0);
  if (srule -> tag == TAGDefs)
    /* To add transduction */
    nr_pushes += code_rule_lhs (srule -> rnr, nr_sons, def -> lhs_pars, 0);
  else code_anonymous_rule_lhs (srule -> rnr, nr_sons);
  code_line ("      /* Alt trailer */");
  code_line ("      cont_pop (hnd, %d);", nr_pushes);
  if (code_stack_checks)
    { code_line ("      if (save_sp != hnd -> cont_sp)");
      code_line ("        dcg_internal_error (\"emp_%d, definition %d\");\n", srule -> rnr, defnr);
    };
  if (srule -> tag == TAGDefs)
    code_free_locals (def -> locals);
  code_line ("    }");
}

static void try_generate_emp_alternatives_for_defs (rule srule)
{ definition_list defs = srule -> Defs.defs;
  int ix;
  for (ix = 0; ix < defs -> size; ix++)
    { definition def = defs -> array[ix];
      alternative alt = def -> grp -> alts -> array[0];
      try_generate_emp_alternative (srule, alt, def, ix + 1);
    };
}

static void try_generate_anonymous_emp_alternatives (rule srule, group grp, definition def)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    try_generate_emp_alternative (srule, alts -> array[ix], def, ix + 1);
}

static void try_generate_emp_alternatives_for_rule (rule srule)
{ switch (srule -> tag)
    { case TAGDefs:
	try_generate_emp_alternatives_for_defs (srule);
	break;
      case TAGAnonymous_option:
	try_generate_anonymous_emp_alternatives (srule, srule -> Anonymous_option.grp,
						 srule -> Anonymous_option.def);
	break;
      case TAGAnonymous_group:
	try_generate_anonymous_emp_alternatives (srule, srule -> Anonymous_group.grp,
						 srule -> Anonymous_group.def);
	break;
      default: dcg_bad_tag (srule -> tag, "try_generate_emp_alternatives_for_rule");
    };
}

static void code_emp_routine (rule srule)
{ code_line ("/*");
  code_line ("   EMP %s", srule -> rspec -> canonic_name);
  code_line ("*/");
  code_line ("static void emp_%d (EagrtsHandle hnd)", srule -> rnr);
  code_line ("{");
  try_generate_lc_trace_enter (srule, "emp");
  try_generate_emp_alternatives_for_rule (srule);
  try_generate_lc_trace_leave (srule, "emp");
  code_line ("  /* Push self reference */");
  code_line ("  cont_push_continuation (hnd, emp_%d);", srule -> rnr);
  code_line ("}\n");
}

/*
   Code red_N parser routine
*/
static void generate_check_goal (rule srule)
{ code_line ("  /* Check if we have achieved our goal */");
  code_line ("  if (cont_top_int (hnd) == %d)", srule -> goalnr);
  code_line ("    { cont_pop (hnd, 1);");
  code_line ("      call_continuation (hnd);");
  code_line ("      cont_push_int (hnd, %d);", srule -> goalnr);
  code_line ("    };");
}

/*
   We have to generate an LCiN alternative for a rule R iff
   R has an alternative R => alfa N beta and alfa =>* empty

   Note that N must be a left corner of R and that we can only
   succesfully recognize if the input of R is on the spine towards
   the parsing goal. Since N has already been recognized, we
   know that we have already parsed a non empty part of the input.
*/
static void try_generate_lcin_alternative_for_member (rule srule, member m, alternative alt,
						      definition def, int defnr, int sonnr,
						      rule rule_N)
{ int nr_sons, nr_pushes;
  if (m -> tag != TAGRes_call) return;
  if (m -> Res_call.rdef != rule_N) return;
  nr_sons = count_sons (alt -> members);
  nr_pushes = 1;	/* exchange */
  code_line ("  if (on_spine_to_goal (%d)) {", srule -> goalnr);
  nr_pushes += code_lc_alternative_header (srule, def, defnr, 0);
  nr_pushes += code_rule_members_after (alt, sonnr, 0);
  nr_pushes += code_emp_members_before (alt, sonnr, 0, 0);
  /* MS: Generate error if qc calls before */
  code_line ("      /* Insert left corner */");
  nr_pushes += code_link_son (sonnr, 0, 0);
  nr_pushes += code_actuals (m -> Res_call.args, 0);
  code_line ("      cont_push_continuation (hnd, erts_exchange_top);\n");
  code_lc_alternative_trailer (rule_nil, srule, def, defnr, nr_sons, nr_pushes);
  code_line ("    }");
}

static void try_generate_lcin_alternative_for_fwo (rule srule, fwo_group fwo, alternative alt,
						   definition def, int defnr, int sonnr,
						   rule rule_N)
{ if (fwo -> tag == TAGFwo)
    dcg_abort ("try_generate_lcin_alternative_for_fwo", "Still very difficult");
  try_generate_lcin_alternative_for_member (srule, fwo -> Single.mem, alt, def,
					    defnr, sonnr, rule_N);
}

static void try_generate_lcin_alternative_for_alt (rule srule, alternative alt,
						   definition def, int defnr, rule rule_N)
{ fwo_group_list fwos = alt -> members;
  int sonnr = 0; 
  int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    { fwo_group fwo = fwos -> array[ix];
      try_generate_lcin_alternative_for_fwo (srule, fwo, alt, def, defnr, sonnr, rule_N);
      sonnr += count_sons_in_fwo (fwo);

      /* If we do not generate empty ourselves we're done */
      if (!fwo -> empty) return;
    };
}

static void try_generate_lcin_alternatives_for_defs (rule srule, rule rule_N)
{ definition_list defs = srule -> Defs.defs;
  int ix;
  for (ix = 0; ix < defs -> size; ix++)
    { definition def = defs -> array[ix];
      alternative alt = def -> grp -> alts -> array[0];
      try_generate_lcin_alternative_for_alt (srule, alt, def, ix + 1, rule_N);
    };
}

static void try_generate_anonymous_lcin_alternatives (rule srule, group grp, definition def,
						      rule rule_N)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    try_generate_lcin_alternative_for_alt (srule, alts -> array[ix], def, ix + 1, rule_N);
}

static void try_generate_lcin_alternatives_for_rule (rule srule, rule rule_N)
{ switch (srule -> tag)
    { case TAGDefs:
	try_generate_lcin_alternatives_for_defs (srule, rule_N);
	break;
      case TAGAnonymous_option:
	try_generate_anonymous_lcin_alternatives (srule, srule -> Anonymous_option.grp,
						  srule -> Anonymous_option.def, rule_N);
	break;
      case TAGAnonymous_group:
	try_generate_anonymous_lcin_alternatives (srule, srule -> Anonymous_group.grp,
						  srule -> Anonymous_group.def, rule_N);
	break;
      default: dcg_bad_tag (srule -> tag, "try_generate_lcin_alternatives_for_rule");
    };
}

static void try_generate_lcin_alternatives (rule lrule)
{ int size = all_syntax_rules -> size;
  int ix;
  for (ix = 0; ix < size; ix++)
    { rule srule = all_syntax_rules -> array[ix];
      if (!srule -> codable) continue;
      if (srule -> empty == e_always_produces_empty) continue;
      if (!rules_lc_rel[srule -> rnr * size + lrule -> rnr]) continue;
      try_generate_lcin_alternatives_for_rule (srule, lrule);
    };
}

static void code_red_routine (rule srule)
{ code_line ("/*");
  code_line ("   RED %s", srule -> rspec -> canonic_name);
  code_line ("*/");
  code_line ("static void red_%d (EagrtsHandle hnd)", srule -> rnr);
  code_line ("{");
  try_generate_lc_trace_enter (srule, "red");
  generate_check_goal (srule);
  try_generate_lcin_alternatives (srule);
  try_generate_lc_trace_leave (srule, "red");
  code_line ("  /* Push self reference */");
  code_line ("  cont_push_continuation (hnd, red_%d);", srule -> rnr);
  code_line ("}\n");
}

static void code_parse_routines_for_rule (rule srule)
{ if (!srule -> codable) return;
  switch (srule -> rspec -> rtype)
    { case r_rule:
	code_rule_routine (srule);
	code_get_routine (srule);
	code_red_routine (srule);
	break;
      case r_option:
	code_rule_routine (srule);
	code_emp_routine (srule);
	code_get_routine (srule);
	code_red_routine (srule);
	break;
      case r_predicate:
	code_predicate (srule);
	break;
      case r_semipredicate:
	code_semipredicate (srule);
	break;
      default: dcg_bad_tag (srule -> rspec -> rtype, "code_parse_routines_for_rule");
    };
}

static void code_parse_routines ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    code_parse_routines_for_rule (all_syntax_rules -> array[ix]);
}

void code_leftcorner_parser ()
{ code_goal_administration ();
  code_rule_declarations ();
  code_parse_routines ();
}

