/*
   File: code_common.c
   Defines functions shared by all code generators

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: code_common.c,v 1.19 2013/01/11 11:05:43 marcs Exp $"
*/

/* include config.h if autoconfigured */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#ifndef WIN32
#include <unistd.h>
#endif

/* include sys/time.h with or without time.h */
#if TIME_WITH_SYS_TIME
#   include <sys/time.h>
#   include <time.h>
#else
#   if HAVE_SYS_TIME_H
#      include <sys/time.h>
#   else
#      include <time.h>
#   endif
#endif

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_fileutil.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "options.h"
#include "globals.h"
#include "code_common.h"

/* TMPDIR is the usual environment variable to consult */
#ifndef TMPDIR
#define TMPDIR "TMPDIR"
#endif

/* Define the usual default temporary directory */
#ifndef DEFAULT_TMP_DIR
#define DEFAULT_TMP_DIR "/tmp"
#endif

/*
   Opening and closing of the C output.
   The file name is exported for the backend.
*/
char *c_code_fname;
static FILE *code;
void open_code_file ()
{ /* Determine where to write the C code */
  if (!generate_c_only && !code_for_debug)
    { /* We generate the C code in a temporary location */
      char *temp_dir = getenv (TMPDIR);
      if (temp_dir == NULL) temp_dir = DEFAULT_TMP_DIR;
      c_code_fname = dcg_new_fmtd_string ("%s%ceag3_%d.c", temp_dir, DIR_SEP, (int) getpid ());
    }
  else
    { /* Generate C code in situ. */
      if (output_fname != NULL) c_code_fname = output_fname;
      else c_code_fname = dcg_new_fmtd_string ("%s%c%s.c", dir_name, DIR_SEP, base_gname);
    }

  /* Open the output */
  code = dcg_fopen (c_code_fname, "w");
}

void close_code_file ()
{ fclose (code);
}

/*
   Useful coding routines
*/
void code_string (char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  fprintf (code, "%s", buf);
};

void code_line (char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  fprintf (code, "%s\n", buf);
};

/*
   Coding of direct constants
*/
static void code_char (char ch)
{ unsigned int ich = (unsigned int) ch;
  switch (ch)
    { case '\f': code_string ("\\f"); break;
      case '\n': code_string ("\\n"); break;
      case '\r': code_string ("\\r"); break;
      case '\t': code_string ("\\t"); break;
      case '"': code_string ("\\\""); break;
      default:
        if ((ich < 32) || (ich >= 127))
          code_string ("\\x%02x", (ich & 0xff));
        else fputc (ch, code);
    };
}

void code_text_constant (char *text)
{ char *ptr;
  fputc ('"', code);
  for (ptr = text; *ptr; ptr++)
    code_char (*ptr);
  fputc ('"', code);
}

void code_direct_affix_value (affix_value value)
{ if (value == affix_value_nil)
    dcg_internal_error ("code_direct_affix_value");
  switch (value -> tag)
    { case TAGInt_value:
	code_line ("      cont_push_int (hnd, %d);", value -> Int_value.ival);
	break;
      case TAGReal_value:
	code_line ("      cont_push_real (hnd, %g);", value -> Real_value.rval);
	break;
      case TAGText_value:
	code_string ("      cont_push_text (hnd, ");
	code_text_constant (value -> Text_value.text);
	code_line (");");
	break;
      default: dcg_bad_tag (value -> tag, "code_direct_affix_value");
    };
}

/*
   Code header includes
*/
void code_include_headers ()
{ time_t thetime;
  char *atime;
  time (&thetime);
  atime = ctime (&thetime);
  code_line ("/*");
  code_line ("   File: %s", c_code_fname);
  code_string ("   Generated on %s", atime);
  code_line ("*/\n");
  code_line ("/* Standard includes */");
  code_line ("#include <stdio.h>");
  code_line ("#include <stdlib.h>");
  code_line ("#include <string.h>\n");
  code_line ("/* libdcg includes */");
  code_line ("#include <dcg.h>");
  code_line ("#include <dcg_error.h>");
  code_line ("#include <dcg_alloc.h>");
  code_line ("#include <dcg_string.h>");
  code_line ("#include <dcg_binfile.h>");
  code_line ("#include <dcg_plist.h>\n");
  code_line ("/* libeagbase includes */");
  code_line ("#include <ebase_ds.h>");
  code_line ("#include <ebase_lexicon.h>\n");
  code_line ("/* Runtime system includes */");
  code_line ("#include <erts_tree.h>");
  code_line ("#include <erts_tree_impl.h>");
  code_line ("#include <erts_trellis.h>");
  code_line ("#include <erts_trellis_impl.h>");
  code_line ("#include <erts_handle.h>");
  code_line ("#include <erts_handle_impl.h>");
  code_line ("#include <erts_leaf_parser.h>");
  code_line ("#include <erts_memory.h>");
  code_line ("#include <erts_predefs.h>");
  code_line ("#include <erts_trace.h>");
  code_line ("#include <erts_cont.h>");
  code_line ("#include <erts_misc.h>");
  code_line ("");
}

/*
   Code common to all coders
*/
void code_rule_name (rule srule, int emp)
{ rule_type type = srule -> rspec -> rtype;
  switch (type)
    { case r_rule:
      case r_option: code_string ("%s_%d", (emp)?"emp":"rule", srule -> rnr); break;
      case r_predicate: code_string ("pred_%d", srule -> rnr); break;
      case r_semipredicate: code_string ("semi_%d", srule -> rnr); break;
      default: dcg_bad_tag (type, "code_rule_name");
    };
}

/*
   Every rule should have a fair announcement
*/
void code_rule_header (rule srule)
{ spec rspec = srule -> rspec;
  code_line ("/*");
  code_line ("   %s %s", string_from_rule_type (rspec -> rtype), rspec -> canonic_name);
  code_line ("*/");
  code_string ("static void ");
  code_rule_name (srule, 0);
  code_line (" (EagrtsHandle hnd)");
  code_line ("{");
}

void code_rule_trailer (rule srule)
{ code_line ("  /* Push self reference */");
  code_string ("  cont_push_continuation (hnd, ");
  code_rule_name (srule, 0);
  code_line (");");
  code_line ("}");
  code_line ("");
};

/*
   Code for tracing support
*/
void try_generate_trace_enter (rule srule)
{ if (!code_tracing) return;
  code_string ("  erts_trace_enter_function (hnd, \"");
  code_rule_name (srule, 0);
  code_line ("\", %d);\n", srule -> rnr);
}

void try_generate_trace_alternative (rule srule, int nr)
{ if (!code_tracing) return;
  code_line ("      erts_trace_alternative (hnd, %d, %d);\n", srule -> rnr, nr);
}

void try_generate_trace_leave (rule srule)
{ if (!code_tracing) return;
  code_string ("  erts_trace_leave_function (hnd, \"");
  code_rule_name (srule, 0);
  code_line ("\", %d);\n", srule -> rnr);
}

/*
   A definition has its own frame of local affixes
*/
void code_make_locals (variable_list locals)
{ int ix;
  code_line ("      /* Allocate affix frame */");
  if (locals -> size)
    code_line ("      AffixNode *frame = erts_make_affix_frame (%d);", locals -> size);
  else code_line ("      AffixNode *frame = affix_frame_nil;");
  for (ix = 0; ix < locals -> size; ix++)
    { variable local = locals -> array[ix];
      code_string ("      frame[%d] = erts_make_affix_", ix);
      if (local -> cnr != -1)
	code_line ("from_constant (\"%s\", hnd, %d);", local -> vname, local -> cnr);
      else code_line ("node (\"%s\");", local -> vname);
    };
  code_line ("");
}

void code_free_locals (variable_list locals)
{ int ix;
  if (!locals -> size) return;
  code_line ("      /* Free affix frame */");
  for (ix = 0; ix < locals -> size; ix++)
    code_line ("      erts_free_affix_node (frame[%d]);", ix);
  code_line ("      erts_free_affix_frame (frame, %d);", locals -> size);
}

/*
   Register a new negative memo
*/
int register_neg_memo ()
{ int nmemo = nr_neg_memos;
  nr_neg_memos++;
  return (nmemo);
}

/*
   When coding rules, we should know how many pushes are associated
   with the coding of affix expressions and sons, so that we may code
   a check on the size of the continuation stack before pushing
*/
int count_sons_in_fwo (fwo_group fwo)
{ int nr_sons = 0;
  switch (fwo -> tag)
    { case TAGSingle:
	if (fwo -> Single.mem -> tag == TAGRes_guard)
	  nr_sons += fwo -> Single.mem -> Res_guard.rconfs -> size;
        else if (fwo -> Single.mem -> tag != TAGOp)
	  nr_sons = 1;
	break;
      case TAGFwo:
	nr_sons = fwo -> Fwo.mems -> size;
	break;
      default: dcg_bad_tag (fwo -> tag, "count_sons_in_fwo");
    };
  return (nr_sons);
}

int count_sons (fwo_group_list fwos)
{ int nr_sons = 0;
  int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    nr_sons += count_sons_in_fwo (fwos -> array[ix]);
  return (nr_sons);
}

/*
   Code an affix expression as the pushing of its constituents plus a tag
*/
int code_affix_expr (affix_term term, int count_only)
{ switch (term -> tag)
    { case TAGDyop:
	{ int nr_pushes = 2;
	  nr_pushes += code_affix_expr (term -> Dyop.arg2, count_only);
	  nr_pushes += code_affix_expr (term -> Dyop.arg1, count_only);
	  if (count_only) return (nr_pushes);
	  code_line ("      cont_push_int (hnd, %d);", term -> Dyop.dop);
	  code_line ("      cont_push_int (hnd, dyop_arith_node);");
	  return (nr_pushes);
	};
      case TAGMonop:
        { int nr_pushes = 2;
          nr_pushes += code_affix_expr (term -> Monop.arg, count_only);
	  if (count_only) return (nr_pushes);
	  code_line ("      cont_push_int (hnd, %d);", term -> Monop.mop);
	  code_line ("      cont_push_int (hnd, monop_arith_node);");
	  return (nr_pushes);
	};
      case TAGAst:
	{ affix_term_list terms = term -> Ast.terms;
	  int typenr = term -> adef -> anr;
	  int altnr = term -> Ast.altnr;
	  int nr_pushes = 4;
          int ix;
	  for (ix = terms -> size - 1; 0 <= ix; ix--)
	    nr_pushes += code_affix_expr (terms -> array[ix], count_only);
	  if (count_only) return (nr_pushes);
	  code_line ("      cont_push_int (hnd, %d);", terms -> size);
	  code_line ("      cont_push_int (hnd, %d);", altnr);
	  code_line ("      cont_push_int (hnd, %d);", typenr);
	  code_line ("      cont_push_int (hnd, ast_node);");
	  return (nr_pushes);
	};
      case TAGConcat:
	{ affix_term_list terms = term -> Concat.terms;
	  int nr_pushes = 2;
          int ix;
	  for (ix = terms -> size - 1; 0 <= ix; ix--)
	    nr_pushes += code_affix_expr (terms -> array[ix], count_only);
	  if (count_only) return (nr_pushes);
	  code_line ("      cont_push_int (hnd, %d);", terms -> size);
	  code_line ("      cont_push_int (hnd, concat_node);");
	  return (nr_pushes);
	};
      case TAGVar:
	{ int vnr = term -> Var.vdef -> vnr;
	  if (count_only) return (2);
	  code_line ("      cont_push_affix_node (hnd, frame[%d]);", vnr);
	  code_line ("      cont_push_int (hnd, single_affix_node);");
	  return (2);
	}; 
      default: dcg_bad_tag (term -> tag, "code_affix_expr");
    };
  return (0);
}

/*
   Code the left hand side of a rule

   We call erts_make_normal_node for a general syntax rule,
   erts_make_simple_node for a rule without positions and
   erts_make_anonymous_node for options and local groups
*/
int code_rule_lhs (int node_nr, int nr_sons, fpar_list lhs_pars, int count_only)
{ int nr_pushes = 0;
  int ix;
  code_line ("      /* Left hand side */");
  for (ix = lhs_pars -> size - 1; 0 <= ix; ix--)
    nr_pushes += code_affix_expr (lhs_pars -> array[ix] -> fexp -> array[0], count_only);
  if (count_only) return (nr_pushes);
  if (lhs_pars -> size != 0)
    code_line ("      erts_make_normal_node (hnd, %d, %d, %d, frame);",
	       node_nr, nr_sons, lhs_pars -> size);
  else code_line ("      erts_make_simple_node (hnd, %d, %d, frame);", node_nr, nr_sons);
  code_line ("");
  return (nr_pushes);
}

void code_anonymous_rule_lhs (int node_nr, int nr_sons)
{ code_line ("      /* Anonymous left hand side */");
  code_line ("      erts_make_anonymous_node (hnd, %d, %d, frame);", node_nr, nr_sons);
  code_line ("");
}

/*
   Code the linking of a son tree under its father
*/
int code_link_son (int nr, int pred, int count_only)
{ int nr_pushes = (pred)?3:2;
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_int (hnd, %d);", nr);
  if (pred)
    { code_line ("      cont_push_tree (hnd, pnode);");
      code_line ("      cont_push_continuation (hnd, erts_link_predicate_son);");
    }
  else code_line ("      cont_push_continuation (hnd, erts_link_son);");
  return (nr_pushes);
}

/*
   Code a single affix expression for rules with only a single position
*/
int code_single_actual (affix_term arg, int count_only)
{ int nr_pushes = 1;
  nr_pushes += code_affix_expr (arg, count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_continuation (hnd, erts_make_affix_link);");
  return (nr_pushes);
}

/*
   Code the actual positions of a son node in the tree
*/
int code_actuals (affix_term_list args, int count_only)
{ int nr_pushes = 1;
  int ix;

  /* If there are no positions, we need not call erts_make_affix_link */
  if (!args -> size) return (0);
  for (ix = args -> size - 1; 0 <= ix; ix--)
    nr_pushes += code_affix_expr (args -> array[ix], count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_continuation (hnd, erts_make_affix_link);");
  return (nr_pushes);
}

/*
   Coding of glue
*/
int code_glue (glue gl, int count_only)
{ if (gl != g_plus) return (0);
  if (!count_only)
    { code_line ("      /* Glue */");
      code_line ("      cont_push_continuation (hnd, erts_glue);\n");
    };
  return (1);
}

/*
   Coding of calls:
   Code a normal call, either to rule_N or emp_N
*/
static int code_normal_call (member m, int nr, int emp, int pred, int count_only)
{ affix_term_list args = m -> Res_call.args;
  rule rdef = m -> Res_call.rdef;
  int anonymous = (rdef -> tag == TAGAnonymous_option) ||
		  (rdef -> tag == TAGAnonymous_group);
  int nr_pushes = 1;
  if (anonymous && pred) nr_pushes++;
  nr_pushes += code_link_son (nr, pred, count_only);
  nr_pushes += code_actuals (args, count_only);
  if (count_only) return (nr_pushes);
  if (anonymous && pred)
    code_line ("      cont_push_affix_frame (hnd, frame);");
  code_string ("      cont_push_continuation (hnd, ");
  code_rule_name (rdef, emp);
  code_line (");\n");
  return (nr_pushes);
}

/*
   Quasi rules are translated using their runtime name, preceded
   by one optional actual, the number of affix positions (0 or 1)
   and for $MATCH/$SKIP the regular expression number. Note that
   $TRACE should be implemented in the runtime system as a predicate.
*/
int code_quasi_call (member m, int nr, int pred, int count_only)
{ affix_term_list args = m -> Res_call.args;
  rule rdef = m -> Res_call.rdef;
  dir_list fdirs = rdef -> rspec -> dirs;
  int nr_apos = args -> size;
  int nr_pushes = 2;
  int ix;

  /* Quasi rules create a tree (needed also for transduction) */
  nr_pushes += code_link_son (nr, pred, count_only);

  /* Handle normal actuals */
  for (ix = args -> size - 1; 0 <= ix; ix--)
    switch (fdirs -> array[ix])
      { case d_static:
	case d_regexp:
	  nr_pushes++;
	  nr_apos--;
	  break;
        default: nr_pushes += code_affix_expr (args -> array[ix], count_only);
      };

  /* If we had an actual position, we must call erts_make_affix_link */
  if (nr_apos) nr_pushes++;
  if (count_only) return (nr_pushes);
  if (nr_apos)
    code_line ("      cont_push_continuation (hnd, erts_make_affix_link);");
  code_line ("      cont_push_int (hnd, %d);", nr_apos);

  /* Handle special arguments */
  for (ix = args -> size - 1; 0 <= ix; ix--)
    switch (fdirs -> array[ix])
      { case d_static:
	  code_direct_affix_value (args -> array[ix] -> value);
	  break;
	case d_regexp:
	  { affix_term arg = args -> array[ix];
	    if (arg -> tag != TAGRegexp)
	      dcg_internal_error ("code_quasi_call");
	    code_line ("      cont_push_int (hnd, %d)", arg -> Regexp.termnr);
	  };
	default: break;
      };

  /* code the call */
  code_line ("      cont_push_continuation (hnd, %s);", rdef -> Quasi_rule.rname);
  code_line ("");
  return (nr_pushes);
}

/*
   Code the call to a fact: to be written
*/
int code_fact_call (member m, int nr, int pred, int count_only)
{ return (0);
}

/*
   Code the call to an external rule
*/
int code_external_call (member m, int nr, int pred, int count_only)
{ affix_term_list args = m -> Res_call.args;
  rule rdef = m -> Res_call.rdef;
  int nr_pushes = 1;
  nr_pushes += code_link_son (nr, pred, count_only);
  nr_pushes += code_actuals (args, count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_continuation (hnd, %s);", rdef -> Ext_rule.ename);
  return (nr_pushes);
}

/*
   Code the call to a lexicon rule
*/
int code_lexicon_call (member m, int nr, int count_only)
{ affix_term_list args = m -> Res_call.args;
  rule rdef = m -> Res_call.rdef;
  int lnr = rdef -> rspec -> lnr;
  int nr_pushes = 2;
  nr_pushes += code_link_son (nr, 0, count_only);
  nr_pushes += code_actuals (args, count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_int (hnd, %d);", lnr);
  code_line ("      cont_push_continuation (hnd, erts_parse_lex_nont);\n");
  return (nr_pushes);
}

int code_call (member m, int nr, int emp, int pred, int count_only)
{ rule rdef = m -> Res_call.rdef;
  if (!count_only)
    code_line ("      /* Call to '%s' */", rdef -> rspec -> canonic_name);
  if (rdef -> tag == TAGQuasi_rule) return (code_quasi_call (m, nr, pred, count_only));
  else
    switch (rdef -> rspec -> rkind)
      { case r_normal:	 return (code_normal_call (m, nr, emp, pred, count_only)); 
	case r_lexicon:	 
	  if (emp) dcg_internal_error ("code_call");
	  else if (pred) dcg_internal_error ("code_call");
	  return (code_lexicon_call (m, nr, count_only));
	case r_fact:	 return (code_fact_call (m, nr, pred, count_only)); 
	case r_external: return (code_external_call (m, nr, pred, count_only));
	default: dcg_bad_tag (rdef -> rspec -> rkind, "code_call");
      };
  return (0);
}

/*
   Code the call to a terminal
*/
int code_terminal (member m, int nr, int count_only)
{ affix_term arg = m -> Res_term.arg;
  int is_regexp = m -> Res_term.is_regexp;
  int termnr = m -> Res_term.termnr;
  int nr_pos = (arg == affix_term_nil)?0:1;
  int nr_pushes = 3;
  if (!count_only)
    code_line ("      /* %s %d: \"%s\" */", (is_regexp)?"Regexp":"Terminal",
	       termnr, m -> Res_term.text);
  nr_pushes += code_link_son (nr, 0, count_only);
  if (nr_pos) nr_pushes += code_single_actual (arg, count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_int (hnd, %d);", nr_pos);
  code_line ("      cont_push_int (hnd, %d);", termnr);
  code_string ("      cont_push_continuation (hnd, erts_parse_");
  if (is_regexp) code_string ("match");
  else code_string ("terminal");
  code_line (");\n");
  return (nr_pushes);
}

/*
   Code a set of confrontations
*/
static propagation_kind propagation_kind_from_guard_type (guard_type typ)
{ switch (typ)
    { case equal:	 return (p_bidirectional);
      case unequal:	 return (p_unequal);
      case restrict:	 return (p_restrict);
      case assign:	 return (p_left_to_right);
      case bidir_assign: return (p_bidirectional);
      case arith_equal:	 return (p_equal);
      default: dcg_bad_tag (typ, "propagation_kind_from_guard_type");
    };
  return (p_bidirectional);
}

static int code_confrontation (res_conf rc, int nr, int pred, int count_only)
{ propagation_kind kind = propagation_kind_from_guard_type (rc -> typ);
  affix_term lhs = rc -> lhs;
  affix_term rhs = rc -> rhs;
  int nr_pushes = 2;
  if (!count_only)
    code_line ("      /* confrontation */");
  nr_pushes += code_link_son (nr, pred, count_only);
  nr_pushes += code_affix_expr (rhs, count_only);
  nr_pushes += code_affix_expr (lhs, count_only);
  if (count_only) return (nr_pushes);
  code_line ("      cont_push_int (hnd, %d);", kind);
  code_line ("      cont_push_continuation (hnd, erts_confrontation);\n");
  return (nr_pushes);
}

static int code_confrontations (member m, int nr, int pred, int count_only)
{ res_conf_list rconfs = m -> Res_guard.rconfs;
  int nr_pushes = 0;
  int ix;

  /*
     We code the confrontations in reverse, so that
     they will be executed in the normal order
  */
  for (ix = rconfs -> size - 1; 0 <= ix; ix--)
    nr_pushes += code_confrontation (rconfs -> array[ix], nr + ix, pred, count_only);
  return (nr_pushes);
}

/*
   Code a control operator (+, -, !, ?)
*/
static int code_control_operator (member m, int count_only)
{ switch (m -> Op.op)
    { case c_success:
	if (count_only) return (0);
	code_line ("      /* + */\n");
	return (0);
      case c_failure:
	if (count_only) return (1);
	code_line ("      /* - */");
	code_line ("      cont_push_continuation (hnd, erts_fail_continuation);\n");
	return (1);
      case c_abort:
	{ char *gname = all_grammars -> array[m -> Op.gnr] -> gname;
          if (count_only) return (3);
	  code_line ("      /* ? */");
 	  code_line ("	    cont_push_int (hnd, %d);", m -> line);
	  code_line ("      cont_push_text (hnd, \"%s\");", gname);
	  code_line ("      cont_push_continuation (hnd, erts_abort);\n");
	  return (3);
	}
      case c_exit:
	if (count_only) return (1);
	code_line ("      cont_push_continuation (hnd, erts_exit);\n");
	return (1);
      default: dcg_bad_tag (m -> Op.op, "code_control_operator");
    };
  return (0);
}

int code_member (member m, int nr, int emp, int pred, int count_only)
{ switch (m -> tag)
    { case TAGRes_call:  return (code_call (m, nr, emp, pred, count_only));
      case TAGRes_term: 
	if (emp) dcg_internal_error ("code_member");
	else if (pred) dcg_internal_error ("code_member");
	return (code_terminal (m, nr, count_only));
      case TAGRes_guard: return (code_confrontations (m, nr, pred, count_only));
      case TAGOp:	 return (code_control_operator (m, count_only));
      default: dcg_bad_tag (m -> tag, "code_member");
    };
  return (0);
}

void code_start_rule ()
{ code_line ("void parse_trellis (EagrtsHandle hnd)");
  code_line ("{ /* Declare top level frame */");
  if (code_stack_checks)
    code_line ("  cont_ptr save_sp = hnd -> cont_sp;");
  code_line ("  cont_push_continuation (hnd, erts_end_of_parse);");
  code_string ("  cont_push_continuation (hnd, ");
  code_rule_name (root_rule, 0);
  code_line (");");
  code_line ("  call_continuation (hnd);");
  code_line ("  cont_pop (hnd, 2);");
  if (code_stack_checks)
    { code_line ("  if (save_sp != hnd -> cont_sp)");
      code_line ("    dcg_internal_error (\"parse_trellis\");");
    };
  code_line ("}");
}

