/*
   File: ast_utils.c
   Defines some support routines to handle the ast

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: ast_utils.c,v 1.9 2013/01/03 15:21:21 marcs Exp $"
*/

/* System includes */
#include <stdio.h>
#include <stdlib.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>

/* Local includes */
#include "eag_ds.h"
#include "affix_rules.h"

string string_from_arule_kind (arule_kind akind)
{ switch (akind)
    { case arule_unknown: return ("unknown");
      case arule_error:   return ("error");
      case arule_int:	  return ("int");
      case arule_real:	  return ("real");
      case arule_text:	  return ("text");
      case arule_lattice: return ("lattice");
      case arule_tree:	  return ("tree");
      case arule_any:	  return ("any");
      default: dcg_bad_tag (akind, "string_from_arule_kind");
    };
  return (NULL);
}

string string_from_rule_type (rule_type type)
{ switch (type)
    { case r_unknown:		return ("unknown");
      case r_rule:		return ("RULE");
      case r_option:		return ("OPTION");
      case r_predicate:		return ("PREDICATE");
      case r_semipredicate:	return ("SEMIPREDICATE");
      default:	dcg_bad_tag (type, "string_from_rule_type");
    };
  return (NULL);
}

string string_from_nullability (nullability nval)
{ switch (nval)
    { case e_may_produce_empty:		return ("may produce empty");
      case e_always_produces_empty:	return ("always produces empty");
      case e_never_produces_empty:	return ("never produces empty");
      default: dcg_bad_tag (nval, "string_from_nullability");
    };
  return (NULL);
}

arule_kind kind_from_operator (operator op)
{ switch (op)
    { case times:               break;
      case modulo:              return (arule_int);
      case divides:             return (arule_int);
      case plus:                break;
      case minus:               break;
      case a_union:		return (arule_lattice);
      case a_part:		return (arule_lattice);
      case shift_left:          return (arule_int);
      case shift_right:         return (arule_int);
      case bitwise_xor:         return (arule_int);
      case bitwise_not:         return (arule_int);
      /* identified operators */
      case bitwise_and:         return (arule_int);
      case bitwise_or:          return (arule_int);
      case int_times_int:       return (arule_int);
      case real_times_real:     return (arule_real);
      case int_times_text:      return (arule_text);
      case int_plus_int:        return (arule_int);
      case real_plus_real:      return (arule_real);
      case text_plus_text:      return (arule_text);
      case int_minus_int:       return (arule_int);
      case real_minus_real:     return (arule_real);
      default: dcg_bad_tag (op, "kind_from_operator");
    };
  return (arule_error);
}

/*
   Shorthands for expression analysis
*/
int is_an_affix_variable (affix_term term)
{ return (term -> tag == TAGVar);
}

int is_a_single_affix_variable (affix_term_list terms)
{ if (terms -> size != 1) return (0);
  return (is_an_affix_variable (terms -> array[0]));
}

/*
   Shorthands for anonymous rules
*/
int is_anonymous_rule (rule srule)
{ switch (srule -> tag)
    { case TAGExt_rule:
      case TAGQuasi_rule:
      case TAGDefs: break;
      case TAGAnonymous_option:
      case TAGAnonymous_group: return (1);
      default: dcg_bad_tag (srule -> tag, "is_anonymous_rule");
    };
  return (0);
}
 
int try_pick_anonymous_group_spec_and_def (rule srule, group *grp,
					   spec *anc_spec, definition *anc_def)
{ switch (srule -> tag)
    { case TAGExt_rule:
      case TAGQuasi_rule:
      case TAGDefs: break;
      case TAGAnonymous_option:
	*grp = srule -> Anonymous_option.grp;
	*anc_spec = srule -> Anonymous_option.anc_spec;
	*anc_def = srule -> Anonymous_option.def;
	return (1);
      case TAGAnonymous_group:
	*grp = srule -> Anonymous_group.grp;
	*anc_spec = srule -> Anonymous_group.anc_spec;
	*anc_def = srule -> Anonymous_group.def;
	return (1);
      default: dcg_bad_tag (srule -> tag, "try_pick_anonymous_spec_and_def");
    };
  return (0);
}
