/*
   File: options.c
   Defines flags and parameters of the eag3 driver

   Copyright (C) 2013 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: options.c,v 1.1 2013/03/13 10:08:52 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>

/* libeagbase include: only for the version */
#include <ebase_version.h>

/* local includes */
#include "options.h"

/* exported flags and pragmats */
int agfl_compatible;
int force_lexgen;
int non_strict;
int td_parser;
int lc_parser;
int lr_parser;
int positive_memoization;
int negative_memoization;

/* Local flags */
static int show_version;

/* Source file names */
string source_name;                     /* single file argument */

/* exported grammar search path */
string_list grammar_search_path;

/* other exported options */
string variants_option;

void init_options ()
{ /* Compiler options */
  agfl_compatible = 0;
  non_strict = 0;

  /* Codegenerator options */
  td_parser = 0;
  lc_parser = 1;
  lr_parser = 0;
  positive_memoization = 0;
  negative_memoization = 0;

  /* DCG options */
  verbose = 0;

  /* Paths and names */
  source_name = string_nil;

  /* Initialize search path for grammars */
  grammar_search_path = new_string_list ();

  /* other exported options */
  variants_option = string_nil;
}

static void print_usage ()
{ dcg_wlog ("usage: eag3 [flags] filename [more_flags]");
  dcg_wlog ("where flags and more_flags may be any of the following:");
  dcg_wlog ("   -h:  provide this help");
  dcg_wlog ("   -V:  show version");
  dcg_wlog ("   -v:  verbose");
  dcg_wlog ("   -f:  force recompilation of lexica");
  dcg_wlog ("   -N:  code for negative memoization");
  dcg_wlog ("   -P:  code for positive memoization");
  dcg_wlog ("   -td: generate a top down parser");
  dcg_wlog ("   -lc: generate a left corner parser");
  dcg_wlog ("   -lr: generate a lr parser");
  dcg_wlog ("   -ns: non strict compile");
  dcg_wlog ("   -ca: allow compatibility with AGFL");
  dcg_wlog ("   -Ipath:      add path to grammar search path");
  dcg_wlog ("   -Xtrp:mname: add mname tp list of triple databases");
  dcg_wlog ("   -Xvar:XYZ:   select variants X, Y and Z");
  dcg_exit (4);
}

static void syntax_error (char *syn_error)
{ dcg_error (0, "error on command line: %s", syn_error);
  print_usage ();
}

static void scan_option (char *ptr, int *i, int argc, char **argv)
{ if (streq (ptr, "V")) show_version = 1;
  else if (streq (ptr, "v"))  { show_version = 1; verbose = 1; }
  else if (streq (ptr, "ns")) non_strict = 1;
  else if (streq (ptr, "ca")) agfl_compatible = 1;
  else if (streq (ptr, "h")) print_usage ();
  else if (streq (ptr, "f")) force_lexgen = 1;
  else if (streq (ptr, "P")) positive_memoization = 1;
  else if (streq (ptr, "N")) negative_memoization = 1;
  else if (streq (ptr, "td")) { td_parser = 1; lc_parser = 0; lr_parser = 0; }
  else if (streq (ptr, "lc")) { td_parser = 0; lc_parser = 1; lr_parser = 0; }
  else if (streq (ptr, "lr")) { td_parser = 0; lc_parser = 0; lr_parser = 1; }
  else if (*ptr == 'I')
    { if (*(ptr + 1))
	append_string_list (grammar_search_path, new_string (ptr + 1));
      else
	{ *i = *i + 1;
	  if ((*i) < argc)
	    append_string_list (grammar_search_path, new_string (argv[*i]));
	  else syntax_error ("missing grammar search path component");
	};
    }
  else if (strncmp (ptr, "Xvar:", 5) == 0)
    { if (*(ptr + 5))
	variants_option = new_string (ptr + 5);
      else syntax_error ("missing variants");
    }
  else syntax_error ("illegal option specified");
}

static void report_version ()
{ if (!show_version) return;
  dcg_wlog ("This is EAG3, C Version %s, (C) M. Seutter", EAG3_VERSION);
}

/*
   Remove the suffix from the source name.
   We can leave the prefix of directories intact:
   eag3-compile and eag3-lexgen will take care of that.
*/
static void remove_suffix_from_source_name ()
{ /* Remember the last '.' in the line, while copying the source name */
  char buf[MAXPATHLEN + 1];
  char *last_dot = NULL;
  char *sptr, *dptr;

  for (sptr = source_name, dptr = buf; (*sptr); )
    { if (*sptr == '.') last_dot = dptr;
      *dptr++ = *sptr++;
    };

  /* Cut off the suffix, if it is there */
  if (last_dot != NULL) *last_dot = '\0';
  else *dptr = '\0';
  if (!buf[0] || (buf[0] == '/') || (buf[0] == DIR_SEP))
    syntax_error ("empty grammar name");

  /* Set base name */
  detach_string (&source_name);
  source_name = new_string (buf);
}

void parse_command_line (int argc, char **argv)
{ int ix;
  for (ix = 1; ix < argc; ix++)
    { char *arg = argv[ix];
      if (arg[0] == '-') scan_option (arg+1, &ix, argc, argv);
      else if (source_name == string_nil) source_name = new_string (arg);
      else syntax_error ("too many arguments");
    };

  /* Try and report stuff */
  report_version ();

  /* Exit if no source given */
  if (source_name == string_nil)
    { if (!show_version)
	dcg_wlog ("no source name given, bailing out...");
      exit (0);
    };
  remove_suffix_from_source_name ();
}
