/* 
   File: erts_propagate.c
   Defines the affix propagation mechanism

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_propagate.c,v 1.13 2013/01/11 14:52:04 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_ds.h>
#include <ebase_lexicon.h>
#include <ebase_lexicon_impl.h>
#include <ebase_affix_value_utils.h>

/* local includes */
#include "erts_handle.h"
#include "erts_handle_impl.h"
#include "erts_tree.h"
#include "erts_tree_impl.h"
#include "erts_propagate.h"
#include "erts_trace.h"
#include "erts_cont.h"

/*
   Calculate the value of an affix expression from its constituents
*/
static affix_value calc_affix_expr (AffixExpr expr)
{ switch (expr -> kind)
    { case single_affix_node:
	return (attach_affix_value (expr -> uni.node -> value));
      case ast_node:
	{ int typenr = expr -> uni.ast.typenr;
	  int marker = expr -> uni.ast.marker;
	  int nr = expr -> uni.ast.nr;
	  AffixExpr *exprs = expr -> uni.ast.exprs;
	  affix_value_list parts = init_affix_value_list (nr);
	  int ix;
	  for (ix = 0; ix < nr; ix++)
	    app_affix_value_list (parts, calc_affix_expr (exprs[ix]));
	  return (new_Composed_value (typenr, marker, parts));
	};
      case concat_node:
	{ int nr = expr -> uni.text_concat.nr;
	  AffixExpr *exprs = expr -> uni.text_concat.exprs;
	  affix_value_list vals = init_affix_value_list (nr);
	  affix_value result;
	  int ix;
	  for (ix = 0; ix < nr; ix++)
	    app_affix_value_list (vals, calc_affix_expr (exprs[ix]));
	  result = ebs_concatenate_text_values (vals);
	  detach_affix_value_list (&vals);
	  return (result);
	};
      case dyop_arith_node:
	{ operator dop = expr -> uni.arith.op;
	  affix_value val1 = calc_affix_expr (expr -> uni.arith.expr1);
	  affix_value val2 = calc_affix_expr (expr -> uni.arith.expr2);
	  affix_value result = ebs_evaluate_dyadic_operation (dop, val1, val2);
	  detach_affix_value (&val1);
	  detach_affix_value (&val2);
	  return (result);
	};
      case monop_arith_node:
	{ operator mop = expr -> uni.arith.op;
	  affix_value val1 = calc_affix_expr (expr -> uni.arith.expr1);
	  affix_value result = ebs_evaluate_monadic_operation (mop, val1);
	  detach_affix_value (&val1);
	  return (result);
	};
      default: dcg_bad_tag (expr -> kind, "erts_calc_affix_value");
    };
  return (affix_value_nil);
}

/*
   erts_calc_affix_value calculates the value of one side of a position
   Its prerequisite is that the sill of this side has dropped to 0, i.e.
   all affix nodes in the expression have a defined value
*/
affix_value erts_calc_affix_value (Position pos, int side)
{ AffixExpr expr = pos -> sides[side].expr;
  return (calc_affix_expr (expr));
}

/*
   Assign_value is called to assign a defined value to an affix node.
   It decreases the sill of every position it is connected to and
   propagates the value to all connected positions from which the
   value did not originate. If its originating position is a delayed
   one, its function will be called (predicates). Before calling all
   propagators, it should check its meta definition if necessary (not yet done)
   After the continuation, everything is undone.

   Calling convention:
     cont_push_affix_node (hnd, affx);
     cont_push_position (hnd, pos);
     cont_push_affix_value (hnd, val);
     cont_push_continuation (hnd, assign_value);
*/
static void assign_value (EagrtsHandle hnd)
{ affix_value value = cont_pop_affix_value (hnd);
  Position orig_pos = cont_pop_position (hnd);
  AffixNode node = cont_pop_affix_node (hnd);

  /* Check if we already have a value */
  if (node -> value == affix_value_nil)
    { /* Save sp because we do not know how many links this one has */
      cont_ptr old_sp = hnd -> cont_sp;
      AffixLink ptr;

      /* Give the node its new value and follow the links to all attached positions */
      node -> value = attach_affix_value (value);
      for (ptr = node -> links; ptr != affix_link_nil; ptr = ptr -> next)
	{ Position dest_pos = ptr -> pos;
	  int dest_side = ptr -> side;
	  dest_pos -> sides[dest_side].sill--;

	  /* Do not follow towards the originating position */
	  if (dest_pos != orig_pos)
    	    { cont_push_position (hnd, dest_pos);
    	      cont_push_continuation (hnd, erts_propagate_affix_value);
    	    };
	};

      /*
	 Check if the original position was a delayed one 
	 The others will be caught by affix propagation
      */
      if (orig_pos -> delayed_func)
	orig_pos -> delayed_func (hnd, orig_pos -> tree_node -> positions);
      else call_continuation (hnd);

      /* Undo the whole cit-and-caboodle */
      hnd -> cont_sp = old_sp;
      for (ptr = node -> links; ptr != affix_link_nil; ptr = ptr -> next)
	{ Position dest_pos = ptr -> pos;
	  int dest_side = ptr -> side;
	  dest_pos -> sides[dest_side].sill++;
	};
      detach_affix_value (&node -> value);
    }
  else if (equal_affix_value (value, node -> value))	/* MS: Check for meta defs */
    call_continuation (hnd);
  
  /* Undo everything */
  cont_push_affix_node (hnd, node);
  cont_push_position (hnd, orig_pos);
  cont_push_affix_value (hnd, value);
  cont_push_continuation (hnd, assign_value);
}

/*
   Unparse an affix value to the other side of the originating position
   We will only unparse a value to a single affix node or match a
   composed value to an ast node. Note that this may be a recursive match
*/
static int decompose_or_assign_affix_value (EagrtsHandle hnd, affix_value value,
					    Position orig_pos, AffixExpr expr)
{ switch (expr -> kind)
    { case single_affix_node:
	{ AffixNode node = expr -> uni.node;
	  cont_push_affix_node (hnd, node);
	  cont_push_position (hnd, orig_pos);
	  cont_push_affix_value (hnd, value);
	  cont_push_continuation (hnd, assign_value);
	  return (1);
	};
      case ast_node:
	{ affix_value_list parts;
	  AffixExpr *exprs;
	  int ix;
	  if (value -> tag != TAGComposed_value) return (0);
	  if (value -> rule_nr != expr -> uni.ast.typenr) return (0);
	  if (value -> Composed_value.marker != expr -> uni.ast.marker) return (0);
	  parts = value -> Composed_value.parts;
	  exprs = expr -> uni.ast.exprs;
	  if (parts -> size != expr -> uni.ast.nr) return (0);
	  for (ix = 0; ix < parts -> size; ix++)
	    if (!decompose_or_assign_affix_value (hnd, parts -> array[ix], orig_pos, exprs[ix]))
	      return (0);

	  /* All parts could be assigned */
	  return (1);
	};
      case concat_node:
      case dyop_arith_node:
      case monop_arith_node: break;
      default: dcg_bad_tag (expr -> kind, "decompose_or_assign_affix_value");
    };
  return (0);
}

static void unparse_affix_value (EagrtsHandle hnd, affix_value value, Position orig_pos, int side)
{ /* Save sp because we do not know how many nodes get assigned */
  cont_ptr old_sp = hnd -> cont_sp;
  if (value == affix_value_nil)
    dcg_internal_error ("unparse_affix_value");
  if (decompose_or_assign_affix_value (hnd, value, orig_pos, orig_pos -> sides[side].expr))
    call_continuation (hnd);
  hnd -> cont_sp = old_sp;
};

/*
   Restrict affix node values by propagation

   Restrict affix position on indicated side. We are called as a result
   of the meet between the value on the lower and the upper side. As a
   result both sides may become more restricted (3 & 6 == 2).

   We use the knowledge that restricting positions are always connected to
   one affix node only, which we have to restrict. Thus a wave of restrictions
   may start until stopped, by being equal (consistent substitution) or by
   being incompatible (a restriction yielding 0).

   Calling convention:
      cont_push_int (hnd, side);
      cont_push_position (hnd, pos);
      cont_push_affix_value (hnd, meet_value);	meet of lower and upper side
      cont_push_continuation (hnd, restrict_position_side)

*/
static void restrict_position_side (EagrtsHandle hnd)
{ /* Pop the arguments */
  affix_value meet_value = cont_pop_affix_value (hnd);
  Position orig_pos = cont_pop_position (hnd);
  int side = cont_pop_int (hnd);
  AffixExpr expr = orig_pos -> sides[side].expr;

  /* If tracing, trace restriction */
  if (hnd -> tracing) erts_trace_restriction (hnd, orig_pos, meet_value);

  /* Pick the affix node, this side is connecting to */
  if (expr == affix_expr_nil)
    call_continuation (hnd)			/* sic */
  else if (expr -> kind != single_affix_node)
    dcg_bad_tag (expr -> kind, "restrict_position_side");
  else
    { AffixNode node = expr -> uni.node;
      affix_value combined_value;
     
      /* If the values agree, we can just continue */
      if (equal_affix_value (meet_value, node -> value))
	call_continuation (hnd)			/* sic */
      else if (ebs_meet_lattice_values (meet_value, node -> value, &combined_value))
	{ /* The confrontation of the meet_value with our own gave a restriction */
	  /* Consequently, our dependents must be restricted as well */
	  cont_ptr old_sp = hnd -> cont_sp;
	  affix_value saved_value = node -> value;
	  AffixLink ptr;

	  /* MS: to add check stack */
	  /* Restrict the node and propagate */
	  node -> value = combined_value;
	  for (ptr = node -> links; ptr != affix_link_nil; ptr = ptr -> next)
	    { Position trgt_pos = ptr -> pos;
	      int my_side = ptr -> side;

	      /* The following line is an optimization, because my_side is already restricted */
	      int other_side = (my_side == lower_side)?upper_side:lower_side;
	      if (trgt_pos != orig_pos)
	        { cont_push_int (hnd, other_side);
	          cont_push_position (hnd, trgt_pos);
	          cont_push_affix_value (hnd, combined_value);
	          cont_push_continuation (hnd, restrict_position_side)
	        };
	    };

	  /* Stage set to play, continue */
	  call_continuation (hnd);

	  /* Local undo */
	  hnd -> cont_sp = old_sp;
	  node -> value = saved_value;
	  detach_affix_value (&combined_value);
	};
    };
      
  /* Undo everything */
  cont_push_int (hnd, side);
  cont_push_position (hnd, orig_pos);
  cont_push_affix_value (hnd, meet_value);
  cont_push_continuation (hnd, restrict_position_side);
}

/*
   Restrict position values on both sides (if necessary)
*/
static void restrict_position_sides (EagrtsHandle hnd, Position pos, affix_value lower_value,
				     affix_value upper_value, affix_value meet_value)
{ /* Save continuation sp since the following code is conditionally called */
  cont_ptr old_sp = hnd -> cont_sp;
  if (!equal_affix_value (lower_value, meet_value))
    { /* Lower side needs to be restricted */
      cont_push_int (hnd, lower_side);
      cont_push_position (hnd, pos);
      cont_push_affix_value (hnd, meet_value);
      cont_push_continuation (hnd, restrict_position_side);
    };
  if (!equal_affix_value (upper_value, meet_value))
    { /* Upper side needs to be restricted */
      cont_push_int (hnd, upper_side);
      cont_push_position (hnd, pos);
      cont_push_affix_value (hnd, meet_value);
      cont_push_continuation (hnd, restrict_position_side);
    };

  /* Call the continuation and undo */
  call_continuation (hnd);
  hnd -> cont_sp = old_sp;
}

/*
   erts_propagate_affix_value tries to propagate the value corresponding
   to one side of a position to its other side when the sill of the side
   drops to 0. If by some reason, both sides have a sill of 0, the affix
   values corresponding to both are compared to implement consistent
   substitution of affix values

   Calling convention:
     cont_push_position (pos);
     cont_push_continuation (erts_propagate_affix_value);
*/
void erts_propagate_affix_value (EagrtsHandle hnd)
{ /* If tracing, trace position */
  Position pos = cont_pop_position (hnd);
  if (hnd -> tracing) erts_trace_position (hnd, pos);

  /* Check if both sides know their expression */
  if ((pos -> sides[upper_side].expr != affix_expr_nil) &&
      (pos -> sides[lower_side].expr != affix_expr_nil))
    { /* Check if one of the sides can be evaluated */
      affix_value upper_value = affix_value_nil;
      affix_value lower_value = affix_value_nil;
      affix_value meet_value;
      if (pos -> sides[upper_side].sill == 0)
	upper_value = calc_affix_expr (pos -> sides[upper_side].expr);
      if (pos -> sides[lower_side].sill == 0)
	lower_value = calc_affix_expr (pos -> sides[lower_side].expr);

      /* If both unknown, we can only continue */
      if ((lower_value == affix_value_nil) && (upper_value == affix_value_nil))
	call_continuation (hnd)		/* sic */
      else
	switch (pos -> kind)
	  /* MS: check what to do with meta defined stuff */
	  { case p_bidirectional:
	      if (lower_value == affix_value_nil)
		unparse_affix_value (hnd, upper_value, pos, lower_side);
	      else if (upper_value == affix_value_nil)
		unparse_affix_value (hnd, lower_value, pos, upper_side);
	      else if (equal_affix_value (lower_value, upper_value))
		{ call_continuation (hnd);
		}
	      break;
	    case p_left_to_right:
	    case p_lower_to_upper:
	      if (lower_value == affix_value_nil)
		{ call_continuation (hnd);
		}
	      else if (upper_value == affix_value_nil)
		unparse_affix_value (hnd, lower_value, pos, upper_side);
	      else if (equal_affix_value (lower_value, upper_value))
		{ call_continuation (hnd);
		}
	      break;
	    case p_equal:
	      if ((lower_value == affix_value_nil) || (upper_value == affix_value_nil))
		{ call_continuation (hnd);
		}
	      else if (equal_affix_value (lower_value, upper_value))
		{ call_continuation (hnd);
		}
	      break;
	    case p_unequal:
	      if ((lower_value == affix_value_nil) || (upper_value == affix_value_nil))
		{ call_continuation (hnd);
		}
	      else if (!equal_affix_value (lower_value, upper_value))
		{ call_continuation (hnd);
		}
	      break;
	    case p_restrict:
	      /* We must restrict: if both sides are equal, there is no penalty */
	      if (equal_affix_value (lower_value, upper_value))
		{ call_continuation (hnd)
		}
	      else if (ebs_meet_lattice_values (lower_value, upper_value, &meet_value))
		{ restrict_position_sides (hnd, pos, lower_value, upper_value, meet_value);
		  detach_affix_value (&meet_value);
		};
	      break;
	    default: dcg_bad_tag (pos -> kind, "erts_propagate_affix_value");
	  };

      /* Free the calculated affix values */
      detach_affix_value (&upper_value);
      detach_affix_value (&lower_value);
    }
  else call_continuation (hnd);

  /* Undo */
  cont_push_position (hnd, pos);
  cont_push_continuation (hnd, erts_propagate_affix_value);
}

/*
   erts_propagate_predicate_value is used to propagate 
   affix values calculated by the predefined predicates

   Calling convention:
     cont_push_position (pos);
     cont_push_affix_value (value)
     cont_push_continuation (erts_propagate_predicate_value);
*/
void erts_propagate_predicate_value (EagrtsHandle hnd)
{ affix_value value = cont_pop_affix_value (hnd);
  Position pos = cont_pop_position (hnd);
  AffixNode node = pos -> sides[lower_side].expr -> uni.node;

  /* If this node does not have a value yet, we can assign it */
  if (node -> value == affix_value_nil)
    { node -> value = attach_affix_value (value);
      pos -> sides[lower_side].sill--;
      cont_push_position (hnd, pos);
      cont_push_continuation (hnd, erts_propagate_affix_value);
      call_continuation (hnd);
      pos -> sides[lower_side].sill++;
      detach_affix_value (&node -> value);
    }
  else if (equal_affix_value (value, node -> value))
    { call_continuation (hnd);
    }

  /* Undo */
  cont_push_position (hnd, pos);
  cont_push_affix_value (hnd, value);
  cont_push_continuation (hnd, erts_propagate_predicate_value);
}
