/*
   File: erts_misc.c
   Defines various miscellaneous functions

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_misc.c,v 1.9 2013/01/11 11:07:29 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_lexicon.h>
#include <ebase_lexicon_impl.h>

/* local includes */
#include "erts_trellis.h"
#include "erts_trellis_impl.h"
#include "erts_handle.h"
#include "erts_handle_impl.h"
#include "erts_trans.h"
#include "erts_cont.h"
#include "erts_misc.h"

/*
   The end point of all continuations
*/
void erts_end_of_parse (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;
  int parse_ok = 0;
  Transition eot_trans = erts_scan_eos_transition (trel, curr_state);
  if (eot_trans != transition_nil)
    { eot_trans -> flags |= TRANS_USED;
      parse_ok = 1;
      trel -> curr_state = eot_trans -> target;
    }
  else parse_ok = hnd -> partial_parse;
  if (parse_ok) erts_pass2 (hnd);
  trel -> curr_state = curr_state;
  cont_push_continuation (hnd, erts_end_of_parse);
}

/*
   The glue operator will force a state jump from state S to I
   to implement the glueing of words in the trellis. For the
   moment we do not generate an explicit transition in the
   trellis for such a state jump
*/
void erts_glue (EagrtsHandle hnd)
{ Trellis trel = hnd -> trellis;
  State curr_state = trel -> curr_state;

  /* Jump if we are in state S */
  trel -> curr_state = erts_handle_glue (trel, curr_state);
  call_continuation (hnd);
  trel -> curr_state = curr_state;
  cont_push_continuation (hnd, erts_glue);
}

/*
   A simple check to clear the negative memo again
*/
void erts_clear_negative_memo (EagrtsHandle hnd)
{ State my_state = cont_pop_state (hnd);
  int nmemo = cont_pop_int (hnd);
  int index = nmemo / 32;
  int bit = nmemo % 32;

#ifdef DEBUG
  if (index >= hnd -> trellis -> neg_memo_size)
    dcg_internal_error ("erts_clear_negative_memo");
#endif
  my_state -> neg_memos[index] &= (~(1 << bit));
  call_continuation (hnd);
  cont_push_int (hnd, nmemo);
  cont_push_state (hnd, my_state);
  cont_push_continuation (hnd, erts_clear_negative_memo);
}

int erts_test_and_set_negative_memo (EagrtsHandle hnd, int nmemo)
{ State curr_state = hnd -> trellis -> curr_state;
  int index = nmemo / 32;
  int bit = nmemo % 32;
  
#ifdef DEBUG
  if (index >= hnd -> trellis -> neg_memo_size)
    dcg_internal_error ("erts_clear_negative_memo");
#endif
  if (curr_state -> neg_memos[index] & (1 << bit))
    return (0);
  curr_state -> neg_memos[index] |= (1 << bit);
  return (1);
}

/*
   Keep track of penalties
*/
#define MAX_PENALTY (1 << 24)
void erts_penalty (EagrtsHandle hnd)
{ int curr_penalty = hnd -> curr_penalty;
  int delta = cont_pop_int (hnd);
  (void) cont_pop_int (hnd);		/* Dummy nr of positions */

  /* Check if we can legally continue */
  if ((delta < 0) && (curr_penalty < -delta))
    dcg_wlog ("This parse accrued too many bonusses to be believable");
  else if ((delta > 0) && (curr_penalty >= MAX_PENALTY - delta))
    /* Do not continue */ ;
  else erts_make_penalty_node (hnd, delta);

  /* Undo everything */
  cont_push_int (hnd, 0);		/* Dummy nr of positions */
  cont_push_int (hnd, delta);
  cont_push_continuation (hnd, erts_penalty);
}

void erts_penalty_one (EagrtsHandle hnd)
{ int curr_penalty = hnd -> curr_penalty;
  (void) cont_pop_int (hnd);		/* Dummy nr of positions */

  /* Check if we can legally continue */
  if (curr_penalty < MAX_PENALTY - 1)
    erts_make_penalty_node (hnd, 1);

  /* Undo everything */
  cont_push_int (hnd, 0);		/* Dummy nr of positions */
  cont_push_continuation (hnd, erts_penalty_one);
}

/*
   Implementation of the control operators
*/
void erts_fail_continuation (EagrtsHandle hnd)
{ cont_push_continuation (hnd, erts_fail_continuation);
}

void erts_abort (EagrtsHandle hnd)
{ char *grammar = cont_pop_text (hnd);
  int linenr = cont_pop_int (hnd);
  dcg_wlog ("\nAbort in grammar '%s', line %d\n", grammar, linenr);
  exit (0);
}

void erts_exit (EagrtsHandle hnd)
{ dcg_wlog ("");
  exit (0);
}

/*
   Implementation of confrontations
*/
void erts_confrontation (EagrtsHandle hnd)
{ propagation_kind kind = cont_pop_int (hnd);

  /* Build the node, propagate its position and continue */
  erts_make_confrontation_node (hnd, kind);

  /* Undo everything */
  cont_push_int (hnd, kind);
  cont_push_continuation (hnd, erts_confrontation);
}
