/*
   File: erts_main.c
   Defines the EAG3 parser driver.

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_main.c,v 1.11 2012/12/26 12:07:54 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>

#ifndef WIN32
#include <unistd.h>
#endif
#include <dlfcn.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_fileutil.h>

/* libebase includes */
#include <ebase_version.h>
#include <ebase_lexicon.h>

/* local includes */
#include "erts_trellis.h"
#include "erts_handle.h"
#include "erts_memory.h"
#include "erts_io.h"
#include "erts_main.h"
#include "erts_handle_impl.h"

static void erts_init_shared_object (EagrtsHandle hnd)
{ /* Locate the shared object */
  char *fname = dcg_new_fmtd_string ("%s.so", hnd -> grammar_name);
  char *path = dcg_construct_path (hnd -> dir_name, fname);
  detach_string (&fname);

  /* Open it and locate parse_trellis */
  hnd -> shared_object_handle = dlopen (path, RTLD_LAZY);
  if (hnd -> shared_object_handle == NULL)
    { dcg_error (1, dlerror ());
      dcg_exit (4);
    };

  /* We have a valid shared object, locate parse trellis */
  hnd -> parse_trellis = dlsym (hnd -> shared_object_handle, "parse_trellis");
  if (hnd -> parse_trellis == NULL)
    { dcg_error (1, dlerror ());
      dcg_exit (4);
    };

  /* Proc has been registered, everything is ok */
}

static void erts_finish_shared_object (EagrtsHandle hnd)
{ dlclose (hnd -> shared_object_handle);
}

void erts_main (EagrtsHandle hnd)
{ /* Prelude */
  erts_init_io (hnd);
  erts_init_stacks (hnd);
  erts_init_memory ();
  erts_init_shared_object (hnd);

  /* Main loop */
  if (debug) dcg_wlog ("Entering main loop of eag runtime system");
  while (erts_read_input (hnd))
    { /* If we have something to parse, do so */
      if (hnd -> input_length)
	{ State start_state;
	  if (debug)
	    { dcg_wlog ("Before initializing trellis");
	      dcg_pool_stats (my_default_pool);
	    };

	  /* Initialize trellis for this piece of input */
	  start_state = erts_init_trellis (hnd, hnd -> input_buffer, hnd -> input_length,
					   hnd -> input_linenr, hnd -> input_colnr);
	  if (debug)
	    { dcg_wlog ("After initializing trellis");
	      dcg_pool_stats (my_default_pool);
	    };

          /* Parse the trellis */
          do
	    { /* Start timers etc. */
	      erts_set_start_state (hnd, start_state);
	      hnd -> parse_trellis (hnd);
	      /* Stop timers etc. */

	      /* Output results and check if we need to restart parsing */
	      erts_output_parse_results (hnd);
	      start_state = erts_determine_next_start_state (hnd);
	      erts_discard_parse_results (hnd);
	    }
	  while (start_state != state_nil);

	  if (debug)
	    { dcg_wlog ("After parsing trellis");
	      dcg_pool_stats (my_default_pool);
	    };

	  /* Done parsing the current input */
	  if (hnd -> show_trellis)
	    erts_dump_trellis (hnd -> trellis);
	  erts_release_trellis (hnd);

	  if (debug)
	    { dcg_wlog ("After releasing trellis");
	      dcg_pool_stats (my_default_pool);
	    };

	  if (debug)
	    erts_dump_memory ();
	};

      /* End of body, try sync */
      erts_try_output_doc_sync (hnd);
    };

  /* Postlude */
  if (debug) dcg_wlog ("Leaving main loop of eag runtime system");
  erts_finish_io (hnd);
  erts_finish_shared_object (hnd);
}
