/*
   File: erts_cont.h
   Defines the basic operations on the continuation stack

   Copyright 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: erts_cont.h,v 1.7 2012/12/26 15:43:16 marcs Exp $"
*/

#ifndef IncErtsCont
#define IncErtsCont

/* local includes */
#include "erts_handle.h"
#include "erts_trellis.h"
#include "erts_tree.h"

/*
   Everything (more or less) can be passed as arguments via the
   continuation stack. Basically, when a future calculation is
   planned and the exact size of the arguments of the calculation
   is known, a check is called to see if enough space is available
   after which all arguments can be pushed.

   The stack grows from low to high memory, so the stack pointer
   will always point to the first free cell that is available.
*/
#include <ebase_ds.h>

typedef union
{ void (*cont)(EagrtsHandle);
  int ival;
  int *iptr;
  real rval;
  char *tval;
  Tree a_tree;
  State a_state;
  Position a_pos;
  AffixNode *a_frame;
  AffixNode a_node;
  affix_value a_value;
} cont_cell, *cont_ptr;

/* Continuation ops */
#define cont_push_continuation(hnd,continuation) { (*hnd -> cont_sp++).cont = continuation; }
#define call_continuation(hnd) { (*--hnd -> cont_sp).cont (hnd); }

/* INT ops */
#define cont_push_int(hnd,nr) { (*hnd -> cont_sp++).ival = (nr); }
#define cont_pop_int(hnd) ((*--hnd -> cont_sp).ival)
#define cont_top_int(hnd) ((*(hnd -> cont_sp - 1)).ival)

/* REAL ops */
#define cont_push_real(hnd,nr) { (*hnd -> cont_sp++).rval = (nr); }
#define cont_pop_real(hnd) ((*--hnd -> cont_sp).rval)

/* TEXT ops */
#define cont_push_text(hnd,txt) { (*hnd -> cont_sp++).tval = (txt); }
#define cont_pop_text(hnd) ((*--hnd -> cont_sp).tval)

/* Tree ops */
#define cont_push_tree(hnd,tree) { (*hnd -> cont_sp++).a_tree = (tree); }
#define cont_pop_tree(hnd) ((*--hnd -> cont_sp).a_tree)

/* State ops */
#define cont_push_state(hnd,state) { (*hnd -> cont_sp++).a_state = (state); }
#define cont_pop_state(hnd) ((*--hnd -> cont_sp).a_state)

/* Position ops */
#define cont_push_position(hnd,pos) { (*hnd -> cont_sp++).a_pos = (pos); }
#define cont_pop_position(hnd) ((*--hnd -> cont_sp).a_pos)

/* Affix node ops */
#define cont_push_affix_node(hnd,node) { (*hnd -> cont_sp++).a_node = (node); }
#define cont_pop_affix_node(hnd) ((*--hnd -> cont_sp).a_node)

/* Affix frame ops */
#define cont_push_affix_frame(hnd,frame) { (*hnd -> cont_sp++).a_frame = (frame); }
#define cont_pop_affix_frame(hnd) ((*--hnd -> cont_sp).a_frame)

/* Affix value ops */
#define cont_push_affix_value(hnd,value) { (*hnd -> cont_sp++).a_value = (value); }
#define cont_pop_affix_value(hnd) ((*--hnd -> cont_sp).a_value)

/* Discarding of stack items */
#define cont_pop(hnd, nr) { hnd -> cont_sp -= nr; }
#define cont_check(hnd, nr) \
{ if ((hnd -> cont_sp + nr) > (hnd -> cont_stack + hnd -> cont_stack_size)) \
    dcg_abort ("cont_check", "Out of continuation stack space"); \
}

#endif /* IncErtsCont */
