/*
   File: ebase_affix_value_utils.c
   Utilities for affix value calculations

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: ebase_affix_value_utils.c,v 1.6 2012/09/21 13:44:01 marcs Exp $
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* libeagbase includes */
#include "ebase_ds.h"
#include "ebase_affix_value_utils.h"

/*
   Concatenate a list of text affix values into a single value
*/
affix_value ebs_concatenate_text_values (affix_value_list vals)
{ /* Calculate length */
  char *dptr, *sptr;
  char *result;
  int len = 0;
  int ix;
  for (ix = 0; ix < vals -> size; ix++)
    { affix_value val = vals -> array[ix];
      if (val == affix_value_nil)
	dcg_abort ("ebs_concatenate_text_values", "Value %d is nil", ix);
      if (val -> tag != TAGText_value)
	dcg_abort ("ebs_concatenate_text_values", "Value %d is not a text value", ix);
      len += strlen (val -> Text_value.text);
    };
 
  result = dcg_malloc (len + 1);
  dptr = result;
  for (ix = 0; ix < vals -> size; ix++)
    for (sptr = vals -> array[ix] -> Text_value.text; *sptr; sptr++, dptr++)
      *dptr = *sptr;
  *dptr = '\0';
  return (new_Text_value (-1, result));
}

/*
   Replicate a text value
*/
affix_value ebs_replicate_text_value (affix_value val1, affix_value val2)
{ char *rhs, *result, *sptr, *dptr;
  int lhs, ix;

  /* Validate lhs */
  if (val1 == affix_value_nil)
    dcg_abort ("ebs_replicate_text_value", "lhs value is nil");
  if (val1 -> tag != TAGInt_value)
    dcg_abort ("ebs_replicate_text_value", "lhs value is not an integer value");
  lhs = val1 -> Int_value.ival;
  if (lhs < 0)
    dcg_abort ("ebs_replicate_text_value", "lhs value is smaller than 0");
  
  /* Validate rhs */
  if (val2 == affix_value_nil)
    dcg_abort ("ebs_replicate_text_value", "rhs value is nil");
  if (val2 -> tag != TAGText_value)
    dcg_abort ("ebs_replicate_text_value", "rhs value is not a text value");
  rhs = val2 -> Text_value.text;

  /* Do the replicate */
  result = dcg_malloc (lhs * strlen (rhs) + 1);
  for (ix = 0, dptr = result; ix < lhs; ix++)
    for (sptr = rhs; *sptr; sptr++, dptr++) *dptr = *sptr;
  *dptr = '\0';
  return (new_Text_value (-1, result));
}

/*
   Evaluate a dyadic operation
   Note: not all operators are allowed in this function
	 especially the lattice operators will abort,
	 since at runtime only restrictions may take place.
*/
affix_value ebs_evaluate_dyadic_operation (operator dop, affix_value val1, affix_value val2)
{ tags_affix_value tag1 = TAGInt_value;
  tags_affix_value tag2 = TAGInt_value;

  /* Validate lhs */
  if (val1 == affix_value_nil)
    dcg_abort ("ebs_evaluate_dyadic_operation", "lhs value is nil");
  if (val2 == affix_value_nil)
    dcg_abort ("ebs_evaluate_dyadic_operation", "rhs value is nil");

  /* Determine the expected type tags */
  switch (dop)
    { case real_times_real:
      case real_plus_real:
      case real_minus_real:
        tag1 = TAGReal_value;
        tag2 = TAGReal_value;
        break;
      case text_plus_text: tag1 = TAGText_value;        /* Fall through */
      case int_times_text: tag2 = TAGText_value;        /* Fall through */
      case modulo:
      case divides:
      case shift_left:
      case shift_right:
      case bitwise_xor:
      case bitwise_or:
      case bitwise_and:
      case int_times_int:
      case int_minus_int:
      case int_plus_int: break;
      default: dcg_bad_tag (dop, "ebs_evaluate_dyadic_operation");
    };

  /* Validate types */
  if (val1 -> tag != tag1)
    dcg_abort ("ebs_evaluate_dyadic_operation", "lhs value has wrong type");
  if (val2 -> tag != tag2)
    dcg_abort ("ebs_evaluate_dyadic_operation", "rhs value has wrong type");

  /* Do the evaluation */
  switch (dop)
    { case real_times_real:
        return (new_Real_value (-1, val1 -> Real_value.rval * val2 -> Real_value.rval));
      case real_plus_real:
        return (new_Real_value (-1, val1 -> Real_value.rval + val2 -> Real_value.rval));
      case real_minus_real:
        return (new_Real_value (-1, val1 -> Real_value.rval - val2 -> Real_value.rval));
      case int_times_text:
        return (ebs_replicate_text_value (val1, val2));
      case modulo:
        if (!val2 -> Int_value.ival)
	  dcg_abort ("ebs_evaluate_dyadic_operation", "division by 0");
        else return (new_Int_value (-1, val1 -> Int_value.ival % val2 -> Int_value.ival));
      case divides:
        if (!val2 -> Int_value.ival)
	  dcg_abort ("ebs_evaluate_dyadic_operation", "division by 0");
        else return (new_Int_value (-1, val1 -> Int_value.ival / val2 -> Int_value.ival));
      case shift_left:
        return (new_Int_value (-1, val1 -> Int_value.ival << val2 -> Int_value.ival));
      case shift_right:
        return (new_Int_value (-1, val1 -> Int_value.ival >> val2 -> Int_value.ival));
      case bitwise_xor:
        return (new_Int_value (-1, val1 -> Int_value.ival ^ val2 -> Int_value.ival));
      case bitwise_or:
        return (new_Int_value (-1, val1 -> Int_value.ival | val2 -> Int_value.ival));
      case bitwise_and:
        return (new_Int_value (-1, val1 -> Int_value.ival & val2 -> Int_value.ival));
      case int_times_int:
        return (new_Int_value (-1, val1 -> Int_value.ival * val2 -> Int_value.ival));
      case int_minus_int:
        return (new_Int_value (-1, val1 -> Int_value.ival - val2 -> Int_value.ival));
      case int_plus_int:
        return (new_Int_value (-1, val1 -> Int_value.ival + val2 -> Int_value.ival));
      default: dcg_bad_tag (dop, "ebs_evaluate_dyadic_operation");
    };

  /* Keep gcc happy */
  return (affix_value_nil);
}

/*
   Evaluate a monadic operation
*/
affix_value ebs_evaluate_monadic_operation (operator mop, affix_value val)
{ tags_affix_value tag = TAGText_value;

  /* Validate argument */
  if (val == affix_value_nil)
    dcg_abort ("ebs_evaluate_monadic_operation", "argument value is nil");

  /* Determine the expected type tag */
  switch (mop)
    { case int_plus_int:
      case int_minus_int:
      case bitwise_not:
        tag = TAGInt_value;
        break;
      case real_plus_real:
      case real_minus_real:
        tag = TAGReal_value;
        break;
      default: dcg_bad_tag (mop, "ebs_evaluate_monadic_operation");
    };

  /* Validate type */
  if (val -> tag != tag)
    dcg_abort ("ebs_evaluate_monadic_operation", "argument value has wrong type");

  /* Do the evaluation */
  switch (mop)
    { case int_plus_int:    return (attach_affix_value (val));
      case int_minus_int:   return (new_Int_value (-1, -val -> Int_value.ival));
      case bitwise_not:     return (new_Int_value (-1, ~val -> Int_value.ival));
      case real_plus_real:  return (attach_affix_value (val));
      case real_minus_real: return (new_Real_value (-1, -val -> Real_value.rval));
      default: dcg_bad_tag (mop, "ebs_evaluate_monadic_operation");
    };

  /* Keep gcc happy */
  return (affix_value_nil);
}

affix_value ebs_join_lattice_values (affix_value_list vals)
{ /* Verify that all values are not nil, of the same size and domain */
  int domain = 0;
  int size = 0;
  int ix, iy;
  affix_value result;
  for (ix = 0; ix < vals -> size; ix++)
    { affix_value val = vals -> array[ix];
      if (val == affix_value_nil)
	dcg_abort ("ebs_unify_lattice_values", "Value %d is nil", ix);
      switch (val -> tag)
	{ case TAGSmall_lattice:
	    if (!ix)
	      { domain = val -> Small_lattice.dom;
		size = 0;
	      }
	    else if ((size != 0) || (domain != val -> Small_lattice.dom))
	      dcg_abort ("ebs_unify_lattice_values", "Value %d is from a different domain", ix);
	    break;
	  case TAGLarge_lattice:
	    if (!ix)
	      { domain = val -> Large_lattice.dom;
		size = val -> Large_lattice.llat -> size;
	      }
	    else if ((size == 0) || (domain != val -> Large_lattice.dom))
	      dcg_abort ("ebs_unify_lattice_values", "Value %d is from a different domain", ix);
	    break;
	  case TAGList_lattice:
	    dcg_abort ("ebs_unify_lattice_values", "No list lattice values yet");
	  default:
	    dcg_abort ("ebs_unify_lattice_values", "Value %d is not a lattice value", ix);
	};
    };

  /* Create new affix value */
  if (size)
    { u_int64_list ul = init_u_int64_list (size);
      result = new_Large_lattice (-1, domain, ul);
      for (iy = 0; iy < size; iy++)
	app_u_int64_list (ul, u_int64_const (0));
    }
  else result = new_Small_lattice (-1, domain, u_int64_const (0));

  /* Join the constituents */
  for (ix = 0; ix < vals -> size; ix++)
    { affix_value val = vals -> array[ix];
      if (size)
	{ for (iy = 0; iy < size; iy++)
	    result -> Large_lattice.llat -> array[iy] |=
	       val -> Large_lattice.llat -> array[iy];
        }
      else result -> Small_lattice.slat |= val -> Small_lattice.slat;
    };

  return (result);
}

int ebs_meet_lattice_values (affix_value val1, affix_value val2, affix_value *resval)
{ /* Verify that the values are not nil, of the same size and domain */
  u_int64 accu = u_int64_const (0);
  affix_value result;
  int domain = 0;
  int size = 0;
  if ((val1 == affix_value_nil) || (val2 == affix_value_nil))
    dcg_abort ("ebs_meet_lattice_values", "Input value is nil");
  if (val1 -> tag != val2 -> tag)
    dcg_abort ("ebs_meet_lattice_values", "Input values are not of the same kind");
  switch (val1 -> tag)
    { case TAGSmall_lattice:
	domain = val1 -> Small_lattice.dom;
	if (val2 -> Small_lattice.dom != domain)
	  dcg_abort ("ebs_meet_lattice_values", "Input values are from different domains");
	break;
      case TAGLarge_lattice:
	domain = val1 -> Large_lattice.dom;
	size = val1 -> Large_lattice.llat -> size;
	if (val2 -> Large_lattice.dom != domain)
	  dcg_abort ("ebs_meet_lattice_values", "Input values are from different domains");
	if (val2 -> Large_lattice.llat -> size != size)
	  dcg_abort ("ebs_meet_lattice_values", "Input values have different sizes");
	break;
      case TAGList_lattice:
	dcg_abort ("ebs_meet_lattice_values", "No list lattice values yet");
      default:
	dcg_abort ("ebs_meet_lattice_values", "Input values are not lattice values");
    };

  /* Create new affix value */
  result = rdup_affix_value (val1);

  /* Meet value2 */
  if (size)
    { int iy;
      for (iy = 0; iy < size; iy++)
	{ result -> Large_lattice.llat -> array[iy] &=
	    val2 -> Large_lattice.llat -> array[iy];
	  accu |= result -> Large_lattice.llat -> array[iy];
	};
    }
  else
    { result -> Small_lattice.slat &= val2 -> Small_lattice.slat;
      accu |= result -> Small_lattice.slat;
    };

  /* If we are not the empty set, return the result */
  if (accu != u_int64_const (0))
    { *resval = result;
      return (1);
    };

  detach_affix_value (&result);
  return (0);
}

int ebs_diff_lattice_values (affix_value val1, affix_value val2, affix_value *resval)
{ /* Verify that the values are not nil, of the same size and domain */
  u_int64 accu = u_int64_const (0);
  affix_value result;
  int domain = 0;
  int size = 0;
  if ((val1 == affix_value_nil) || (val2 == affix_value_nil))
    dcg_abort ("ebs_diff_lattice_values", "Input value is nil");
  if (val1 -> tag != val2 -> tag)
    dcg_abort ("ebs_diff_lattice_values", "Input values are not of the same kind");
  switch (val1 -> tag)
    { case TAGSmall_lattice:
	domain = val1 -> Small_lattice.dom;
	if (val2 -> Small_lattice.dom != domain)
	  dcg_abort ("ebs_diff_lattice_values", "Input values are from different domains");
	break;
      case TAGLarge_lattice:
	domain = val1 -> Large_lattice.dom;
	size = val1 -> Large_lattice.llat -> size;
	if (val2 -> Large_lattice.dom != domain)
	  dcg_abort ("ebs_diff_lattice_values", "Input values are from different domains");
	if (val2 -> Large_lattice.llat -> size != size)
	  dcg_abort ("ebs_diff_lattice_values", "Input values have different sizes");
	break;
      case TAGList_lattice:
	dcg_abort ("ebs_diff_lattice_values", "No list lattice values yet");
      default:
	dcg_abort ("ebs_diff_lattice_values", "Input values are not lattice values");
    };

  /* Create new affix value */
  result = rdup_affix_value (val1);

  /* Meet value2 */
  if (size)
    { int iy;
      for (iy = 0; iy < size; iy++)
	{ result -> Large_lattice.llat -> array[iy] &=
	   ~val2 -> Large_lattice.llat -> array[iy];
	  accu |= result -> Large_lattice.llat -> array[iy];
	};
    }
  else
    { result -> Small_lattice.slat &= ~val2 -> Small_lattice.slat;
      accu |= result -> Small_lattice.slat;
    };

  /* If we are not the empty set, return the result */
  if (accu != u_int64_const (0))
    { *resval = result;
      return (1);
    };

  detach_affix_value (&result);
  return (0);
}

int ebs_lattice_value_is_subset (affix_value val1, affix_value val2)
{ if (val1 == affix_value_nil)
    dcg_abort ("ebs_lattice_value_is_subset", "Value 1 is nil");
  if (val2 == affix_value_nil)
    dcg_abort ("ebs_lattice_value_is_subset", "Value 2 is nil");
  if (val1 -> tag != val2 -> tag)
    dcg_abort ("ebs_lattice_value_is_subset", "Tag mismatch");
  switch (val1 -> tag)
    { case TAGSmall_lattice:
	{ u_int64 v1 = val1 -> Small_lattice.slat;
          u_int64 v2 = val2 -> Small_lattice.slat;
	  if ((v1 & ~v2) != u_int64_const (0)) return (0);
	}; break;
      case TAGLarge_lattice:
	{ u_int64_list v1s = val1 -> Large_lattice.llat;
	  u_int64_list v2s = val2 -> Large_lattice.llat;
	  int ix;
	  if (v1s -> size != v2s -> size)
	    dcg_abort ("ebs_lattice_value_is_subset", "Size mismatch");
	  for (ix = 0; ix < v1s -> size; ix++)
	    { u_int64 v1 = v1s -> array[ix];
              u_int64 v2 = v2s -> array[ix];
	      if ((v1 & ~v2) != u_int64_const (0)) return (0);
	    };
	}; break;
      case TAGList_lattice:
         dcg_abort ("ebs_lattice_value_is_subset", "No list lattice values yet");
      default:
         dcg_abort ("ebs_lattice_value_is_subset", "Values are not of a lattice type");
    };
  return (1);
}
