/*
   File: options.c
   Defines flags and parameters of the eag3 lexicon generator compiler

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: options.c,v 1.5 2013/03/13 10:07:34 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* libeagbase include: only for the version */
#include <ebase_version.h>

/* local includes */
#include "globals.h"
#include "options.h"

/* exported flags */
int lexgen_forced;
int generate_info;

/* Source file names */
string lexicon_fname;
string source_name;
string info_fname;
string base_gname;
string dir_name;

/* Hash sizes */
int affix_hash_size;
int call_hash_size;
int fact_hash_size;

/* Local flags */
static int show_version;

void init_options ()
{ /* Lexicon generator options */
  generate_info = 0;
  lexgen_forced = 0;
  show_version = 0;

  /* DCG options */
  full_verbose = 0;
  verbose = 0;
  debug = 0;

  /* Paths and names */
  lexicon_fname = string_nil;
  info_fname = string_nil;
  base_gname = string_nil;
  dir_name = string_nil;

  /* Hash sizes */
  affix_hash_size = 65536;
  call_hash_size = 65536;
  fact_hash_size = 65536;
}

static void print_usage ()
{ dcg_wlog ("usage: eag3-lexgen [flags] gname [more_flags]");
  dcg_wlog ("where flags and more_flags may be any of the following:");
  dcg_wlog ("   -gi: generate lexicon information");
  dcg_wlog ("   -f:  force lexicon generation");
  dcg_wlog ("   -h:  provide this help");
  dcg_wlog ("   -V:  show version");
  dcg_wlog ("   -d:  set debugging");
  dcg_wlog ("   -v:  verbose");
  dcg_wlog ("   -fv: full verbose");
  dcg_wlog ("   -o ofile: output lexicon in file 'ofile'");
  dcg_wlog ("   -i ifile: output lexicon information in file 'ifile'");
  dcg_wlog ("   -ahs sz:  specify affix hash size");
  dcg_wlog ("   -chs sz:  specify call hash size");
  dcg_wlog ("   -fhs sz:  specify fact hash size");
  dcg_exit (4);
}

static void syntax_error (char *syn_error)
{ dcg_error (0, "error on command line: %s", syn_error);
  print_usage ();
}

static void scan_option (char *ptr, int *i, int argc, char **argv)
{ if (streq (ptr, "V")) show_version = 1;
  else if (streq (ptr, "v"))  { verbose = 1; }
  else if (streq (ptr, "fv")) { show_version = 1; verbose = 1; full_verbose = 1; }
  else if (streq (ptr, "gi")) generate_info = 1;
  else if (streq (ptr, "f")) lexgen_forced = 1;
  else if (streq (ptr, "h")) print_usage ();
  else if (streq (ptr, "d")) debug = 1;
  else if (streq (ptr, "i"))
    { *i = *i + 1;
      if ((*i) < argc)
        { info_fname = new_string (argv[*i]);
          generate_info = 1;
        }
      else syntax_error ("missing lexicon info file name");
    }
  else if (streq (ptr, "o"))
    { *i = *i + 1;
      if ((*i) < argc) { lexicon_fname = new_string (argv[*i]); }
      else syntax_error ("missing lexicon file name");
    }
  else if (streq (ptr, "ahs"))
    { *i = *i + 1;
      if ((*i) < argc) affix_hash_size = atoi (argv[*i]);
      else syntax_error ("missing affix hash size");
    }
  else if (streq (ptr, "chs"))
    { *i = *i + 1;
      if ((*i) < argc) call_hash_size = atoi (argv[*i]);
      else syntax_error ("missing call hash size");
    }
  else if (streq (ptr, "fhs"))
    { *i = *i + 1;
      if ((*i) < argc) fact_hash_size = atoi (argv[*i]);
      else syntax_error ("missing fact hash size");
    }
  else
    syntax_error ("illegal option specified");
}

static void report_version ()
{ if (!show_version) return;
  dcg_wlog ("This is the EAG3 lexicon generator, C Version %s, (C) M. Seutter", EAG3_VERSION);
}

static void set_basename_and_dirname ()
{ char buf[MAXPATHLEN + 1];
  char *sptr, *dptr;
  char *last_slash;
  char *last_dot;

  /* Locate the last slash or DIR_SEP in the line */
  last_slash = NULL;
  for (sptr = source_name; (*sptr); sptr++)
    if ((*sptr == '/') || (*sptr == DIR_SEP)) last_slash = sptr;

  /* Determine the directory name and cut it from the source name */
  if (last_slash == NULL) dir_name = new_string (".");
  else if (last_slash == source_name)
    { dir_name = new_string ("/");
      source_name++;
    }
  else
    { size_t delta = last_slash - source_name;
      char buf2[MAXPATHLEN + 1];
      strncpy (buf2, source_name, delta);
      buf2[delta] = '\0';
      dir_name = new_string (buf2);
      source_name = last_slash + 1;
    };

  /* Remember the last '.' in the line, while copying the source name */
  last_dot = NULL;
  for (sptr = source_name, dptr = buf; (*sptr); )
    { if (*sptr == '.') last_dot = dptr;
      *dptr++ = *sptr++;
    };

  /* Cut off the suffix, if it is there */
  if (last_dot != NULL)
    { /* Check that the suffix is bli: this is the only one allowed */
      if (!streq (last_dot, BLI_SUFFIX))
	syntax_error ("Illegal source file type");
      else *last_dot = '\0';
    }
  else *dptr = '\0';
  if (!buf[0])
    syntax_error ("empty module name");

  /* Set base name */
  base_gname = new_string (buf);
}

void parse_command_line (int argc, char **argv)
{ int ix;
  for (ix = 1; ix < argc; ix++)
    { char *arg = argv[ix];
      if (arg[0] == '-') scan_option (arg+1, &ix, argc, argv);
      else if (source_name == string_nil) source_name = new_string (arg);
      else syntax_error ("too many arguments");
    };

  /* Try and report stuff */
  report_version ();

  /* Exit if no source given */
  if (source_name == string_nil)
    { if (!show_version)
	dcg_wlog ("no source name given, bailing out...");
      exit (0);
    };

  /* Set the basename and dirname for later use */
  set_basename_and_dirname ();
}
