/*
   File: main.c
   Driver for eag3 lexicon generator

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: main.c,v 1.16 2013/03/13 10:07:34 marcs Exp $"
*/

/* global includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_alloc.h>
#include <dcg_fileutil.h>

/* libeagbase includes */
#include <ebase_input.h>

/* local includes */
#include "globals.h"
#include "options.h"
#include "bli_parser.h"
#include "blx_writer.h"
#include "info_writer.h"
#include "parser.h"
#include "calls.h"
#include "lexer.h"
#include "entries.h"
#include "affix_values.h"

static void init_lexicon_generator (void)
{ dcg_init_error (stderr, "eag3-lexgen", NULL);
  dcg_init_alloc ("eag3-lexgen", 0);
  init_options ();
}

static int source_newer (char *src, file_kind kind, time_t blx_age)
{ char *fname = dcg_new_fmtd_string ("%s.%s", src, suffix_from_file_kind (kind));
  char *fpath = dcg_construct_path (dir_name, fname);
  time_t age;
  if (!dcg_file_mtime (fpath, &age))
    dcg_panic ("Could not stat source file '%s'", fpath);
  detach_string (&fname);
  detach_string (&fpath);
  return (age > blx_age);
}

static int source_list_newer (string_list srcs, file_kind kind, time_t blx_age)
{ int ix;
  for (ix = 0; ix < srcs -> size; ix++)
    if (source_newer (srcs -> array[ix], kind, blx_age))
      return (1);
  return (0);
}

static int translations_newer (time_t blx_age)
{ char *fpath;
  time_t age;
  if (translate_option != TranslationsByFile) return (0);
  fpath = dcg_construct_path (dir_name, translate_fname);
  if (!dcg_file_mtime (fpath, &age))
    dcg_panic ("Could not stat source file '%s'", fpath);
  detach_string (&fpath);
  return (age > blx_age);
}

static int recompilation_necessary (void)
{ time_t age;
  if (lexgen_forced) return (1);
  if (!new_lexicon_interface_compatible ()) return (1);
  age = get_binary_lexicon_age ();
  if (source_list_newer (all_lexica, lexicon, age)) return (1);
  if (source_list_newer (all_fact_tables, fact, age)) return (1);
  if (source_list_newer (all_triple_databases, triple, age)) return (1);
  if (translations_newer (age)) return (1);
  dcg_warning (0, "   no changes in interface or input files detected");
  return (0);
}

static void prepare_parsing (void)
{ prepare_globals ();
  prepare_affix_values ();
  prepare_nonterminals ();
  prepare_facts ();
}

static void try_report_statistics (void)
{ if (!full_verbose) return;
  report_affix_values ();
  dcg_wlog ("   collected %d grammar terminals, %d lexicon/fact nonterminals",
	    all_terminals -> size, all_lex_nonts -> size);
  dcg_wlog ("   collected %d different calls of lexicon/fact nonterminals",
	    all_calls -> size);
  dcg_wlog ("   generated %d entries on %d entry lists",
	    nr_entries, all_entry_lists -> size);
}

int main (int argc, char **argv)
{ init_lexicon_generator ();
  parse_command_line (argc, argv);
  parse_lexgen_interface_file ();
  if (!recompilation_necessary ())
    return (0);
  prepare_parsing ();
  parse_files ();
  write_binary_lexicon ();
  try_write_information_file ();
  try_report_statistics ();
  return (0);
}
