/*
   File: info_writer.c
   Generates a humanly readable report on the generated binary lexicon

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: info_writer.c,v 1.10 2012/08/11 20:19:32 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_fileutil.h>

/* libeagbase includes */
#include <ebase_version.h>
#include <ebase_ds.h>
#include <ebase_utils.h>

/* Local includes */
#include "options.h"
#include "globals.h"
#include "info_writer.h"

static FILE *open_information_file ()
{ FILE *fd;
  if (info_fname == string_nil)
    { /* No command line overruling */
      char *fname = dcg_new_fmtd_string ("%s.%s", base_gname, INFO_SUFFIX);
      fd = dcg_fopen_path (dir_name, fname, "w");
      dcg_detach ((void **) &fname);
    }
  else if (info_fname[0] != '/')
    fd = dcg_fopen_path (dir_name, info_fname, "w");
  else fd = dcg_fopen (info_fname, "w");
  return (fd);
}

static void write_sources_list (FILE *info, char *kind, string_list sl)
{ int ix;
  if (!sl -> size) return;
  fprintf (info, "%s:\n", kind);
  for (ix = 0; ix < sl -> size; ix++)
    fprintf (info, "  %s\n", sl -> array[ix]);
  fprintf (info, "\n");
}

static void write_translations (FILE *info)
{ int ix;
  fprintf (info, "Translations:\n");
  for (ix = 0; ix < translation_sources -> size; ix++)
    { fprintf (info, "%d\t->\t%d\t%d\n", 
	       translation_sources -> array[ix],
	       translation_targets -> array[ix],
	       translation_penalties -> array[ix]);
    };
  fprintf (info, "\n");
}

static void write_domains (FILE *info)
{ int ix;
  fprintf (info, "Affix domains:\n");
  for (ix = 0; ix < all_rt_domains -> size; ix++)\
    ebs_dump_domain (info, ix, all_rt_domains -> array[ix], full_verbose);
  fprintf (info, "\n");
}

static void write_type_system (FILE *info)
{ int ix;
  fprintf (info, "Runtime type system:\n");
  for (ix = 0; ix < all_rt_types -> size; ix++)
    ebs_dump_type (info, all_rt_types -> array[ix], full_verbose);
  fprintf (info, "\n");
}

static void write_terminals (FILE *info)
{ int ix;
  fprintf (info, "All terminals:\n");
  for (ix = 0; ix < all_terminals -> size; ix++)
    { terminal term = all_terminals -> array[ix];
      fprintf (info, "Terminal %d: ", ix);
      ebs_dump_text (info, all_terminal_texts -> array[ix]);
      if (term -> marker) fprintf (info, ", marker = %02x", term -> marker);
      fprintf (info, "\n");
    };
  fprintf (info, "\n");
}

static void write_regexps (FILE *info)
{ int ix;
  fprintf (info, "All regular expressions:\n");
  for (ix = 0; ix < all_regexp_nfas -> size; ix++)
    { nfa regexp = all_regexp_nfas -> array[ix];
      fprintf (info, "Regexp %d: '%s'", ix, regexp -> origin);
      if (regexp -> marker) fprintf (info, ", marker = %02x", regexp -> marker);
      fprintf (info, "\n");
    };
  fprintf (info, "\n");
}

static void write_lex_nonts (FILE *info)
{ int ix;
  fprintf (info, "All lexicon nonterminals/facts:\n");
  for (ix = 0; ix < all_lex_nonts -> size; ix++)
    { ebs_dump_lex_nont (info, all_lex_nonts -> array[ix]);
      fprintf (info, "\n");
    };
  fprintf (info, "\n");
}

static void write_call (FILE *info, int_list call)
{ int nont_id = call -> array[0];
  if (nont_id >= all_lex_nonts -> size)
    { int term_id = nont_id - all_lex_nonts -> size;
      terminal term = all_terminals -> array[term_id];
      fprintf (info, "Terminal %d (", term_id);
      ebs_dump_text (info, all_terminal_texts -> array[term_id]);
      if (term -> marker) fprintf (info, ", marker = %02x", term -> marker);
      fprintf (info, ")");
    }
  else
    { lex_nont lt = all_lex_nonts -> array[nont_id];
      int_list name_chars = lt -> name_chars;
      int nidx = 0, fidx = 0, aidx = 0, in_args = 0, ix;
      for (ix = 0; ix < name_chars -> size; ix++)
	if (name_chars -> array[ix])
	  { /* dump name part */
	    if (in_args) fprintf (info, ") ");
	    else if (nidx) fprintf (info, " ");
	    fprintf (info, "%s", lt -> name_parts -> array[nidx]);
	    in_args = 0;
	    nidx++;
	  }
	else
	  { /* dump actual argument */
	    if (in_args) fprintf (info, ", ");
	    else fprintf (info, " (");
	    if (lt -> crits -> array[fidx]) fprintf (info, ">TEXT");
	    else
	      { int actual_id = call -> array[aidx + 1];		/* Note +1 */
	        affix_value value = all_actuals -> array[actual_id];
	        ebs_dump_affix_value (info, value, all_rt_domains);
	        aidx++;
	      }
	    in_args = 1;
	    fidx++;
	  }
      if (in_args) fprintf (info, ")");
    };
  fprintf (info, "\n");
}

static void write_calls (FILE *info)
{ int ix;
  fprintf (info, "All collected calls:\n");
  for (ix = 0; ix < all_calls -> size; ix++)
    { fprintf (info, "Call %d: ", ix);
      write_call (info, all_calls -> array[ix]);
    };
  fprintf (info, "\n");
}

static void write_entry_lists (FILE *info)
{ int ix, iy;
  fprintf (info, "All compiled entry lists:\n");
  for (ix = 1; ix < all_entry_lists -> size; ix++)
    { int_list entries = all_entry_lists -> array[ix];
      fprintf (info, "Entry list %d: ", ix);
      for (iy = 0; iy < entries -> size; iy += 2)
	fprintf (info, "(%d,%d) ", entries -> array[iy], entries -> array[iy + 1]);
      fprintf (info, "\n");
    };
}

static void write_vocabulary_entry (void *arg, char *key, int info)
{ FILE *file = (FILE *) arg;
  fprintf (file, "\"%s\": %d\n", key, info);
}

static void write_vocabulary_entries (FILE *info)
{ int ix;
  for (ix = 0; ix < MAX_MARKERS; ix++)
    { Vocabulary voc = all_vocabularies[ix];
      if (voc == vocabulary_nil) continue;
      fprintf (info, "Vocabulary with marker %d:\n", ix);
      ebs_iterate_over_vocabulary (voc, write_vocabulary_entry, (void *) info);
    };
}

static void write_fact_bucket (FILE *info, int idx, int_list bucket, int ncrits)
{ int iy;
  if (bucket == int_list_nil) return;
  fprintf (info, "Bucket %d ", idx);
  for (iy = 0; iy < bucket -> size; iy += (ncrits + 1))
    { int iz;
      fprintf (info, "(");
      for (iz = 0; iz < ncrits; iz++)
	{ int crit = bucket -> array[iy + iz];
	  if (iz) fprintf (info, ", ");
	  ebs_dump_text (info, all_critical_texts -> array[crit]);
	  fprintf (info, " %d", crit);
	}
      fprintf (info, "): %d", bucket -> array[iy + ncrits]);
      fprintf (info, "\n");
    };
}

static void write_this_fact_table_entries (FILE *info, int fact_nr, int ncrits)
{ int_list_list all_buckets = all_compiled_fact_tables [fact_nr];
  int ix;
  fprintf (info, "Fact table %d\n", fact_nr);
  for (ix = 0; ix < all_buckets -> size; ix++)
    write_fact_bucket (info, ix, all_buckets -> array[ix], ncrits);
}

static void write_fact_table_entries (FILE *info)
{ int ix;
  for (ix = 0; ix < all_lex_nonts -> size; ix++)
    { lex_nont nt = all_lex_nonts -> array[ix];
      if (nt -> fact_nr != -1)
        write_this_fact_table_entries (info, nt -> fact_nr, nt -> ncrits);
    };
}

void try_write_information_file ()
{ FILE *info;
  if (!generate_info) return;
  info = open_information_file ();
  write_sources_list (info, "Lexica", all_lexica);
  write_sources_list (info, "Fact files", all_fact_tables);
  write_sources_list (info, "Triple databases", all_triple_databases);
  write_translations (info);
  write_domains (info);
  write_type_system (info);
  write_terminals (info);
  write_regexps (info);
  write_lex_nonts (info);
  write_calls (info);
  write_entry_lists (info);
  write_vocabulary_entries (info);
  write_fact_table_entries (info);
  fclose (info);
}
