/*
   File: calls.c
   Identifies nonterminals and collects all calls

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: calls.c,v 1.4 2012/04/21 16:22:38 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>

/* libeagbase includes */
#include <ebase_version.h>
#include <ebase_ds.h>
#include <ebase_hash.h>

/* Local includes */
#include "options.h"
#include "globals.h"
#include "calls.h"

/*
   Identification of nonterminals
*/
static int_list_list nonterminal_hash_container;
static int_list_list call_hash_container;
static void allocate_hash_containers ()
{ int ix;
  nonterminal_hash_container = init_int_list_list (all_lex_nonts -> size);
  for (ix = 0; ix < all_lex_nonts -> size; ix++)
    app_int_list_list (nonterminal_hash_container, new_int_list ());
  call_hash_container = init_int_list_list (call_hash_size);
  for (ix = 0; ix < call_hash_size; ix++)
    app_int_list_list (call_hash_container, new_int_list ());
}

static int calculate_nonterminal_hash (string_list name_parts, int_list formals)
{ unsigned int value = 0;
  int ix;
  for (ix = 0; ix < name_parts -> size; ix++)
    value = ebs_hash_continued_text (name_parts -> array[ix], value);
  value = ebs_hash_continued_int (formals -> size, value);
  return ((int) value % all_lex_nonts -> size);
}

static void enter_nonterminals ()
{ int ix;
  for (ix = 0; ix < all_lex_nonts -> size; ix++)
    { lex_nont nont = all_lex_nonts -> array[ix];
      int hash = calculate_nonterminal_hash (nont -> name_parts, nont -> formals);
      int_list bucket = nonterminal_hash_container -> array[hash];
      app_int_list (bucket, ix);
    };
}

void prepare_nonterminals ()
{ allocate_hash_containers ();
  enter_nonterminals ();
}

static int ident_matches (lex_nont nont, string_list name_parts, int_list formals)
{ int_list his_formals = nont -> formals;
  int ix;
  if (!equal_string_list (nont -> name_parts, name_parts)) return (0);
  if (his_formals -> size != formals -> size) return (0);
  for (ix = 0; ix < formals -> size; ix++)
    { int his_formal = his_formals -> array[ix];
      rt_type his_type = all_rt_types -> array[his_formal];
      if (his_type -> tag == TAGSynonym_type)
	his_type = all_rt_types -> array[his_type -> Synonym_type.snr];
      if (his_type -> tag == TAGLattice_type)
        his_formal = ENCODE_DOMAIN_FORMAL (his_type -> Lattice_type.dom);
      if (his_formal != formals -> array[ix])
	return (0);
    };
  return (1);
}

int try_identify_nonterminal (string_list name_parts, int_list formals)
{ int hash = calculate_nonterminal_hash (name_parts, formals);
  int_list bucket = nonterminal_hash_container -> array[hash];
  int ix;
  for (ix = 0; ix < bucket -> size; ix++)
    { int my_id = bucket -> array[ix];
      if (ident_matches (all_lex_nonts -> array[my_id], name_parts, formals))
        return (my_id);
    };
  return (-1);
}

/*
   Identification of calls
   Note: a call is saved as an int_list. Element 0 contains the nonterminal id
   The rest of the list are the ids of the actual arguments.
*/
static int calculate_call_hash (int nont_id, int_list actuals)
{ unsigned value = (unsigned) nont_id;
  int ix;
  for (ix = 0; ix < actuals -> size; ix++)
    value = (997 * value + (unsigned) actuals -> array[ix]) & 0x7ffffff;
  return ((int) value % call_hash_size);
}

static int call_matches (int my_id, int nont_id, int_list actuals)
{ int_list my_call = all_calls -> array[my_id];
  int ix;
  if (my_call -> array[0] != nont_id)
    return (0);

  /* The length of the call should be one bigger than the length of the actuals */
  if (my_call -> size != actuals -> size + 1)
    dcg_internal_error ("call_matches");

  for (ix = 0; ix < actuals -> size; ix++)
    if (my_call -> array[ix + 1] != actuals -> array[ix])
      return (0);
  return (1);
}

int enter_call (int nont_id, int_list actuals)
{ int hash = calculate_call_hash (nont_id, actuals);
  int_list bucket = call_hash_container -> array[hash];
  int_list new_call;
  int my_id, ix;

  /* Search my call in the bucket */
  for (ix = 0; ix < bucket -> size; ix++)
    { my_id = bucket -> array[ix];
      if (call_matches (my_id, nont_id, actuals))
	return (my_id);
    };

  /* We have a new call */
  my_id = all_calls -> size;
  new_call = init_int_list (actuals -> size + 1);
  app_int_list (new_call, nont_id);
  for (ix = 0; ix < actuals -> size; ix++)
    app_int_list (new_call, actuals -> array[ix]);

  /* Register call */
  app_int_list_list (all_calls, new_call);
  app_int_list (bucket, my_id);
  return (my_id);
}
