/*
   File: rules.c
   Analyzes the syntax rules

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: rules.c,v 1.25 2012/12/26 15:43:08 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_alloc.h>
#include <dcg_string.h>
#include <dcg_dstring.h>
#include <dcg_plist.h>

/* local includes */
#include "eag_ds.h"
#include "globals.h"
#include "options.h"
#include "affix_rules.h"
#include "rule_tree.h"
#include "match.h"
#include "expr.h"
#include "contsens.h"
#include "groups.h"
#include "rules.h"

/*
   Rule checking phase 1
   Collect all rules and specs into the rule space and all quasi rules into the quasi rule space

   Note1: rules are identified by their typing alone. During resolution of the formal
   parameters as well as the rule calls, we need the direction of the affixes as well
   to know where we have to introduce implicit confrontation guards.

   Note2: Definitions may be preceded by a specification to indicate the intended typing.
   As a rule, if a definition is preceded (either directly, or through some rules or specs)
   by a spec that matches its name parts and arity it is taken as the formal spec of the
   definition to take the formal typing from. The other aspects of the spec (flow, lex layer
   and rule type) may not conflict with the definition, although the definition may leave
   them undecided. When constructing the specification of a rules, its concatenated name
   and its canonical name without the intended affix direction is created. At a later moment
   the canonical name may be recreated with affix directions because flow analysis may give
   additional information. Likewise, the analysis of the grammar may yield additional rule
   typing and/or layering information.

   The defines list of a grammar is only checked after the rule space is
   constructed. The reason for this is that the defines list is only checked
   as an additional spec, not as a true export.
*/
static rule_tree syntax_rule_space;
static rule_tree quasi_rule_space;
rdecl_list pre_identify_rule (string_list rname_parts, int arity)
{ return (lookup_rule_tree (syntax_rule_space, rname_parts, arity));
}

rdecl_list pre_identify_quasi_rule (string_list rname_parts, int arity)
{ return (lookup_rule_tree (quasi_rule_space, rname_parts, arity));
}

static string make_concat_name (string_list rname_parts, int_list rname_chars)
{ dstring ds = dcg_init_dstring (64);
  int ix, cidx;

  for (ix = 0, cidx = 0; ix < rname_chars -> size; ix++)
    if (rname_chars -> array[ix])
      { dcg_sprintfa_dstring (ds, "%s%s", (cidx)?"_":"", rname_parts -> array[cidx]);
	cidx++;
      };
  return (dcg_finish_dstring (&ds));
}

static string make_canonic_name (string_list rname_parts, int_list rname_chars, signature sig)
{ dstring ds = dcg_init_dstring (64);
  int ix, cidx, fidx;
  int in_args = 0;

  for (ix = 0, cidx = 0, fidx = 0; ix < rname_chars -> size; ix++)
    if (rname_chars -> array[ix])
      { /* Add a new name part */
	dcg_sprintfa_dstring (ds, "%s%s", (in_args)?") ":(cidx)?" ":"", rname_parts -> array[cidx]);
	in_args = 0;
        cidx++;
      }
    else
      { /* Add a new formal */
	affix_rule arule = sig -> array[fidx];
	string aname;
	if (arule == affix_rule_nil) aname = "?UNKNOWN?";
	else if (arule == AFFIX_ERROR) aname = "?ERROR?";
	else aname = arule -> aname;

	dcg_sprintfa_dstring (ds, "%s%s", (in_args)?",":" (", aname);
	in_args = 1;
	fidx++;
      };

  if (in_args) dcg_append_dstring_c (ds, ')');
  return (dcg_finish_dstring (&ds));
}

/*
   When a formal affix parameter in a specification is a single affix variable,
   we can determine the formal typing (also through synonyms)
*/
static affix_rule determine_type_from_fpar (fpar fp)
{ affix_term_list fexp = fp -> fexp;
  affix_term fvar;
  affix_rule adef;
  if (fexp -> size != 1) return (affix_rule_nil);
  fvar = fexp -> array[0];
  if (fvar -> tag != TAGVar) return (affix_rule_nil);
  adef = identify_affix_instance (fvar -> Var.aname);
  fvar -> adef = adef;
  return (adef);
}

static spec determine_spec_from_lhs (grammar gra, plhs lhs, int forced)
{ fpar_list fpars = lhs -> lhs_pars;
  signature sig = init_affix_rule_list (fpars -> size);
  dir_list dirs = init_dir_list (fpars -> size);
  spec new_sp;
  int ix;
  for (ix = 0; ix < fpars -> size; ix++)
    { fpar fp = fpars -> array[ix];
      affix_rule arule = determine_type_from_fpar (fp);
      if (arule == affix_rule_nil)
	{ /* We could not determine the full signature */
	  if (!forced)
	    { detach_affix_rule_list (&sig);
	      detach_dir_list (&dirs);
	      return (spec_nil);
	    };
	  contsens_error (gra, lhs -> line, lhs -> col,
			  "Cannot determine typing of formal parameter %d of spec", ix + 1);
	  app_affix_rule_list (sig, AFFIX_ERROR);
	}
      else app_affix_rule_list (sig, attach_affix_rule (arule));
      app_dir_list (dirs, fp -> fdir);
    };
  new_sp = new_spec (lhs -> rlayer, lhs -> rkind, lhs -> rtype,
		     attach_string_list (lhs -> rname_parts),
		     attach_int_list (lhs -> rname_chars), sig, dirs);
  new_sp -> concat_name = make_concat_name (lhs -> rname_parts, lhs -> rname_chars);
  new_sp -> canonic_name = make_canonic_name (lhs -> rname_parts, lhs -> rname_chars, sig);
  new_sp -> lnr = -1;
  return (new_sp);
}

/*
   Sometimes we have to confront a spec with a second spec or a definition against a specification
*/
static void try_confront_layer (rule_layer *prim, rule_layer sec, int gnr, int line, int col)
{ if (sec == r_syntax) return;
  if (sec == *prim) return;
  if (*prim == r_syntax)
    { *prim = sec;
      contsens_warning_by_gnr (gnr, line, col,
			       "Layer specification strengthens previous specification");
    }
  else contsens_error_by_gnr (gnr, line, col,
			      "Layer specification contradicts previous specification");
}

static void try_confront_kind (rule_kind *prim, rule_kind sec, int gnr, int line, int col)
{ if (sec == *prim) return;
  contsens_error_by_gnr (gnr, line, col, "Rule kind contradicts previous specification");
}

static void try_confront_type (rule_type *prim, rule_type sec, int gnr, int line, int col)
{ if (sec == r_unknown) return;
  if (sec == *prim) return;
  if (*prim == r_unknown)
    { *prim = sec;
      contsens_warning_by_gnr (gnr, line, col, "Rule type strengthens previous specification");
    }
  else contsens_error_by_gnr (gnr, line, col, "Rule type contradicts previous specification");
}

static void try_confront_dir (dir *prim, dir sec, int nr, int gnr, int line, int col)
{ if (sec == d_unknown) return;
  if (sec == *prim) return;
  if (*prim == d_unknown)
    { *prim = sec;
      contsens_warning_by_gnr (gnr, line, col,
		"Direction of affix position %d strengthens previous specification", nr);
    }
  else contsens_error_by_gnr (gnr, line, col,
		"Direction of affix position %d contradicts previous specification", nr);
}

static void try_confront_dirs (dir_list prim, dir_list sec, int gnr, int line, int col)
{ int ix;
  if (prim -> size != sec -> size)
    dcg_internal_error ("try_confront_dirs");
  for (ix = 0; ix < prim -> size; ix++)
    try_confront_dir (&prim -> array[ix], sec -> array[ix], ix + 1, gnr, line, col);
}

static void try_combine_specifications (spec prim, spec sec, prule pr)
{ /* collect line info for error messages */
  int gnr = pr -> gnr;
  int line = pr -> lhs -> line;
  int col = pr -> lhs -> col; 
  try_confront_layer (&prim -> rlayer, sec -> rlayer, gnr, line, col);
  try_confront_kind (&prim -> rkind, sec -> rkind, gnr, line, col);
  try_confront_type (&prim -> rtype, sec -> rtype, gnr, line, col);
  try_confront_dirs (prim -> dirs, sec -> dirs, gnr, line, col);
}

static void allocate_external_rule_entry (spec sp, prule pr)
{ signature sig = sp -> rsig;
  rdecl_list rdecls = enter_rule_tree (&syntax_rule_space, sp -> rname_parts, sig -> size);
  rdecl new_rd;
  rule rl;
  int ix;
  for (ix = 0; ix < rdecls -> size; ix++)
    { rdecl rd = rdecls -> array[ix];
      if (similar_signatures (sig, rd -> rsig))
	{ contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
				 "Multiple specification of external rule");
	  return;
	};
    };

  /* New rule entry */
  rl = new_Ext_rule (sp, attach_string (pr -> Ext_prule.ename));
  new_rd = new_rdecl ();
  new_rd -> rsig = sig;
  new_rd -> rdef = rl;
  app_rdecl_list (rdecls, new_rd);
  rl -> rnr = all_syntax_rules -> size;
  app_rule_list (all_syntax_rules, rl);

  /* Remember pointer to rule and original proto rule */
  sp -> drule = rl;
  sp -> pr = pr;
  pr -> drule = rl;
}

static void allocate_quasi_rule_entry (spec sp, prule pr)
{ signature sig = sp -> rsig;
  rdecl_list rdecls = enter_rule_tree (&quasi_rule_space, sp -> rname_parts, sig -> size);
  rdecl new_rd;
  rule rl;
  int ix;
  for (ix = 0; ix < rdecls -> size; ix++)
    { rdecl rd = rdecls -> array[ix];
      if (similar_signatures (sig, rd -> rsig))
	{ contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
				 "Multiple specification of quasi rule %s",
				 pr -> lhs -> rname_parts -> array[0]);
	  return;
	};
    };

  /* New rule entry */
  rl = new_Quasi_rule (sp, attach_string (pr -> Quasi_prule.rname));
  new_rd = new_rdecl ();
  new_rd -> rsig = sig;
  new_rd -> rdef = rl;
  app_rdecl_list (rdecls, new_rd);
  rl -> rnr = all_quasi_rules -> size;
  app_rule_list (all_quasi_rules, rl);

  /* Remember pointer to rule and original proto rule */
  sp -> drule = rl;
  sp -> pr = pr;
  pr -> drule = rl;
}

static void allocate_specification_entry (spec sp, prule pr)
{ signature sig = sp -> rsig;
  rdecl_list rdecls = enter_rule_tree (&syntax_rule_space, sp -> rname_parts, sig -> size);
  rdecl new_rd;
  rule rl;
  int ix;
  for (ix = 0; ix < rdecls -> size; ix++)
    { rdecl rd = rdecls -> array[ix];
      if (similar_signatures (sig, rd -> rsig))
	{ /* We must check the compatibility of the old and new spec */
	  try_combine_specifications (rd -> rdef -> rspec, sp, pr); 
	  return;
	};
    };

  /* New rule entry with an empty definition list */
  rl = new_Defs (sp, new_definition_list ());
  new_rd = new_rdecl ();
  new_rd -> rsig = sig;
  new_rd -> rdef = rl;
  app_rdecl_list (rdecls, new_rd);
  rl -> rnr = all_syntax_rules -> size;
  app_rule_list (all_syntax_rules, rl);

  /* Remember pointer to rule and original proto rule */
  sp -> drule = rl;
  sp -> pr = pr;
  pr -> drule = rl;
}

/*
   While adding to the definitions list, we make a copy of the formal
   parameters since the type resolution will overwrite the original one.
*/
static void add_to_definitions (definition_list defs, prule pr)
{ int arity = pr -> lhs -> lhs_pars -> size;
  fpar_list fpars = init_fpar_list (arity);
  definition new_def;
  int ix;
  for (ix = 0; ix < arity; ix++)
    { fpar old = pr -> lhs -> lhs_pars -> array[ix];
      fpar new = new_fpar (old -> line, old -> col, old -> fdir,
			   attach_affix_term_list (old -> fexp));
      app_fpar_list (fpars, new);
    };
  new_def = new_definition (fpars, new_variable_list (), attach_group (pr -> Proto_rule.grp));
  new_def -> pr = pr;
  app_definition_list (defs, new_def);
}

static void allocate_rule_entry (spec sp, prule pr)
{ signature sig = sp -> rsig;
  rdecl_list rdecls = enter_rule_tree (&syntax_rule_space, sp -> rname_parts, sig -> size);
  rdecl new_rd;
  rule rl;
  int ix;
  for (ix = 0; ix < rdecls -> size; ix++)
    { rdecl rd = rdecls -> array[ix];
      if (similar_signatures (sig, rd -> rsig))
	{ contsens_error_by_gnr (pr -> gnr, pr -> lhs -> line, pr -> lhs -> col,
				 "Multiple declaration of this rule in different files");
	  return;
	};
    };

  /* New rule entry with an empty definition list */
  rl = new_Defs (sp, new_definition_list ());
  new_rd = new_rdecl ();
  new_rd -> rsig = sig;
  new_rd -> rdef = rl;
  app_rdecl_list (rdecls, new_rd);
  rl -> rnr = all_syntax_rules -> size;
  app_rule_list (all_syntax_rules, rl);

  /* Remember pointer to rule and original proto rule */
  sp -> drule = rl;
  sp -> pr = pr;
  pr -> drule = rl;

  /* Add ourself as the first definition */
  add_to_definitions (rl -> Defs.defs, pr);
}

static void allocate_definition_entry (grammar gra, prule proto, prule_list prules, int idx)
{ plhs lhs = proto -> lhs;
  string_list rname_parts = lhs -> rname_parts;
  int arity = lhs -> lhs_pars -> size;
  spec new_sp;
  int iy;
  for (iy = idx - 1; 0 <= iy; iy--)
    { prule ps = prules -> array[iy];
      if (ps -> tag == TAGExt_prule) continue;
      if (cmp_string_list (ps -> lhs -> rname_parts, rname_parts) != 0) continue;
      if (ps -> lhs -> lhs_pars -> size != arity) continue;

      /*
	 We have a previous proto rule or spec that matches our name parts and arity
	 It is a defining spec if it was a specification or it gave rise to an
	 implicit specification.
      */
      if (ps -> drule == rule_nil) continue;
      if (ps -> drule -> rspec -> pr != ps) continue;

      /*
	 We have a previous specification that matches the tag and arity
	 For the moment, we assume that they are compatible.
	 The later check in phase 2 must confirm this.
      */
      proto -> drule = ps -> drule;
      add_to_definitions (ps -> drule -> Defs.defs, proto);
      return;
    };

  /* If we get here, we must be able to derive the spec from the proto rule itself */
  new_sp = determine_spec_from_lhs (gra, lhs, 0);
  if (new_sp == spec_nil)
    { contsens_error (gra, lhs -> line, lhs -> col, "Cannot determine signature of rule");
      return;
    };

  allocate_rule_entry (new_sp, proto);
}

static void collect_rule_and_spec_for_proto (grammar gra, prule proto, prule_list prules, int idx)
{ proto -> gnr = gra -> gnr;
  switch (proto -> tag)
    { case TAGExt_prule:
	{ spec sp = determine_spec_from_lhs (gra, proto -> lhs, 1);
	  allocate_external_rule_entry (sp, proto);
	}; break;
      case TAGQuasi_prule:
	{ spec sp = determine_spec_from_lhs (gra, proto -> lhs, 1);
	  allocate_quasi_rule_entry (sp, proto);
	}; break;
      case TAGPspec:
	{ spec sp = determine_spec_from_lhs (gra, proto -> lhs, 1);
	  allocate_specification_entry (sp, proto);
	  /* Note: redundant specs will give a memory leak */
        }; break;
      case TAGProto_rule:
	allocate_definition_entry (gra, proto, prules, idx);
	break;
      default: dcg_bad_tag (proto -> tag, "collect_rule_and_spec_for_proto");
    };
}

static void collect_rules_and_specs_for_grammar (grammar gra)
{ prule_list prules = gra -> prules;
  int ix;
  for (ix = 0; ix < prules -> size; ix++)
    collect_rule_and_spec_for_proto (gra, prules -> array[ix], prules, ix);
}

static void collect_rules_and_specs ()
{ int ix;
  dcg_hint ("      collecting rules and specifications");
  for (ix = 0; ix < all_grammars -> size; ix++)
    collect_rules_and_specs_for_grammar (all_grammars -> array[ix]);
}

/* Try dump all rules */
static void try_dump_rules ()
{ int ix;
  if (!dump_rules) return;
  dcg_wlog ("Syntax rules");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule tr = all_syntax_rules -> array[ix];
      dcg_eprint ("\t %d: %s", ix, tr -> rspec -> canonic_name);
      if (tr -> tag == TAGExt_rule) dcg_eprint (", predefined");
      dcg_wlog ("");
    };

  dcg_wlog ("Quasi rules");
  for (ix = 0; ix < all_quasi_rules -> size; ix++)
    dcg_wlog ("\t %d: %s", ix, all_quasi_rules -> array[ix] -> rspec -> canonic_name);
}

/*
   Rule checking phase 2:
   After the collection of the rules and specs, the definitions that
   collectively make up a single rule must now be checked.
*/
static void check_formal_parameters (definition def, spec rspec, int gnr)
{ fpar_list lhs_pars = def -> lhs_pars;
  int ix;
  if (lhs_pars -> size != rspec -> rsig -> size)
    dcg_internal_error ("check_formal_parameters");
  for (ix = 0; ix < lhs_pars -> size; ix++)
    { affix_rule arule = rspec -> rsig -> array[ix];
      affix_term res_expr = affix_term_nil;
      fpar fp = lhs_pars -> array[ix];
      int status;
      try_confront_dir (&rspec -> dirs -> array[ix], fp -> fdir, ix + 1,
			gnr, fp -> line, fp -> col);

      /* Check the formal parameter against the specified one */
      (void) check_expr (def, fp -> fexp);
      status = match_expr (arule, fp -> fexp, &res_expr);
      if (status == 0)
        { contsens_error_by_gnr (gnr, fp -> line, fp -> col,
				 "Formal parameter %d does not conform to affix rule %s",
				 ix + 1, arule -> aname);
	  fp -> err = 1;
	}
      else if (status > 1)
	{ contsens_error_by_gnr (gnr, fp -> line, fp -> col,
				 "Formal parameter %d conforms in multiple ways to affix rule %s",
				 ix + 1, arule -> aname);
	  fp -> err = 1;
	}
      else
	{ detach_affix_term_list (&fp -> fexp);
	  fp -> fexp = new_affix_term_list ();
	  app_affix_term_list (fp -> fexp, res_expr);
	  fp -> adef = arule;
	};
    };
}
 
static void check_spec_and_lhs (definition def, spec rspec)
{ prule proto = def -> pr;
  plhs lhs = proto -> lhs;
  int gnr = proto -> gnr;
  try_confront_layer (&rspec -> rlayer, lhs -> rlayer, gnr, lhs -> line, lhs -> col);
  try_confront_kind (&rspec -> rkind, lhs -> rkind, gnr, lhs -> line, lhs -> col);
  try_confront_type (&rspec -> rtype, lhs -> rtype, gnr, lhs -> line, lhs -> col);
  check_formal_parameters (def, rspec, gnr);
}

static fpar_list duplicate_formal_parameters (fpar_list old_formals, variable_list new_locals)
{ fpar_list new_formals = init_fpar_list (old_formals -> size);
  int ix;
  for (ix = 0; ix < old_formals -> size; ix++)
    { fpar old_formal = old_formals -> array[ix];
      affix_term_list new_fexp;
      fpar new_formal;
      if (old_formal -> err)
        new_fexp = attach_affix_term_list (old_formal -> fexp);
      else new_fexp = remap_expr (old_formal -> fexp, new_locals);
      new_formal = new_fpar (old_formal -> line, old_formal -> col, old_formal -> fdir, new_fexp);
      new_formal -> adef = old_formal -> adef;
      app_fpar_list (new_formals, new_formal);
    };
  return (new_formals);
}

static int check_and_distribute_group (definition_list defs, int index, spec rspec)
{ definition orig_def = defs -> array[index];
  alternative_list alts = orig_def -> grp -> alts;
  int nr_alts = alts -> size;
  int ix;

  /* Check spec and lhs parameters */
  check_spec_and_lhs (orig_def, rspec);
  while (alts -> size > 1)
    { /* Promote this alternative to a new definition */
      alternative alt = alts -> array[alts -> size - 1];
      variable_list new_locals = duplicate_locals (orig_def -> locals);
      fpar_list new_lhs = duplicate_formal_parameters (orig_def -> lhs_pars, new_locals);
      alternative_list new_alts = new_alternative_list ();
      group new_grp = new_group (new_alts);
      definition new_def = new_definition (new_lhs, new_locals, new_grp);
      new_def -> pr = orig_def -> pr;

      /* Move alternative and insert new definition */
      app_alternative_list (new_alts, alt);
      del_alternative_list (alts, alts -> size - 1);
      ins_definition_list (defs, index + 1, new_def);
    };

  /* All alts have been distributed, check each underlying group */
  for (ix = 0; ix < nr_alts; ix++)
    { definition def = defs -> array[index + ix];
      check_group (def, def -> grp);
    };

  return (nr_alts);
}

static void check_rule (rule tr)
{ definition_list defs = tr -> Defs.defs;
  int ix = 0;
  while (ix < defs -> size)
    { int nr_alts = check_and_distribute_group (defs, ix, tr -> rspec);
      ix += nr_alts;
    };
}

static void check_rules ()
{ int ix;
  dcg_hint ("      checking rules");
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule tr = all_syntax_rules -> array[ix];
      if (tr -> tag != TAGDefs) continue;
      check_rule (tr);
    };
}

/*
   While checking anonymous rules are created for options and groups
*/
static int anon_group_nr;
static int anon_option_nr;
static string_list make_anonymous_name (int option)
{ string_list parts = new_string_list ();
  string part;
  if (option)
    { part = dcg_new_fmtd_string ("_anonymous_option_%d", anon_option_nr);
      anon_option_nr++;
    }
  else
    { part = dcg_new_fmtd_string ("_anonymous_group_%d", anon_group_nr);
      anon_group_nr++;
    }
  app_string_list (parts, part);
  return (parts);
}

rule make_anonymous_rule (definition def, group grp, int option)
{ spec father_spec = def -> pr -> drule -> rspec;
  rule_layer rlayer = father_spec -> rlayer;
  rule_kind rkind = r_normal;
  rule_type rtype = (option)?r_option:r_unknown;
  string_list rname_parts = make_anonymous_name (option);
  int_list rname_chars = new_int_list ();
  affix_rule_list rsig = new_affix_rule_list ();
  dir_list dirs = new_dir_list ();
  spec child_spec;
  rule child_rule;

  /* Create the spec of the new rule */
  app_int_list (rname_chars, 1);
  child_spec = new_spec (rlayer, rkind, rtype, rname_parts, rname_chars, rsig, dirs);
  child_spec -> concat_name = attach_string (rname_parts -> array[0]);
  child_spec -> canonic_name = attach_string (rname_parts -> array[0]);
  child_spec -> drule = def -> pr -> drule;
  child_spec -> pr = def -> pr;
  child_spec -> lnr = -1;
  
  /* Create the anonymous rule and register it */
  if (option)
    { child_rule = new_Anonymous_option (child_spec, grp);
      child_rule -> Anonymous_option.anc_spec = father_spec;	/* Remember the ancestor */
      child_rule -> Anonymous_option.def = def;
    }
  else
    { child_rule = new_Anonymous_group (child_spec, grp);
      child_rule -> Anonymous_group.anc_spec = father_spec;	/* Remember the ancestor */
      child_rule -> Anonymous_group.def = def;
    };
  
  child_rule -> rnr = all_syntax_rules -> size;
  app_rule_list (all_syntax_rules, child_rule);

  /* Add an additional alternative to options */
  if (option)
    { alternative first_alt = grp -> alts -> array[0];
      int line = first_alt -> line;
      int col = first_alt -> col;
      fwo_group_list fwos = new_fwo_group_list ();
      member extra = new_Op (line, col, c_success);
      fwo_group fwo = new_Single (g_none, extra);
      alternative new_alt;
      app_fwo_group_list (fwos, fwo);
      new_alt = new_alternative (line, col, -1.0, fwos, 0, trans_member_list_nil);
      app_alternative_list (grp -> alts, new_alt);
    };

  /* The following call may create even more local anonymous rules */
  check_group (def, grp);
  return (child_rule);
}

/*
   Checking of the root call
*/
static definition make_root_definition (member m)
{ fwo_group fwo = new_Single (g_none, attach_member (m));
  fwo_group_list fwos = new_fwo_group_list ();
  alternative nalt = new_alternative (m -> line, m -> col, -1.0, fwos, 0, trans_member_list_nil);
  alternative_list alts = new_alternative_list ();
  group grp = new_group (alts);
  app_fwo_group_list (fwos, fwo);
  app_alternative_list (alts, nalt);
  return (new_definition (new_fpar_list (), new_variable_list (), grp));
}

#define ROOT_NAME "_ROOT_"
static prule make_proto_rule (int line, int col)
{ string_list nsl = new_string_list ();
  int_list ncl = new_int_list ();
  fpar_list npl = new_fpar_list ();
  plhs proto_lhs;
  prule proto;
  string rname = new_string (ROOT_NAME);
  app_string_list (nsl, rname);
  app_int_list (ncl, 1);
  proto_lhs = new_plhs (line, col, r_syntax, r_unknown, r_normal, nsl, ncl, npl);
  proto = new_Pstart (proto_lhs);
  proto -> gnr = root_grammar -> gnr;
  return (proto);
}

static void combine_root_definition (int line, int col, definition rdef, prule proto)
{ spec pspec;
  rule drule;
  pspec = new_spec (r_syntax, r_unknown, r_normal,
		    attach_string_list (proto -> lhs -> rname_parts),
		    attach_int_list (proto -> lhs -> rname_chars),
		    new_affix_rule_list (), new_dir_list ());
  pspec -> concat_name = new_string (ROOT_NAME);
  pspec -> canonic_name = new_string (ROOT_NAME);
  pspec -> lnr = -1;
  drule = new_Defs (pspec, new_definition_list ());
  app_definition_list (drule -> Defs.defs, rdef);
  rdef -> pr = proto;
  pspec -> pr = proto;
  proto -> drule = drule;
  drule -> rnr = all_syntax_rules -> size;
  app_rule_list (all_syntax_rules, drule);
  root_rule = attach_rule (drule);
}

static void create_root_rule_from_call ()
{ member m = root_grammar -> root_call;
  definition rdef = make_root_definition (m);
  prule proto = make_proto_rule (m -> line, m -> col);
  combine_root_definition (m -> line, m -> col, rdef, proto);
  check_group (rdef, rdef -> grp);
}

static rule find_first_defining_rule ()
{ int ix;
  prule_list prules = root_grammar -> prules;
  for (ix = 0; ix < prules -> size; ix++)
    { prule pr = prules -> array[ix];
      if ((pr -> drule != rule_nil) &&
	  ((pr -> tag == TAGPspec) || (pr -> tag == TAGProto_rule)))
	return (pr -> drule);
    };
  contsens_error (root_grammar, 0, 0, "Could not locate a proper start rule in grammmar %s");
  return (rule_nil);
}

static member make_root_call (rule drule)
{ spec rspec = drule -> rspec;
  signature sig = rspec -> rsig;
  apar_list actuals = init_apar_list (sig -> size);
  plhs pr = rspec -> pr -> lhs;
  int ix;
  for (ix = 0; ix < sig -> size; ix++)
    { /* Create actual argument */
      string aname = dcg_new_fmtd_string ("%s%d", sig -> array[ix] -> aname, ix);
      affix_term_list al = new_affix_term_list ();
      app_affix_term_list (al, new_Var (pr -> line, pr -> col, aname));
      apar act = new_apar (d_unknown, al);
      app_apar_list (actuals, act);
    };
  return (new_Call (pr -> line, pr -> col,
		    attach_string_list (rspec -> rname_parts),
		    attach_int_list (rspec -> rname_chars), actuals));
}

static void create_root_rule_from_first_rule ()
{ rule drule = find_first_defining_rule ();
  if (drule == rule_nil) return;
  root_grammar -> root_call = make_root_call (drule);
  create_root_rule_from_call ();
}

static void check_root_call ()
{ dcg_hint ("      checking root call");
  if (root_grammar -> root_call != member_nil)
    create_root_rule_from_call ();
  else create_root_rule_from_first_rule ();
}

static void init_rules ()
{ syntax_rule_space = rule_tree_nil;
  quasi_rule_space = rule_tree_nil;
  anon_group_nr = 0;
  anon_option_nr = 0;
}

/* Exported functions */
void analyze_rules ()
{ dcg_warning (0, "   analyzing grammar rules...");
  init_rules ();
  collect_rules_and_specs ();
  dcg_hint ("      collected %d rules, %d quasi rules",
	    all_syntax_rules -> size, all_quasi_rules -> size);
  dcg_panic_if_errors ();
  check_rules ();
  check_root_call ();
  dcg_hint ("      created %d anonymous option rules", anon_option_nr);
  dcg_hint ("      created %d anonymous group rules", anon_group_nr);
  try_dump_rules ();
  dcg_panic_if_errors ();
}
