/*
   File: properties.c
   Calculates the properties of the grammar

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: properties.c,v 1.11 2013/01/03 11:48:14 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "lexemes.h"
#include "grammars.h"
#include "affix_rules.h"
#include "lattices.h"
#include "rules.h"
#include "transform.h"
#include "nullability.h"
#include "flow.h"
#include "lc_rel.h"
#include "constant.h"
#include "lookahead.h"
#include "reachability.h"
#include "lexicon_info.h"
#include "runtime_info.h"
#include "properties.h"

static void try_dump_properties_tree ()
{ if (!dump_properties_tree) return;
  dcg_wlog ("Affix rules:");
  pp_affix_rule_list (dcg_error_file (), all_affix_rules);
  dcg_wlog ("\nSyntax rules:");
  pp_rule_list (dcg_error_file (), all_syntax_rules);
}

/*
   Calculate properties

   Note that the values of the affix nonterminals need to be calculated before they
   are saved as runtime types. Likewise the lexicon nonterminals must have been
   collected before the first sets can be calculated.

   After experiments with anonymous rules and leftcorner parsing it became clear
   that is is impossible to derive a leftcorner parser in which anonymous rules must
   be executed in the frame of their definition ancestor. Consequently, the
   grammar must be transformed in such a way that all anonymous options and rules
   have their own frame. This should take place before the evaluation of the constants.
*/
void calculate_properties ()
{ dcg_warning (0, "   determining properties...");
  construct_lattices ();
  determine_nullability_in_rules ();
  determine_lc_relation_in_rules ();
  determine_reachability ();
  try_transform_anonymous_rules ();
  try_evaluate_constants ();
  collect_lexicon_info ();
  dcg_warning (0, "   determining more properties...");
  determine_lookahead_sets ();
  do_flow_analysis ();
  collect_runtime_info ();
  try_dump_properties_tree ();
  check_not_freed_affix_rule_list (all_affix_rules);
  check_not_freed_rule_list (all_syntax_rules);
  check_not_freed_rule_list (all_quasi_rules);
  dcg_panic_if_errors ();
}
