/*
   File: options.c
   Defines flags and parameters of the eag3 compiler
   It also specifies some compiler defaults

   Copyright (C) 2010-2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: options.c,v 1.25 2013/03/13 10:08:24 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>

/* libeagbase include: only for the version */
#include <ebase_version.h>

/* local includes */
#include "globals.h"
#include "options.h"

/* exported flags and pragmats */
int agfl_compatible;
int generate_c_only;
int code_negative_memoization;
int code_positive_memoization;
int code_stack_checks;
int code_for_debug;
int code_tracing;
int parse_only;
int no_code;
int strict;

/* exported codegen flags */
int td_parser;
int lc_parser;
int lr_parser;

/* exported debug flags */
int debug;
int dump_parser;
int dump_checker;
int dump_intermed;
int dump_flow;
int dump_lattices;
int dump_affix_rules;
int dump_rules;
int dump_lexemes;
int dump_properties;
int dump_properties_tree;

/* Local flags */
static int show_version;
static int show_include_path;
static int show_link_path;

/* exported grammar search path */
string_list grammar_search_path;

/* Source file names */
string dir_name;                        /* Directory name */
string base_gname;			/* This should match the grammar name */
string source_name;                     /* single file argument */

/* target configuration */
string output_fname;			/* implicit from opts */

/* other exported options */
string variants_option;

/* set default location of predef.eag3 and runtime libraries */
#ifndef INCDIR
#define INCDIR "/home/marcs/share/eag3/include"
#endif
#ifndef LIBDIR
#define LIBDIR "/home/marcs/lib"
#endif

/*
   Read a column separated path from an environment variable
   and store it into the paths list
*/
static void append_paths_from_environment (string_list paths, char *env_var)
{ char *env_paths = getenv (env_var);
  char buf[MAXPATHLEN + 1];
  char *ptr, *dptr;

  /* Environment var has no value, ignore */
  if (env_paths == NULL) return;
  ptr = env_paths;
  if (!(*ptr)) return;
  dptr = buf;
	  
  /* As long as we have something to copy do so */
  while (1)
    if ((*ptr == ':') || !(*ptr))
      { *dptr = 0;
	if (strlen (buf))
	  { /* We have some path in the buffer */
#ifdef WIN32
	    struct _stat sbuf;
	    if ((_stat (buf, &sbuf) == 0) && (sbuf.st_mode & _S_IFDIR))
#else
	    struct stat sbuf;
	    if ((stat (buf, &sbuf) == 0) && S_ISDIR (sbuf.st_mode))
#endif
	    app_string_list (paths, new_string (buf));
	  };
	ptr++;
	if (!(*ptr)) return;
	dptr = buf;	
      }
    else *dptr++ = *ptr++;
}

void init_options ()
{ /* Compiler options */
  agfl_compatible = 0;
  generate_c_only = 0;
  code_negative_memoization = 0;
  code_positive_memoization = 0;
  code_stack_checks = 0;
  code_for_debug = 0;
  code_tracing = 0;
  parse_only = 0;
  no_code = 0;
  strict = 1;

  /* Codegenerator options */
  td_parser = 0;
  lc_parser = 1;
  lr_parser = 0;

  /* Debug options */
  dump_parser = 0;
  dump_checker = 0;
  dump_intermed = 0;
  dump_lattices = 0;
  dump_affix_rules = 0;
  dump_rules = 0;
  dump_lexemes = 0;
  dump_properties = 0;
  show_version = 0;
  show_include_path = 0;
  show_link_path = 0;

  /* DCG options */
  full_verbose = 0;
  verbose = 0;
  debug = 0;

  /* Paths and names */
  dir_name = string_nil;
  base_gname = string_nil;
  source_name = string_nil;
  output_fname = string_nil;

  /*
     Initialize search path for grammars (and libraries)
     We will add the default paths after parsing the command line
  */
  grammar_search_path = new_string_list ();

  /* other exported options */
  variants_option = string_nil;
}

static void print_usage ()
{ dcg_wlog ("usage: eag3-compile [flags] filename [more_flags]");
  dcg_wlog ("where flags and more_flags may be any of the following:");
  dcg_wlog ("   -h:  provide this help");
  dcg_wlog ("   -V:  show version");
  dcg_wlog ("   -d:  set debugging");
  dcg_wlog ("   -v:  verbose");
  dcg_wlog ("   -fv: full verbose");
  dcg_wlog ("   -po: only do context-free analysis of grammar");
  dcg_wlog ("   -td: generate a top down parser");
  dcg_wlog ("   -lc: generate a left corner parser");
  dcg_wlog ("   -lr: generate a lr parser");
  dcg_wlog ("   -ip: show default include path");
  dcg_wlog ("   -lp: show default link path");
  dcg_wlog ("   -ns: non strict compile");
  dcg_wlog ("   -ca: allow compatibility with AGFL");
  dcg_wlog ("   -sc: code with stack checks");
  dcg_wlog ("   -tr: code for tracing"); 
  dcg_wlog ("   -C:  compile to C code");
  dcg_wlog ("   -N:  code for negative memoization");
  dcg_wlog ("   -P:  code for positive memoization");
  dcg_wlog ("   -n:  do not generate code");
  dcg_wlog ("   -g:  code for debugging");
  dcg_wlog ("   -o  oname:   place output in file 'oname'");
  dcg_wlog ("   -Ipath:      add path to grammar search path");
  dcg_wlog ("   -Xtrp:mname: add mname tp list of triple databases");
  dcg_wlog ("   -Xvar:XYZ:   select variants X, Y and Z");
  dcg_wlog ("Debug flags:");
  dcg_wlog ("   -dc: dump checker output");
  dcg_wlog ("   -df: dump flow analysis output");
  dcg_wlog ("   -di: dump intermediate code");
  dcg_wlog ("   -da: dump affix rules");
  dcg_wlog ("   -dr: dump rules");
  dcg_wlog ("   -dla: dump lattices");
  dcg_wlog ("   -dlx: dump lexemes");
  dcg_wlog ("   -dpa: dump parser output");
  dcg_wlog ("   -dpr: dump properties");
  dcg_wlog ("   -dpt: dump tree after properties calculation");
  dcg_exit (4);
}

static void syntax_error (char *syn_error)
{ dcg_error (0, "error on command line: %s", syn_error);
  print_usage ();
}

static void scan_option (char *ptr, int *i, int argc, char **argv)
{ if (streq (ptr, "dc")) { dump_checker = 1; verbose = 1; }
  else if (streq (ptr, "df")) { dump_flow = 1; verbose = 1; }
  else if (streq (ptr, "di")) { dump_intermed = 1; verbose = 1; }
  else if (streq (ptr, "da")) { dump_affix_rules = 1; verbose = 1; }
  else if (streq (ptr, "dr")) { dump_rules = 1; verbose = 1; }
  else if (streq (ptr, "dla")) { dump_lattices = 1; verbose = 1; }
  else if (streq (ptr, "dlx")) { dump_lexemes = 1; verbose = 1; }
  else if (streq (ptr, "dpa")) { dump_parser = 1; verbose = 1; }
  else if (streq (ptr, "dpr")) { dump_properties = 1; verbose = 1; }
  else if (streq (ptr, "dpt")) { dump_properties_tree = 1; verbose = 1; }
  else if (streq (ptr, "d")) debug = 1;
  else if (streq (ptr, "V")) show_version = 1;
  else if (streq (ptr, "v"))  { verbose = 1; }
  else if (streq (ptr, "fv")) { show_version = 1; verbose = 1; full_verbose = 1; }
  else if (streq (ptr, "po")) parse_only = 1;
  else if (streq (ptr, "ip")) show_include_path = 1;
  else if (streq (ptr, "lp")) show_link_path = 1;
  else if (streq (ptr, "ns")) strict = 0;
  else if (streq (ptr, "ca")) agfl_compatible = 1;
  else if (streq (ptr, "h")) print_usage ();
  else if (streq (ptr, "sc")) code_stack_checks = 1;
  else if (streq (ptr, "tr")) code_tracing = 1;
  else if (streq (ptr, "td")) { td_parser = 1; lc_parser = 0; lr_parser = 0; }
  else if (streq (ptr, "lc")) { td_parser = 0; lc_parser = 1; lr_parser = 0; }
  else if (streq (ptr, "lr")) { td_parser = 0; lc_parser = 0; lr_parser = 1; }
  else if (streq (ptr, "n"))  { no_code = 1; td_parser = 0; lc_parser = 0; lr_parser = 0; }
  else if (streq (ptr, "g")) code_for_debug = 1;
  else if (streq (ptr, "C")) generate_c_only = 1;
  else if (streq (ptr, "N")) code_negative_memoization = 1;
  else if (streq (ptr, "P")) code_positive_memoization = 1;
  else if (streq (ptr, "o"))
    { *i = *i + 1;
      if ((*i) < argc)
	{ detach_string (&output_fname);
	  output_fname = new_string (argv[*i]);
	}
      else syntax_error ("missing file name");
    }
  else if (*ptr == 'I')
    { if (*(ptr + 1))
	ins_string_list (grammar_search_path, 0, new_string (ptr + 1));
      else
	{ *i = *i + 1;
	  if ((*i) < argc)
	    ins_string_list (grammar_search_path, 0, new_string (argv[*i]));
	  else syntax_error ("missing grammar search path component");
	};
    }
  else if (strncmp (ptr, "Xvar:", 5) == 0)
    { if (*(ptr + 5))
	variants_option = new_string (ptr + 5);
      else syntax_error ("missing variants");
    }
  else syntax_error ("illegal option specified");
}

static void report_version ()
{ if (!show_version) return;
  dcg_wlog ("This is the EAG3 compiler, C Version %s, (C) M. Seutter", EAG3_VERSION);
}

static void try_report_paths ()
{ if (show_include_path)
    { fprintf (stdout, "%s\n", INCDIR);
      exit (0);
    };
  if (show_link_path)
    { fprintf (stdout, "%s\n", LIBDIR);
      exit (0);
    };
}

/*
   Split the source name into a directory name and a basic (root) grammar name.
   The directory name is then added to the grammar search path list.
*/
static void set_basename_and_dirname ()
{ char buf[MAXPATHLEN + 1];
  char *sptr, *dptr;
  char *last_slash;
  char *last_dot;

  /* Locate the last slash or DIR_SEP in the line */
  last_slash = NULL;
  for (sptr = source_name; (*sptr); sptr++)
    if ((*sptr == '/') || (*sptr == DIR_SEP)) last_slash = sptr;

  /* Determine the directory name and cut it from the source name */
  if (last_slash == NULL) dir_name = new_string (".");
  else if (last_slash == source_name)
    { dir_name = new_string ("/");
      source_name++;
    }
  else
    { size_t delta = last_slash - source_name;
      char buf2[MAXPATHLEN + 1];
      strncpy (buf2, source_name, delta);
      buf2[delta] = '\0';
      dir_name = new_string (buf2);
      source_name = last_slash + 1;
    };

  /* Remember the last '.' in the line, while copying the source name */
  last_dot = NULL;
  for (sptr = source_name, dptr = buf; (*sptr); )
    { if (*sptr == '.') last_dot = dptr;
      *dptr++ = *sptr++;
    };

  /* Cut off the suffix, if it is there */
  if (last_dot != NULL) *last_dot = '\0';
  else *dptr = '\0';
  if (!buf[0])
    syntax_error ("empty grammar name");

  /* Set base name */
  base_gname = new_string (buf);
}

/*
   After parsing the command line, we can add the defaults for the search paths
*/
static void add_default_search_paths ()
{ /*
     Add dir_name (the location where we searched the main grammar)
     in the front of the search path, then append everything in the
     EAG_INCLUDE_PATH and finally append our default predef.eag3
     search path at the end. If someone wants to overrule the
     predef.eag3, that is fine with me: at your own risk... ;-)
  */
  ins_string_list (grammar_search_path, 0, attach_string (dir_name));
  append_paths_from_environment (grammar_search_path, "EAG_INCLUDE_PATH");
  app_string_list (grammar_search_path, new_string (INCDIR));
}

void parse_command_line (int argc, char **argv)
{ int ix;
  for (ix = 1; ix < argc; ix++)
    { char *arg = argv[ix];
      if (arg[0] == '-') scan_option (arg+1, &ix, argc, argv);
      else if (source_name == string_nil) source_name = new_string (arg);
      else syntax_error ("too many arguments");
    };

  /* Try and report stuff */
  try_report_paths ();
  report_version ();

  /* Exit if no source given */
  if (source_name == string_nil)
    { if (!show_version)
	dcg_wlog ("no source name given, bailing out...");
      exit (0);
    };

  /* Set the basename for later use */
  set_basename_and_dirname ();
  add_default_search_paths ();
}
