/*
   File: nullability.c
   Determines which syntax rules and affix rules may produce empty and
   which will always produce empty. Then check the user's declared intentions.

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: nullability.c,v 1.10 2012/12/05 14:11:49 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>
#include <dcg_plist.h>
#include <dcg_plist_ops.h>
#include <dcg_warshall.h>

/* libeagbase includes */
#include <ebase_ds.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "contsens.h"
#include "ast_utils.h"
#include "nullability.h"

/*
   We must determine which text affix rules can produce empty
   This is a typical closure calculation assuming that the marking
   of the initial empty producing rules has taken place.

   The reader should note however that the marking of the affix
   alternatives is not complete. This will be corrected by a
   final marking when we have determined the nullability of all
   affix rules.
*/
static int affix_element_may_produce_empty (affix_element elem)
{ nullability empty = elem -> empty;
  if (empty != e_unknown)
    return (empty != e_never_produces_empty);
  if (elem -> tag != TAGAffix_var)
    dcg_bad_tag (elem -> tag, "element_may_produce_empty");
  empty = elem -> Affix_var.vdef -> empty;
  elem -> empty = empty;	/* Preliminary marking */

  /* The following is overspecified */
  return ((empty == e_may_produce_empty) || (empty == e_always_produces_empty));
}

static int affix_alternative_may_produce_empty (affix_alternative alt, int *change)
{ affix_element_list elems = alt -> elems;
  int ix;
  if (alt -> tag != TAGAffix_concat)
    dcg_internal_error ("affix_alternative_may_produces_empty");
  if (alt -> empty) return (1);
  for (ix = 0; ix < elems -> size; ix++)
    if (!affix_element_may_produce_empty (elems -> array[ix]))
      return (0);
  alt -> empty = 1;
  *change = 1;
  return (1);
}

static int affix_alternatives_may_produce_empty (affix_alternative_list alts, int *change)
{ int ix;
  for (ix = 0; ix < alts -> size; ix++)
    if (affix_alternative_may_produce_empty (alts -> array[ix], change))
      return (1);
  return (0);
}

static void determine_empty_producing_affix_rules ()
{ int asize = all_affix_rules -> size;
  int change, ix;

  dcg_hint ("      determining empty producing text affix rules");
  do
    { change = 0;
      for (ix = 0; ix < asize; ix++)
        { affix_rule arule = all_affix_rules -> array[ix];
	  if (arule -> kind != arule_text) continue;
	  if (arule -> tag != TAGAffix_alts) continue;

	  /* If the empty property of this rule is already known, we do not need to check */
	  if (arule -> empty != e_unknown)
	    continue;

	  /* Check all alternatives if one may produce empty */
	  if (affix_alternatives_may_produce_empty (arule -> Affix_alts.alts, &change))
	    { arule -> empty = e_may_produce_empty;
	      change = 1;
	    };
	};
    }
  while (change);
}

static void mark_never_empty_producing_affix_rules ()
{ /* All rules that are unmarked will by fixed point calculation never produce empty */
  int asize = all_affix_rules -> size;
  int ix;
  for (ix = 0; ix < asize; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      if (arule -> empty == e_unknown)
	arule -> empty = e_never_produces_empty;
      if (arule -> empty == e_never_produces_empty)
	arule -> npred = 1;
    };
}

/*
   The second step is to determine which user defined rules always produce
   empty. This too is a fixed point calculation but from the opposite side.
   We mark those affix rules that are certainly not predicates and then
   take the closure.
*/
static int affix_element_always_produces_empty (affix_element elem)
{ if (elem -> tag == TAGAffix_var)
    { affix_rule vdef = elem -> Affix_var.vdef;
      if (vdef -> empty == e_never_produces_empty) return (0);
      return (!vdef -> npred);
    }
  else return (elem -> empty == e_always_produces_empty);
}
  
static int affix_alternative_always_produces_empty (affix_alternative alt)
{ affix_element_list elems = alt -> elems;
  int ix;
  if (alt -> tag != TAGAffix_concat)
    dcg_internal_error ("affix_alternative_always_produces_empty");
  for (ix = 0; ix < elems -> size; ix++)
    if (!affix_element_always_produces_empty (elems -> array[ix]))
      return (0);
  return (1);
}

static int affix_alternatives_always_produce_empty (affix_alternative_list alts)
{ int ix;
  for (ix = 0; ix < alts -> size; ix++)
    if (!affix_alternative_always_produces_empty (alts -> array[ix]))
      return (0);
  return (1);
}

static void determine_always_empty_producing_affix_rules ()
{ int asize = all_affix_rules -> size;
  int change, ix;
  do
    { change = 0;
      for (ix = 0; ix < asize; ix++)
        { affix_rule arule = all_affix_rules -> array[ix];
	  if (arule -> kind != arule_text) continue;
	  if (arule -> tag != TAGAffix_alts) continue;

	  /*
	     If we already know that an affix rule does not always produce empty,
	     we don't need to check
	  */
	  if (arule -> npred) continue;
	  if (!affix_alternatives_always_produce_empty (arule -> Affix_alts.alts))
	    { arule -> npred = 1;
	      change = 1;
	    };
	};
    }
  while (change);
}

/*
   All affix rules that are still not marked as non predicates always
   produce empty. Adjust the empty flag appropriately and warn the
   user about this affix rule. On the second level, affix rules that
   always produce empty, do not make sense. Perhaps we should make
   an exception for a predefined affix rule EMPTY.
*/
static void mark_always_empty_producing_affix_rules ()
{ int asize = all_affix_rules -> size;
  int ix;
  for (ix = 0; ix < asize; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      if (arule -> empty == e_may_produce_empty && !arule -> npred)
	{ arule -> empty = e_always_produces_empty;
	  contsens_warning_by_gnr (arule -> gnr, arule -> line, arule -> col,
				   "Affix rule %s always produces empty", arule -> aname);
	};
    };
}

/*
   Finish the marking of the elements and alternatives by visiting
   all alternatives and elements and mark the emptyness appropriately
*/
static int finish_empty_marking_in_affix_element (affix_element elem)
{ nullability empty = elem -> empty;
  if (elem -> tag == TAGAffix_var)
    { empty = elem -> Affix_var.vdef -> empty;
      elem -> empty = empty;
    };
  if (empty == e_unknown)
    dcg_internal_error ("finish_empty_marking_in_affix_element");
  return (empty != e_never_produces_empty);
}

static void finish_empty_marking_in_affix_alternative (affix_alternative alt)
{ affix_element_list elems = alt -> elems;
  int result = 1;
  int ix;
  if (alt -> tag != TAGAffix_concat)
    dcg_internal_error ("finish_empty_marking_in_affix_alternative");
  for (ix = 0; ix < elems -> size; ix++)
    result &= finish_empty_marking_in_affix_element (elems -> array[ix]);
  if (alt -> empty && !result)
    dcg_internal_error ("finish_empty_marking_in_affix_alternative");
  alt -> empty = result;
}

static void finish_empty_marking_in_affix_alternatives (affix_alternative_list alts)
{ int ix;
  for (ix = 0; ix < alts -> size; ix++)
    finish_empty_marking_in_affix_alternative (alts -> array[ix]);
}

static void finish_empty_marking_in_affix_rules ()
{ int ix;
  for (ix = 0; ix < all_affix_rules -> size; ix++)
    { affix_rule arule = all_affix_rules -> array[ix];
      if (arule -> kind != arule_text) continue;
      if (arule -> tag != TAGAffix_alts) continue;
 
      finish_empty_marking_in_affix_alternatives (arule -> Affix_alts.alts);
    };
}

void determine_nullability_in_affix_rules ()
{ determine_empty_producing_affix_rules ();
  mark_never_empty_producing_affix_rules ();
  determine_always_empty_producing_affix_rules ();
  mark_always_empty_producing_affix_rules ();
  finish_empty_marking_in_affix_rules ();
}

/*
   For the predefined rules or quasi rules, we determine the nullability from
   the declared rule type, assuming that the library writer knew what he was doing.
*/
static void determine_classification_from_spec (rule srule)
{ nullability empty = e_unknown;
  spec rspec = srule -> rspec;
  int npred = 0;
  switch (rspec -> rtype)
    { case r_rule:
	empty = e_never_produces_empty;
	npred = 1;
	break;
      case r_option:
	empty = e_may_produce_empty;
	npred = 1;
	break;
      case r_predicate:
	empty = e_always_produces_empty;
	npred = 0;
	break;
      default: dcg_bad_tag (rspec -> rtype, "determine_classification_from_spec");
    };

  srule -> empty = empty;
  srule -> npred = npred;
}

static void determine_classification_for_special_rules (rule srule)
{ spec rspec = srule -> rspec;
  switch (rspec -> rkind)
    { case r_lexicon:
	srule -> empty = e_never_produces_empty;
	srule -> npred = 1;
	break;
      case r_fact:
	srule -> empty = e_always_produces_empty;
	srule -> npred = 0;
      default: break;
    };
}

static void do_initial_rule_classification (rule srule)
{ switch (srule -> tag)
    { case TAGExt_rule:
      case TAGQuasi_rule: determine_classification_from_spec (srule);
      case TAGDefs: determine_classification_for_special_rules (srule);
      default: break;
    };
}

static void determine_initial_rule_classification ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    do_initial_rule_classification (all_syntax_rules -> array[ix]);
  for (ix = 0; ix < all_quasi_rules -> size; ix++)
    do_initial_rule_classification (all_quasi_rules -> array[ix]);
}

/*
   For the user defined rules (Defs, groups and options), we must perform
   a fixed point calculation. The previous code already marked every member
   that is not a call with the appropriate nullability. From this marking
   we determine those rules that may produce empty.

   The reader should note that this fixed point calculation does not resolve
   the full empty calculation since not all members are reached during the
   empty calculation. Following the empty calculation is the predicate
   determination (check which rules always produce empty).

   Finally all members, fwos, alternatives and defs are appropriately marked.
*/
static int member_may_produce_empty (member m)
{ nullability empty = m -> empty;
  if (empty != e_unknown)
    return (empty != e_never_produces_empty);
  if (m -> tag != TAGRes_call)
    dcg_bad_tag (m -> tag, "member_may_produce_empty");
  empty = m -> Res_call.rdef -> empty;
  m -> empty = empty;			/* Preliminary marking */
  return ((empty == e_may_produce_empty) || (empty == e_always_produces_empty));
}

static int fwo_may_produce_empty (fwo_group fwo)
{ switch (fwo -> tag)
    { case TAGSingle: return (member_may_produce_empty (fwo -> Single.mem));
      case TAGFwo:
	{ member_list fwo_mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < fwo_mems -> size; ix++)
	    if (member_may_produce_empty (fwo_mems -> array[ix]))
	      return (1);
	  return (0);
	};
      default: dcg_bad_tag (fwo -> tag, "fwo_may_produce_empty");
    };
  return (0);
}

/* Empty alternatives produce empty */
static int alt_may_produce_empty (alternative alt, int *change)
{ fwo_group_list members = alt -> members;
  int ix;
  if (alt -> empty) return (1);
  for (ix = 0; ix < members -> size; ix++)
    if (!fwo_may_produce_empty (members -> array[ix]))
      return (0);
  alt -> empty = 1;
  *change = 1;
  return (1);
}

static int group_may_produce_empty (group grp, int *change)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    if (alt_may_produce_empty (alts -> array[ix], change))
      return (1);
  return (0);
}

static int defs_may_produce_empty (definition_list defs, int *change)
{ int ix;
  for (ix = 0; ix < defs -> size; ix++)
    if (group_may_produce_empty (defs -> array[ix] -> grp, change))
      return (1);
  return (0);
}

static void determine_if_rule_produces_empty (rule srule, int *change)
{ int may_produce_empty = 0;
  if (srule -> empty != e_unknown) return;
  switch (srule -> tag)
    { case TAGDefs:
	may_produce_empty = defs_may_produce_empty (srule -> Defs.defs, change);
	break;
      case TAGAnonymous_option:
        may_produce_empty = group_may_produce_empty (srule -> Anonymous_option.grp, change);
	break;
      case TAGAnonymous_group:
	may_produce_empty = group_may_produce_empty (srule -> Anonymous_group.grp, change);
      case TAGExt_rule: break;
      default: dcg_bad_tag (srule -> tag, "determine_if_rule_produces_empty");
    };

  if (may_produce_empty)
    { srule -> empty = e_may_produce_empty;
      *change = 1;
    };
}

static void determine_empty_producing_rules ()
{ int nr_passes = 0;
  int ix, change;
  dcg_hint ("      determining empty producing rules");
  do
    { change = 0;
      for (ix = 0; ix < all_syntax_rules -> size; ix++)
	determine_if_rule_produces_empty (all_syntax_rules -> array[ix], &change);
      nr_passes++;
    }
  while (change);
  dcg_hint ("      needed %d pass%s for empty detection",
	    nr_passes, (nr_passes == 1)?"":"es");
}

/*
   All rules that are not yet known, are marked as never producing empty.
   All rules that never produce empty, either by detection or by inference,
   are also marked as being non predicates.
*/
static void mark_never_empty_producing_rules ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule mrule = all_syntax_rules -> array[ix];
      if (mrule -> empty == e_unknown)
	mrule -> empty = e_never_produces_empty;
      if (mrule -> empty == e_never_produces_empty)
	mrule -> npred = 1;
    };
}

/*
   The second step is to determine those user defined rules that always
   produce empty. This too is a fixed point calculation, but from the
   opposite side. We mark those rules that are certainly not predicates
   and then take the closure. All rules that may produce empty but are
   not predicates are still syntax rules, but not predicates.
*/
static int member_always_produces_empty (member m)
{ if (m -> tag == TAGRes_call)
    { rule rdef = m -> Res_call.rdef;
      if (rdef -> empty == e_never_produces_empty) return (0);
      return (!rdef -> npred);
    }
  else return (m -> empty == e_always_produces_empty);
}

static int fwo_always_produces_empty (fwo_group fwo)
{ switch (fwo -> tag)
    { case TAGSingle: return (member_always_produces_empty (fwo -> Single.mem));
      case TAGFwo:
	/* You must be really sick to write a free word order predicate */
	{ member_list fwo_mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < fwo_mems -> size; ix++)
	    if (!member_always_produces_empty (fwo_mems -> array[ix]))
	      return (0);
	  return (1);
	};
      default: dcg_bad_tag (fwo -> tag, "fwo_always_produces_empty");
    };
  return (0);
}

static int alt_always_produces_empty (alternative alt)
{ fwo_group_list members = alt -> members;
  int ix;
  for (ix = 0; ix < members -> size; ix++)
    if (!fwo_always_produces_empty (members -> array[ix]))
      return (0);
  return (1);
}

static int group_always_produces_empty (group grp)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    if (!alt_always_produces_empty (alts -> array[ix]))
      return (0);
  return (1);
}

static int defs_always_produce_empty (definition_list defs)
{ int ix;
  for (ix = 0; ix < defs -> size; ix++)
    if (!group_always_produces_empty (defs -> array[ix] -> grp))
      return (0);
  return (1);
}

static void determine_if_rule_always_produces_empty (rule srule, int *change)
{ int always_produces_empty = 0;
  if (srule -> npred) return;
  switch (srule -> tag)
    { case TAGDefs:
	always_produces_empty = defs_always_produce_empty (srule -> Defs.defs);
	break;
      case TAGAnonymous_option:
	always_produces_empty = group_always_produces_empty (srule -> Anonymous_option.grp);
	break;
      case TAGAnonymous_group:
	always_produces_empty = group_always_produces_empty (srule -> Anonymous_group.grp);
	break;
      case TAGExt_rule: return;
      default: dcg_bad_tag (srule -> tag, "determine_if_rule_always_produces_empty");
    };

  if (!always_produces_empty)
    { srule -> npred = 1;
      *change = 1;
    };
}

static void determine_always_empty_producing_rules ()
{ int nr_passes = 0;
  int ix, change;
  do
     { change = 0;
       for (ix = 0; ix < all_syntax_rules -> size; ix++)
	 determine_if_rule_always_produces_empty (all_syntax_rules -> array[ix], &change);
       nr_passes++;
     }
  while (change);
  dcg_hint ("      needed %d pass%s for always empty detection",
            nr_passes, (nr_passes == 1)?"":"es");
}

/*
   Complete the empty marking in the syntax rules. Currently it is
   only done partially and we need the full marking for the leftcorner
   calculation. Members become full marked, alts become marked whether
   that may produce empty to provide a proper warning for options with
   empty alternatives.
*/
static int finish_empty_marking_in_member (member m)
{ if (m -> tag == TAGRes_call)
    m -> empty = m -> Res_call.rdef -> empty;
  return ((m -> empty == e_always_produces_empty) ||
	  (m -> empty == e_may_produce_empty));
}

static int finish_empty_marking_in_fwo (fwo_group fwo)
{ int result = 0;
  switch (fwo -> tag)
    { case TAGSingle:
	result = finish_empty_marking_in_member (fwo -> Single.mem);
	break;
      case TAGFwo:
	{ member_list mems = mems = fwo -> Fwo.mems;
	  int ix;
	  for (ix = 0; ix < mems -> size; ix++)
	    if (finish_empty_marking_in_member (mems -> array[ix]))
	      result = 1;
	}; break;
      default: dcg_bad_tag (fwo -> tag, "finish_empty_marking_in_fwo");
    };
  fwo -> empty = result;
  return (result);
}

static void finish_empty_marking_in_alt (alternative alt)
{ fwo_group_list fwos = alt -> members;
  int result = 1;
  int ix;
  for (ix = 0; ix < fwos -> size; ix++)
    if (!finish_empty_marking_in_fwo (fwos -> array[ix]))
      result = 0;
  alt -> empty = result;
}

static void finish_empty_marking_in_group (group grp)
{ alternative_list alts = grp -> alts;
  int ix;
  for (ix = 0; ix < alts -> size; ix++)
    finish_empty_marking_in_alt (alts -> array[ix]);
}

static void finish_empty_marking_in_defs (definition_list defs)
{ int ix;
  for (ix = 0; ix < defs -> size; ix++)
    finish_empty_marking_in_group (defs -> array[ix] -> grp);
}

static void finish_empty_marking_in_rule (rule srule)
{ switch (srule -> tag)
    { case TAGDefs:
	finish_empty_marking_in_defs (srule -> Defs.defs);
	break;
      case TAGAnonymous_option:
	finish_empty_marking_in_group (srule -> Anonymous_option.grp);
	break;
      case TAGAnonymous_group:
	finish_empty_marking_in_group (srule -> Anonymous_group.grp);
      case TAGExt_rule:
      case TAGQuasi_rule: break;
      default: dcg_bad_tag (srule -> tag, "finish_empty_marking_in_rule");
    };
}

static void finish_empty_marking_in_rules ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    finish_empty_marking_in_rule (all_syntax_rules -> array[ix]);
}

/*
   Warn about alternatives in options that may produce empty
*/
static void check_for_empty_producing_alts_in_options ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    { rule srule = all_syntax_rules -> array[ix];
      prule proto = srule -> rspec -> pr;
      alternative_list alts;
      int iy;
      if (srule -> tag != TAGAnonymous_option) continue;
      alts = srule -> Anonymous_option.grp -> alts;
      for (iy = 0; iy < alts -> size - 1; alts++)	/* Last alt is empty by default */ 
	{ alternative alt = alts -> array[iy];
	  if (alt -> empty)
	    contsens_warning_by_gnr (proto -> gnr, alt -> line, alt -> col,
				     "Alternative %d of option may produce empty", iy);
	};
    };
}

/*
   Since all derived nullability properties are known, we can give
   all rules their final derived marking. This is also the time,
   when we can confront the user's annotation with the derived marking.
*/
static void determine_final_derived_rule_marking (rule srule)
{ spec rspec = srule -> rspec;
  prule proto = rspec -> pr;
  int line = proto -> lhs -> line;
  int col = proto -> lhs -> col;
  nullability expected = e_unknown;
  if ((srule -> empty == e_may_produce_empty) && !srule -> npred)
    srule -> empty = e_always_produces_empty;
  switch (rspec -> rtype)
    { case r_unknown: break;
      case r_rule:	expected = e_never_produces_empty; break;
      case r_option:	expected = e_may_produce_empty; break;
      case r_predicate:	expected = e_always_produces_empty;
      default: break; 
    };

  if (expected == e_unknown)
    { switch (srule -> empty)
	{ case e_unknown: dcg_internal_error ("determine_final_derived_rule_marking");
	  case e_may_produce_empty:	rspec -> rtype = r_option; break;
	  case e_always_produces_empty:	rspec -> rtype = r_predicate; break;
	  case e_never_produces_empty:	rspec -> rtype = r_rule;
	  default: break;
	};
      if (dump_properties)
        contsens_warning_by_gnr (proto -> gnr, line, col,
				 "Rule %s has been determined to be a %s",
			         rspec -> canonic_name, string_from_rule_type (rspec -> rtype));
    }
  else if (expected != srule -> empty)
    contsens_error_by_gnr (proto -> gnr, line, col, "Rule %s was declared as %s but %s",
			   rspec -> canonic_name, string_from_rule_type (rspec -> rtype),
			   string_from_nullability (srule -> empty));
  else if (dump_properties)
    contsens_warning_by_gnr (proto -> gnr, line, col,
			     "Rule %s has been checked to be a %s",
			     rspec -> canonic_name, string_from_rule_type (rspec -> rtype));
}

static void determine_final_derived_marking ()
{ int ix;
  for (ix = 0; ix < all_syntax_rules -> size; ix++)
    determine_final_derived_rule_marking (all_syntax_rules -> array[ix]);
}
 
void determine_nullability_in_rules ()
{ determine_initial_rule_classification ();
  determine_empty_producing_rules (); 
  mark_never_empty_producing_rules ();
  determine_always_empty_producing_rules ();
  finish_empty_marking_in_rules ();
  check_for_empty_producing_alts_in_options ();
  determine_final_derived_marking ();
}
