/*
   File: expr.c
   Checks simple affix expressions and resolves binary operators

   Copyright (C) 2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: expr.c,v 1.14 2012/10/05 19:50:29 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>
#include <string.h>

/* support lib includes */
#include <dcg.h>
#include <dcg_alloc.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* libeagbase includes */
#include <ebase_ds.h>
#include <ebase_utils.h>

/* local includes */
#include "eag_ds.h"
#include "ast_utils.h"
#include "globals.h"
#include "options.h"
#include "affix_rules.h"
#include "lattices.h"
#include "contsens.h"
#include "expr.h"

/*
   Checking and identification of affix variables
*/
static int is_a_local_variable (definition d, string vname, variable *vdef)
{ variable_list locals = d -> locals;
  int ix;
  for (ix = 0; ix < locals -> size; ix++)
    { variable local = locals -> array[ix];
      if (streq (vname, local -> vname))
	{ *vdef = local;
	  return (1);
	};
    };
  return (0);
}

/*
   Simple checking of expressions, affix variables and affix instances

   The result of a check is that every affix formula becomes typed with the exception
   of enclosed expressions for which resolution might be needed. In effect it is a
   kind of precomputation that is only done once, leaving the type pointer in the
   formula node. Upon returning null, we still do not know the type yet.
*/
static int make_typed_operator (affix_term term, arule_kind kind1, arule_kind kind2)
{ switch (term -> tag)
    { case TAGDyop:
	switch (term -> Dyop.dop)
	  { case plus:
	      if (kind1 != kind2) return (0);
	      switch (kind1)
		{ case arule_int:  term -> Dyop.dop = int_plus_int; return (1);
		  case arule_real: term -> Dyop.dop = real_plus_real; return (1);
		  case arule_text: term -> Dyop.dop = text_plus_text; return (1);
		  default: return (0);
		};
	    case minus:
	      if (kind1 != kind2) return (0);
	      switch (kind1)
		{ case arule_int:  term -> Dyop.dop = int_minus_int; return (1);
		  case arule_real: term -> Dyop.dop = real_minus_real; return (1);
		  default: return (0);
		};
	    case times:
	      switch (kind1)
		{ case arule_int:
		    switch (kind2)
		      { case arule_int:  term -> Dyop.dop = int_times_int; return (1);
			case arule_text: term -> Dyop.dop = int_times_text; return (1);
			default: return (0);
		      };
		  case arule_real:
		    if (kind2 != arule_real) return (0);
		    term -> Dyop.dop = real_times_real;
		    return (1);
		  case arule_text:
		    { affix_term left = term -> Dyop.arg1;
		      affix_term right = term -> Dyop.arg2;
		      if (kind2 != arule_int) return (0);
		      term -> Dyop.dop = int_times_text;
		      term -> Dyop.arg1 = right;
	              term -> Dyop.arg2 = left;
		    };
		  default: return (0);
		};
	    case a_union:
	      if (kind1 != kind2) return (0);
	      switch (kind1)
		{ case arule_int: term -> Dyop.dop = bitwise_or; return (1);
		  case arule_lattice: return (1);	/* It is the union operator */
		  default: return (0);
		}
	    case a_part:
	      if (kind1 != kind2) return (0);
	      switch (kind1)
		{ case arule_int: term -> Dyop.dop = bitwise_or; return (1);
		  case arule_lattice: return (1);	/* It is the partition operator */
		  default: return (0);
		}
	    default: /* All other binary operators are between ints */
	      if (kind1 != kind2) return (0);
	      return (kind1 == arule_int);
	  };
      case TAGMonop:
	switch (term -> Monop.mop)
	  { case plus:
	      switch (kind1)
		{ case arule_int:  term -> Monop.mop = int_plus_int; return (1);
		  case arule_real: term -> Monop.mop = real_plus_real; return (1);
		  default: return (0);
		}
	    case minus:
	      switch (kind1)
	        { case arule_int:  term -> Monop.mop = int_minus_int; return (1);
		  case arule_real: term -> Monop.mop = real_minus_real; return (1);
		  default: return (0);
	        }
            case bitwise_not:
	      if (kind1 == arule_int) return (1);
	      /* fall thru */
	    default: return (0);
	  };
      default: return (0);
    };
  return (0);
}

static arule_kind check_term (definition d, affix_term term)
{ arule_kind result = arule_unknown;
  switch (term -> tag)
    { case TAGTerminal:
	term -> adef = affix_rule_nil;
	/* Tentative; can do better */
	if (identify_lattice_element (term -> Terminal.marker) != element_nil)
	  return (arule_lattice);
	break;
      case TAGDyop:
	{ arule_kind kind1 = check_term (d, term -> Dyop.arg1);
	  arule_kind kind2 = check_term (d, term -> Dyop.arg2);

	  /* Check argument 1 */
	  if (kind1 == arule_unknown)
	    { contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
				     "Operator '%s' cannot be applied to first argument",
				     ebs_string_from_operator (term -> Dyop.dop));
	      kind1 = arule_error;
	    };

	  /* Check argument 2 */
	  if (kind2 == arule_unknown)
	    { contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
				     "Operator '%s' cannot be applied to second argument",
				     ebs_string_from_operator (term -> Dyop.dop));
	      kind2 = arule_error;
	    };

	  /* Check for recovery */
	  if ((kind1 == arule_error) || (kind2 == arule_error))
	    { term -> adef = AFFIX_ERROR; 
	      return (arule_error);
	    };

	  /* Check for compatibility of operator to both operands */
	  if (!make_typed_operator (term, kind1, kind2))
	    { contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
				     "can't apply dyadic operator '%s' to these operands",
				     ebs_string_from_operator (term -> Dyop.dop));
	      return (arule_error);
	    };

	  /* Return the correct (identified) type, unknown or AFFIX_ERROR */
	  result = kind_from_operator (term -> Dyop.dop);
	  term -> adef = get_primitive_type (result);
	}; break;
      case TAGMonop:
	{ arule_kind kind = check_term (d, term -> Monop.arg);

	  if (kind == arule_unknown)
	    { contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
				     "can't apply monadic operator '%s' to these operands",
				     ebs_string_from_operator (term -> Monop.mop));
	      term -> adef = AFFIX_ERROR; 
	      return (arule_error);
	    };

	  /* Check for compatibility of operator and operand */
	  if (!make_typed_operator (term, kind, arule_unknown))
	    { contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
				     "can't apply monadic operator %s to this operand",
				     ebs_string_from_operator (term -> Monop.mop));
	      return (arule_error);
	    };

	  /* Return the correct (identified) type or AFFIX_ERROR */
	  result = kind_from_operator (term -> Monop.mop);
	  term -> adef = get_primitive_type (result);
	  return (result);
	};
      case TAGEnclosed: 
	result = check_expr (d, term -> Enclosed.terms);
	term -> adef = get_primitive_type (result);
	return (result);
	break;
      case TAGInt:  term -> adef = get_primitive_type (arule_int);  break;
      case TAGReal: term -> adef = get_primitive_type (arule_real); break;
      case TAGText: term -> adef = get_primitive_type (arule_text); break;
      case TAGVar:
	{ string aname = term -> Var.aname;
	  variable vdef;
	  if (!is_a_local_variable (d, aname, &vdef))
	    { /* New local variable, add to list of locals */
	      affix_rule adef = identify_affix_instance (aname);
	      vdef = new_variable (aname);
	      vdef -> vnr = d -> locals -> size;
	      vdef -> cnr = -1;			/* We are not constant (yet) */
	      app_variable_list (d -> locals, vdef);
	      if (adef == affix_rule_nil)
		{ contsens_error_by_gnr (d -> pr -> gnr, term -> line, term -> col,
					 "could not identify local affix %s", aname);
		  adef = AFFIX_ERROR;
		};
	      vdef -> adef = adef;
	    };
	  term -> Var.vdef = vdef;
	  term -> adef = vdef -> adef;
	}; break;
      default: dcg_bad_tag (term -> tag, "check_term");
    };

  /* Return the result if available */
  if ((term -> adef == affix_rule_nil) && (result == arule_lattice)) return (arule_lattice);
  if (term -> adef == affix_rule_nil) return (arule_unknown);
  if (term -> adef == AFFIX_ERROR) return (arule_error);
  return (term -> adef -> kind);
}

arule_kind check_expr (definition d, affix_term_list terms)
{ int ix;
  if (terms -> size == 1)
    return (check_term (d, terms -> array[0]));
  for (ix = 0; ix < terms -> size; ix++)
    (void) check_term (d, terms -> array[ix]);

  /* A list of terms must have some tree structure */
  return (arule_tree);
}

/*
   To distribute the checked left hand side of a definition among the
   alternatives in the right hand side, we must be able to remap the
   references to the variables while keeping the identified parts
   correct.
*/
variable_list duplicate_locals (variable_list locals)
{ variable_list new_locals = init_variable_list (locals -> size);
  int ix;
  for (ix = 0; ix < locals -> size; ix++)
    { variable old_local = locals -> array[ix];
      variable new_local = new_variable (attach_string (old_local -> vname));
      new_local -> vnr = old_local -> vnr;
      if (new_local -> vnr != ix)
	dcg_internal_error ("duplicate_locals");
      new_local -> adef = old_local -> adef;
      new_local -> cnr = -1;
      app_variable_list (new_locals, new_local);
    };
  return (new_locals);
}

static affix_term_list remap_affix_term_list (affix_term_list terms, variable_list new_locals);
static affix_term remap_affix_term (affix_term term, variable_list new_locals)
{ affix_term new_term = affix_term_nil;
  int line = term -> line;
  int col = term -> col;
  switch (term -> tag)
    { case TAGTerminal:
	{ new_term = new_Terminal (line, col, attach_string (term -> Terminal.marker));
	  new_term -> Terminal.edef = term -> Terminal.edef;
	}; break;
      case TAGDyop:
	new_term = new_Dyop (line, col, term -> Dyop.dop,
			     remap_affix_term (term -> Dyop.arg1, new_locals),
			     remap_affix_term (term -> Dyop.arg2, new_locals));
	break;
      case TAGMonop:
	new_term = new_Monop (line, col, term -> Monop.mop,
			      remap_affix_term (term -> Monop.arg, new_locals));
	break;
      case TAGAst:
	new_term = new_Ast (line, col, remap_affix_term_list (term -> Ast.terms, new_locals),
			    term -> Ast.altnr);
	break;
      case TAGInt:
	new_term = new_Int (line, col, term -> Int.num);
	break;
      case TAGReal:
	new_term = new_Real (line, col, term -> Real.num);
	break;
      case TAGText:
	new_term = new_Text (line, col, attach_string (term -> Text.txt));
	break;
      case TAGVar:
	{ variable old_vdef = term -> Var.vdef;
	  variable new_vdef = new_locals -> array[old_vdef -> vnr];
	  new_term = new_Var (line, col, attach_string (term -> Var.aname));
	  new_term -> Var.vdef = new_vdef;
	}; break;
      default: dcg_bad_tag (term -> tag, "remap_affix_term");
    };

  new_term -> adef = term -> adef;
  return (new_term);
} 

static affix_term_list remap_affix_term_list (affix_term_list terms, variable_list new_locals)
{ affix_term_list new_terms = init_affix_term_list (terms -> size);
  int ix;
  for (ix = 0; ix < terms -> size; ix++)
    app_affix_term_list (new_terms, remap_affix_term (terms -> array[ix], new_locals));
  return (new_terms);
}

affix_term_list remap_expr (affix_term_list terms, variable_list new_locals)
{ return (remap_affix_term_list (terms, new_locals));
}
