/*
   File: contsens.c
   Does context dependent analysis

   Copyright (C) 2009 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: contsens.c,v 1.16 2012/08/23 21:06:57 marcs Exp $"
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>

/* local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "lexemes.h"
#include "grammars.h"
#include "affix_rules.h"
#include "lattices.h"
#include "rules.h"
#include "nullability.h"
#include "lc_rel.h"
#include "contsens.h"

/* error administration */
void contsens_error (grammar gra, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_error (0, "Grammar %s: %s", gra -> gname, buf);
  else dcg_error (0, "Grammar %s, line %d, col %d: %s", gra -> gname, line, col, buf);
}

void contsens_error_by_gnr (int gnr, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_error (0, "Grammar %s: %s", all_grammars -> array[gnr] -> gname, buf);
  else dcg_error (0, "Grammar %s, line %d, col %d: %s",
		  all_grammars -> array[gnr] -> gname, line, col, buf);
}

void contsens_warning (grammar gra, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_warning (0, "Grammar %s: %s", gra -> gname, buf);
  else dcg_warning (0, "Grammar %s, line %d, col %d: %s", gra -> gname, line, col, buf);
}

void contsens_warning_by_gnr (int gnr, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_warning (0, "Grammar %s: %s", all_grammars -> array[gnr] -> gname, buf);
  else dcg_warning (0, "Grammar %s, line %d, col %d: %s",
		    all_grammars -> array[gnr] -> gname, line, col, buf);
}

void contsens_hint (grammar gra, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_hint ("Grammar %s: %s", gra -> gname, buf);
  else dcg_hint ("Grammar %s, line %d, col %d: %s", gra -> gname, line, col, buf);
}

void contsens_hint_by_gnr (int gnr, int line, int col, char *format, ...)
{ char buf[MAXSTRLEN];
  va_list arg_ptr;
  va_start (arg_ptr, format);
  vsprintf (buf, format, arg_ptr);
  va_end (arg_ptr);
  if (!line && !col) dcg_hint ("Grammar %s: %s", all_grammars -> array[gnr] -> gname, buf);
  else dcg_hint ("Grammar %s, line %d, col %d: %s",
		 all_grammars -> array[gnr] -> gname, line, col, buf);
}

static void try_dump_checker ()
{ if (!dump_checker) return;
  dcg_wlog ("Affix rules:");
  pp_affix_rule_list (dcg_error_file (), all_affix_rules);
  dcg_wlog ("\nSyntax rules:");
  pp_rule_list (dcg_error_file (), all_syntax_rules);
  dcg_wlog ("\nQuasi rules:");
  pp_rule_list (dcg_error_file (), all_quasi_rules);
}

/*
   Context sensitive analysis
*/
void context_sensitive_analysis ()
{ prepare_lexeme_gathering ();
  pre_analyze_grammars ();
  analyze_affix_rules ();
  analyze_lattices_in_affix_rules ();
  analyze_rules ();
  check_not_freed_affix_rule_list (all_affix_rules);
  check_not_freed_rule_list (all_syntax_rules);
  check_not_freed_rule_list (all_quasi_rules);
  try_dump_lexemes ();
  try_dump_checker ();
}
