/*
   File: contfree.c
   Defines the context free analysis

   This module is responsible for parsing the grammars

   Copyright (C) 2008-2011 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: contfree.c,v 1.7 2012/02/01 12:29:06 marcs Exp $"
*/

/* global includes */
#include <stdio.h>   
#include <stdlib.h>
#include <string.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>
#include <dcg_string.h>   
#include <dcg_plist.h>
#include <dcg_plist_ops.h>
               
/* local includes */ 
#include "eag_ds.h"
#include "options.h"
#include "lexer.h"
#include "parser.h"
#include "grammars.h"
#include "contfree.h"

/*
   Locate the source file;
   If found, initialize the lexer
*/
static int try_locate (char *fname, int predef_flag)
{ FILE *fd = fopen (fname, "r");
  if (fd == NULL) return (0);
  init_lexer (fd, fname, predef_flag);
  return (1);
};

static int locate_and_init_grammar (string dir, string sname)
{ char buf[MAXPATHLEN+1];
  int predef_flag;
  if (streq (dir, ".")) sprintf (buf, "%s", sname);
  else sprintf (buf, "%s%c%s", dir, DIR_SEP, sname);
  if (try_locate (buf, 0)) return (1);

  predef_flag = streq (sname, PREDEF_NAME);
  if (streq (dir, ".")) sprintf (buf, "%s.eag3", sname);
  else sprintf (buf, "%s%c%s.eag3", dir, DIR_SEP, sname);
  if (try_locate (buf, predef_flag)) return (1);

  if (streq (dir, ".")) sprintf (buf, "%s.eag", sname);
  else sprintf (buf, "%s%c%s.eag", dir, DIR_SEP, sname);
  if (try_locate (buf, 0)) return (1);

  if (streq (dir, ".")) sprintf (buf, "%s.gra", sname);
  else sprintf (buf, "%s%c%s.gra", dir, DIR_SEP, sname);
  return (try_locate (buf, 0));
};

static void parse_main_grammar (grammar *gra)
{ dcg_hint ("      parsing grammar %s", base_gname);
  if (!locate_and_init_grammar (dir_name, source_name))
    dcg_panic ("could not open source file '%s'", source_name);
  should_be_grammar (base_gname, 1, gra);
  finish_lexer ();
}

static void parse_grammar (string gname, grammar *gra)
{ int ix;
  dcg_hint ("      parsing grammar %s", gname);
  for (ix = 0; ix < grammar_search_path -> size; ix++)
    if (locate_and_init_grammar (grammar_search_path -> array[ix], gname))
      { should_be_grammar (gname, 0, gra);
        finish_lexer ();
	return;
      };
  dcg_panic ("could not locate grammar '%s'", gname);
};

static void add_uses_to_grammar_names (string_list uses, string_list gnames)
{ int ix;
  for (ix = 0; ix < uses -> size; ix++)
    { string gnm = uses -> array[ix];
      add_uniquely_to_string_list (gnames, gnm);
    };
};

/*
   The actual context free analysis
*/
void context_free_analysis ()
{ string_list gnames = new_string_list ();
  string predef_name = new_string (PREDEF_NAME);
  grammar gra;
  int ix = 1;

  /* parse the main module */
  dcg_warning (0, "   parsing grammars...");
  parse_main_grammar (&gra);
  if (parse_only && dump_parser)
    { dcg_wlog ("   Dump of grammar parse tree:");
      pp_grammar (dcg_error_file (), gra);
      dcg_wlog ("");
    };
  dcg_panic_if_errors ();

  /* if we're only parsing, we stop now */
  if (parse_only) exit (0);

  /*
     We have succesfully parsed the first source grammar
     Register our name and add the predefines to our uses
  */
  register_grammar (gra);
  app_string_list (gnames, gra -> gname); 
  add_uniquely_to_string_list (gra -> uses, predef_name);
  add_uses_to_grammar_names (gra -> uses, gnames);

  /*
     Read the grammars on the grammar name list as well as the closure of it:
     we need all the grammars it may reference to setup the typing system
  */
  while (ix < gnames -> size)
    { string gname = gnames -> array[ix];
      parse_grammar (gname, &gra);
      register_grammar (gra);
      dcg_panic_if_errors ();

      /* Add predef to the uses list unless we are predef itself */
      if (!streq (gname, predef_name))
        add_uniquely_to_string_list (gra -> uses, predef_name);
      add_uses_to_grammar_names (gra -> uses, gnames);
      ix++;
    };

  if (dump_parser)
    dump_grammars ();
};
