/*
   File: bli_writer.c
   Writes the binary lexicon generator interface file

   Copyright (C) 2012 Marc Seutter

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.

   CVS ID: "$Id: bli_writer.c,v 1.8 2012/07/28 20:02:24 marcs Exp $"
*/

/* standard includes */
#include <stdio.h>

/* libdcg includes */
#include <dcg.h>
#include <dcg_error.h>

/* libeagbase includes */
#include <ebase_version.h>

/* Local includes */
#include "eag_ds.h"
#include "options.h"
#include "globals.h"
#include "bli_writer.h"

/*
   The lexicon interface file is a binary encoded file containing 
   all necessary information for eag3-lexgen to generate the runtime
   information file. It will have the following parts.

   1) A list of lexicon names
   2) A list of fact table names
   3) A list of triples file names
   4) All affix super domains
   5) The type table (in full but types are marked with their usability)
   6) Those pragmat settings that influence the lexicalization process
   7) The number of facts
   8) The fact/lexicon nonterminal table (defining all nonterminals with their typing)
   9) The terminal table (defining all grammar terminals)
   10) The character set table (defining all characters sets)
   11) The regexp nfa table (defining all nfas for the regexps)
*/
static BinFile open_lexgen_interface_file ()
{ char *fname = dcg_new_fmtd_string ("%s.%s", base_gname, BLI_SUFFIX);
  BinFile bf = dcg_bin_fopen_path (dir_name, fname, "w", 1);
  dcg_detach ((void **) &fname);
  return (bf);
}

static void save_pragmats (BinFile bf)
{ save_int (bf, hyphen_convention_option);
  save_int (bf, hybrid_parsing_option);
  save_int (bf, utf8_processing);
  save_int (bf, translate_option);
  save_int (bf, empty_white_space_option);
  save_string (bf, translate_fname);
  save_int_list (bf, white_spaces);
  save_int_list (bf, separators);
}

void write_lexgen_interface_file ()
{ BinFile bf = open_lexgen_interface_file ();
  dcg_bin_save_header (bf, "EAG3", "Lexicon Interface", EAG3_VERSION);
  save_string_list (bf, all_lexica);
  save_string_list (bf, all_fact_tables);
  save_string_list (bf, all_triple_databases);
  save_rt_domain_list (bf, all_rt_domains);
  save_rt_type_list (bf, all_rt_types);
  save_pragmats (bf);
  save_int (bf, nr_facts);
  save_lex_nont_list (bf, all_lex_nonts);
  save_terminal_list (bf, all_terminals); 
  save_cset_list (bf, all_csets);
  save_nfa_list (bf, all_regexp_nfas);
  dcg_bin_save_eof (bf);
  dcg_bin_fclose (&bf);
}
