from __future__ import print_function

from . import absetup
import os
import math
import numpy
import scipy.interpolate as interpolate
import matplotlib.pyplot as pyplot
import operator
import subprocess
import stat
import logging


# Parameters for ATLAS models.
# ATLAS code must be properly dimensioned for NDEPTH, otherwise crash ensues

LOGTAU0 = -6.875
DLOGTAU =  0.125
NDEPTH  =  72

AFE     = 'A'          # Use alpha-enhanced ('A') or solar-scaled ('S') initial models
A9_NREP = 1            # Number of times the 15 ATLAS9 iterations are repeated
A9_INIT_EXT  = None    # Filename extension for ATLAS9 initial models
A9_FROMSCRATCH = False # Calculate ATLAS9 models from scratch?
A12_NREP = 2           # Number of times to repeat 15 extra ATLAS12 iterations
A12_INIT_EXT = None    # Filename extension for ATLAS12 initial models


# Various constants

mvsun =   4.82
bcsun =  -0.194
teffsun = 5778.
loggsun = 4.44


# Metallicities and filename prefixes for ATLAS9 color tables
# The prefixes in 'metstr' must match the filenames present in
# the directory 'colpath'

metcol = [-2.5, -2.0, -1.5, -1.0, -0.5, 0.0, 0.2, 0.5]
metstr = ['m25','m20','m15','m10','m05','p00','p02','p05']   
# colpath = os.getenv('HOME')+'/cats/Kurucz/colors/'
colpath = absetup.catpref+'/cats/Kurucz/colors/'

# Paths and filenames for model atmosphere calculations

# molfile = os.getenv('HOME')+'/cats/Castelli2015/molecules.dat'
molfile = absetup.catpref+'/cats/Castelli2015/molecules.dat'

# The lists of pre-computed ODFs and models below are hardwired for now 
# and these models must be present in the ${HOME}/cats/Castelli
# directory. 

# odfdir = os.getenv('HOME')+'/cats/Castelli2015/odfsl/'
odfdir = absetup.catpref+'/cats/Castelli2015/odfsl/'
# odfdir = os.getenv('HOME')+'/cats/Castelli2015/odfnew/'
# odfdir = os.getenv('HOME')+'/cats/Castelli/odfnew/'

# Solar-scaled ODFs:

# modfs = [ -5.0, -4.0,  -3.5,  -3.0,  -2.5,  -2.0,  -1.5,  -1.0,  -0.5,  -0.4,  -0.3,  -0.2,  -0.1,   0.0,   0.2,   0.3,   0.4,   0.5,   1.0]
# podfs = ['m50', 'm40', 'm35', 'm30', 'm25', 'm20', 'm15', 'm10', 'm05', 'm04', 'm03', 'm02', 'm01', 'p00', 'p02', 'p03', 'p04', 'p05', 'p10']
modfs = [-3.50, -3.25, -3.00, -2.75, -2.50, -2.25, -2.00, -1.75, -1.50,
         -1.25, -1.00, -0.75, -0.50, -0.25,  0.00,  0.25,  0.50]
podfs = ['m350','m325','m300','m275','m250','m225','m200','m175','m150',
         'm125','m100','m075','m050','m025','p000','p025','p050']

# Alpha-enhanced ODFs:
# modfa = [-5.5, -4.5, -4.0, -3.5, -3.0, -2.5, -2.0, -1.5, -1.0, -0.5, 0.0, 0.2, 0.5]
# podfa = ['m55a', 'm45a', 'm40a', 'm35a', 'm30a', 'm25a', 'm20a', 'm15a', 'm10a', 'm05a', 'p00a', 'p02a', 'p05a']
modfa = [-3.50, -3.25, -3.00, -2.75, -2.50, -2.25, -2.00, -1.75, -1.50,
         -1.25, -1.00, -0.75, -0.50, -0.25,  0.00,  0.25,  0.50]
podfa = ['m350a','m325a','m300a','m275a','m250a','m225a','m200a','m175a','m150a',
         'm125a','m100a','m075a','m050a','m025a','p000a','p025a','p050a']

matm = [-2.5, -2.0, -1.5, -1.0, -0.5, 0.0, 0.2, 0.5]
patm = { 'S': ['am25' ,'am20' ,'am15' ,'am10' ,'am05' ,'ap00' ,'ap02' ,'ap05' ], 
         'A': ['am25a', 'am20a', 'am15a', 'am10a','am05a','ap00a','ap02a','ap05a'] }
# atmdir = os.getenv('HOME')+'/cats/Castelli2015/atlas9/'
atmdir = absetup.catpref+'/cats/Castelli2015/atlas9/'
# patm = ['am25','am20','am10','am05','ap00','ap02','ap05']
# atmdir = os.getenv('HOME')+'/cats/Castelli/atlas9/'



def interp_col(colx, coly, uselogg, m, xval, fpref='coubes', fsuff='k2odfnew.dat'):

# colx and coly can each be a single column or a list of two columns.
# In the latter case, the difference between the two columns is calculated and used.

#  Find best matching metallicity

    m1a = metcol[0]
    for mi in metcol:
        if abs(m - mi) < abs(m - m1a):
            m1a = mi
            
    mstr1a = metstr[metcol.index(m1a)]

#    print 'Best matching [m/H] for VI-Teff = '+str(m1a)+', '+p1a
 
    logg = []
    xlst = [[]]
    ylst = [[]]
    i    = 0
    
    with open(colpath+fpref+mstr1a+fsuff,'r') as f:
        for s in f:
            if (i > 1):
                ss = s.split()
                if isinstance(colx,list):
                    x = float(ss[colx[0]]) - float(ss[colx[1]])
                else:    
                    x = float(ss[colx])
                    
                if isinstance(coly,list):
                    y = float(ss[coly[0]]) - float(ss[coly[1]])
                else:    
                    y = float(ss[coly])
         
                xlogg = float(ss[2])

                if (logg.count(xlogg) == 0):
                    logg = logg + [xlogg]
                    xlst = xlst + [[x]]
                    ylst = ylst + [[y]]
                else:
                    j = logg.index(xlogg)
                    xlst[j] = xlst[j] + [x]
                    ylst[j] = ylst[j] + [y]            
                
            i = i + 1
      
# Find x-y relation for best matching log g

    jbest = -1
    loggb = -99.99
    for j in range(len(logg)):
        if (max(xlst[j]) > xval) and (min(xlst[j]) < xval) and (abs(logg[j] - uselogg) < abs(loggb - uselogg)):
            loggb = logg[j]
            jbest = j
            
    if (jbest == -1):
        print("ERROR: Requested value outside interpolation range")
        return float('NaN')
        
# Make sure the "x"-array is sorted in ascending order

    xy = []
    for i in range(len(xlst[jbest])):
        xy = xy + [[xlst[jbest][i], ylst[jbest][i]]]
    
    xy = sorted(xy, key=operator.itemgetter(0))
#    print xy    
    xbest, ybest = [], []
    for jxy in xy:
        xbest.append(jxy[0])
        ybest.append(jxy[1])

    fy = interpolate.interp1d(xbest, ybest)
    yval = fy(xval)

#    print 'Input     : ', obsvi, obslogg
#    print 'Best match: ', loggb
    
    return float(yval)


def vi2teff(vi, logg, mh):
# Wrapper to return Teff given Johnson-Cousins V-I, Logg and [m/H] as input
    teff = interp_col(11, 1, logg, mh, vi, fpref='coubes',fsuff='k2odfnew.dat')
    return teff

def acs_f606f814_2teff(vi, logg, mh):
# As vi2teff, but for ACS F606W-F814W VEGAMAG
    teff = interp_col([16,13], 1, logg, mh, vi, fpref='acs',fsuff='k2.dat')
    return teff


def teff2bc(teff, logg, mh):
    bc = interp_col(1, 11, logg, mh, teff, fpref='ubv',fsuff='k2odfnew.dat')
    return bc
    
    
def obs2phys(vi, v, mh, mass):
    
#   We use the "Kurucz" BC (1979, ApJ 40, 1). This is slightly non-standard, but doesn't 
#   really matter since all scalings are relative anyway.
    mbolsun = mvsun + bcsun
    
    logg = 3.0
    
# Iterate a few times -  teff and bc depend (weakly) on log g, but log g is only known 
# at the end.

    for i in range(3):
        teff = vi2teff(vi, logg, mh)
        bc = teff2bc(teff, logg, mh)
        mbol = v + bc
        lbol = 10**(-0.4*(mbol - mbolsun))
        rstar = (lbol**0.5) * (teffsun/teff)**2.
        gstar = mass * rstar**(-2.0)
        logg  = loggsun + math.log10(gstar)
#        print teff, logg, rstar

#  Return the Effective temperature, Surface Gravity and Radius of the star
#  The two former are needed to select the appropriate atmosphere, the latter to
#  scale the flux.
   
#    rr = [teff, logg, astar]
    rr = [teff, logg, rstar]

    return rr
    

def acs_obs2phys(vi, m606, mh, mass):

# Same as obs2phys, but using ACS F606W and F814W colours as input
# (here vi = m(f606w) - m(f814w))
    
#   We use the "Kurucz" BC (1979, ApJ 40, 1). This is slightly non-standard, but doesn't 
#   really matter since all scalings are relative anyway.
    mbolsun = mvsun + bcsun
    
    logg = 3.0
    
# Iterate a few times -  teff and bc depend (weakly) on log g, but log g is only known 
# at the end.

    for i in range(3):
        teff = acs_f606f814_2teff(vi, logg, mh)
        f606w2v = interp_col([16,13], 13, logg, mh, vi, fpref='acs',fsuff='k2.dat')
        v = m606 + f606w2v
        bc = teff2bc(teff, logg, mh)
        mbol = v + bc
        lbol = 10**(-0.4*(mbol - mbolsun))
        rstar = (lbol**0.5) * (teffsun/teff)**2.
        gstar = mass * rstar**(-2.0)
        logg  = loggsun + math.log10(gstar)
#        print teff, logg, rstar
#        print vi, m606, v

#  Return the Effective temperature, Surface Gravity and Radius of the star
#  The two former are needed to select the appropriate atmosphere, the latter to
#  scale the flux.
   
#    rr = [teff, logg, astar]
    rr = [teff, logg, rstar]

    return rr



def tlogglst(fname):
    f = open(fname,'r')
    s = ' '
    teff, logg = [], []
    while len(s) > 0:
        s = f.readline()
        if (s.find('TEFF') > -1):
            ss = s.split()
            rteff, rlogg = float(ss[1]), float(ss[3])
            teff = teff + [rteff]
            logg = logg + [rlogg]
    
    return [teff, logg]
    
    
def fhcorr(m, atoms=[], abun=[]):

# Calculate the hydrogen abundance for given composition
# and correction factor for abundance scale (i.e. [X/tot] for given [X/H])

    fhe0 = 0.07834     # Corresponds to Y=0.248; Grevesse & Sauval 1998
    ab0 = numpy.array([ab[2] for ab in absetup.stdabun])
    fz0 = sum(10**ab0)
    fh0 = 1 - fhe0 - fz0

    fhe = fhe0      
    if ('He' in atoms): fhe = abun[atoms.index('He')]
    
    rh = 1.

    for i in range(3):
        fz = 0.

        for aa in absetup.stdabun:
            ida = 0.
            selem = aa[0]
            ielem = aa[1]

            if (ielem in atoms):
                ida = abun[atoms.index(ielem)]

            if (selem in atoms):
                ida = abun[atoms.index(selem)]

            iabun = aa[2] + ida
            fz = fz + 10**(iabun+m)

        fh = 1 - fhe - rh*fz
        rh = fh/fh0

    return fh, math.log10(rh)    


def wrabun(file, m, atoms=[], abun=[]):
#    fh = 0.92156
#    fhe = 0.078438
    fhe = 0.07834      # Corresponds to Y=0.248; Grevesse & Sauval 1998
    if ('He' in atoms): fhe = abun[atoms.index('He')]

# First find the H fraction

    fh, dm = fhcorr(m, atoms, abun)

#    fz = 0
#    for aa in stdabun:
#        ida = 0.
#        selem = aa[0]
#        ielem = aa[1]
#        
#        if (ielem in atoms):
#            ida = abun[atoms.index(ielem)]
#                   
#        if (selem in atoms):
#            ida = abun[atoms.index(selem)]
# 
#        iabun = aa[2] + ida       
#        fz = fz + 10**(iabun+m)
#
#    fh = 1 - fhe - fz

# If the user insists:        

    if ('H' in atoms):  fh = abun[atoms.index('H')]

# Then write to file
    
    file.write('ABUNDANCE SCALE %9.5f ABUNDANCE CHANGE 1 %7.5f 2 %8.6f' % (10**(m+dm), fh, fhe))
    j = 0
    for aa in absetup.stdabun:
        if (j % 6 == 0):
            file.write('\n ABUNDANCE CHANGE')
        ida = 0.
        j += 1

        selem = aa[0]
        ielem = aa[1]
        
        if (ielem in atoms):
            ida = abun[atoms.index(ielem)]
                   
        if (selem in atoms):
            ida = abun[atoms.index(selem)]
        
        iabun = aa[2] + ida
        file.write('%3d %6.2f' % (ielem, iabun))

    file.write('\n')
    

def odfintp(odfdir,p1,p2,w1,w2,k):

    print('Interpolating in ODF files:')
    print('  '+odfdir+p1+'big'+k+'.bdf')
    print('  '+odfdir+p2+'big'+k+'.bdf')
    print('  Weights: %0.5f %0.5f' % (w1, w2))

    logging.info('Interpolating in ODF files:')
    logging.info('  %s' % (odfdir+p1+'big'+k+'.bdf'))
    logging.info('  %s' % (odfdir+p2+'big'+k+'.bdf'))
    logging.info('  Weights: %0.5f %0.5f' % (w1, w2))
    
    comf = open('dfinterpbig.com','w')
    comf.write('ln -fs '+odfdir+p1+'big'+k+'.bdf fort.1\n')
    comf.write('ln -fs '+odfdir+p2+'big'+k+'.bdf fort.2\n')
    comf.write(absetup.binpath+'dfinterpbig.exe<<EOF>dfinterpbig.out\n')
    comf.write('{0:5f} {1:5f}\n'.format(w1,w2))
    comf.write('EOF\n')
    comf.write('mv -f fort.3 odfbig'+k+'.bdf\n')
    comf.write('\\rm -f fort.*\n')
    comf.close()
    
    os.system("bash -c 'source dfinterpbig.com'")
        
        
def rossmean(m,k,atoms=[],abun=[]):
    
    comf = open('kappa.com','w')
    comf.write('ln -fs '+molfile+' fort.2\n')
    comf.write('ln -fs odfbig'+k+'.bdf fort.9\n')
    comf.write(absetup.binpath+'kappa9.exe<<EOF>kappa.out\n')
    comf.write('MOLECULES ON\n')
    comf.write('READ MOLECULES\n')
    comf.write('FREQUENCIES 337 1 337 BIG\n')
    comf.write('ITERATIONS 1 PRINT 1 PUNCH 0\n')
    comf.write('TITLE ROSSELAND OPACITY FOR VTURB '+k+' KM/S\n')
    comf.write(' OPACITY IFOP 1 1 1 1 1 1 1 1 1 1 1 1 1 0 1 0 0 0 0 0\n')
    comf.write(' CONVECTION ON   1.25 TURBULENCE OFF  0.00  0.00  0.00  0.00\n')
    comf.write('TEFF   5777.  GRAVITY 4.43770 LTE\n')
    wrabun(comf,m,atoms,abun)
#    comf.write('ABUNDANCE SCALE {0:f} ABUNDANCE CHANGE 1 0.92156 2 0.078438\n'.format(10**m))
#    comf.write(' ABUNDANCE CHANGE  3 -10.94  4 -10.64  5  -9.49  6  -3.52  7  -4.12  8  -3.21\n')
#    comf.write(' ABUNDANCE CHANGE  9  -7.48 10  -3.96 11  -5.71 12  -4.46 13  -5.57 14  -4.49\n')
#    comf.write(' ABUNDANCE CHANGE 15  -6.59 16  -4.71 17  -6.54 18  -5.64 19  -6.92 20  -5.68\n')
#    comf.write(' ABUNDANCE CHANGE 21  -8.87 22  -7.02 23  -8.04 24  -6.37 25  -6.65 26  -4.54\n')
#    comf.write(' ABUNDANCE CHANGE 27  -7.12 28  -5.79 29  -7.83 30  -7.44 31  -9.16 32  -8.63\n')
#    comf.write(' ABUNDANCE CHANGE 33  -9.67 34  -8.63 35  -9.41 36  -8.73 37  -9.44 38  -9.07\n')
#    comf.write(' ABUNDANCE CHANGE 39  -9.80 40  -9.44 41 -10.62 42 -10.12 43 -20.00 44 -10.20\n')
#    comf.write(' ABUNDANCE CHANGE 45 -10.92 46 -10.35 47 -11.10 48 -10.27 49 -10.38 50 -10.04\n')
#    comf.write(' ABUNDANCE CHANGE 51 -11.04 52  -9.80 53 -10.53 54  -9.87 55 -10.91 56  -9.91\n')
#    comf.write(' ABUNDANCE CHANGE 57 -10.87 58 -10.46 59 -11.33 60 -10.54 61 -20.00 62 -11.03\n')
#    comf.write(' ABUNDANCE CHANGE 63 -11.53 64 -10.92 65 -11.69 66 -10.90 67 -11.78 68 -11.11\n')
#    comf.write(' ABUNDANCE CHANGE 69 -12.04 70 -10.96 71 -11.98 72 -11.16 73 -12.17 74 -10.93\n')
#    comf.write(' ABUNDANCE CHANGE 75 -11.76 76 -10.59 77 -10.69 78 -10.24 79 -11.03 80 -10.91\n')
#    comf.write(' ABUNDANCE CHANGE 81 -11.14 82 -10.09 83 -11.33 84 -20.00 85 -20.00 86 -20.00\n')
#    comf.write(' ABUNDANCE CHANGE 87 -20.00 88 -20.00 89 -20.00 90 -11.95 91 -20.00 92 -12.54\n')
#    comf.write(' ABUNDANCE CHANGE 93 -20.00 94 -20.00 95 -20.00 96 -20.00 97 -20.00 98 -20.00\n')
#    comf.write(' ABUNDANCE CHANGE 99 -20.00\n')
    comf.write('READ DECK6 72 RHOX,T,P,XNE,ABROSS,ACCRAD,VTURB\n')
    comf.write(' 5.12838287E-04   3709.1 1.405E+01 2.797E+09 2.600E-04 7.028E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 6.71148215E-04   3732.6 1.839E+01 3.602E+09 3.019E-04 7.387E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 8.53583147E-04   3754.9 2.339E+01 4.517E+09 3.483E-04 7.641E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.06469772E-03   3778.6 2.917E+01 5.568E+09 4.009E-04 7.802E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.30968473E-03   3803.4 3.588E+01 6.782E+09 4.596E-04 7.825E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.59577136E-03   3828.9 4.372E+01 8.192E+09 5.233E-04 7.674E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.93076383E-03   3853.7 5.290E+01 9.828E+09 5.962E-04 7.544E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.32272392E-03   3877.9 6.364E+01 1.173E+10 6.797E-04 7.446E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.78100631E-03   3901.7 7.619E+01 1.392E+10 7.755E-04 7.381E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.31641180E-03   3924.9 9.086E+01 1.647E+10 8.855E-04 7.356E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.94186646E-03   3947.9 1.080E+02 1.941E+10 1.010E-03 7.334E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 4.67302051E-03   3970.8 1.280E+02 2.283E+10 1.152E-03 7.318E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 5.52856151E-03   3993.7 1.515E+02 2.679E+10 1.312E-03 7.309E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 6.53054813E-03   4016.5 1.789E+02 3.139E+10 1.494E-03 7.303E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 7.70352445E-03   4039.3 2.111E+02 3.673E+10 1.703E-03 7.337E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 9.07514808E-03   4062.0 2.486E+02 4.292E+10 1.943E-03 7.406E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.06774833E-02   4085.1 2.925E+02 5.010E+10 2.219E-03 7.515E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.25488575E-02   4108.9 3.438E+02 5.843E+10 2.533E-03 7.610E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.47351857E-02   4133.5 4.037E+02 6.811E+10 2.891E-03 7.695E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.72881496E-02   4158.4 4.736E+02 7.934E+10 3.303E-03 7.816E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.02692929E-02   4184.1 5.553E+02 9.239E+10 3.770E-03 7.923E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.37530891E-02   4210.6 6.508E+02 1.076E+11 4.302E-03 8.011E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.78229282E-02   4237.5 7.623E+02 1.252E+11 4.913E-03 8.119E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.25736063E-02   4264.9 8.924E+02 1.456E+11 5.614E-03 8.251E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.81169004E-02   4292.8 1.044E+03 1.694E+11 6.416E-03 8.394E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 4.45872937E-02   4321.3 1.222E+03 1.970E+11 7.328E-03 8.527E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 5.21389910E-02   4349.7 1.428E+03 2.289E+11 8.376E-03 8.716E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 6.09518129E-02   4378.4 1.670E+03 2.660E+11 9.568E-03 8.930E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 7.12439100E-02   4407.3 1.952E+03 3.090E+11 1.092E-02 9.155E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 8.32644240E-02   4436.2 2.281E+03 3.588E+11 1.247E-02 9.432E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 9.72971259E-02   4465.0 2.666E+03 4.165E+11 1.425E-02 9.770E-02 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.13672868E-01   4493.8 3.114E+03 4.834E+11 1.629E-02 1.018E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.32786810E-01   4523.1 3.638E+03 5.610E+11 1.860E-02 1.062E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.55098562E-01   4552.3 4.249E+03 6.509E+11 2.126E-02 1.115E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.81139153E-01   4581.5 4.963E+03 7.549E+11 2.428E-02 1.176E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.11544893E-01   4611.0 5.796E+03 8.755E+11 2.773E-02 1.244E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.47052781E-01   4640.6 6.768E+03 1.015E+12 3.167E-02 1.323E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.88507698E-01   4670.5 7.904E+03 1.177E+12 3.617E-02 1.414E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.36895159E-01   4701.0 9.230E+03 1.365E+12 4.133E-02 1.520E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 3.93366856E-01   4732.7 1.078E+04 1.584E+12 4.723E-02 1.642E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 4.59259744E-01   4765.8 1.258E+04 1.839E+12 5.398E-02 1.782E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 5.36129287E-01   4800.8 1.469E+04 2.136E+12 6.171E-02 1.943E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 6.25814366E-01   4838.3 1.715E+04 2.484E+12 7.053E-02 2.122E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 7.30438050E-01   4879.0 2.001E+04 2.893E+12 8.064E-02 2.328E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 8.52425560E-01   4924.1 2.335E+04 3.376E+12 9.225E-02 2.564E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 9.94567773E-01   4974.7 2.725E+04 3.951E+12 1.056E-01 2.834E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.16008551E+00   5032.4 3.178E+04 4.641E+12 1.210E-01 3.143E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.35274770E+00   5099.6 3.706E+04 5.480E+12 1.387E-01 3.498E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.57666446E+00   5177.6 4.320E+04 6.515E+12 1.593E-01 3.908E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 1.83606723E+00   5269.2 5.030E+04 7.829E+12 1.838E-01 4.392E-01 1.500E+05 0.000E+00 0.000E+00\n')
    comf.write(' 2.13471080E+00   5375.8 5.848E+04 9.551E+12 2.139E-01 4.990E-01 1.500E+05 1.240E-09 0.000E+00\n')
    comf.write(' 2.47332025E+00   5504.6 6.776E+04 1.198E+13 2.545E-01 5.827E-01 1.500E+05 3.933E-08 0.000E+00\n')
    comf.write(' 2.84881471E+00   5650.2 7.805E+04 1.554E+13 3.098E-01 7.002E-01 1.500E+05 3.308E-06 0.000E+00\n')
    comf.write(' 3.24852750E+00   5843.7 8.900E+04 2.190E+13 4.021E-01 9.037E-01 1.500E+05 5.162E-04 0.000E+00\n')
    comf.write(' 3.63863046E+00   6100.3 9.968E+04 3.491E+13 5.768E-01 1.244E+00 1.500E+05 1.547E-02 1.626E+03\n')
    comf.write(' 3.99186702E+00   6359.7 1.094E+05 5.674E+13 8.514E-01 1.662E+00 1.500E+05 8.664E-02 2.998E+03\n')
    comf.write(' 4.32516876E+00   6560.2 1.185E+05 8.310E+13 1.163E+00 2.017E+00 1.500E+05 1.667E-01 5.056E+03\n')
    comf.write(' 4.64876290E+00   6778.0 1.274E+05 1.237E+14 1.621E+00 2.550E+00 1.500E+05 2.433E-01 2.240E+04\n')
    comf.write(' 4.95288884E+00   7030.4 1.357E+05 1.908E+14 2.340E+00 3.362E+00 1.500E+05 3.121E-01 5.586E+04\n')
    comf.write(' 5.23070832E+00   7308.0 1.433E+05 2.979E+14 3.453E+00 4.545E+00 1.500E+05 3.719E-01 8.952E+04\n')
    comf.write(' 5.47879852E+00   7619.0 1.501E+05 4.727E+14 5.229E+00 6.441E+00 1.500E+05 4.181E-01 1.403E+05\n')
    comf.write(' 5.68988391E+00   7981.7 1.559E+05 7.723E+14 8.308E+00 8.682E+00 1.500E+05 5.058E-01 2.085E+05\n')
    comf.write(' 5.87591173E+00   8275.3 1.610E+05 1.118E+15 1.195E+01 8.487E+00 1.500E+05 6.602E-01 2.211E+05\n')
    comf.write(' 6.05637485E+00   8508.5 1.659E+05 1.478E+15 1.586E+01 7.902E+00 1.500E+05 7.612E-01 2.254E+05\n')
    comf.write(' 6.23957016E+00   8729.1 1.709E+05 1.901E+15 2.059E+01 7.414E+00 1.500E+05 8.290E-01 2.261E+05\n')
    comf.write(' 6.43191469E+00   8918.9 1.762E+05 2.343E+15 2.570E+01 6.910E+00 1.500E+05 8.718E-01 2.240E+05\n')
    comf.write(' 6.63749632E+00   9111.8 1.818E+05 2.874E+15 3.200E+01 6.559E+00 1.500E+05 9.031E-01 2.216E+05\n')
    comf.write(' 6.86094505E+00   9283.7 1.880E+05 3.433E+15 3.879E+01 6.111E+00 1.500E+05 9.250E-01 2.180E+05\n')
    comf.write(' 7.10613604E+00   9466.6 1.947E+05 4.118E+15 4.724E+01 5.827E+00 1.500E+05 9.417E-01 2.148E+05\n')
    comf.write(' 7.37808777E+00   9631.9 2.021E+05 4.840E+15 5.619E+01 5.417E+00 1.500E+05 9.541E-01 2.106E+05\n')
    comf.write(' 7.68129689E+00   9813.4 2.104E+05 5.740E+15 6.752E+01 5.181E+00 1.500E+05 9.638E-01 2.072E+05\n')
    comf.write(' 8.02184660E+00   9977.4 2.197E+05 6.688E+15 7.936E+01 5.099E+00 1.500E+05 9.695E-01 2.027E+05\n')
    comf.write('PRADK 1.4828E+00\n')
    comf.write('VTURB {0:5.1E}\n'.format(float(k)))
    comf.write('BEGIN                    ITERATION  15 COMPLETED\n')
    comf.write('END\n')
    comf.write('EOF\n')
    comf.write('mv fort.7 kappak'+k+'.dat\n')
    comf.write('\\rm -f fort.*\n')
    
    comf.close()
    
    os.system("bash -c 'source kappa.com'")


def rosscmb():
    comf = open('kapreadts.com','w')
    comf.write('ln -fs kappak0.dat fort.11\n')
    comf.write('ln -fs kappak1.dat fort.12\n')
    comf.write('ln -fs kappak2.dat fort.13\n')
    comf.write('ln -fs kappak4.dat fort.14\n')
    comf.write('ln -fs kappak8.dat fort.15\n')
    comf.write(absetup.binpath+'kapreadts.exe\n')
    comf.write('mv -f fort.2 kappa.ros\n')
    comf.write('\\rm -f fort.*\n')
    comf.close()
    
    os.system("bash -c 'source kapreadts.com'")


def selatm(fname, teff, logg, mname):
    print('Searching in '+fname)
    f = open(fname,'r')

    mu = 1.0

    rteff = -99.99
    rlogg = -99.99
    s = ' '

    while ((rteff != teff) or (rlogg != logg)) and (len(s) > 0):
        s = f.readline()
        if (s.find('TEFF') > -1):
            ss = s.split()
            rteff, rlogg = float(ss[1]), float(ss[3])

    if (len(s) == 0):
        print('EOF reached')
    else:
        print('Found Teff = '+str(rteff)+' log g = '+str(rlogg))
        fm = open(mname,'w')
        while (ss[0] != 'BEGIN'):
            fm.write(s)
            s = f.readline()
            ss = s.split()
        
        fm.write(s)
        fm.close()
        
    f.close()
    
    
def mkodf(m,atoms=[],abun=[], odfs='S'):    

    if (odfs=='A'):
        print('Using Alpha-enhanced ODFs')
        logging.info('Using Alpha-enhanced ODFs')
        modf = modfa
        podf = podfa
    else:
        print('Using Solar-scaled ODFs')
        logging.info('Using Solar-scaled ODFs')
        modf = modfs
        podf = podfs

# Find the closest existing ODF files
    
    m1o = min(modf)
    m2o = max(modf)
    for mi in modf:
        if (mi > m1o) and (mi < m):
            m1o = mi
        if (mi < m2o) and (mi >= m):
            m2o = mi
    
    p1o = podf[modf.index(m1o)]
    p2o = podf[modf.index(m2o)]
    
# Interpolate in them to make an ODF for desired metallicity

    print('ATLAS9: Generating ODF files for [m/H] = '+str(m))
    logging.debug('ATLAS9: Generating ODF files for [m/H] = '+str(m))
    w1 = (m2o - m)/(m2o-m1o)
    w2 = (m - m1o)/(m2o-m1o)
    for k in ['0','1','2','4','8']:
        odfintp(odfdir,p1o,p2o,w1,w2,k)

# Make Rosseland opacity table

    print('ATLAS9: Computing Rosseland opacity tables')
    logging.debug('ATLAS9: Computing Rosseland opacity tables')
    for k in ['0','1','2','4','8']:
        rossmean(m,k,atoms=atoms,abun=abun)
    rosscmb()
    

def mkatm(teff, logg, m, atmname, workdir='.', wait=True, atoms=[], abun=[], nlte=False, vturb=0.0, firstguess=None):

# This routine produces an ATLAS9 model atmosphere for the desired
# Teff, logg and metallicity.
# vturb is the "turbulent" velocity (in km/s).

    fromscratch = A9_FROMSCRATCH

    if (fromscratch): 
        print('Computing new model from scratch')
        logging.debug('Computing new model from scratch')
    elif (firstguess is not None):
        # Firstguess -> atm_in.dat
        print('Using '+firstguess+' as starting point for iterations')
        logging.debug('Using '+firstguess+' as starting point for iterations')
        os.system('cp -f '+firstguess+' '+workdir+'/atm_in.dat')
    elif (A9_INIT_EXT is not None):
        # atmname+A9_INIT_EXT -> atm_in.dat
        print('Using '+atmname+A9_INIT_EXT+' as starting point for iterations')
        logging.debug('Using '+atmname+A9_INIT_EXT+' as starting point for iterations')
        os.system('cp -f '+atmname+A9_INIT_EXT+' '+workdir+'/atm_in.dat')
    else:        
        # Find the closest existing ATLAS atmosphere

        m1a = matm[0]
        for mi in matm:
            if abs(m - mi) < abs(m - m1a):
                m1a = mi
        print('Best matching existing ATLAS [m/H] = '+str(m1a))
        logging.debug('Best matching existing ATLAS [m/H] = %0.3f' % m1a)
    
        p1a = patm[AFE][matm.index(m1a)]
        
        tlogg = tlogglst(atmdir+p1a+'k2odfnew.dat')
    
        t1a = tlogg[0][0]
        for ti in tlogg[0]:
            if abs(teff - ti) < abs(teff - t1a):
                t1a = ti
  
        print('Best matching existing ATLAS Teff  = '+str(t1a))
        logging.debug('Best matching existing ATLAS Teff  = %0.1f' % t1a)
    
        logg1a = tlogg[1][tlogg[0].index(t1a)]
        for i in range(len(tlogg[1])):
            if tlogg[0][i] == t1a:
                if (abs(logg - tlogg[1][i]) < abs(logg - logg1a)):
                    logg1a = tlogg[1][i]
        print('Best matching existing ATLAS log g = '+str(logg1a))
        logging.debug('Best matching existing ATLAS log g = %0.3f' % logg1a)
    
        selatm(atmdir+p1a+'k2odfnew.dat', t1a, logg1a, workdir+'/atm_in.dat')

# Run ATLAS9

    if (nlte):
        snlte = 'NLTE'
    else:
        snlte = 'LTE'

    print('ATLAS9: Computing model atmosphere for Teff='+str(teff)+', logg='+str(logg)+', [m/H]='+str(m)+', '+snlte)
    logging.info('ATLAS9: Computing model atmosphere for Teff=%0.1f, Logg=%0.2f, [m/H]=%0.2f, %s' % (teff, logg, m, snlte))
    for at,ab in zip(atoms,abun):
        if (at in ['H','He']):
            print('  %-2s:  %0.3f' % (at,ab) )
            logging.info('  %-2s:  %0.3f' % (at,ab))
        else:
            print('  %-2s: %+0.3f' % (at,ab) )
            logging.info('  %-2s: %+0.3f' % (at,ab))

    cwd = os.getcwd()

    fc = open(workdir+'/atlas9.com','w')
#    fc.write('#!/bin/csh -f\n')
    fc.write('#!/bin/bash\n')
    fc.write('cd '+workdir+'\n')
    fc.write('ln -fs '+cwd+'/kappa.ros fort.1\n')
    
    if (vturb < 0.5):
        fc.write('ln -fs '+cwd+'/odfbig0.bdf fort.9\n')
        logging.info('  Using '+cwd+'/odfbig0.bdf')
    elif (vturb < 1.5):
        fc.write('ln -fs '+cwd+'/odfbig1.bdf fort.9\n')
        logging.info('  Using '+cwd+'/odfbig1.bdf')
    elif (vturb < 3.0):
        fc.write('ln -fs '+cwd+'/odfbig2.bdf fort.9\n')
        logging.info('  Using '+cwd+'/odfbig2.bdf')
    elif (vturb < 6.0):
        fc.write('ln -fs '+cwd+'/odfbig4.bdf fort.9\n')
        logging.info('  Using '+cwd+'/odfbig4.bdf')
    else:
        fc.write('ln -fs '+cwd+'/odfbig8.bdf fort.9\n')
        logging.info('  Using '+cwd+'/odfbig8.bdf')
        
    fc.write('ln -fs '+molfile+' fort.2\n')
    if (not fromscratch): fc.write('ln -fs atm_in.dat fort.3\n')
    fc.write(absetup.binpath+'atlas9mem.exe<<EOF>atlas9.out\n')
    if (fromscratch): fc.write('TEFF %0.1f GRAVITY %0.2f\n' % (teff, logg))
    fc.write('READ KAPPA\n')
    if (not fromscratch): fc.write('READ PUNCH\n')
    fc.write('MOLECULES ON\n')
    fc.write('READ MOLECULES\n')
    fc.write('FREQUENCIES 337 1 337 BIG\n')
#    fc.write('VTURB 2.0E+5\n')
    fc.write('VTURB %0.2E\n' % (vturb*1e5))
    fc.write('CONVECTION OVER 1.25 0 36\n')
    if (nlte):
        fc.write('NLTE\n')
    fc.write('TITLE  [%0.1f] VTURB=%0.1f L/H=1.25 NOVER NEW ODF\n' % (m, vturb))
#    fc.write('SCALE 72 -6.875 0.125 %0.1f %4.2f\n' % (teff,logg))
    if (fromscratch):
        fc.write('CALCULATE %d %0.3f %0.3f\n' % (NDEPTH, LOGTAU0, DLOGTAU))
    else:
        fc.write('SCALE %d %0.3f %0.3f %0.1f %0.2f\n' % (NDEPTH,LOGTAU0,DLOGTAU,teff,logg))
    wrabun(fc, m, atoms, abun)
    
#    fc.write('ABUNDANCE SCALE {0:f} ABUNDANCE CHANGE 1 0.92156 2 0.078438\n'.format(10**m))
#    fc.write(' ABUNDANCE CHANGE  3 -10.94  4 -10.64  5  -9.49  6  -3.52  7  -4.12  8  -3.21\n')
#    fc.write(' ABUNDANCE CHANGE  9  -7.48 10  -3.96 11  -5.71 12  -4.46 13  -5.57 14  -4.49\n')
#    fc.write(' ABUNDANCE CHANGE 15  -6.59 16  -4.71 17  -6.54 18  -5.64 19  -6.92 20  -5.68\n')
#    fc.write(' ABUNDANCE CHANGE 21  -8.87 22  -7.02 23  -8.04 24  -6.37 25  -6.65 26  -4.54\n')
#    fc.write(' ABUNDANCE CHANGE 27  -7.12 28  -5.79 29  -7.83 30  -7.44 31  -9.16 32  -8.63\n')
#    fc.write(' ABUNDANCE CHANGE 33  -9.67 34  -8.63 35  -9.41 36  -8.73 37  -9.44 38  -9.07\n')
#    fc.write(' ABUNDANCE CHANGE 39  -9.80 40  -9.44 41 -10.62 42 -10.12 43 -20.00 44 -10.20\n')
#    fc.write(' ABUNDANCE CHANGE 45 -10.92 46 -10.35 47 -11.10 48 -10.27 49 -10.38 50 -10.04\n')
#    fc.write(' ABUNDANCE CHANGE 51 -11.04 52  -9.80 53 -10.53 54  -9.87 55 -10.91 56  -9.91\n')
#    fc.write(' ABUNDANCE CHANGE 57 -10.87 58 -10.46 59 -11.33 60 -10.54 61 -20.00 62 -11.03\n')
#    fc.write(' ABUNDANCE CHANGE 63 -11.53 64 -10.92 65 -11.69 66 -10.90 67 -11.78 68 -11.11\n')
#    fc.write(' ABUNDANCE CHANGE 69 -12.04 70 -10.96 71 -11.98 72 -11.16 73 -12.17 74 -10.93\n')
#    fc.write(' ABUNDANCE CHANGE 75 -11.76 76 -10.59 77 -10.69 78 -10.24 79 -11.03 80 -10.91\n')
#    fc.write(' ABUNDANCE CHANGE 81 -11.14 82 -10.09 83 -11.33 84 -20.00 85 -20.00 86 -20.00\n')
#    fc.write(' ABUNDANCE CHANGE 87 -20.00 88 -20.00 89 -20.00 90 -11.95 91 -20.00 92 -12.54\n')
#    fc.write(' ABUNDANCE CHANGE 93 -20.00 94 -20.00 95 -20.00 96 -20.00 97 -20.00 98 -20.00\n')
#    fc.write(' ABUNDANCE CHANGE 99 -20.00\n')
    fc.write('ITERATIONS 15 PRINT 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
    fc.write('PUNCH 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
    fc.write('BEGIN                    ITERATION  10 COMPLETED\n')
#    fc.write('SCALE 72 -6.875 0.125 {0:.1f} {1:3.1f}\n'.format(teff,logg))
    for j in range(A9_NREP):
#        fc.write('SCALE %d %0.3f %0.3f %0.1f %0.2f\n' % (NDEPTH,LOGTAU0,DLOGTAU,teff,logg))
        fc.write('ITERATIONS 15 PRINT 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
        fc.write('PUNCH 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
        fc.write('BEGIN                    ITERATION  10 COMPLETED\n')
    fc.write('END\n')
    fc.write('EOF\n')
#    fc.write('mv -f fort.7 '+cwd+'/'+atmname+'.dat\n')
    fc.write('mv -f fort.7 '+cwd+'/'+atmname+'.A9\n')
    fc.write('mv -f atlas9.out '+cwd+'/'+atmname+'.out\n')
    fc.write('\\rm -f fort.*\n')
    fc.close()
    
    os.chmod(workdir+'/atlas9.com',stat.S_IXUSR | stat.S_IRUSR | stat.S_IWUSR)
    
    p = subprocess.Popen(workdir+'/atlas9.com')
    
    if (wait):
        os.waitpid(p.pid,0)
    else:
        return p
#    os.system('source atlas9.com')

    print('Model atmosphere stored in '+atmname+'.A9')
    print('   ATLAS9 output stored in '+atmname+'.out')
    
    
def mkatm_a12(teff, logg, m, atmname, workdir='.', wait=True, atoms=[], abun=[], nlte=False, vturb=2.0, firstguess=None):

    if (firstguess is not None):
        # Firstguess -> atm_in.dat
        print('Using '+firstguess+' as starting point for iterations')
        logging.debug('Using '+firstguess+' as starting point for iterations')
        os.system('cp -f '+firstguess+' '+workdir+'/atm_in.dat')
    elif (A12_INIT_EXT is not None):
        # atmname+A12_INIT_EXT -> atm_in.dat
        print('Using '+atmname+A12_INIT_EXT+' as starting point for iterations')
        logging.debug('Using '+atmname+A12_INIT_EXT+' as starting point for iterations')
        os.system('cp -f '+atmname+A12_INIT_EXT+' '+workdir+'/atm_in.dat')
    else:        
        # First find the closest existing ATLAS atmosphere

        m1a = matm[0]
        for mi in matm:
            if abs(m - mi) < abs(m - m1a):
                m1a = mi
        print('Best matching existing ATLAS [m/H] = '+str(m1a))
        logging.debug('Best matching existing ATLAS [m/H] = %0.3f' % m1a)

        p1a = patm[AFE][matm.index(m1a)]

        tlogg = tlogglst(atmdir+p1a+'k2odfnew.dat')

        t1a = tlogg[0][0]
        for ti in tlogg[0]:
            if abs(teff - ti) < abs(teff - t1a):
                t1a = ti
        print('Best matching existing ATLAS Teff  = '+str(t1a))
        logging.debug('Best matching existing ATLAS Teff  = %0.1f' % t1a)

        logg1a = tlogg[1][tlogg[0].index(t1a)]
        for i in range(len(tlogg[1])):
            if tlogg[0][i] == t1a:
                if (abs(logg - tlogg[1][i]) < abs(logg - logg1a)):
                    logg1a = tlogg[1][i]
        print('Best matching existing ATLAS log g = '+str(logg1a))
        logging.debug('Best matching existing ATLAS log g = %0.3f' % logg1a)

        selatm(atmdir+p1a+'k2odfnew.dat', t1a, logg1a, workdir+'/atm_in.dat')

    if (nlte):
        snlte = 'NLTE'
    else:
        snlte = 'LTE'

    print('ATLAS12: Computing model atmosphere for Teff='+str(teff)+', logg='+str(logg)+', [m/H]='+str(m)+', '+snlte)
    logging.info('ATLAS12: Computing model atmosphere for Teff=%0.1f, Logg=%0.2f, [m/H]=%0.2f, %s' % (teff, logg, m, snlte))

    for at,ab in zip(atoms,abun):
        if (at in ['H','He']):
            print('  %-2s:  %0.3f' % (at,ab) )
            logging.info('  %-2s:  %0.3f' % (at,ab))
        else:
            print('  %-2s: %+0.3f' % (at,ab) )
            logging.info('  %-2s: %+0.3f' % (at,ab))


    cwd = os.getcwd()

    fc = open(workdir+'/atlas12.com','w')
#    fc.write('#!/bin/csh -f\n')
    fc.write('#!/bin/bash\n')
    fc.write('cd '+workdir+'\n')
#    fc.write('ln -fs ${HOME}/cats/Castelli2015/molecules.dat fort.2\n')
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/molecules.dat fort.2\n')
    fc.write('ln -fs atm_in.dat fort.3\n')
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/lowlines.bin fort.11\n')   
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/hilines.bin fort.21\n')   
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/diatomicsiwl.bin fort.31\n')    
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/schwenke.bin fort.41\n')
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/h2ofast.bin fort.51\n')
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/fclowlines.bin fort.11\n')   
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/fchilines.bin fort.21\n')   
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/diatomics.pck fort.31\n')    
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/tioschwenke.bin fort.41\n')
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/h2ofastfix.bin fort.51\n')

    fc.write(absetup.binpath+'atlas12_sl.exe<<EOF>'+atmname+'a.out\n')
    
    
    fc.write('MOLECULES ON\n')  
    fc.write('READ MOLECULES\n')  
    fc.write('READ PUNCH\n') 
    fc.write('READ LINES\n') 
    fc.write('CONVECTION OVER 1.25 0 36\n')
    if (nlte):
        fc.write('NLTE\n')
    fc.write('ITERATIONS 1 PRINT 1 PUNCH 0\n')

    wrabun(fc, m, atoms, abun)

    fc.write('BEGIN\n')  
    fc.write('END\n')  
    fc.write('EOF\n')  
    fc.write('mv -f fort.12 sellin.bin\n') 
    fc.write('\\rm -f fort.*\n')
    
    
    fc.write('ln -fs atm_in.dat fort.3\n')  
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/molecules.dat fort.2\n')
#    fc.write('ln -fs ${HOME}/cats/Castelli/linelists/nltelines.bin fort.19\n')
    fc.write('ln -fs '+absetup.catpref+'/cats/Castelli2015/linelists/nltelines.bin fort.19\n') 
    fc.write('ln -fs sellin.bin fort.12\n')
    fc.write(absetup.binpath+'atlas12_sl.exe<<EOF>atlas12b.out\n')  
    fc.write('MOLECULES ON\n')  
    fc.write('READ MOLECULES\n')  
    fc.write('READ PUNCH\n')  
    fc.write('TITLE ATLAS12\n') 
    fc.write('OPACITY ON LINES\n') 
    fc.write('OPACITY ON XLINES\n')  
    fc.write('CONVECTION OVER 1.25 0 36\n')
    if (nlte):
        fc.write('NLTE\n')
    fc.write('ITERATIONS 15\n')  
    fc.write('PRINT 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n') 
    fc.write('PUNCH 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
    fc.write('SCALE MODEL %d %0.3f %0.3f %0.1f %0.2f\n' % (NDEPTH,LOGTAU0,DLOGTAU,teff,logg))
    fc.write('VTURB %0.2E\n' % (vturb*1e5))
    
    wrabun(fc, m, atoms, abun)
    
    fc.write('BEGIN\n')
    
    for j in range(A12_NREP):
        fc.write('ITERATIONS 15\n')
        fc.write('PRINT 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
        fc.write('PUNCH 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1\n')
        fc.write('BEGIN\n')
                
    fc.write('END\n')
    fc.write('EOF\n')
#    fc.write('mv -f fort.7 '+atmname+'.dat\n')
#    fc.write('mv -f fort.7 '+cwd+'/'+atmname+'.dat\n')
    fc.write('mv -f fort.7 '+cwd+'/'+atmname+'.A12\n')
    fc.write('mv -f atlas12b.out '+cwd+'/'+atmname+'b.out\n')
    fc.write('\\rm -f fort.*\n')
    fc.close()
    
    os.chmod(workdir+'/atlas12.com',stat.S_IXUSR | stat.S_IRUSR | stat.S_IWUSR)

    p = subprocess.Popen(workdir+'/atlas12.com')
    
    if (wait):
        os.waitpid(p.pid,0)
    else:
        return p

    print('Model atmosphere stored in '+atmname+'.A12')
    print('  ATLAS12 output stored in '+atmname+'b.out')


def interp(fn1, fn2, fnout, tout, loggout):
    f1 = open(fn1,'r')
    f2 = open(fn2,'r')

    s = ' '
    s = f1.readline()
    s = f1.readline()
    ss = s.split()
    teff1, logg1 = float(ss[2]), float(ss[6])
    s = f1.readline()
    ss = s.split()
    ntau1 = int(ss[1])

    s = f2.readline()
    s = f2.readline()
    ss = s.split()
    teff2, logg2 = float(ss[2]), float(ss[6])
    s = f2.readline()
    ss = s.split()
    ntau2 = int(ss[1])

    print(fn1+': Teff = '+str(teff1)+' Logg = '+str(logg1)+' ntau = '+str(ntau1))
    print(fn2+': Teff = '+str(teff2)+' Logg = '+str(logg2)+' ntau = '+str(ntau2))
    print('Requested: Teff = '+str(tout)+' Logg = '+str(loggout))
    
    if (tout < teff1) or (tout > teff2) or (loggout < logg1) or (loggout > logg2):
        print('Error: Requested Teff and Log g not bracketed by input files')
        return

    rhox1, t1, p1, xne1, abross1, accrad1, vturb1 = [], [], [], [], [], [], []
    for i1 in range(ntau1):
        s = f1.readline()
        ss = s.split()
        rhox1   = rhox1   + [float(ss[0])]
        t1      = t1      + [float(ss[1])]
        p1      = p1      + [float(ss[2])]
        xne1    = xne1    + [float(ss[3])]
        abross1 = abross1 + [float(ss[4])]
        accrad1 = accrad1 + [float(ss[5])]
        vturb1  = vturb1  + [float(ss[6])]


    rhox2, t2, p2, xne2, abross2, accrad2, vturb2 = [], [], [], [], [], [], []
    for i2 in range(ntau2):
        s = f2.readline()
        ss = s.split()
        rhox2   = rhox2   + [float(ss[0])]
        t2      = t2      + [float(ss[1])]
        p2      = p2      + [float(ss[2])]
        xne2    = xne2    + [float(ss[3])]
        abross2 = abross2 + [float(ss[4])]
        accrad2 = accrad2 + [float(ss[5])]
        vturb2  = vturb2  + [float(ss[6])]
        
    f1.close()
    f2.close()


    frhox   = interpolate.interp1d(t2, rhox2, bounds_error=False)
    fp      = interpolate.interp1d(t2, p2, bounds_error=False)
    fxne    = interpolate.interp1d(t2, xne2, bounds_error=False)
    fabross = interpolate.interp1d(t2, abross2, bounds_error=False)
    faccrad = interpolate.interp1d(t2, accrad2, bounds_error=False)
    fvturb  = interpolate.interp1d(t2, vturb2, bounds_error=False)

    rhox2i   = frhox(t1)
    p2i      = fp(t1)
    xne2i    = fxne(t1)
    abross2i = fabross(t1)
    accrad2i = faccrad(t1)
    vturb2i  = fvturb(t1)

    rhoxi, pi, xnei, abrossi, accradi, vturbi = [], [], [], [], [], []
    if logg1 == logg2:
        w2 = (tout - teff1)/(teff2-teff1)
        w1 = (teff2 - tout)/(teff2-teff1)
        print('Interpolating in Teff')
    else:
        w2 = (loggout - logg1)/(logg2-logg1)
        w1 = (logg2 - loggout)/(logg2-logg1)
        print('Interpolating in logg')
    
    ntaui = 0
    
    for i in range(ntau1):
        rhoxi   = rhoxi   + [math.exp( w1*math.log(rhox1[i]) + w2*math.log(rhox2i[i]) )]
        pi      = pi      + [math.exp( w1*math.log(p1[i]) + w2*math.log(p2i[i]) )]
        xnei    = xnei    + [math.exp( w1*math.log(xne1[i]) + w2*math.log(xne2i[i]) )]
        abrossi = abrossi + [math.exp( w1*math.log(abross1[i]) + w2*math.log(abross2i[i]) )]
        accradi = accradi + [math.exp( w1*math.log(accrad1[i]) + w2*math.log(accrad2i[i]) )]
        vturbi  = vturbi  + [w1*vturb1[i] + w2*vturb2i[i]]
        if not math.isnan(rhoxi[i]):
            ntaui = ntaui+1

    pyplot.semilogx(rhox1, t1)
    pyplot.semilogx(rhox2, t2)
    pyplot.semilogx(rhoxi, t1)
    pyplot.xlabel('RhoX')
    pyplot.ylabel('Temp')
    
    fout = open(fnout,'w')
    fout.write('KURUCZ\n')
    fout.write('Teff = '+str(tout)+' K, Logg = '+str(logg1)+'\n')
    fout.write('NTAU'+str(ntaui).rjust(11)+'\n')
    
    for i in range(ntau1):
        if not math.isnan(rhoxi[i]):
            fout.write('{0:14.8E} {1:8.1f} {2:9.3E} {3:9.3E} '.format(rhoxi[i], t1[i], pi[i], xnei[i]))
            fout.write('{0:9.3E} {1:9.3E} {2:9.3E}\n'.format(abrossi[i], accradi[i], vturbi[i]))

    fout.close()
  
  

def getatm(fname, teff, logg, mname):
    f = open(fname,'r')

    mu = 1.0

    rteff = -99.99
    rlogg = -99.99
    s = ' '

    while ((rteff != teff) or (rlogg != logg)) and (len(s) > 0):
        s = f.readline()
        if (s.find('EFF') > -1):
            ss = s.split()
            rteff, rlogg = float(ss[1]), float(ss[3])

    if (len(s) == 0):
        print('EOF reached')
    else:
        fm = open(mname,'w')

    fm.write('KURUCZ\n')
    fm.write('Teff = '+str(teff)+' K, Logg = '+str(logg)+'\n')

    while (s.find('EAD') != 0):
        s = f.readline()
        
    ss = s.split()
    ntau = int(ss[2])
    fm.write('NTAU'+str(ntau).rjust(11)+'\n')

    itau = 1
    while (itau <= ntau):
        s = f.readline()
        fm.write(s)
        itau = itau+1
	  
    fm.close()
    print('Success: Model --> '+mname)

    f.close()
    
    
def tau500(atmname, tauname):

# Use ATLAS9 to generate a table with Tau(500 nm) vs. RHOX for an ATLAS9/ATLAS12 model
# atmosphere.

    fatm = open(atmname,'r')
    fout = open(atmname+'.tmp_t500','w')
    
    s = fatm.readline()
    shdr = s
    while (s != ''):
    
        if ('ABUNDANCE TABLE' in s):   # Skip these ATLAS12 model cards
            while not ('READ DECK' in s): s = fatm.readline()    

        if ('READ DECK' in s):
            ss = s.split()
            ntau = int(ss[2])

        fout.write(s)
        s = fatm.readline()

    fout.close()
    fatm.close()
    
    print("TAU500: %s --> %s (%d layers)" % (atmname,tauname,ntau))
    logging.info("TAU500: %s --> %s (%d layers)" % (atmname,tauname,ntau))
        
    fc = open('tau500.com','w')
    fc.write('\\rm -f fort.*\n')
    fc.write('ln -fs '+odfdir+podfs[0]+'big2.bdf fort.9\n')  # ODF file needs to be present
                                                             # but is not actually used since
                                                             # we do not include line opacities
#    fc.write('ln -fs /Users/soeren/playpen/kur_mar/odfbig2.bdf fort.9\n')
    fc.write('ln -fs '+atmname+'.tmp_t500 fort.3\n')
    fc.write(absetup.binpath+'atlas9mem.exe<<EOF>tau500.out\n')
    fc.write('WAVELENGTHS 500.00 0.1 500.00\n')
    fc.write('READ PUNCH\n')
    fc.write('OPACITY IFOP 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0\n')
    fc.write('ITERATIONS 1\n')
    fc.write('PRINT 3\n')
    fc.write('BEGIN\n')
    fc.write('END\n')
    fc.write('EOF\n')
    fc.write('\\rm -f fort.*\n')
    fc.close()
            
    os.system("bash -c 'source tau500.com'")
    
    ft = open(tauname,'w')
    ft.write(shdr)
    flog = open('tau500.out','r')
    s = ''
    while (not 'WAVELENGTH' in s): s = flog.readline()
    ft.write(s)
 
    for itau in range(ntau+2):
        s = flog.readline()
        ft.write(s)
   
    ft.close()
    flog.close()
    os.remove(atmname+'.tmp_t500')
    

def calcodf(m, odfname, atoms=[], abun=[]):

    temps = [1995., 2089., 2188., 2291., 2399., 2512., 2630., 2754., 2884., 
             3020., 3162., 3311., 3467., 3631., 3802., 3981., 4169., 4365.,
             4571., 4786., 5012., 5370., 5754., 6166., 6607., 7079., 7586.,
             8128., 8710., 9333., 10000.,11220.,12589.,14125.,15849.,17783.,
             19953.,22387.,25119.,28184.,31623.,35481.,39811.,44668.,50119.,
             56234.,63096.,70795.,79433.,89125.,
             100000.,112202.,125893.,141254.,158489.,177828.,199526.]

    # First write command file for xnfdf.exe

    fx = open('xnfdf.com','w')
    fx.write('\\rm -f fort.*\n')
    fx.write('ln -s '+absetup.catpref+'/cats/Castelli2015/molecules.dat fort.2\n')
    fx.write('ln -s '+absetup.catpref+'/cats/Castelli2015/pfiron.dat fort.4\n')
    fx.write('ln -s '+absetup.catpref+'/cats/Castelli2015/continua.dat fort.17\n')
    fx.write(absetup.binpath+'xnfdf.exe<<EOF>xnfdf.out\n')
    fx.write('READ MOLECULES\n')
    fx.write('MOLECULES ON\n')
    fx.write('ITERATIONS 1 PRINT 0 PUNCH 0\n')
    fx.write('READ FREQUENCIES 1 1 1 TEST\n')
    fx.write('1 1 1\n')

    for t in temps:
        fx.write('TEFF   5777.  GRAVITY 4.43770 LTE\n')
        fx.write('TITLE TEMPERATURES AND PRESSURES FOR ODF CALCULATION\n')
        fx.write(' OPACITY IFOP 1 1 1 1 1 1 1 1 1 1 1 1 1 0 1 0 0 0 0 0\n')
        fx.write(' CONVECTION ON   1.25 TURBULENCE OFF  0.00  0.00  0.00  0.00\n')
        wrabun(fx,m,atoms,abun)
        fx.write('READ DECK6 25 RHOX,T,P,XNE,ABROSS,ACCRAD,VTURB,CONVFRAC,VCONV\n')
        for p in 10**(numpy.arange(-4., 8.5, 0.5)):
            sp = '%0.3E' % p
            spf = sp.replace('E+0','E+').replace('E-0','E-')
            fx.write('   %s %6.0f.  %s  0 0 0 0 0\n' % (spf, t, spf))
        fx.write(' 1\n')
        fx.write('BEGIN\n')

    fx.write('END\n')
    fx.write('EOF\n')
    fx.write('mv fort.10  xnfpdf.dat\n')
    fx.write('mv fort.11 xnfpdfmax.dat\n')
    fx.write('\\rm -f fort.*\n')
    fx.close()

    # Write command file for dfsynthe.exe

    fd = open('dfsyn.com','w')
    fd.write('\\rm -f fort.*\n')
    j = 0
    sw = '000000000000000000000000000000000000000000000000000000000'
    for t in temps:
        fd.write('ln -s xnfpdf.dat fort.10\n')
        fd.write('ln -s xnfpdfmax.dat fort.22\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/lowlinesdf.bin fort.11\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/highlinesdf.bin fort.21\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/diatomicsdf.bin fort.31\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/tiolinesdf.bin fort.41\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/h2olinesdf.bin fort.43\n')
        fd.write('ln -s '+absetup.catpref+'/cats/Castelli2015/dflines/nltelinesdf.bin fort.51\n')
        fd.write(absetup.binpath+'dfsynthe.exe<<EOF>dft%d.out\n' % t)
        fd.write(sw[:j]+'1'+sw[j+1:]+'\n')
        j+=1
        fd.write('EOF\n')
        fd.write('mv fort.15 dft%dvt0.bin\n' % t)
        fd.write('mv fort.16 dft%dvt1.bin\n' % t)
        fd.write('mv fort.17 dft%dvt2.bin\n' % t)
        fd.write('mv fort.18 dft%dvt4.bin\n' % t)
        fd.write('mv fort.20 dft%dvt8.bin\n' % t)
        fd.write('\\rm -f fort.*\n')

    fd.close()

    # Write command files for dfsortp.exe (vt=0,1,2,4,8)

    fd0 = open('dfsortp0.com','w')
    fd1 = open('dfsortp1.com','w')
    fd2 = open('dfsortp2.com','w')
    fd4 = open('dfsortp4.com','w')
    fd8 = open('dfsortp8.com','w')

    fd0.write('\\rm -f fort.*\n')
    fd1.write('\\rm -f fort.*\n')
    fd2.write('\\rm -f fort.*\n')
    fd4.write('\\rm -f fort.*\n')
    fd8.write('\\rm -f fort.*\n')

    for t in temps:
        fd0.write('ln -s dft%dvt0.bin fort.1\n' % t)
        fd0.write(absetup.binpath+'dfsortp.exe>dfsortp.out\n')
        fd0.write('mv fort.2 dft%dvt0sortp.asc\n' % t)
        fd0.write('\\rm -f fort.*\n\n')

        fd1.write('ln -s dft%dvt1.bin fort.1\n' % t)
        fd1.write(absetup.binpath+'dfsortp.exe>dfsortp.out\n')
        fd1.write('mv fort.2 dft%dvt1sortp.asc\n' % t)
        fd1.write('\\rm -f fort.*\n\n')

        fd2.write('ln -s dft%dvt2.bin fort.1\n' % t)
        fd2.write(absetup.binpath+'dfsortp.exe>dfsortp.out\n')
        fd2.write('mv fort.2 dft%dvt2sortp.asc\n' % t)
        fd2.write('\\rm -f fort.*\n\n')

        fd4.write('ln -s dft%dvt4.bin fort.1\n' % t)
        fd4.write(absetup.binpath+'dfsortp.exe>dfsortp.out\n')
        fd4.write('mv fort.2 dft%dvt4sortp.asc\n' % t)
        fd4.write('\\rm -f fort.*\n\n')

        fd8.write('ln -s dft%dvt8.bin fort.1\n' % t)
        fd8.write(absetup.binpath+'dfsortp.exe>dfsortp.out\n')
        fd8.write('mv fort.2 dft%dvt8sortp.asc\n' % t)
        fd8.write('\\rm -f fort.*\n\n')

    fd0.close()
    fd1.close()
    fd2.close()
    fd4.close()
    fd8.close()

    # Write command file for separatedf.exe

    fs0 = open('separatedfvt0.com','w')
    fs1 = open('separatedfvt1.com','w')
    fs2 = open('separatedfvt2.com','w')
    fs4 = open('separatedfvt4.com','w')
    fs8 = open('separatedfvt8.com','w')

    fs0.write('\\rm -f fort.*\n')
    fs1.write('\\rm -f fort.*\n')
    fs2.write('\\rm -f fort.*\n')
    fs4.write('\\rm -f fort.*\n')
    fs8.write('\\rm -f fort.*\n')

    j = 10
    for t in temps:
        fs0.write('ln -s dft%dvt0sortp.asc fort.%d\n' % (t, j))
        fs1.write('ln -s dft%dvt1sortp.asc fort.%d\n' % (t, j))
        fs2.write('ln -s dft%dvt2sortp.asc fort.%d\n' % (t, j))
        fs4.write('ln -s dft%dvt4sortp.asc fort.%d\n' % (t, j))
        fs8.write('ln -s dft%dvt8sortp.asc fort.%d\n' % (t, j))
        j += 1

    fs0.write(absetup.binpath+'separatedf.exe\n')
    fs0.write('mv fort.2 '+odfname+'big0.bdf\n')
    fs0.write('mv fort.3 '+odfname+'lit0.bdf\n')
    fs0.write('\\rm -f dft*vt0sortp.asc\n')
    fs0.write('\\rm -f fort.*\n')

    fs1.write(absetup.binpath+'separatedf.exe\n')
    fs1.write('mv fort.2 '+odfname+'big1.bdf\n')
    fs1.write('mv fort.3 '+odfname+'lit1.bdf\n')
    fs1.write('\\rm -f dft*vt1sortp.asc\n')
    fs1.write('\\rm -f fort.*\n')

    fs2.write(absetup.binpath+'separatedf.exe\n')
    fs2.write('mv fort.2 '+odfname+'big2.bdf\n')
    fs2.write('mv fort.3 '+odfname+'lit2.bdf\n')
    fs2.write('\\rm -f dft*vt2sortp.asc\n')
    fs2.write('\\rm -f fort.*\n')

    fs4.write(absetup.binpath+'separatedf.exe\n')
    fs4.write('mv fort.2 '+odfname+'big4.bdf\n')
    fs4.write('mv fort.3 '+odfname+'lit4.bdf\n')
    fs4.write('\\rm -f dft*vt4sortp.asc\n')
    fs4.write('\\rm -f fort.*\n')

    fs8.write(absetup.binpath+'separatedf.exe\n')
    fs8.write('mv fort.2 '+odfname+'big8.bdf\n')
    fs8.write('mv fort.3 '+odfname+'lit8.bdf\n')
    fs8.write('\\rm -f dft*vt8sortp.asc\n')
    fs8.write('\\rm -f fort.*\n')

    fs0.close()
    fs1.close()
    fs2.close()
    fs4.close()
    fs8.close()

    # Now execute all the command files

    os.system("bash -c 'source xnfdf.com'")
    os.system("bash -c 'source dfsyn.com'")
    os.system("bash -c 'source dfsortp0.com'")
    os.system("bash -c 'source dfsortp1.com'")
    os.system("bash -c 'source dfsortp2.com'")
    os.system("bash -c 'source dfsortp4.com'")
    os.system("bash -c 'source dfsortp8.com'")
    os.system("bash -c 'source separatedfvt0.com'")
    os.system("bash -c 'source separatedfvt1.com'")
    os.system("bash -c 'source separatedfvt2.com'")
    os.system("bash -c 'source separatedfvt4.com'")
    os.system("bash -c 'source separatedfvt8.com'")



