from __future__ import print_function

from . import absetup
from . import kurucz
from . import marcs
from . import syntmoog
from . import syntherk
from . import turbospec
import math
import numpy
import scipy.signal as signal
import scipy.ndimage as ndimage
import scipy.interpolate as interpolate
import tempfile
import shutil
import subprocess
import os
import logging
import operator
import random
import utils

def parsehdr(shdr):
    ss = shdr.split()
    i = 0
    ivi, imv, imass, iwgt, ilogvt, isynt = -1, -1, -1, -1, -1, -1
    
# V-I = V-I colour of the star
# MV  = Absolute V magnitude
# MASS = Mass (used to compute logg)
# WEIGHT = Weight when co-adding spectra
# LOGVT = microturbulent velocity
# SYNT = Method used for spectral synthesis. Allowed values:
#     A9S  = ATLAS9 + SYNTHE
#     A12S = ATLAS12 + SYNTHE
#     MT   = MARCS + Turbospectrum
#     A9T  = ATLAS9 + Turbospectrum    
    
    for s in ss:
        if (s == 'V-I'):
            ivi = i
        elif (s == 'MV'):
            imv = i
        elif (s == 'MASS'):
            imass = i
        elif (s == 'WEIGHT'):
            iwgt = i
        elif (s == 'LOGVT'):
            ilogvt = i
        elif (s == 'SYNT'):
            isynt = i
            
        i+=1
        
    pp = {'ivi': ivi, 'imv': imv, 'imass': imass, 'iwgt': iwgt, 'ilogvt': ilogvt, 'isynt': isynt}
    return pp       


def vi2phys(cmdfile, mh, teffsort=True):
    '''Calculate physical parameters for stars with VI photometry. 
  "cmdfile" is the name of a text file with the following information
  for each star, one star (or group of stars) per line:
   V-I  M_V  M/Msun  [N]
  The last column is optional and indicates the number of stars
  with the given parameters. Returns a list with
  [[teff1, logg1, rstar1, nstar1, logvt1, synt1], [teff2, logg2, rstar2, nstar2, logvt2, synt2], ...]
  ''' 

    logging.info('VI2PHYS: Reading photometry from %s' % cmdfile)   

    fcmd = open(cmdfile,'r')
    s = fcmd.readline()
    hdr = parsehdr(s)

    if (hdr['ivi'] == -1 or hdr['imv'] == -1 or hdr['imass'] == -1):
        logging.error('VI2PHYS: header of %s does not specify mandatory V-I, MV and MASS fields' % cmdfile)
        fcmd.close()
        raise Exception('Error reading photometry file')

    stelpar = []

    s = fcmd.readline()
    while (s != ''):
        ss = s.split()
        vi = float(ss[hdr['ivi']])
        mv = float(ss[hdr['imv']])
        mass = float(ss[hdr['imass']])

        nstar = 1.0
        if (hdr['iwgt'] > -1):
            nstar = float(ss[hdr['iwgt']])
  
        logvt = None
        if (hdr['ilogvt'] > -1):
            logvt = float(ss[hdr['ilogvt']])  
            
        synt = 'A9S'
        if (hdr['isynt'] > -1):
            if (ss[hdr['isynt']] == 'MT'):
                synt = 'MT'
            elif (ss[hdr['isynt']] == 'A9S'):
                synt = 'A9S'
            elif (ss[hdr['isynt']] == 'A12S'):
                synt = 'A12S'
            elif (ss[hdr['isynt']] == 'A9T'):
                synt = 'A9T'
            else:
                logging.error('VI2PHYS: %s is not an allowed value for SYNT (must be A9S, A12S, A9T, or MT)' % ss[hdr['isynt']])
                fcmd.close()
                raise Exception('Error reading photometry file')  
  
              
        phys = kurucz.obs2phys(vi, mv, mh, mass) + [nstar, logvt, synt]
        if (numpy.isnan(phys[0])): 
            logging.info('   Colour outside range of Kurucz models (V-I=%0.3f)' % vi)
            print('Colour outside range of Kurucz models (V-I=%0.3f)' % vi)
        else:
            stelpar.append(phys)
        
        if (logvt == None):
            logging.info('    (V-I, MV) = %0.3f, %0.3f --> (Teff, Logg, r) = %0.1f %0.2f %0.2f  N= %0.2e' % (vi, mv, phys[0], phys[1], phys[2], phys[3]))
        else:
            logging.info('    (V-I, MV) = %0.3f, %0.3f --> (Teff, Logg, r) = %0.1f %0.2f %0.2f  N= %0.2e LogVT= %0.3f' % (vi, mv, phys[0], phys[1], phys[2], phys[3], phys[4]))
        
        if (synt == 'A9S'):
            logging.info('                Use ATLAS9/SYNTHE for this star.')
        elif (synt == 'A12S'):
            logging.info('                Use ATLAS12/SYNTHE for this star.')    
        if (synt == 'A9T'):
            logging.info('                Use ATLAS9/Turbospectrum for this star.')
        elif (synt == 'MT'):
            logging.info('                Use MARCS/Turbospectrum for this star.')    
            
        s = fcmd.readline()
        
    fcmd.close()

    if (teffsort):
        spsort = sorted(stelpar, key=operator.itemgetter(0))
    else:
        spsort = stelpar
   
    return spsort    
    
    
    
def parsephyshdr(shdr):
    ss = shdr.split()
    i = 0
    iteff, ilogg, imass, irstar, iwgt, ilogvt, isynt = -1, -1, -1, -1, -1, -1, -1
    
# MASS = Mass colour of the star
# TEFF  = Effective temperature
# LOGG  = surface gravity
# RSTAR  = Radius
# WEIGHT = Weight when co-adding spectra
# LOGVT = microturbulent velocity
# SYNT = Method used for spectral synthesis. Allowed values:
#     A9S  = ATLAS9 + SYNTHE
#     A12S = ATLAS12 + SYNTHE
#     A9T  = ATLAS9 + Turbospectrum
#     MT   = MARCS + Turbospectrum    
    
    for s in ss:
        if (s == 'MASS'):
            imass = i
        elif (s == 'TEFF'):
            iteff = i
        elif (s == 'LOGG'):
            ilogg = i
        elif (s == 'RSTAR'):
            irstar = i
        elif (s == 'WEIGHT'):
            iwgt = i
        elif (s == 'LOGVT'):
            ilogvt = i
        elif (s == 'SYNT'):
            isynt = i
            
        i+=1
        
    pp = {'imass': imass, 'iteff': iteff, 'ilogg': ilogg, 'irstar': irstar, 'iwgt': iwgt, 'ilogvt': ilogvt, 'isynt': isynt}
    return pp       
    


def rdphys(hrdfile,teffsort=True):

    logging.info('RDPHYS: Reading HRD data from %s' % hrdfile)

    fcmd = open(hrdfile,'r')
    s = fcmd.readline()
    hdr = parsephyshdr(s)

    if (hdr['iteff'] == -1 or hdr['ilogg'] == -1 or hdr['irstar'] == -1 or hdr['iwgt'] == -1):
        logging.error('RDPHYS: header of %s does not specify mandatory TEFF, LOGG, RSTAR and WEIGHT fields' % hrdfile)
        fcmd.close()
        raise Exception('Error reading HRD file')


    sptmp = []

    s = fcmd.readline()
    while (s != ''):
        ss = s.split()
        teff = float(ss[hdr['iteff']])
        logg = float(ss[hdr['ilogg']])
        rstar = float(ss[hdr['irstar']])
        nstar = float(ss[hdr['iwgt']])
       
        mass = None
        if (hdr['imass'] > -1):
            mass = float(ss[hdr['imass']])
            
        logvt = None
        if (hdr['ilogvt'] > -1):
            logvt = float(ss[hdr['ilogvt']])  
            
            
        synt = 'A9S'
        if (hdr['isynt'] > -1):
            if (ss[hdr['isynt']] == 'MT'):
                synt = 'MT'
            elif (ss[hdr['isynt']] == 'A9S'):
                synt = 'A9S'
            elif (ss[hdr['isynt']] == 'A12S'):
                synt = 'A12S'
            elif (ss[hdr['isynt']] == 'A9T'):
                synt = 'A9T'
            else:
                logging.error('RDPHYS: %s is not an allowed value for SYNT (must be A9S, A12S, A9T, or MT)' % ss[hdr['isynt']])
                fcmd.close()
                raise Exception('Error reading HRD file')  
    
        sptmp.append([teff,logg,rstar,nstar,logvt,synt])
        s = fcmd.readline()
        
    fcmd.close()


    if (teffsort):
        spsort = sorted(sptmp, key=operator.itemgetter(0))
    else:
        spsort = sptmp

    return spsort        


def chksynt(stelpar,synt):
    match = False
    for sp in stelpar:
        if (sp[5] == synt): match = True
    return match
        

def hrd2atm(stelpar, mh, atoms=[], abun=[], tmproot='.', nlte=False, odfs='S'):
    '''Calculate ATLAS model atmospheres for the stellar parameters given by stelpar
  and metallicity.
  stelpar: [[teff1, logg1, rstar1, nstar1, logvt1, synt1], [teff2, logg2, rstar2, nstar2, logvt2, synt2], ...]
  (but only the teff and logg values are used by this routine).
  odfs: valid options are 'S' for Solar scaled, or 'A' for Alpha enhanced.'''
    
    kurucz.AFE = odfs
    
    if chksynt(stelpar,'A9S') or chksynt(stelpar,'A9T'):
        kurucz.mkodf(mh,atoms,abun,odfs)
        
#    if chksynt(stelpar,'MT'):
#       No action required for Marcs models at this point

    speclst = []
    
    procs = []
    tmpdir = []
    
    if (nlte):
        slte = 'NLTE'
    else:
        slte = 'LTE'
    
    logging.info('HRD2ATM: Computing %i model atmospheres' % len(stelpar))
    logging.info('  [m/H] = %0.3f' % mh)
    for at,ab in zip(atoms,abun):
        logging.info('  %-2s: %+0.3f' % (at,ab))

    for i in range(len(stelpar)):
        if len(procs) == absetup.MAXPROC:
            logging.debug('MAXPROC reached, waiting..')
            pt = os.wait()
            for j in range(absetup.MAXPROC):
                logging.debug('Checking processes writing in %s' % tmpdir[j])
                if (procs[j].pid == pt[0]): jt = j

            logging.debug('Deleting tmpdir: %s' % tmpdir[jt])
            shutil.rmtree(tmpdir[jt])
            del(procs[jt])
            del(tmpdir[jt])
#            os.waitpid(procs[0].pid, 0)
#            shutil.rmtree(tmpdir[0])
#            procs = procs[1:]
#            tmpdir = tmpdir[1:]
            
        teff = stelpar[i][0]
        logg = stelpar[i][1]
        rstar = stelpar[i][2]
        vturb = 10**(stelpar[i][4])
        mstar = 10**(logg-kurucz.loggsun + 2*math.log10(rstar))
        fname = 'star%04d' % i
        wd = tempfile.mkdtemp(dir=tmproot)
        logging.debug('Making new tmpdir for star %04d: %s ' % (i, wd))
        tmpdir.append(wd)
        logging.info('    %s: (Teff,Logg,[m/H]) = %0.1f %0.2f %0.2f %s' % (fname, teff, logg, mh, slte))
        
        if (stelpar[i][5] == 'A9S' or stelpar[i][5] == 'A9T'):
#            logging.info('      Computing ATLAS9 model')
            pp = kurucz.mkatm(teff, logg, mh, fname, workdir=wd, wait=False, atoms=atoms, abun=abun, nlte=nlte, vturb=vturb)
        elif (stelpar[i][5] == 'A12S'):
#            logging.info('      Computing ATLAS12 model')
            pp = kurucz.mkatm_a12(teff, logg, mh, fname, workdir=wd, wait=False, atoms=atoms, abun=abun, nlte=nlte, vturb=vturb)
        elif(stelpar[i][5] == 'MT'):
#            logging.info('      Interpolating in MARCS model grid')
            pp = marcs.intpatm(teff, logg, mh, mstar, fname, workdir=wd, wait=False, atoms=atoms, abun=abun)

        procs.append(pp)
        speclst.append(fname)
        
    for j in range(len(procs)):
        os.waitpid(procs[j].pid,0)
        logging.debug('Deleting tmpdir: %s' % tmpdir[j])
        shutil.rmtree(tmpdir[j])
            
    return speclst    


def chkspec(speclst):
    
    ok = True
    for specfile in speclst:
        lam, flx = utils.getcol(specfile, (0,2))
        nl = len(lam)
        if nl < 3: 
            ok = False
            strok = '[ERROR]'
        else:
            strok = ' '
            
        print('  Checking %s:  %d lines %s' % (specfile, nl, strok))
        logging.debug('  Checking %s:  %d lines %s' % (specfile, nl, strok))
        
    return ok


def hrd2spec(stelpar, outspec, synlimits, mh, atoms=[], abun=[], logvt=0.30103, 
             tmproot='.', initialize=True, initdir='.'):
    '''stelpar: [[teff1, logg1, rstar1, nstar1, logvt1], [teff2, logg2, rstar2, nstar2, logvt2], ...]'''

            
    if (initialize):
        logging.debug('HRD2SPEC: Initializing')           
        print('HRD2SPEC: Initializing')
        if (chksynt(stelpar,'A9S') or chksynt(stelpar,'A12S')):
            syntherk.synbeg(synlimits[0], synlimits[1], synlimits[2], initdir=initdir)
        if (chksynt(stelpar,'MT') or chksynt(stelpar,'A9T')):
            turbospec.turboinit(synlimits[0], synlimits[1], synlimits[2], initdir=initdir)
    else:
        print('HRD2SPEC: Skipping initialization')
        logging.debug('HRD2SPEC: Skipping initialization')

# Calculate synthetic spectra and equivalent widths

    specok = False
    ntry = 1
    
    while (not specok):

        logging.debug('HRD2SPEC: Computing synthetic spectra')           
        
        procs = []
        tmpdir = []
        speclst = []
        weights = []
        syntid = []
        
        for i in range(len(stelpar)):
            fname = 'star%04d' % i
            rstar = stelpar[i][2]
            nstar = stelpar[i][3]
            if (logvt == 'USER'):
                vturb = 10**(stelpar[i][4])
            else:
                vturb = 10**logvt

            if len(procs) == absetup.MAXPROC:
                pt = os.wait()
                for j in range(absetup.MAXPROC): 
                    if (procs[j].pid == pt[0]): jt = j
                shutil.rmtree(tmpdir[jt])
                del(procs[jt])
                del(tmpdir[jt])    
#                os.waitpid(procs[0].pid,0)        
#                shutil.rmtree(tmpdir[0])
#                procs = procs[1:]
#                tmpdir = tmpdir[1:]                  
                        
            wd = tempfile.mkdtemp(dir=tmproot)
            tmpdir.append(wd)
            if (stelpar[i][5] == 'A9S'):
                logging.debug('Using ATLAS9/SYNTHE for %s' % fname)
                pp = syntherk.synthespec(mh, fname+'.A9', fname+'.asc',
                                   atoms=atoms, abun=abun, vturb=vturb,
                                   workdir=wd, initdir=initdir, wait=False)
            elif (stelpar[i][5] == 'A12S'):
                logging.debug('Using ATLAS12/SYNTHE for %s' % fname)
                pp = syntherk.synthespec(mh, fname+'.A12', fname+'.asc',
                                   atoms=atoms, abun=abun, vturb=vturb,
                                   workdir=wd, initdir=initdir, wait=False)
            elif (stelpar[i][5] == 'MT'):
                logging.debug('Using MARCS/TURBOSPECTRUM for %s' % fname)
                pp = turbospec.turbospec(mh, fname+'.marcs', fname+'.asc',
                                   atoms=atoms, abun=abun, vturb=vturb,
                                   workdir=wd, initdir=initdir, wait=False)
            elif (stelpar[i][5] == 'A9T'):
                logging.debug('Using ATLAS9/TURBOSPECTRUM for %s' % fname)
                pp = turbospec.turbospec(mh, fname+'.A9', fname+'.asc',
                                   atoms=atoms, abun=abun, vturb=vturb,
                                   workdir=wd, initdir=initdir, wait=False)
            else:
                print('Invalid SYNT option, oops')
                logging.info('Invalid SYNT option, oops')
                raise Exception('Invalid SYNT option')
                                          
            procs.append(pp)
            speclst.append(fname+'.asc')
            weights.append(nstar * rstar**2.0)
            syntid.append(stelpar[i][5])
        
        for j in range(len(procs)):
            os.waitpid(procs[j].pid,0)
            shutil.rmtree(tmpdir[j])

        specok = chkspec(speclst)        
        if (not specok):
            ntry+=1
            if (ntry < 4):
                print('  Problems with synthetic spectra - trying again (attempt=%d)' % ntry)
                logging.info('  Problems with synthetic spectra - trying again (attempt=%d)' % ntry)
            else:
                print('  Problems with synthetic spectra - aborting (attempt=%d)' % ntry)
                logging.info('  Problems with synthetic spectra - aborting (attempt=%d)' & ntry)
                raise Exception('Problems with synthetic spectra')
            

    logging.debug('HRD2SPEC: Co-adding synthetic spectra')   

            
    sumspec(speclst, weights, syntid, outspec)
    logging.debug('HRD2SPEC: Done')
    
    

def cmd2spec(cmdfile, outspec, outew, synlimits, mh, atoms=[], abun=[], existing_atm=False, synt='SYNTHE'):

# This subroutine is obsolete and probably won't work any more.

    if not synt in ['MOOG','SYNTHE']:
        print('ERROR in cmd2spec: synt must be either SYNTHE or MOOG')
        return
        

    mass = 0.9
    stelpar = []
    nstar = []
    
    if (synt == 'MOOG'):
        syntmoog.linelst(synlimits[0:2], 'linelist.txt')
    else:
        syntherk.synbeg(synlimits[0], synlimits[1], synlimits[2])

# Determine the physical parameters for each star

    with open(cmdfile,'r') as f:
        for s in f:
            ss = s.split()
            vi = float(ss[0])
            mv = float(ss[1])
            if len(ss) > 2:
                nstar.append(float(ss[2]))
            else:
                nstar.append(1.0)
              
            phys = kurucz.obs2phys(vi, mv, mh, mass)
            stelpar.append(phys)
        
#    print stelpar
    
# Calculate ATLAS9 model atmospheres

    if (existing_atm):
        print('Using existing model atmospheres!')
    else:
        kurucz.mkodf(mh)
    
    speclst = []
    ewlst   = []
    weights  = []
        
    for i in range(len(stelpar)):
        teff = stelpar[i][0]
        logg = stelpar[i][1]
        fname = 'star%04d' % i
        if existing_atm:
            print('Using existing model atmosphere: '+fname)
        else:
            kurucz.mkatm(teff, logg, mh, fname)
        
# Calculate synthetic spectra and equivalent widths
        
    procs = []
    tmpdir = []
        
    for i in range(len(stelpar)):
        fname = 'star%04d' % i
        rstar = stelpar[i][2]

        if (synt == 'MOOG'):
            syntmoog.moogspec(mh, fname+'.dat', fname+'.moog', \
                              synlimits=synlimits, linelist='linelist.txt', \
                              atoms=atoms, abun=abun)
            syntmoog.moogew(mh, fname+'.dat', fname+'.ew', \
                              synlimits=synlimits, linelist='linelist.txt', \
                              atoms=atoms, abun=abun)
            speclst.append(fname+'.moog')
            ewlst.append(fname+'.ew')
        else:
            if len(procs) == absetup.MAXPROC:
                os.waitpid(procs[0].pid,0)        
                shutil.rmtree(tmpdir[0])
                procs = procs[1:]
                tmpdir = tmpdir[1:]                  
                        
            wd = tempfile.mkdtemp(dir='.')
            tmpdir.append(wd)
            pp = syntherk.synthespec(mh, fname+'.dat', fname+'.asc' , \
                               atoms=atoms, abun=abun, workdir=wd, wait=False)
            procs.append(pp)
            speclst.append(fname+'.asc')
                
        weights.append(nstar[i] * rstar**2.0)
        
    for j in range(len(procs)):
        os.waitpid(procs[j].pid,0)
        shutil.rmtree(tmpdir[j])
        
    sumspec(speclst, weights, outspec)
    
    if (synt=='MOOG'):
        syntmoog.sumew(ewlst, weights, outew)
        
        
def lamreb(lam,flux,lamref):
    fnc = interpolate.interp1d(lam, flux, bounds_error=False, kind='linear')
    flux = fnc(lamref)
    if math.isnan(flux[0]): 
        j = 0
        while (math.isnan(flux[j])): j = j+1
        for k in range(j): flux[k] = flux[j]
#        flux[0] = flux[1]
    if math.isnan(flux[-1]): 
        j = len(flux)-1
        while (math.isnan(flux[j])): j = j-1
        for k in range(j+1, len(flux)): flux[k] = flux[j]
#        flux[-1] = flux[-2]

    return flux
    
    
def sumspec(specfiles, weights, syntid, outfile):

    nspec = len(specfiles)
    lam = []
    flux = []
    
#    if ('A9S' in syntid) & ('MT' in syntid):
#        synref='A9S'
#    elif ('MT' in syntid):
#        synref='MT'
#    else:
#        synref='A9S'            
    if ('A12S' in syntid):
        synref='A12S'
    elif ('A9S' in syntid):
        synref='A9S'
    elif ('MT' in syntid):
        synref='MT'
    elif ('A9T' in syntid):
        synref='A9T'


    iref = syntid.index(synref)
    print('Using %s as reference for wavelength scale' % specfiles[iref])
    lamref = utils.getcol(specfiles[iref],(0))
    
    of = open(outfile,'w')
    of.write('# '+specfiles[0]+' {0:0.4f}  {1:s}\n'.format(weights[0], syntid[0]))

    with open(specfiles[0],'r') as f:
        for s in f:
            ss = s.split()
            if (ss[0] != '#'):
                lam.append(float(ss[0]))
                flux.append(float(ss[2]))

    if (syntid[0] != synref): flux = lamreb(lam, flux, lamref)
    
    sflux = numpy.array(flux) * weights[0]

    for i in range(1,nspec):
        flux = []
        lam = []
        of.write('# '+specfiles[i]+' {0:0.4f}  {1:s}\n'.format(weights[i], syntid[i]))
  
        with open(specfiles[i],'r') as f:
            for s in f:  
                ss = s.split()
                if (ss[0] != '#'):
                    lam.append(float(ss[0]))
                    flux.append(float(ss[2]))

        if (syntid[i] != synref): flux = lamreb(lam, flux, lamref)

        sflux2 = numpy.array(flux) * weights[i]
 #       print len(sflux), len(sflux2)
        sflux = sflux + sflux2

    for i in range(len(lamref)):
        of.write('{0:0.4f}   {1:0.4e}\n'.format(lamref[i], sflux[i]))
        
    of.close()
    
    
    
def smoothspec(infile, outfile, sigsmooth, lamcol=0, flxcol=1):

    fout = open(outfile,'w')
    lam, flx = [], []

    with open(infile,'r') as f:
        for s in f:
            ss = s.split()
            if (ss[0] == '#'):
                fout.write(s)
            else:
                lam.append(float(ss[lamcol]))
                flx.append(float(ss[flxcol]))
            
    fout.write('# Smoothing: Sigma={0:0.4f} AA\n'.format(sigsmooth))        
        
    dlam = (max(lam)-min(lam)) / (len(lam) - 1)
#    print min(lam), max(lam), len(lam)
#    print 'Wavelength step = '+str(dlam)
    
    flxcnv = ndimage.gaussian_filter1d(flx, sigsmooth/dlam)
#    print flxcnv
    
    for i in range(len(lam)):
        fout.write('{0:0.4f}   {1:0.4e}\n'.format(lam[i], flxcnv[i]))
        
    fout.close()
    
    
def rebspec(infile, outfile, dlam, npint=10, lamcol=0, flxcol=1):
    '''Rebin a spectrum to a sampling of dlam per pixel.
    This is done by interpolating in the original input spectrum and
    integrating over npint sampling points per output pixel.'''
   
    fout = open(outfile,'w')
    lam, flx = [], []
    
    with open(infile,'r') as f:
        for s in f:
            ss = s.split()
            if (ss[0] == '#'):
                fout.write(s)
            else:
                lam.append(float(ss[lamcol]))
                flx.append(float(ss[flxcol]))

    fout.write('# abutils.rebspec:\n')
    fout.write('# Binning: dlam=%0.4f AA\n' % (dlam))
    fout.write('#          npint=%i\n' % (npint))
    
    fncflx = interpolate.interp1d(lam, flx, bounds_error=False, kind='linear')
            
    l1 = min(lam)
    while (l1 < max(lam)-dlam):
        l2 = l1 + dlam
        dlsub = (l2 - l1)/npint
        lsub = l1 + dlsub/2 + numpy.array(range(npint))*dlsub
        flxarr = fncflx(lsub)
        flxi = sum(flxarr)/npint
        fout.write('%0.4f   %0.4e\n' % ((l1+l2)/2,flxi))
        l1 = l1 + dlam
    
    fout.close()


def addnoise(infile, outfile, s2n=10):
    '''Add random (gaussian) noise to a spectrum. 
    For now, a constant sigma (based on the mean flux) is added.
    In the future, a more sophisticated noise model may be included.'''
    
    fout = open(outfile,'w')
    lam, flx = [], []
    
    with open(infile,'r') as f:
        for s in f:
            ss = s.split()
            if (ss[0] == '#'):
                fout.write(s)
            else:
                lam.append(float(ss[0]))
                flx.append(float(ss[1]))

    fmean = sum(flx)/len(flx)
    snoise = fmean / s2n

    fout.write('# abutils.addnoise:\n')
    fout.write('# s2n = %0.1f\n' % (s2n))
    
    for l,f in zip(lam,flx):
        fout.write('%0.4f   %0.4e  %0.4e\n' % (l, f+random.gauss(0,snoise), snoise))
    
    fout.close()
 
    
def rvcorrect(spec, rv):
    lscl = 1 - rv/299792.458
    sp2 = []
    for pixdat in spec: 
        sp2.append( [float(pixdat[0])*lscl] + [float(xx) for xx in pixdat[1:]] )
    return sp2
    
    
def bsnoise(spec):
    '''Duplicate input spectrum but add noise to each pixel based on the errors'''
    spn = []    
    for pixdat in spec:
        err = float(pixdat[2])
        pixval = float(pixdat[1]) + random.gauss(0, err)
        spn.append( [pixdat[0], pixval] + [float(xx) for xx in pixdat[2:]])
    return spn    
    
    
def air2vac(spec):
    '''Air to vacuum wavelength conversion. The formula used is from Morton 1991, 
ApJS 77, 119. Wavelengths must be in Angstrom.'''
    
    sp2 = []
    for pixdat in spec:
        lair = float(pixdat[0])
        s2 = (1e4/lair)**2.
        n  = 1.0 + 6.4328e-5 + 2.94981e-2/(146.0 - s2) + 2.5540e-4/(41.0 - s2)
        lvac = lair * n
        sp2.append( [lvac] + [float(xx) for xx in pixdat[1:]] )
    return sp2    
    
    
def vac2air(spec):
    '''Vacuum to air wavelength conversion. The formula used is from Morton 1991, 
ApJS 77, 119. Wavelengths must be in Angstrom.'''

    sp2 = []
    for pixdat in spec:
        lvac = float(pixdat[0])
        s2 = (1e4/lvac)**2.
        n  = 1.0 + 6.4328e-5 + 2.94981e-2/(146.0 - s2) + 2.5540e-4/(41.0 - s2)
        lair = lvac / n
        sp2.append( [lair] + [float(xx) for xx in pixdat[1:]] )
    return sp2    
    
    
    
def hrd2ew(stelpar, elem, lam, nablog=1, minlog=0, dablog=0, logvt=0.300, 
           lamtol=0.001, lines=None):

# Calculated integrated-light equivalent widths.
# Input: stelpar = array of stellar parameters
#        elem    = element code, e.g. 11.00 = Na I
#        lam     = wavelength of line, in AA
#        nablog  = Number of abundance offsets (with respect to value in atmospheres)
#                  nablog must be <= 9
#        minlog  = Minimum relative abundance
#        dablog  = Log abundance step
#        lamtol  = Tolerance when matching lambda to line list, in AA
#        lines   = File with line data (Kurucz format). Default: Castelli list
# Output: Integrated-light EWs

    wsum = 0.
    ewsum = numpy.zeros(nablog)
    
    for i in range(len(stelpar)):
        if (stelpar[i][5] == 'A9S'):
            fname = 'star%04d.A9' % i
        elif (stelpar[i][5] == 'A12S'):
            fname = 'star%04d.A12' % i
        else:
            print('Invalid SYNT option, oops')
            logging.info('Invalid SYNT option, oops')
            raise Exception('Invalid SYNT option')

            
        rstar = stelpar[i][2]
        nstar = stelpar[i][3]
        if (logvt == 'USER'):
            logvturb = stelpar[i][4]
        else:
            logvturb = logvt

        ewdat = syntherk.calcew(fname, elem, lam, nablog, minlog, dablog, logvturb, 
                               lamtol=lamtol, lines=lines)

        ewi = ewdat['ew']
        cflxi = ewdat['contin']
        wgti = cflxi * nstar * rstar**2.0
        
        print("%s: WGT= %0.3e  EW= " % (fname, wgti), end='')
        for ee in ewi: print("%6.1f " % ee, end='')
        print("mAA")
        
        ewsum += wgti * numpy.array(ewi)
        wsum += wgti
        
    ew = ewsum / wsum
    return ew
