/*
   File: tuples.c
   Handles the storage, reading, writing and updating
   of the unparsing rules
*/

/* Global includes */
#include <stdio.h>
#include <string.h>
#include <ctype.h>

/* libeag includes */
#include <export.h>
#include <memalloc.h>
#include <textstorage.h>
#include <textparsing.h>

/* local includes */
#include <editor.h>
#include <tuples.h>

/*
   Allocation of tuples. The tuples may change but
   the number of rules never.
*/
private tuple free_tuples;
private tuple new_tuple (int type, char *text, int relates,
			 int hor_off, int vert_off, int sonnr)
	{ tuple new;
	  if (free_tuples != tuple_nil)
	     { new = free_tuples;
	       free_tuples = free_tuples -> next;
	     }
	  else new = (tuple) ckmalloc (sizeof (struct tuple_rec));
	  new -> type = type;
	  new -> text = text;
	  new -> relates = relates;
	  new -> hor_offset = hor_off;
	  new -> vert_offset = vert_off;
	  new -> sonnr = sonnr;
	  new -> next = tuple_nil;
	  return (new);
	};

private void deallocate_tuples (tuple old)
	{ tuple ptr;
	  if (old == tuple_nil) return;
	  for (ptr = old; ptr -> next != tuple_nil; ptr = ptr -> next);
	  ptr -> next = free_tuples;
	  free_tuples = old;
	};

private tuple append_tuple (tuple src, int type, char *text, int relates,
			    int hor_off, int vert_off, int sonnr)
	{ tuple ptr;
	  if (src == tuple_nil)
	     return (new_tuple (type, text, relates, hor_off, vert_off, sonnr));
	  for (ptr = src; ptr -> next != tuple_nil; ptr = ptr -> next);
	  ptr -> next = new_tuple (type, text, relates, hor_off,
				vert_off, sonnr);
	  return (src);
	};

private rule new_rule (tuple hrule, tuple vrule)
	{ rule new = (rule) ckmalloc (sizeof (struct rule_rec));
	  new -> hrule = hrule;
	  new -> vrule = vrule;
	  return (new);
	};

public int changed_rules;
private void replace_rule (rule nrule, int dir, tuple ntuple)
	{ tuple old;
	  if (dir == ver_rule)
	     { old = nrule -> vrule; nrule -> vrule = ntuple; }
	  else { old = nrule -> hrule; nrule -> hrule = ntuple; };
	  deallocate_tuples (old);
	  changed_rules = 1;
	};

/*
   Register the correct filename for the unparsing rules
*/
private char *fname;
private void establish_tuplefile (char *gram)
	{ strcpy (strstore, gram);		/* will become gram */
	  strcat (strstore, ".rules");
	  fname = addto_names (strstore);
	};

private FILE *open_tuple_file (char *mode)
	{ FILE *f = fopen (fname, mode);
	  if (f) return (f);
	  fprintf (stderr, "could not open file '%s' for %s\n", fname,
				(mode[0] == 'r')?"reading":"writing");
	  exit (4);
	};

/*
   Tuples may be read or written either from/to file
   or from/to the tuple_edit_buffer
*/
public char *tuple_edit_buffer;
private char *editptr;

private int tuple_error;
public char *tuple_error_buffer;

private int use_file;
private FILE *input;
private FILE *output;

#define MAXBUFLEN 250
private char inputbuffer[MAXBUFLEN];
private char *inputptr;

private int refill_input_buffer ()
	{ inputptr = inputbuffer;
	  if (use_file) return (!fgets (inputbuffer, MAXBUFLEN, input));
	  else
	     { char *dptr;
	       if (!(*editptr)) return (1);
	       for (dptr = inputbuffer;
		    (*editptr != '\0') && (*editptr != '\n');
		    editptr++, dptr++) *dptr = *editptr;
	       *dptr++ = '\n';
	       *dptr = '\0';
	       if ((*editptr)) editptr++;
	       return (0);
	     };
	};

/*
   The tuples are read by an ordinary LL1 parser
   'inputptr' will always point to the first unrecognized character
*/
private int linenr;
#define colnr (inputptr - inputbuffer)
private int eof;
private void may_read_next_line ()
	{ while (!eof && !(*inputptr))
	     if (refill_input_buffer ())
		{ eof = 1;
		  *inputptr = '\0';
		}
	     else linenr++;
	};

#define lookahead_char(c) (*inputptr == c)
private int is_char (char c)
	{ if (lookahead_char (c))
	     { inputptr++;
	       return (1);
	     };
	  return (0);
	};

private int lookahead_string (char *s)
	{ char *ptr, *ptr2;
	  for (ptr = s, ptr2 = inputptr; *ptr; ptr++, ptr2++)
	     if (*ptr != *ptr2) return (0);
	  return (1);
	};
	
private int is_string (char *s)
	{ char *ptr, *ptr2;
	  for (ptr = s, ptr2 = inputptr; *ptr; ptr++, ptr2++)
	     if (*ptr != *ptr2) return (0);
	  inputptr = ptr2;
	  return (1);
	};

private void skip_layout ()
	{ may_read_next_line ();
	  while (is_char (' ') || is_char ('\t') || is_char ('\n'))
	     may_read_next_line ();
	};

private int is_token (char *s)
	{ if (is_string (s))
	     { skip_layout ();
	       return (1);
	     };
	  return (0);
	};

private void expected_string (char *s)
	{ sprintf (tuple_error_buffer, "%s expected at line %d, col %d",
			s, linenr, colnr + 1);
	  tuple_error = 1;
	  if (use_file)
	     { fprintf (stderr, "error while reading tuples: %s\n",
			tuple_error_buffer);
	       exit (4);
	     };
	};

private void should_be_string (char *s)
	{ if (is_string (s)) return;
	  expected_string (s);
	};

private void should_be_token (char *s)
	{ if (is_token (s)) return;
	  expected_string (s);
	};

private int should_be_number ()
	{ int nr;
	  if (!isdigit (*inputptr)) expected_string ("number");
	  if (tuple_error) return (0);
	  nr = *inputptr - '0';
	  inputptr++;
	  while (isdigit (*inputptr))
	     { nr = 10 * nr + *inputptr - '0';
	       inputptr++;
	     };
	  skip_layout ();
	  return (nr);
	};

#define eoln (eof || !(*inputptr))
private char *should_be_string_ending_with (char *close)
	{ char *ptr = strstore;
	  while (!tuple_error && !lookahead_string (close) && !eoln)
	     if (iscntrl (*inputptr)) expected_string (close);
	     else if (*inputptr == '\\')
		{ inputptr ++;
		  if (iscntrl (*inputptr)) expected_string (close);
		  else if (*inputptr == '"') *ptr++ = '"';
	          else if (*inputptr == '}') *ptr++ = '}';
		  else if (*inputptr == 'n') *ptr++ = '\n';
		  else if (*inputptr == 't') *ptr++ = '\t';
		  else if (*inputptr == '\\') *ptr++ = '\\';
		  inputptr ++;
		}
	     else *ptr++ = *inputptr++;
	  *ptr = '\0';
	  if (tuple_error) return (addto_names (strstore));
	  should_be_string (close);
	  return (addto_names (strstore));
	};

/*
   Initialize reading and writing of tuples
*/
private void init_input_buffer ()
	{ tuple_error = 0;
	  eof = 0;
	  linenr = 0;
	  inputptr = inputbuffer;
	  editptr = tuple_edit_buffer;
	  *inputptr = '\0';
	  skip_layout ();
	};

private void init_reading ()
	{ use_file = 1;
	  input = open_tuple_file ("r");
	  free_tuples = tuple_nil;
	  init_input_buffer ();
	};

private void init_writing ()
	{ use_file = 1;
	  output = open_tuple_file ("w");
	};

/*
   Finish reading and writing of tuples
*/
private void finish_reading ()
	{ fclose (input);
	  use_file = 0;
	  changed_rules = 0;
	};

private void finish_writing ()
	{ fclose (output);
	  use_file = 0;
	  changed_rules = 0;
	};

/*
   These variables are needed for intermediate storage during parsing
   The editor generator will indicate the amount of storage necessary
*/
public int max_nr_of_tuples;
public int max_nr_of_rules;
public rule *ruletable;

private int *x_pos;
private int *y_pos;
private int *used_nrs;

private tuple read_alternative ()
	{ tuple current = tuple_nil;
	  int count = 0;		/* Keeps track of sons */
	  int i;
	  int lasty = linenr;
	  for (i = 0;;i++)
	     { char *text;
	       int type;
	       int sonnr;
	       int relates;
	       int hor_off = 0;
	       int vert_off = 0;
	       x_pos [i] = colnr;
	       y_pos [i] = linenr;

	       /* find out to whom we are related */
	       for (relates = 0; x_pos[i] != x_pos[relates]; relates++);

	       /* if we find a '.' we really want to insert layout,
		  related either to the previous position or to
		  a related position
	       */
	       if (is_token ("."))
		  { hor_off = colnr - x_pos[i];
		    vert_off = linenr - y_pos[i];
		  }

	       /* if we are not related to a previous tuple and
		  we went to a new line register the horizontal offset
		  from the start of the line
	       */
	       if ((relates != i) && y_pos[i] != lasty) hor_off = colnr;

	       /* add the vertical difference between the starts of
		  the current and the previous tuple to the vertical
		  offset
	       */
	       vert_off += (i==0)?0:y_pos[i]-y_pos[i-1];

	       if (is_char ('"'))
		  { text = should_be_string_ending_with ("\"");
		    type = type_terminal;
		    sonnr = -1;
		  }
	       else if (is_char ('#'))
		  { should_be_string (typed_open_symbol);
		    if (tuple_error) break;
		    text = should_be_string_ending_with (typed_close_symbol);
		    if (tuple_error) break;
		    type = type_forcednonterminal;
		    sonnr = used_nrs [count];
		    count++;
		  }
	       else if (is_string (typed_open_symbol))
		  { text = should_be_string_ending_with (typed_close_symbol);
		    if (tuple_error) break;
		    type = type_nonterminal;
		    sonnr = used_nrs [count];
		    count++;
		  }
	       else if (is_char ('{'))
		  { text = should_be_string_ending_with ("}");
		    if (tuple_error) break;
		    type = type_set;
		    sonnr = used_nrs [count];
		    count++;
		  }
	       else break;

	       current = append_tuple (current, type, text, relates,
				hor_off, vert_off, sonnr);
	       lasty = linenr;
	       skip_layout ();
	     };
	  if (tuple_error)
	     { deallocate_tuples (current);
	       return ((tuple) NULL);
	     }
	  else return (current);
	};

private int rules_consistent (tuple otuple, tuple ntuple)
	{ tuple optr, nptr;
	  for (optr = otuple, nptr = ntuple;
	       (optr != tuple_nil) && (nptr != tuple_nil);
	       optr = optr -> next, nptr = nptr -> next)
	    { if ((optr -> type == type_terminal) &&
		  (nptr -> type == type_terminal))
		 { if (strcmp (optr -> text, nptr -> text) != 0) return (0);
		 }
	      else if (((optr -> type == type_forcednonterminal) ||
			(optr -> type == type_nonterminal)) &&
		       ((nptr -> type == type_forcednonterminal) ||
			(nptr -> type == type_nonterminal)))
		 { if (strcmp (optr -> text, nptr -> text) != 0) return (0);
	         }
	      else if ((optr -> type == type_set) &&
		       (nptr -> type == type_set))
		 { if (strcmp (optr -> text, nptr -> text) != 0) return (0);
		 }
	      else return (0);
	    };
	  return (optr == nptr);	/* should equal to nil */
	};

public int try_and_replace_rule (rule lrule, int dir)
	{ tuple otuple = (dir == hor_rule)?lrule -> hrule:lrule -> vrule;
	  tuple ntuple;

	  init_input_buffer ();
	  ntuple = read_alternative ();
	  if (tuple_error) return (0);
	  if (!rules_consistent (otuple, ntuple))
	     { strcpy (tuple_error_buffer,
		"inconsistency between old and new rule");
	       return (0);
	     };
	  replace_rule (lrule, dir, ntuple);
	  return (1);
	};

/*
   Writing of a rule
*/
private void write_string_to_buffer (char *s, int *ccol)
	{ char *ptr;
	  int len = 1;
	  strcat (tuple_edit_buffer, "\"");
	  for (ptr = s; *ptr; ptr++)
	     if (*ptr == '"')
		{ strcat (tuple_edit_buffer, "\\\""); len += 2; }
	     else if (*ptr == '}')
		{ strcat (tuple_edit_buffer, "\\}"); len += 2; }
	     else if (*ptr == '\n')
		{ strcat (tuple_edit_buffer, "\\n"); len += 2; }
	     else if (*ptr == '\t')
		{ strcat (tuple_edit_buffer, "\\t"); len += 2; }
	     else if (*ptr == '\\')
		{ strcat (tuple_edit_buffer, "\\\\"); len += 2; }
	     else
		{ char buf[2];
		  buf[0] = *ptr;
		  buf[1] = '\0';
		  strcat (tuple_edit_buffer, buf);
		  len++;
		};
	  strcat (tuple_edit_buffer, "\"");
	  len++;
	  *ccol += len;
	};

private void write_spaces_to_buffer (int amount, int *ccol)
	{ int i;
	  for (i = 0;i < amount; i++) strcat (tuple_edit_buffer, " ");
	  *ccol += amount;
	};

private void write_alternative (tuple current)
	{ tuple ptr;
	  int ccol = 0;
	  int i, j;

	  tuple_edit_buffer[0] = '\0';
	  for (ptr = current, i = 0; ptr != tuple_nil; ptr = ptr -> next, i++)
	     { /* layout stuff */
	       if (ptr -> vert_offset)
		  { for (j = 0; j < ptr -> vert_offset; j++)
		       strcat (tuple_edit_buffer, "\n");
		    ccol = 0;
		  };
	       if (ptr -> relates < i)
		  write_spaces_to_buffer (x_pos[ptr -> relates], &ccol);
	       if (ptr -> hor_offset)
		  { strcat (tuple_edit_buffer, ".");
		    ccol += 1;
		    write_spaces_to_buffer (ptr -> hor_offset - 1, &ccol);
		  };

	       /* save position for related tuples */
	       x_pos[i] = ccol;
	       switch (ptr -> type)
		  { case type_terminal:
		       write_string_to_buffer (ptr -> text, &ccol);
		       break;
		    case type_forcednonterminal:
		       { strcat (tuple_edit_buffer, "#");
			 ccol++;
		       }
		    case type_nonterminal:
		       { strcat (tuple_edit_buffer, typed_open_symbol);
			 ccol += strlen (typed_open_symbol);
			 strcat (tuple_edit_buffer, ptr -> text);
			 ccol += strlen (ptr -> text);
			 strcat (tuple_edit_buffer, typed_close_symbol);
			 ccol += strlen (typed_close_symbol);
		       };
		       break;
		    case type_set:
		       { strcat (tuple_edit_buffer, "{}");
			 ccol += 2;
		       }
		    default: break;
		  };
	     };
	};

public void write_rule_to_buffer (rule lrule, int dir)
	{ if (dir == hor_rule) write_alternative (lrule -> hrule);
	  else write_alternative (lrule -> vrule);
	};

/*
   Reading and writing of rules
*/
private void read_rule ()
	{ int rulenr;
	  int count = 0;
	  tuple hrule = tuple_nil;
	  tuple vrule = tuple_nil;
	  should_be_token ("type");
	  rulenr = should_be_number ();

	  while (is_token (","))
	     { used_nrs [count] = should_be_number ();
	       count++;
	     };
	  if (is_token ("::"))
	     { hrule = read_alternative ();
	       vrule = hrule;
	     }
	  else if (is_token (":H:"))
	     { hrule = read_alternative ();
	       should_be_token (":V:");
	       vrule = read_alternative ();
	     }
	  else expected_string ("::");
	  ruletable [rulenr] = new_rule (hrule, vrule);
	};

private void write_rule (int i)
	{ tuple ptr;
	  rule wrule = ruletable[i];
	  if (wrule == rule_nil) return;
	  fprintf (output, "type %d", i);
	  for (ptr = wrule -> hrule; ptr != tuple_nil; ptr = ptr -> next)
	     if (ptr -> sonnr != -1) fprintf (output, ", %d", ptr -> sonnr);
	  fputs ("\n:H:\n", output);
	  write_alternative (wrule -> hrule);
	  fputs (tuple_edit_buffer, output);
	  fputs ("\n:V:\n", output);
	  write_alternative (wrule -> vrule);
	  fputs (tuple_edit_buffer, output);
	  fputc ('\n', output);
	};

/*
   The header of the tuple file contains some info on the storage
   needed during parsing.
*/
private void read_header ()
	{ int i;
	  should_be_token (max_rule_header);
	  max_nr_of_rules = should_be_number ();
	  should_be_token (max_tuple_header);
	  max_nr_of_tuples = should_be_number ();
	  ruletable = (rule *) ckcalloc (max_nr_of_rules, sizeof (rule));
	  for (i = 0; i < max_nr_of_rules; i++) ruletable[i] = rule_nil;
	  x_pos = (int *) ckcalloc (max_nr_of_tuples, sizeof (int));
	  y_pos = (int *) ckcalloc (max_nr_of_tuples, sizeof (int));
	  used_nrs = (int *) ckcalloc (max_nr_of_tuples, sizeof (int));
	  tuple_edit_buffer = (char *) ckmalloc (max_nr_of_rules * MAXBUFLEN);
	  tuple_error_buffer = (char *) ckmalloc (MAXBUFLEN);
	};

private void write_header ()
	{ fprintf (output, "%s %d\n", max_rule_header, max_nr_of_rules);
	  fprintf (output, "%s %d\n", max_tuple_header, max_nr_of_tuples);
	};

/*
   Reading and writing of the tuples
*/
private void read_tuples ()
	{ init_reading ();
	  read_header ();
	  while (!eof) read_rule ();
	  finish_reading ();
	};

public void write_tuples ()
	{ int i;
	  init_writing ();
	  write_header ();
	  for (i = 0; i < max_nr_of_rules; i++) write_rule (i);
	  finish_writing ();
	};

public void init_tuples (char *gram)
	{ establish_tuplefile (gram);
	  read_tuples ();
	};
