/*
   File: propagate.c
   Defines the affix propagation mechanism
*/

/* global includes */
#include <stdio.h>
#include <string.h>

/* local includes */
#include <export.h>
#include <ds.h>
#include <trace.h>
#include <textstorage.h>
#include <textparsing.h>
#include <buildaffixgraph.h>
#include <propagate.h>

public valuenode calc_affix_value (posnode pos, int side)
	{ switch (pos -> sides[side].type)
	     { case singleaffix:
		  { valuenode v = pos -> sides[side].a.affx -> value;
		    attach_valuenode (v);
		    return (v);
		  };
	       case composaffix:
		  { valuenode v = new_valuenode ();
		    int nr = pos -> sides[side].a.co.nraffs;
		    affixnode *affs = pos -> sides[side].a.co.affs;
		    valuenode *vals = new_valuespace (nr);
		    int i;

		    v -> type = compostype;
		    v -> v.co.nr = nr;
		    v -> v.co.vals = vals;
		    for (i=0; i<nr; i++)
		       { valuenode val = affs[i] -> value;
			 attach_valuenode (val);
			 vals[i] = val;
		       };
		    return (v);
		  };
	       case concataffix:
		  { int nr = pos -> sides[side].a.co.nraffs;
		    affixnode *affs = pos -> sides[side].a.co.affs;
		    int type = affs[0] -> value -> type;
		    int i;

		    for (i=1; i <nr; i++)
		       if (affs[i] -> value -> type != type)
			  return (new_valuenode ());	/* is undefined */
		    switch (type)
		       { case stringtype:
			    { register char *dptr = strstore;
			      char *string;
			      for (i=0; i < nr; i++)
				 { register char *sptr =
					affs[i] -> value -> v.string;
				   while (*sptr) *dptr++ = *sptr++;
				 };
			      *dptr = '\0';
			      string = addto_names (strstore);
			      return (string_to_value (string));
			    };
			 case numbertype:
			    { int val = 0;
			      for (i=0; i < nr; i++)
				 val += affs[i] -> value -> v.number;
			      return (number_to_value (val));
			    };
			 default: return (new_valuenode ());
		       };
		  };
	     };
	  return (new_valuenode ());		/* undefined affix */
	};

public int equal_affix_value (valuenode v1, valuenode v2)
	{ if (v1 == v2) return (1);
	  if (v1 -> type != v2 -> type) return (0);
	  switch (v1 -> type)
	     { case stringtype:
		  return (strcmp (v1 -> v.string, v2 -> v.string) == 0);
	       case numbertype:
		  return (v1 -> v.number == v2 -> v.number);
	       case compostype:
		  { int i;
		    if (v1 -> v.co.nr != v2 -> v.co.nr) return (0);
		    for (i = 0; i < v1 -> v.co.nr; i++)
		       if (!equal_affix_value (v1 -> v.co.vals[i],
				v2 -> v.co.vals[i])) return (0);
		    return (1);
		  };
	     };
	  return (0);
	};

private void make_link_applied_and_propagate ()
	{ affixnode affx = popa ();
	  posnode pos = popp ();
	  linknode lk = make_link_applied (affx,pos);
	  pushp (pos);
	  pushq (propagate_affix_value);
	  callq ();
	  pop (2);
	  restore_link_to_undefined (pos,lk);
	  pushp (pos);
	  pusha (affx);
	  pushq (make_link_applied_and_propagate);
	};

private int connected_to_node_with_undefined_value (posnode pos, int side)
	{ int otherside = (side == lower_side)?upper_side:lower_side;
	  switch (pos -> sides [otherside].type)
	     { case singleaffix:
		  { affixnode afx = pos -> sides [otherside].a.affx;
		    return (afx -> defined && afx -> hasval &&
			    (afx -> value -> type == undefinedtype));
		  };
	       case concataffix:
	       case composaffix:
		  { affixnode *affxs = pos -> sides [otherside].a.co.affs;
		    int nr = pos -> sides [otherside].a.co.nraffs;
		    int i;
		    for (i=0; i < nr; i++)
		       { affixnode afx = affxs[i];
			 if (afx -> defined && afx -> hasval &&
				(afx -> value -> type == undefinedtype))
			    return (1);
		       };
		  };
	     };
	  return (0);
	};

private void move_over_undefined_affix_value ()
	{ affixnode affx = popa ();
	  posnode pos = popp ();
	  linknode ptr;
	  /* save qptr since we do not know how many items will be pushed */
	  cont *lqptr = qptr;
	  for (ptr = affx -> links; ptr != linknode_nil; ptr = ptr -> next)
	     if ((ptr -> type == undefined_link) ||
		 (((ptr -> type == applied_link) ||
		   (ptr -> type == first_defined_link)) &&
		  connected_to_node_with_undefined_value
			(ptr -> pos, ptr -> side)))
	        { pushp (ptr -> pos);
		  pusha (affx);
		  pushq (make_link_applied_and_propagate);
		};
	  if (pos -> delayed) pos -> dfunc (pos -> args);
	  else callq ();
	  qptr = lqptr;
	  pushp (pos);
	  pusha (affx);
	  pushq (move_over_undefined_affix_value);
	};

private void move_affix_value ()
	{ affixnode affx = popa ();
	  posnode pos = popp ();
	  linknode ptr;
	  /* save qptr since we do not know how many items will be pushed */
	  cont *lqptr = qptr;
	  for (ptr = affx -> links; ptr != linknode_nil; ptr = ptr -> next)
	     if (ptr -> type == undefined_link)
		{ pushp (ptr -> pos);
		  pusha (affx);
		  pushq (make_link_applied_and_propagate);
		};
	  if (pos -> delayed) pos -> dfunc (pos -> args);
	  else callq ();
	  qptr = lqptr;
	  pushp (pos);
	  pusha (affx);
	  pushq (move_affix_value);
	};

private void assignvalue ()
	{ valuenode val = popv();
	  posnode pos = popp ();
	  affixnode affx = popa ();
	  if (affx -> defined && affx -> hasval)
	     { if (affx -> value -> type == undefinedtype)
	          { valuenode lval = affx -> value;
		    linknode lk = make_link_first_defined (affx, pos);
		    affx -> value = val;
		    pushp (pos);
		    pusha (affx);
		    pushq (move_over_undefined_affix_value);
		    if (affx -> mfunc)
		       { pushv (val);
			 pushq (affx -> mfunc);
			 callq ();
			 pop (2);
		       }
		    else callq ();
		    pop (3);
		    affx -> value = lval;
		    restore_link_to_undefined (pos,lk);
		    change_link_type (lk, applied_link);
		  }
	       /* consistent substitution */
	       else if (equal_affix_value (val, affx -> value))
	 	  { linknode lk = make_link_defined (affx, pos);
	 	    callq ();
		    restore_link_to_undefined (pos, lk);
		  };
	     }
	  else 
	     { linknode lk = make_link_first_defined (affx, pos);
	       affx -> value = val;
	       attach_valuenode (val);
	       pushp (pos);
	       pusha (affx);
	       pushq (move_affix_value);	/* propagate further */
	       if (affx -> defined)
		  { /* check metadefinition */
		    affx -> hasval = 1;
		    pushv (val);
		    pushq (affx -> mfunc);
		    callq ();
		    pop (2);
		    affx -> hasval = 0;
		  }
	       else
		  { affx -> defined = 1;
		    affx -> hasval = 1;
		    callq ();
		    affx -> hasval = 0;
		    affx -> defined = 0;
		  };
	       pop (3);
	       detach_valuenode (val);
	       affx -> value = valuenode_nil;
	       restore_link_to_undefined (pos, lk);
	     };
	  pusha (affx);
	  pushp (pos);
	  pushv (val);
	  pushq (assignvalue);
	}

private void assign_undef_value ()
	{ valuenode val = popv ();
	  posnode pos = popp ();
	  affixnode affx = popa ();
	  if (affx -> defined)
	     { if (affx -> hasval) callq ();	/* undefined meets defined */
	       else
		  { linknode lk = make_link_first_defined (affx, pos);
		    affx -> hasval = 1;
		    affx -> value = val;
		    attach_valuenode (affx -> value);
		    pushp (pos);
		    pusha (affx);
		    pushq (move_affix_value);
		    callq ();	
		    pop (3);
		    detach_valuenode (affx -> value);
		    affx -> value = valuenode_nil;
		    affx -> hasval = 0;
		    restore_link_to_undefined (pos, lk);
		  };
	     }
	  else
	     { linknode lk = make_link_first_defined (affx, pos);
	       affx -> defined = 1;
	       affx -> hasval = 1;
	       affx -> value = val;
	       attach_valuenode (affx -> value);
	       pushp (pos);
	       pusha (affx);
	       pushq (move_affix_value);
	       callq ();
	       pop (3);
	       detach_valuenode (affx -> value);
	       affx -> hasval = 0;
	       affx -> defined = 0;
	       restore_link_to_undefined (pos, lk);
	     };
	  pusha (affx);
	  pushp (pos);
	  pushv (val);
	  pushq (assign_undef_value);
	};

private void assignsinglevalue (valuenode val, posnode pos, affixnode aff)
	{ pusha (aff);
	  pushp (pos);
	  pushv (val);
	  pushq (assignvalue);
	  callq ();
	  pop (4);
	};

private void assign_undef_singlevalue (valuenode val,
				       posnode pos, affixnode aff)
	{ pusha (aff);
	  pushp (pos);
	  pushv (val);
	  pushq (assign_undef_value);
	  callq ();
	  pop (4);
	};

private void assigncomposvalue (valuenode val, posnode pos,
			       int nr, affixnode *affs)
	{ int i;
	  cont *lqptr;
	  if (val -> type != compostype) return;
	  if (val -> v.co.nr != nr) return;	/* Mismatch in number */
	  lqptr = qptr;
	  for (i = 0; i < nr; i++)
	     { pusha (affs[i]);
	       pushp (pos);
	       pushv (val -> v.co.vals[i]);
	       pushq (assignvalue);
	     };
	  callq ();
	  qptr = lqptr;
	};

private void assignconcatstringvalue (char *string, posnode pos,
				      int nr, affixnode *affs, int from)
	{ valuenode ival;
	  int i;
	  if (from == nr - 1)
	     { ival = string_to_value (string);
	       pusha (affs[from]);
	       pushp (pos);
	       pushv (ival);
	       pushq (assignvalue);
	       callq ();
	       pop (4);
	       detach_valuenode (ival);
	     }
	  else
	     for (i=0; i <= strlen (string); i++)
	        { char *head, *tail;
		  strncpy (strstore, string, i);
		  strstore[i] = '\0';
		  head = addto_names (strstore);
		  tail = addto_names (string + i);
		  ival = string_to_value (head);
		  pusha (affs[from]);
		  pushp (pos);
		  pushv (ival);
		  pushq (assignvalue);
		  assignconcatstringvalue (tail, pos, nr, affs, from + 1);
		  pop (4);
		  detach_valuenode (ival);
		};
	};

private void assignconcatintvalue (int number, posnode pos,
				   int nr, affixnode *affs, int from)
	{ valuenode ival;
	  int i;
	  if (number < 0) return;	/* split only naturals */
	  if (from == nr - 1)
	     { ival = number_to_value (number);
	       pusha (affs[from]);
	       pushp (pos);
	       pushv (ival);
	       pushq (assignvalue);
	       callq ();
	       pop (4);
	       detach_valuenode (ival);
	     }
	  else
	     for (i=0; i <= number; i++)
		{ ival = number_to_value (i);
		  pusha (affs[from]);
		  pushp (pos);
		  pushv (ival);
		  pushq (assignvalue);
		  assignconcatintvalue (number - i, pos, nr, affs, from + 1);
		  pop (4);
		  detach_valuenode (ival);
		};
	};

private void assignconcatvalue (valuenode val, posnode pos,
				int nr, affixnode *affs)
	{ if (val -> type == compostype) return;
	  if (val -> type == stringtype)
	     assignconcatstringvalue (val -> v.string, pos, nr, affs, 0);
	  else
	     assignconcatintvalue (val -> v.number, pos, nr, affs, 0);
	};

private void assign_undef_values (valuenode val, posnode pos,
				  int nr, affixnode *affs)
	{ int i;
	  cont *lqptr = qptr;
	  for (i=0; i < nr; i++)
	     { pusha (affs[i]);
	       pushp (pos);
	       pushv (val);
	       pushq (assign_undef_value);
	     };
	  callq ();
	  qptr = lqptr;
	};

public void unparse_affix_value (valuenode val, posnode pos, int side)
	{ if (val -> type == undefinedtype)
	     { switch (pos -> sides[side].type)
		 { case singleaffix:
		      assign_undef_singlevalue (val, pos,
					pos -> sides[side].a.affx);
		      break;
		   case composaffix:
		   case concataffix:
		      assign_undef_values (val, pos,
					pos -> sides[side].a.co.nraffs,
					pos -> sides[side].a.co.affs);
		      break;
		 };
	     }
	  else 
	     { switch (pos -> sides[side].type)
	         { case singleaffix:
		      assignsinglevalue (val, pos, pos -> sides[side].a.affx);
		      break;
	           case composaffix:
		      assigncomposvalue (val, pos,
					pos -> sides[side].a.co.nraffs,
					pos -> sides[side].a.co.affs);
		      break;
	           case concataffix:
		      assignconcatvalue (val, pos,
					pos -> sides[side].a.co.nraffs,
					pos -> sides[side].a.co.affs);
		      break;
	         };
	     };
	};

/*
   Propagate affix value
   The value corresponding to an affix position is propagated
   when its sill drops to zero.
   Calling convention:
	pushp (pos);
	pushq (propagate_affix_value);
	callq ();
	pop (2);
*/
public void propagate_affix_value ()
	{ posnode pos = popp ();
	  if (tracing) trace_pos (pos);
	  if ((pos -> sides[lower_side].type != undefinedaffix) &&
	      (pos -> sides[upper_side].type != undefinedaffix))
	     { valuenode upper_val = valuenode_nil;
	       valuenode lower_val = valuenode_nil;
	       if (pos -> sides[upper_side].sill == 0)
		  upper_val = calc_affix_value (pos, upper_side);
	       if (pos -> sides[lower_side].sill == 0)
		  lower_val = calc_affix_value (pos, lower_side);
	       if ((lower_val == valuenode_nil) &&
		   (upper_val == valuenode_nil))
		  callq ();
	       else if (upper_val == valuenode_nil)
		  unparse_affix_value (lower_val, pos, upper_side);
	       else if (lower_val == valuenode_nil)
		  unparse_affix_value (upper_val, pos, lower_side);
	       else if ((lower_val -> type != undefinedtype) &&
			(upper_val -> type == undefinedtype))
		  unparse_affix_value (lower_val, pos, upper_side);
	       else if ((lower_val -> type == undefinedtype) &&
			(upper_val -> type != undefinedtype))
		  unparse_affix_value (upper_val, pos, lower_side);
	       else if (equal_affix_value (lower_val, upper_val))
		  callq ();
	       detach_valuenode (upper_val);
	       detach_valuenode (lower_val);
	     }
	  else callq();
	  pushp (pos);
	  pushq (propagate_affix_value);
	};

/*
   Propagate values calculated by standard predicates.
   Calling convention:
	pushp (pos);
	pushv (value);
	pushq (propagate_predicate_value);
	callq ();
	pop (3);
*/
public void propagate_predicate_value ()
	{ valuenode val = popv ();
	  posnode pos = popp ();
	  affixnode affx = pos -> sides[lower_side].a.affx;
	  if (affx -> hasval == 0)
	     { linknode lk = make_link_first_defined (affx, pos);
	       affx -> value = val;
	       affx -> defined = 1;
	       affx -> hasval = 1;
	       pushp (pos);
	       pushq (propagate_affix_value);
	       callq ();
	       pop (2);
	       affx -> hasval = 0;
	       affx -> defined = 0;
	       affx -> value = valuenode_nil;
	       restore_link_to_undefined (pos, lk);
	     }
	  else if (equal_affix_value (val, affx -> value)) callq ();
	  pushp (pos);
	  pushv (val);
	  pushq (propagate_predicate_value);
	};

public void make_node_delayed ()
	{ int i;
	  void (*p)() = popq();
	  treenode top = top_treenode();
	  posnode *ps = top -> affs;
	  for (i = 0; i < top -> nraffs; i++)
	     { ps[i] -> delayed = 1;
	       ps[i] -> args = ps;
	       ps[i] -> dfunc = p;
	     };
	  p (ps);
	  for (i = 0; i < top -> nraffs; i++)
	     { ps[i] -> delayed = 0;
	       ps[i] -> args = posarray_nil;
	       ps[i] -> dfunc = NULL;
	     };
	  pushq (p);
	  pushq (make_node_delayed);
	};
