/*
   File: ds.c
   Keeps track of the incore datastructures
*/

/* global includes */
#include <stdio.h>

/* local includes */
#include <memalloc.h>
#include <ds.h>

/*
   Manage the tree stack
*/
public treenode *tptr;
private treenode *tstack;
public int tstack_size = 10000;

private void init_tstack ()
	{ tstack = (treenode *) ckcalloc (tstack_size, sizeof (treenode));
	  tptr = tstack;
	};

/*
   Memory management routines
*/

/* management of tree nodes */
private treenode free_treenode_list;
public treenode new_treenode ()
	{ treenode new;
	  if (free_treenode_list == treenode_nil)
	     { new = (treenode) ckmalloc (sizeof (struct tree_rec));
	     }
	  else
	     { new = free_treenode_list;
	       free_treenode_list = free_treenode_list -> father;
	     };
	  new -> type = undefinednode;
	  new -> name = "";
	  new -> nodenr = 0;
	  new -> nrsons = 0;
	  new -> father = treenode_nil;
	  new -> nraffs = 0;
	  new -> copied = treenode_nil;
	  return (new);
	};

public void free_treenode (treenode old)
	{ old -> father = free_treenode_list;
	  free_treenode_list = old;
	};

/* management of arrays of tree nodes */
#define MAXMANAGEDSONS 40
private treenode *free_son_list[MAXMANAGEDSONS];
public treenode *new_sonspace (int nrsons)
	{ int i;
	  treenode *new;
	  if (nrsons == 0) return (treearray_nil);

	  if ((nrsons > MAXMANAGEDSONS) ||
	      (free_son_list[nrsons-1] == treearray_nil))
	     { new = (treenode *) ckcalloc (nrsons, sizeof (treenode));
	     }
	  else
	     { new = free_son_list[nrsons-1];
	       free_son_list[nrsons-1] =
			(treenode *)(free_son_list[nrsons-1][0]);
	     };
	  for (i=0; i < nrsons; i++) new[i] = treenode_nil;
	  return (new);
	};

public void free_sonspace (int nrsons, treenode *old)
	{ if (nrsons == 0) return;
	  if (nrsons > MAXMANAGEDSONS)
	     { free ((char *) old);
	     }
	  else
	     { old[0] = (treenode) free_son_list[nrsons-1];
	       free_son_list[nrsons-1] = old;
	     };
	};

/* management of affix positions */
private posnode free_posnode_list;
public posnode new_posnode ()
	{ posnode new;
	  if (free_posnode_list == posnode_nil)
	     { new = (posnode) ckmalloc (sizeof (struct pos_rec));
	     }
	  else
	     { new = free_posnode_list;
	       free_posnode_list = (posnode) free_posnode_list -> node;
	     };
	  new -> node = treenode_nil;
	  new -> sides[0].type = undefinedaffix;
	  new -> sides[1].type = undefinedaffix;
	  new -> delayed = 0;
	  new -> args = posarray_nil;
	  new -> dfunc = NULL;
	  return (new);
	};

public void free_posnode (posnode old)
	{ old -> node = (treenode) free_posnode_list;
	  free_posnode_list = old;
	}; 

/* management of affix position arrays */
#define MAXMANAGEDPOS 40
private posnode *free_posspace_list[MAXMANAGEDPOS];
public posnode *new_posspace (int nrofps)
	{ int i;
	  posnode *new;
	  if (nrofps == 0) return (posarray_nil);

	  if ((nrofps > MAXMANAGEDPOS) ||
	      (free_posspace_list[nrofps-1] == posarray_nil))
	     { new = (posnode *) ckcalloc (nrofps, sizeof (posnode));
	     }
	  else
	     { new = free_posspace_list[nrofps-1];
	       free_posspace_list[nrofps-1] =
			(posnode *)(free_posspace_list[nrofps-1][0]);
	     };
	  for (i=0; i < nrofps; i++) new[i] = posnode_nil;
	  return (new);
	};

public void free_posspace (int nrofps, posnode *old)
	{ if (nrofps == 0) return;
	  if (nrofps > MAXMANAGEDPOS)
	     { free ((char *) old);
	     }
	  else
	     { old[0] = (posnode) free_posspace_list[nrofps-1];
	       free_posspace_list[nrofps-1] = old;
	     };
	};

/* management of values */
private valuenode free_valuenode_list;
public valuenode new_valuenode ()
	{ valuenode new;
	  if (free_valuenode_list == valuenode_nil)
	     { new = (valuenode) ckmalloc (sizeof (struct value_rec));
	     }
	  else
	     { if ((int) free_valuenode_list == 1)
		  { new = (valuenode) ckmalloc (sizeof (struct value_rec));
		    free_valuenode_list = valuenode_nil;
	          }
	       else
		  { new = free_valuenode_list;
	            free_valuenode_list =
			(valuenode) free_valuenode_list -> v.string;
		  };
	     };
	  new -> type = undefinedtype;
	  new -> ref_count = 1;
	  return (new);
	};

public void free_valuenode (valuenode old)
	{ old -> v.string = (char *) free_valuenode_list;
	  free_valuenode_list = old;
	};

public void detach_valuenode (valuenode old)
	{ if (old == valuenode_nil) return;
	  old -> ref_count--;
	  if (old -> ref_count == 0)
	     { if (old -> type == compostype)
		  { int i;
		    for (i = 0; i < old -> v.co.nr; i++)
		       detach_valuenode (old -> v.co.vals[i]);
		    free_valuespace (old -> v.co.nr, old -> v.co.vals);
		  };
	       free_valuenode (old);
	     };
	};

/* management of value arrays */
#define MAXMANAGEDVALUES 40
private valuenode *free_valuespace_list[MAXMANAGEDVALUES];
public valuenode *new_valuespace (int nrofvs)
	{ int i;
	  valuenode *new;
	  if (nrofvs == 0) return (valuearray_nil);

	  if ((nrofvs > MAXMANAGEDVALUES) ||
	      (free_valuespace_list[nrofvs-1] == valuearray_nil))
	     { new = (valuenode *) ckcalloc (nrofvs, sizeof (valuenode));
	     }
	  else
	     { new = free_valuespace_list[nrofvs-1];
	       free_valuespace_list[nrofvs-1] =
			(valuenode *)(free_valuespace_list[nrofvs-1][0]);
	     };
	  for (i=0; i < nrofvs; i++) new[i] = valuenode_nil;
	  return (new);
	};

public void free_valuespace (int nrofvs, valuenode *old)
	{ if (nrofvs == 0) return;
	  if (nrofvs > MAXMANAGEDVALUES)
	     { free ((char *) old);
	     }
	  else
	     { old[0] = (valuenode) free_valuespace_list[nrofvs-1];
	       free_valuespace_list[nrofvs-1] = old;
	     };
	};

/* management of affixes */
private affixnode free_affixnode_list;
public affixnode new_affixnode (char *name)
	{ affixnode new;
	  if (free_affixnode_list == affixnode_nil)
	     { new = (affixnode) ckmalloc (sizeof (struct affix_rec));
	     }
	  else
	     { new = free_affixnode_list;
	       free_affixnode_list = free_affixnode_list -> copied;
	     };
	  new -> name = name;
	  new -> defined = 0;
	  new -> hasval = 0;
	  new -> value = valuenode_nil;
	  new -> mfunc = NULL;
	  new -> links = linknode_nil;
	  new -> copied = affixnode_nil;
	  return (new);
	};

public void free_affixnode (affixnode old)
	{ old -> copied = free_affixnode_list;
	  free_affixnode_list = old;
	};

/* management of affix arrays */
#define MAXMANAGEDAFFIXES 40
private affixnode *free_affixspace_list[MAXMANAGEDAFFIXES];
public affixnode *new_affixspace (int nraffs)
	{ int i;
	  affixnode *new;
	  if (nraffs == 0) return (affixarray_nil);

	  if ((nraffs > MAXMANAGEDAFFIXES) ||
	      (free_affixspace_list[nraffs-1] == affixarray_nil))
	     { new = (affixnode *) ckcalloc (nraffs, sizeof (affixnode));
	     }
	  else
	     { new = free_affixspace_list[nraffs-1];
	       free_affixspace_list[nraffs-1] =
			(affixnode *)(free_affixspace_list[nraffs-1][0]);
	     };
	  for (i=0; i < nraffs; i++) new[i] = affixnode_nil;
	  return (new);
	};

public void free_affixspace (int nraffs, affixnode *old)
	{ if (nraffs == 0) return;
	  if (nraffs > MAXMANAGEDAFFIXES)
	     { free ((char *) old);
	     }
	  else
	     { old[0] = (affixnode) free_affixspace_list[nraffs-1];
	       free_affixspace_list[nraffs-1] = old;
	     };
	};

/* management of links */
private linknode free_linknode_list;
public linknode new_linknode ()
	{ linknode new;
	  if (free_linknode_list == linknode_nil)
	     { new = (linknode) ckmalloc (sizeof (struct link_rec));
	     }
	  else
	     { new = free_linknode_list;
	       free_linknode_list = free_linknode_list -> next;
	     };
	  new -> node = treenode_nil;
	  new -> pos = posnode_nil;
	  new -> side = 0;
	  new -> type = undefined_link;
	  new -> next = linknode_nil;
	  return (new);
	};

public void free_linknode (linknode old)
	{ old -> next = free_linknode_list;
	  free_linknode_list = old;
	};

/*
   Manage the continuation stack
*/
public cont *qptr;
private cont *qstack;
public int qstack_size = 100000;

private void init_qstack ()
	{ qstack = (cont *) ckcalloc (qstack_size, sizeof (cont));
	  qptr = qstack;	/* can others initialize qptr? */
	};

public void reinit_ds ()
	{ tptr = tstack;
	  qptr = qstack;
	};

public void dump_affixvalue_on_file (valuenode v, FILE *out)
	{ if (v == valuenode_nil) fputs ("<value_nil>", out);
	  else switch (v -> type)
	     { case undefinedtype: break;
	       case stringtype: fputs (v -> v.string, out); break;
	       case compostype: 
		  { int i;
		    fputc ('<', out);
		    for (i=0; i < v -> v.co.nr; i++)
		       { dump_affixvalue_on_file (v -> v.co.vals[i], out);
			 if (i != v -> v.co.nr - 1) fputs (" * ", out);
		       };
		    fputc ('>', out);
		  };
		  break;
	       case numbertype: fprintf (out, "%d", v -> v.number); break;
	     };
	};

public void output_affix ()
	{ affixnode affx = popa();
	  dump_affixvalue_on_file (affx -> value, stdout);
	  callq ();
	  pusha (affx);
	  pushq (output_affix);
	};

public void init_ds ()
	{ int i;
	  init_tstack ();
	  init_qstack ();
	  free_treenode_list = treenode_nil;
	  for (i=0; i<MAXMANAGEDSONS; i++) free_son_list[i] = treearray_nil;
	  free_posnode_list = posnode_nil;
	  for (i=0; i<MAXMANAGEDPOS; i++) free_posspace_list[i] = posarray_nil;
	  free_valuenode_list = valuenode_nil;
	  for (i=0; i<MAXMANAGEDVALUES; i++)
	     free_valuespace_list[i] = valuearray_nil;
	  free_affixnode_list = affixnode_nil;
	  for (i=0; i<MAXMANAGEDAFFIXES; i++)
	     free_affixspace_list[i] = affixarray_nil;
	  free_linknode_list = linknode_nil;
	};
