/*
   File: buildtree.c
   Defines the actions necessary for building the syntax tree
   as well as the linkage to the affix tree
*/

/* global includes */
#include <stdio.h>

/* local includes */
#include <export.h>
#include <error.h>
#include <ds.h>
#include <textparsing.h>
#include <textstorage.h>
#include <buildaffixgraph.h>
#include <propagate.h>
#include <nodenrs.h>
#include <buildtree.h>

/*
   make_affix_link will build the upper side affix expressions
   corresponding to applications of syntax rules (i.e. right
   hand sides). For every position it will pop a representation of the
   corresponding affix expression from the q stack, which it will build
   at the upper side of the position the corresponding affix graph.
   After building it will try and propagate the affixes through
   the affix graph. 
*/
public void make_affix_link ()
	{ int i;
	  cont *lqptr;
	  int nr = popi();		/* redundant wegens topnode? */
	  treenode topnode = top_treenode();
	  posnode *save_ps = new_posspace (nr);
	  for (i=0; i<nr; i++)
	     { posnode save_pos = new_posnode ();
	       posnode old_pos = topnode -> affs[i];
	       int type;
	       save_pos -> sides[upper_side].type =
			old_pos -> sides[upper_side].type;
	       save_pos -> sides[upper_side].sill =
			old_pos -> sides[upper_side].sill;
	       save_pos -> sides[upper_side].a = old_pos -> sides[upper_side].a;
	       save_ps[i] = save_pos;
	       type = popi();
	       old_pos -> sides[upper_side].type = type;
	       if (type == singleaffix)
		  { affixnode affxs = popa();
		    old_pos -> sides[upper_side].sill = 1;
		    old_pos -> sides[upper_side].a.affx = affxs;
		    add_link (affxs,old_pos,upper_side);
		  }
	       else
		  { int nras = popi ();
		    affixnode *affs = new_affixspace (nras);
		    int j;
		    old_pos -> sides[upper_side].sill = nras;
		    old_pos -> sides[upper_side].a.co.nraffs = nras;
		    old_pos -> sides[upper_side].a.co.affs = affs;
		    for (j=0; j < nras; j++)
		       { affixnode affx = popa();
			 affs[j] = affx;
			 add_link (affx,old_pos,upper_side);
		       };
		  };
	     };
	  lqptr = qptr;
	  for (i=0; i<nr; i++)
	     { pushp (topnode -> affs[i]);
	       pushq (propagate_affix_value);
	     };
	  callq ();
	  qptr = lqptr;
	  for (i=nr - 1; 0 <= i; i--)
	     { posnode save_pos = save_ps[i];
	       posnode old_pos = topnode -> affs[i];
	       int type = old_pos -> sides[upper_side].type;
	       if (type == singleaffix)
		  { affixnode affxs = old_pos -> sides[upper_side].a.affx;
		    delete_link (affxs, old_pos);
		    pusha (affxs);
		  }
	       else
		  { int nras = old_pos -> sides[upper_side].a.co.nraffs;
		    affixnode *affs = old_pos -> sides[upper_side].a.co.affs;
		    int j;
		    for (j=nras - 1; 0 <= j; j--)
		        { affixnode affx = affs[j];
			  delete_link (affx, old_pos);
			  pusha (affx);
			};
		    free_affixspace (nras, affs);
		    pushi (nras);
		  };
	       pushi (type);
	       old_pos -> sides[upper_side].type =
			save_pos -> sides[upper_side].type;
	       old_pos -> sides[upper_side].sill =
			save_pos -> sides[upper_side].sill;
	       old_pos -> sides[upper_side].a = save_pos -> sides[upper_side].a;
	       free_posnode (save_pos);
	     };
	  free_posspace (nr, save_ps);
	  pushi (nr);
	  pushq (make_affix_link);
	};

/*
   make_affix_pos builds the lower side affix positions corresponding
   to the definition of a syntax node (i.e. its left hand side)
   For every position it will pop a representation of the corresponding
   affix expression from the q stack, which it will build at the lower
   side of the position the corresponding affix graph
*/
private void make_affix_pos (treenode node, int nrps)
	{ int nr;
	  posnode *ps = new_posspace (nrps);
	  node -> nraffs = nrps;
	  node -> affs = ps;

	  for (nr = 0; nr < nrps; nr++)
	     { posnode pos = new_posnode ();
	       int type = popi();
	       ps[nr] = pos;
	       pos -> node = node;
	       pos -> sides[lower_side].type = type;
	       if (type == singleaffix)
		  { /* add link will decrease the sill if possible */
		    affixnode affx = popa();
		    pos -> sides[lower_side].sill = 1;
		    pos -> sides[lower_side].a.affx = affx;
		    add_link (affx, pos, lower_side);
		  }
	       else
		  { int i;
		    int nras = popi();
		    affixnode *affs = new_affixspace (nras);
		    pos -> sides[lower_side].sill = nras;
		    pos -> sides[lower_side].a.co.nraffs = nras;
		    pos -> sides[lower_side].a.co.affs = affs;
		    for (i = 0; i < nras; i++)
		       { affixnode affx = popa();
			 affs[i] = affx;
			 add_link (affx, pos, lower_side);
		       };
		  };
	     };
	};

/*
   clear affix pos will undo the actions of make affix pos
*/
private void clear_affix_pos (treenode node, int nrps)
	{ int nr;
	  posnode *ps = node -> affs;
	  for (nr = 0; nr < nrps; nr++)
	     { posnode pos = ps[nr];
	       int type = pos -> sides[lower_side].type;
	       if (type == singleaffix)
		  { affixnode affx = pos -> sides[lower_side].a.affx;
		    delete_link (affx, pos);
		    pusha (affx);
		  }
	       else
		  { int i;
		    int nras = pos -> sides[lower_side].a.co.nraffs;
		    affixnode *affs = pos -> sides[lower_side].a.co.affs;
		    for (i=0; i < nras; i++)
		       { affixnode affx = affs[i];
			 delete_link (affx, pos);
			 pusha (affx);
		       };
		    free_affixspace (nras, affs);
		    pushi (nras);
		  };
	       pushi (type);
	       free_posnode (pos);
	     };
	  free_posspace (nrps, ps);
	};

/*
   leaf nodes are used to administer the values delivered by parsing
   semi terminals: they always have one affix position
*/
public void make_leafnode ()
	{ treenode new = new_treenode ();
	  char *string = addto_names (strstore);
	  posnode *ps = new_posspace (1);
	  posnode newpos = new_posnode ();
	  affixnode aff = string_to_affix ("predef_leaf", string); 

	  new -> type = leafnode;
	  new -> name = string;
	  new -> nodenr = 1;
	  new -> nrsons = 0;
	  new -> nraffs = 1;
	  new -> affs = ps;
	  ps[0] = newpos;
	  newpos -> node = new;
	  newpos -> sides[lower_side].type = singleaffix;
	  /* add_link will decrease the sill! */
	  newpos -> sides[lower_side].sill = 1;
	  newpos -> sides[lower_side].a.affx = aff;
	  newpos -> delayed = 0;
	  add_link (aff, newpos, lower_side);	/* can be optimized */
	  push_treenode (new);
	  callq ();
	  (void) pop_treenode ();
	  delete_link (aff, newpos);
	  free_posnode (newpos);
	  free_posspace (1, ps);
	  detach_valuenode (aff -> value);
	  free_affixnode (aff);
	  free_treenode (new);
	  pushq (make_leafnode);
	};

/*
   make typed node will create a new node for the syntax tree and
   attribute it with the necessary affixes. After backup call it
   will undo any action necessary
*/
private void make_typed_node (int type)
	{ treenode new = new_treenode ();
	  int nrsons;
	  int nrps;

	  new -> type = type;
	  new -> nodenr = popi();
	  new -> name = name_from_nodenr (new -> nodenr);
	  nrsons = popi();
	  new -> nrsons = nrsons;
	  new -> sons = new_sonspace (nrsons);
	  nrps = popi ();
	  make_affix_pos (new, nrps);
	  push_treenode (new);
	  callq ();
	  (void) pop_treenode ();
	  clear_affix_pos (new, nrps);
	  pushi (nrps);
	  free_sonspace (nrsons, new -> sons);
	  pushi (nrsons);
	  pushi (new -> nodenr);
	  free_treenode (new);
	};

public void make_normalnode ()
	{ make_typed_node (normalnode);
	  pushq (make_normalnode);
	};

public void make_predicatenode ()
	{ make_typed_node (predicatenode);
	  pushq (make_predicatenode);
	};

public void make_semipredicatenode ()
	{ make_typed_node (semipredicatenode);
	  pushq (make_semipredicatenode);
	};

/*
   make simple node will create a new (normal) node for the syntax tree.
   It is intended to build those tree nodes that have zero affixes.
   After backup call it will undo any action necessary.
*/
public void make_simplenode ()
	{ treenode new = new_treenode ();
	  int nrsons;
	  new -> type = normalnode;
	  new -> nodenr = popi();
	  new -> name = name_from_nodenr (new -> nodenr);
	  nrsons = popi();
	  new -> nrsons = nrsons;
	  new -> sons = new_sonspace (nrsons);
	  push_treenode (new);
	  callq ();
	  (void) pop_treenode ();
	  free_sonspace (nrsons, new -> sons);
	  pushi (nrsons);
	  pushi (new -> nodenr);
	  free_treenode (new);
	  pushq (make_simplenode);
	};

private void make_placeholder_node (int type)
	{ treenode new = new_treenode ();
	  int nrps;

	  new -> type = type;
	  new -> nodenr = popi();
	  new -> name = name_from_nodenr (new -> nodenr);
	  new -> nrsons = 0;
	  new -> sons = treearray_nil;
	  nrps = popi ();
	  make_affix_pos (new, nrps);
	  push_treenode (new);
	  callq ();
	  (void) pop_treenode ();
	  clear_affix_pos (new, nrps);
	  pushi (nrps);
	  pushi (new -> nodenr);
	  free_treenode (new);
	};

public void make_typedplaceholdernode ()
	{ make_placeholder_node (typedplaceholdernode);
	  pushq (make_typedplaceholdernode);
	};

public void make_untypedplaceholdernode ()
	{ make_placeholder_node (untypedplaceholdernode);
	  pushq (make_untypedplaceholdernode);
	};

public void update_predicatenode ()
	{ int nodenr = popi();
	  int nrsons = popi();
	  posnode *args = poppp();
	  treenode node = args[0] -> node;
	  int oldnodenr = node -> nodenr;
	  int nraffs = popi();
	  posnode *save_pafs;
	  cont *lqptr;
	  int i;

	  node -> nodenr = nodenr;
	  node -> nrsons = nrsons;
	  node -> sons = new_sonspace (nrsons);
	  save_pafs = new_posspace (nraffs);
	  for (i=0; i<nraffs; i++)
	     { posnode save_pos = new_posnode ();
	       posnode old_pos = args[i];
	       int type;
	       save_pos -> sides[lower_side].type =
			old_pos -> sides[lower_side].type;
	       save_pos -> sides[lower_side].sill =
			old_pos -> sides[lower_side].sill;
	       save_pos -> sides[lower_side].a = old_pos -> sides[lower_side].a;
	       save_pafs[i] = save_pos;
	       type = popi();
	       old_pos -> sides[lower_side].type = type;
	       if (type == singleaffix)
		  { affixnode affxs = popa();
		    old_pos -> sides[lower_side].sill = 1;
		    old_pos -> sides[lower_side].a.affx = affxs;
		    add_link (affxs,old_pos,lower_side);
		  }
	       else
		  { int nras = popi ();
		    affixnode *affs = new_affixspace (nras);
		    int j;
		    old_pos -> sides[lower_side].sill = nras;
		    old_pos -> sides[lower_side].a.co.nraffs = nras;
		    old_pos -> sides[lower_side].a.co.affs = affs;
		    for (j=0; j < nras; j++)
		       { affixnode affx = popa();
			 affs[j] = affx;
			 add_link (affx,old_pos,lower_side);
		       };
		  };
	     };
	  lqptr = qptr;
	  for (i=0; i<nraffs; i++)
	     { pushp (args[i]);
	       pushq (propagate_affix_value);
	     };
	  callq ();
	  qptr = lqptr;
	  for (i=nraffs - 1; 0 <= i; i--)
	     { posnode save_pos = save_pafs[i];
	       posnode old_pos = args[i];
	       int type = old_pos -> sides[lower_side].type;
	       if (type == singleaffix)
		  { affixnode affxs = old_pos -> sides[lower_side].a.affx;
		    delete_link (affxs, old_pos);
		    pusha (affxs);
		  }
	       else
		  { int nras = old_pos -> sides[lower_side].a.co.nraffs;
		    affixnode *affs = old_pos -> sides[lower_side].a.co.affs;
		    int j;
		    for (j=nras - 1; 0 <= j; j--)
		        { affixnode affx = affs[j];
			  delete_link (affx, old_pos);
			  pusha (affx);
			};
		    free_affixspace (nras, affs);
		    pushi (nras);
		  };
	       pushi (type);
	       old_pos -> sides[lower_side].type =
			save_pos -> sides[lower_side].type;
	       old_pos -> sides[lower_side].sill =
			save_pos -> sides[lower_side].sill;
	       old_pos -> sides[lower_side].a = save_pos -> sides[lower_side].a;
	       free_posnode (save_pos);
	     };
	  free_posspace (nraffs, save_pafs);
	  free_sonspace (nrsons, node -> sons);
	  node -> sons = treearray_nil;
	  node -> nrsons = 0;
	  node -> nodenr = oldnodenr;
	  pushi (nraffs);
	  pushpp (args);
	  pushi (nrsons);
	  pushi (nodenr);
	  pushq (update_predicatenode);
	};

/*
   exchange tops of tree stacks
*/
public void exchange_top ()
	{ treenode top = pop_treenode ();
	  treenode top2 = pop_treenode ();
	  push_treenode (top);
	  push_treenode (top2);
	  callq ();
	  tptr -= 2;
	  push_treenode (top2);
	  push_treenode (top);
	  pushq (exchange_top);
	};
 
/*
   link son will pop a son from the tree stack and insert him
   as son of the top node of the tree stack. care is taken to
   undo any necessary action after backup.
*/
public void link_son ()
	{ treenode son;
	  int nr = popi();
	  son = pop_treenode ();
	  if (nr != 0)
	     { treenode father = top_treenode ();
	       treenode save_son = father -> sons[nr-1];
	       treenode save_father = son -> father;
	       father -> sons[nr-1] = son;
	       son -> father = father;
	       callq();
	       son -> father = save_father;
	       father -> sons[nr-1] = save_son;
	     }
	  else callq();
	  push_treenode (son);
	  pushi (nr);
	  pushq (link_son);
	};

public void link_predicateson ()
	{ treenode son;
	  treenode father = popt();
	  int nr = popi();
	  son = pop_treenode ();
	  if (nr != 0)
	     { treenode save_son = father -> sons[nr-1];
	       treenode save_father = son -> father;
	       father -> sons[nr-1] = son;
	       son -> father = father;
	       callq();
	       son -> father = save_father;
	       father -> sons[nr-1] = save_son;
	     }
	  else callq();
	  push_treenode (son);
	  pushi (nr);
	  pusht (father);
	  pushq (link_predicateson);
	};

public void dump_parse_tree_indented (treenode tnode, int indent)
	{ int i;
	  if (tnode == treenode_nil) return;	/* layout? */
	  if ((tnode -> type != normalnode) &&
	      (tnode -> type != ambiguousnode) &&
	      (tnode -> type != typedplaceholdernode) &&
	      (tnode -> type != untypedplaceholdernode)) return;
	  for (i=0; i < indent; i++) eprint_log ("%c", ' ');
	  eprint_log ("%s%s (%d)\n", 
			(tnode -> type == ambiguousnode)?"ambi: ":"",
			tnode -> name, tnode -> nodenr);
	  for (i=0; i < tnode -> nrsons; i++)
	     dump_parse_tree_indented (tnode -> sons[i], indent+1);
	};

public void dump_parse_tree ()
	{ wlog ("\nDump of parse tree:");
	  dump_parse_tree_indented (top_treenode (),0);
	  callq ();
	  pushq (dump_parse_tree);
	};

public void dummy_continuation ()
	{ pushq (dummy_continuation);
	};
