/*
   File: typecheck.c
   Type checks the syntax rules
*/

/* Global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* libeag includes */
#include <export.h>
#include <error.h>

/* local includes */
#include <sizes.h>
#include <tree.h>
#include <numbering.h>
#include <common.h>
#include <typecheck.h>
#include <main.h>

/*
   error administration
*/
private int typing_errors;
private void init_typecheck ()
	{ typing_errors = 0;
	};

private void type_error (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);

	  typing_errors++;
	  error ("typing error: %s", buf);
	};

/*
   initial typecheck
*/
public char* string_from_type (int t)
	{ switch (t)
	     { case undefined_type: return ("undefined");
	       case string_type: return ("string");
	       case integer_type: return ("int");
	       case integer_or_string_type: return ("int or string");
	       case tuple_type: return ("tuple");
	       case any_type: return ("any");
	     };
	  return ("error");
	};

/*
   Type coercion
*/
#define conflict_type -1
private int try_coerce_type (int demanded_type, int offered_type)
	{ if (demanded_type == undefined_type) return (offered_type);
	  if (offered_type == undefined_type) return (demanded_type);
	  if (offered_type == error_type) return (error_type);
	  if (demanded_type == error_type) return (error_type);
	  if (demanded_type == offered_type) return (demanded_type);
	  if (offered_type == any_type) return (conflict_type);
	  if (offered_type & demanded_type)
	     return (offered_type & demanded_type);
	  return (conflict_type);
	};

private int coerce_type (hyper_rule rule, int i,
			 int demanded_type, int offered_type)
	{ int result_type = try_coerce_type (demanded_type, offered_type);
	  if (result_type == conflict_type)
	     { type_error (
		   "in rule %s, alt %i: cannot coerce type %s to type %s",
		   rule -> nonterminal, i + 1, string_from_type (offered_type),
		   string_from_type (demanded_type));
	       return (error_type);
	     };
	  return (result_type);
	};

/*
   Type balancing
*/
private int try_balance_type (int type1, int type2)
	{ if (type1 == undefined_type) return (type2);
	  if (type2 == undefined_type) return (type1);
	  if (type2 == error_type) return (error_type);
	  if (type1 == error_type) return (error_type);
	  if (type1 == type2) return (type1);
	  if ((type1 == integer_or_string_type) &&
	      ((type2 == string_type) || (type2 == integer_type)))
	     return (type2);
	  if ((type2 == integer_or_string_type) &&
	      ((type1 == string_type) || (type1 == integer_type)))
	     return (type1);
	  return (conflict_type);
	};

private int balance_type (hyper_rule rule, int i,
			  int type1, int type2)
	{ int result_type = try_balance_type (type1, type2);
	  if (result_type == conflict_type)
	     { type_error (
		  "in rule %s, alt %i: unbalance between type %s and type %s",
		  rule -> nonterminal, i + 1, string_from_type (type1),
		  string_from_type (type2));
	        return (error_type);
	     };
	  return (result_type);
	};

/*
   affix analysis
*/
private int change;
private int typecheck_affix_variable (hyper_rule rule, int i, affix a,
				      int demanded_type)
	{ int result_type;
	  meta_rule mrule = a -> u.var -> def;
	  if ((a -> type == undefined_type) && (mrule != meta_rule_nil))
	     { a -> type = mrule -> type;
	       change = 1;
	     };
	  result_type = try_coerce_type (demanded_type, a -> type);
	  if (result_type == conflict_type)
	     { type_error (
	   "in rule %s, alt %d: cannot coerce type of affix %s from %s to %s",
			rule -> nonterminal, i + 1, a -> name,
			string_from_type (a -> type),
			string_from_type (demanded_type));
	       a -> type = error_type;
	       change = 1;
	       return (error_type);
	     }
	  else if (a -> type != result_type)
	     { a -> type = result_type;
	       change = 1;
	     };
	  return (result_type);
	};

private int typecheck_affix (hyper_rule rule, int i, affix a,
			     int demanded_type)
	{ int affix_type;
	  switch (a -> tag)
	     { case tag_affix_nonterminal:
		  return (typecheck_affix_variable (rule, i, a, demanded_type));
	       case tag_affix_terminal: affix_type = string_type; break;
	       case tag_affix_number: affix_type = integer_type; break;
	       case tag_affix_set: affix_type = string_type; break;
	       default: affix_type = undefined_type;
	     };
	  a -> type = affix_type;
	  return (coerce_type (rule, i, demanded_type, affix_type));
	};

/*
   typechecking expressions
*/
private int typecheck_compos (hyper_rule rule, int i, affix_list affl,
			      int demanded_type)
	{ int j, result_type;
	  for (j=0; j < affl -> nrofas; j++)
	     (void) typecheck_affix (rule, i, affl -> as[j], undefined_type);
	  result_type = coerce_type (rule, i, demanded_type, tuple_type);
	  return (result_type);
	};

private int typecheck_concat (hyper_rule rule, int i, affix_list affl,
			      int demanded_type)
	{ int j;
	  int demanded_type2;
	  int result_type = undefined_type;

	  demanded_type2 = coerce_type (rule, i,
				demanded_type, integer_or_string_type);
	  for (j=0; j < affl -> nrofas; j++)
	     { result_type = typecheck_affix (rule, i, affl -> as[j],
				demanded_type2);
	       if (result_type != demanded_type2) break;
	     };
	  for (j=0; j < affl -> nrofas; j++)
	     { (void) typecheck_affix (rule, i, affl -> as[j], result_type);
	     };
	  return (result_type);
	};

/*
   type checking positions
*/
private int typecheck_position (hyper_rule rule, int i, pos p,
				int demanded_type)
	{ int ctype, dtype, result_type;
	  expr ex = p -> ex;

	  dtype = balance_type (rule, i, demanded_type, p -> type);
	  switch (ex -> tag)
	     { case tag_single:
		  ctype = typecheck_affix (rule, i, ex -> u.single, dtype);
		  break;
	       case tag_compos:
		  ctype = typecheck_compos (rule, i, ex -> u.compos, dtype);
		  break;
	       case tag_concat:
		  ctype = typecheck_concat (rule, i, ex -> u.concat, dtype);
		  break;
	       default: ctype = undefined_type;
	     };

	  result_type = balance_type (rule, i, ctype, p -> type);
	  if (p -> type != result_type)
	     { p -> type = result_type;
	       change = 1;
	     };
	  return (result_type);
	};

private void update_protopos_type (hyper_rule rule, int i, int postype,
				   pos protopos, int actual)
	{ int prototype = protopos -> type;
	  if (prototype == error_type) return;
	  if (postype == error_type)
	     { protopos -> type = error_type;
	       change = 1;
	       return;
	     };
	  if (postype == undefined_type) return;
	  if (prototype == undefined_type)
	     { protopos -> type = postype;
	       change = 1;
	       return;
	     };
	  if (prototype == postype) return;
	  if (postype == any_type)
	     { type_error (
     "in rule %s, alt %d: cannot coerce type of position from any type to %s",
			rule -> nonterminal, i, string_from_type (prototype));
	       protopos -> type = error_type;
	       change = 1;
	       return;
	     };
	  if ((prototype == integer_or_string_type) &&
	      ((postype == string_type) || (postype == integer_type)))
	     { protopos -> type = postype;
	       change = 1;
	       return;
	     };
	  if ((prototype == any_type) && (postype != any_type) && !actual)
	     { type_error (
  "in rule %s, alt %d: unbalance in type of position between %s and any type",
			rule -> nonterminal, i, string_from_type (postype));
	       protopos -> type = error_type;
	       change = 1;
	     };
	};

private void typecheck_display (hyper_rule rule, int i, pos_list pl,
			        pos_list protodisplay, int actual)
	{ int j;
	  if (pl == pos_list_nil) return;
	  for (j=0; j < pl -> nrofps; j++)
	     { pos thispos = pl -> ps[j];
	       pos protopos = protodisplay -> ps[j];
	       int demanded_type = ((protopos -> type == any_type) && actual)?
			undefined_type:protopos -> type;
	       int postype = typecheck_position (rule, i,
				thispos, demanded_type);
	       update_protopos_type (rule, i, postype, protopos, actual);
	     };
	};

private void typecheck_call (hyper_rule rule, int i, call c)
	{ typecheck_display (rule, i, c -> display,
				c -> def -> proto_display, 1);
	};

private void typecheck_semiterminal (hyper_rule rule, int i, semiterminal s)
	{ (void) typecheck_position (rule, i,
		s -> display -> ps[0], string_type);
	};

private void typecheck_member (hyper_rule rule, int i, member m)
	{ switch (m -> tag)
	     { case tag_call: typecheck_call (rule, i, m -> u.cl); break;
	       case tag_terminal: break;
	       case tag_semiterminal:
		  typecheck_semiterminal (rule, i, m -> u.semi); break;
	       case tag_cut:
	       default: break;
	     };
	};

private void typecheck_members (hyper_rule rule, int i, member_list mems)
	{ int j;
	  if (mems == member_list_nil) return;
	  for (j=0; j < mems -> nrofms; j++)
	     typecheck_member (rule, i, mems -> ms[j]);
	};

private void initial_typecheck_alt (hyper_rule rule, int i, alt a)
	{ typecheck_display (rule, i, a -> display, rule -> proto_display, 0);
	};

private void typecheck_alt (hyper_rule rule, int i, alt a)
	{ typecheck_display (rule, i, a -> display, rule -> proto_display, 0);
	  typecheck_members (rule, i, a -> members);
	  typecheck_display (rule, i, a -> display, rule -> proto_display, 0);
	};

private void typecheck_alts (hyper_rule rule, alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     typecheck_alt (rule, i, alts -> as[i]);
	};

private void initial_typecheck_hyper_rule (hyper_rule rule)
	{ int i;
	  for (i=0; i < rule -> alts -> nrofas; i++)
	     initial_typecheck_alt (rule, i, rule -> alts -> as[i]);
	};

private void initial_typecheck_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     initial_typecheck_hyper_rule (all_hyper_rules[i]);
	};

/*
   do an incremental typecheck
*/
private int display_completely_typed (pos_list pl)
	{ int i;
	  if (pl == pos_list_nil) return (1);
	  for (i=0; i < pl -> nrofps; i++)
	     if ((pl -> ps[i] -> type == undefined_type) ||
		 (pl -> ps[i] -> type == integer_or_string_type)) return (0);
	  return (1);
	};

private int alt_completely_typed (alt a)
	{ affix ptr;
	  for (ptr = a -> locals; ptr != affix_nil; ptr = ptr -> next)
	     if ((ptr -> type == undefined_type) ||
		 (ptr -> type == integer_or_string_type))
		return (0);
	  return (1);
	};

private int alts_completely_typed (alt_list alts)
	{ int i;
	  for (i=0; i < alts -> nrofas; i++)
	     if (!alt_completely_typed (alts -> as[i])) return (0);
	  return (1);
	};

private int rule_completely_typed (hyper_rule rule)
	{ if (!display_completely_typed (rule -> proto_display)) return (0);
	  if (!alts_completely_typed (rule -> alts)) return (0);
	  return (1);
	};

private void typecheck_hyper_rule (hyper_rule rule)
	{ if (rule -> ext) return;
	  typecheck_alts (rule, rule -> alts);
	};

private void typecheck_hyper_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     typecheck_hyper_rule (all_hyper_rules[i]);
	};

/*
   dump those affixes whose type could not be resolved
*/
private void dump_untyped_affixes_in_alt (int i, alt a)
	{ affix ptr;
          for (ptr = a -> locals; ptr != affix_nil; ptr = ptr -> next)
             if ((ptr -> type == undefined_type) ||
		 (ptr -> type == integer_or_string_type))
		wlog ("affix %s in alt %d is not completely typed",
			ptr -> name, i + 1);
	};

private void dump_untyped_affixes_in_rule (hyper_rule rule)
	{ alt_list alts = rule -> alts;
	  int i;
	  for (i=0; i < alts -> nrofas; i++)
	     dump_untyped_affixes_in_alt (i, alts -> as[i]);
	};

private void final_check_if_rule_typed (hyper_rule rule)
	{ if (!rule_completely_typed (rule))
	     { type_error ("rule %s could not be completely typed",
			rule -> nonterminal);
	       dump_untyped_affixes_in_rule (rule);
	     };
	};

private void final_typecheck ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     final_check_if_rule_typed (all_hyper_rules[i]);
	};

/*
   dump typecheck results
*/
private void dump_protodisplay (pos_list pl)
	{ int i;
	  if (pl == pos_list_nil) return;
	  if (pl -> nrofps == 0) return;
	  eprint_log (" (");
	  for (i=0; i < pl -> nrofps; i++)
	     { if (pl -> ps[i] -> kind == inherited) eprint_log (">");
	       eprint_log ("%s", string_from_type (pl -> ps[i] -> type));
	       if (pl -> ps[i] -> kind == derived) eprint_log (">");
	       if (i < pl -> nrofps - 1) eprint_log (", ");
	     };
	  eprint_log (")");
	};

private void dump_rule_typing (hyper_rule rule)
	{ eprint_log ("%s %s", rule_qualifier (rule), rule -> nonterminal);
	  dump_protodisplay (rule -> proto_display);
	  eprint_log (".\n");
	};

private void try_dump_grammar_typing ()
	{ int i;
	  if (!full_verbose) return;
	  wlog ("Hyper rules have been typed as follows:");
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     dump_rule_typing (all_hyper_rules[i]);
	};

#define MAXPASSES 20
public void type_check ()
	{ int nr_passes = 0;
	  warning ("type checking syntax rules...");
	  init_typecheck ();
	  initial_typecheck_hyper_rules ();
	  do
	     { change = 0;
	       nr_passes++;
	       typecheck_hyper_rules ();
	     }
	  while (change && (nr_passes < MAXPASSES));
	  hint ("needed %d pass%s for typecheck",
		nr_passes, (nr_passes == 1)?"":"es");
	  final_typecheck ();
	  try_dump_grammar_typing ();
	  if (typing_errors)
	     panic ("%d typing error%s w%s found", typing_errors,
		    (typing_errors==1)?"":"s", (typing_errors==1)?"as":"ere");
	};
