/*
   File: recursion.c
   Determines the primary leftcorner relation and
   checks left recursion for topdown parsers
*/

/* global includes */
#include <stdio.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <memalloc.h>
#include <warshall.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <main.h>
#include <recursion.h>

public int leftrecursive;
private void init_recursion_check ()
	{ leftrecursive = 0;
	};

/*
   Determine which non predicate rules call each other either on the left
   calls_left [j * nr_of_hyper_rules + i] = 1 <=>
   j is called by i in one step as a (possibly) leftmost member <=>
   rule j <LC1 rule i
*/
private char *calls_left;
private void allocate_space_for_calls ()
	{ int i,j;
	  calls_left = (char *) ckcalloc
		       (nr_of_hyper_rules * nr_of_hyper_rules, sizeof (char));
	  for (i=0; i < nr_of_hyper_rules; i++)
	     for (j = 0; j < nr_of_hyper_rules; j++)
	        calls_left[i * nr_of_hyper_rules + j] = 0;
	};

private void detect_call_in_member (hyper_rule rule, member m)
	{ if (m -> tag == tag_call)
	     { int rulenumber = rule -> number;
	       hyper_rule callee = m -> u.cl -> def;
	       if (!callee -> ext)
	          calls_left [callee -> number * nr_of_hyper_rules +
			      rulenumber] = 1;
	     };
	};

private void detect_calls_in_alt (hyper_rule rule, alt a)
	{ member_list mems = a -> members;
	  int i;
	  if (mems == member_list_nil) return;
	  for (i=0; i < mems -> nrofms; i++)
	     { detect_call_in_member (rule, mems -> ms[i]);
	       if (mems -> ms[i] -> empty == h_neverproducesempty) return;
	     };
	};

private void detect_calls_in_rule (hyper_rule rule)
	{ alt_list alts = rule -> alts;
	  int i;
	  if (rule -> ext) return;
	  if (rule -> kind & (h_predicate | h_semipredicate)) return;
	  if (!rule -> reachable) return;
	  for (i=0; i < alts -> nrofas; i++)
	     detect_calls_in_alt (rule, rule -> alts -> as[i]);
	};

private void detect_calls_in_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     detect_calls_in_rule (all_hyper_rules[i]);
	};

/*
   Determine which non predicate rules call each other either on the left
   in one or more steps by taking the closure of the calls_left relation
   leftc_relation [j * nr_of_hyper_rules + i] = 1 <=>
   j is called by i in one or more steps as a (possibly) leftmost member <=>
   rule j <LC+ rule i
*/
public char *leftc_relation;
private void take_calls_closure ()
	{ leftc_relation = warshall (nr_of_hyper_rules, calls_left);
	};

/*
   report primary lc relation
*/
private void try_report_primary_lc_relation ()
	{ int i,j;
	  if (!full_verbose) return;
	  warning ("Dump of primary lc relation:");
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     { char *nont_i = all_hyper_rules[i] -> nonterminal;
	       for (j = 0; j < nr_of_hyper_rules; j++)
		  if (leftc_relation [i * nr_of_hyper_rules + j])
		     { char *nont_j = all_hyper_rules[j] -> nonterminal;
		       warning ("rule %s <LC+ rule %s", nont_i, nont_j);
		     };
	     };
	};

private void complain_on_left_recursive_rules ()
	{ int i;
	  for (i = 0; i < nr_of_hyper_rules; i++)
	     if (leftc_relation [i * nr_of_hyper_rules + i])
	        { hyper_rule rule = all_hyper_rules[i];
	          error ("rule %s is left recursive", rule -> nonterminal);
		  leftrecursive ++;
		};
	};

public void check_recursion ()
	{ warning ("computing primary leftcorner relation...");
	  init_recursion_check ();
	  allocate_space_for_calls ();
	  detect_calls_in_rules ();
	  take_calls_closure ();
	  try_report_primary_lc_relation ();
	  if (!topdown) return;
	  complain_on_left_recursive_rules ();
	  if (leftrecursive)
	     panic ("found %d left recursive rule%s", leftrecursive,
		    (leftrecursive > 1)?"s":"");
	};
