/*
   File: meta.c
   Analyzes the meta grammar
   Calculates the value of meta rules with a single production
   Detects left recursive meta rules
*/

/* global includes */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

/* libeag includes */
#include <export.h>
#include <error.h>
#include <memalloc.h>
#include <textstorage.h>

/* local includes */
#include <tree.h>
#include <numbering.h>
#include <warshall.h>
#include <common.h>
#include <meta.h>
#include <typecheck.h>
#include <sizes.h>
#include <main.h>

private int meta_errors;
private void meta_error (char *format, ...)
	{ char buf[MAXSTRLEN];
	  va_list arg_ptr;
	  va_start (arg_ptr, format);
	  vsprintf (buf, format, arg_ptr);
	  va_end (arg_ptr);

	  meta_errors++;
	  error ("error in metagrammar: %s", buf);
	};

/*
   Some storage administration of values
*/
private value free_values;
private value new_value (int type)
	{ value new;
	  if (free_values != value_nil)
	     { new = free_values;
	       free_values = (value) free_values -> v.string;
	     }
	  else new = (value) ckmalloc (sizeof (struct value_rec));
	  new -> ref_count = 1;
	  new -> type = type;
	  return (new);
	};

/* Announce to use 'room' values in an value_list */
private void room_value_list (value_list vl, int room)
	{ if (room <= vl -> room) return;
	  vl -> vals = (value *) ckrecalloc (vl -> vals, room, sizeof (value));
	  vl -> room = room;
	};

/* Allocate a new value_list */
public value_list new_value_list ()
	{ value_list new =
			(value_list) ckmalloc (sizeof (struct value_list_rec));
	  new -> nrofvs = 0;
	  new -> room = 2;
	  new -> vals = (value *) ckcalloc (2, sizeof (value));
	  return (new);
	};

/* Free a value list */
private void free_value_list (value_list old)
	{ if (old == value_list_nil) return;
	  ckfree (old -> vals);
	  ckfree (old);
	};

/* Append value to value_list */
public void app_value_list (value_list vl, value v)
	{ if (vl -> nrofvs == vl -> room)
	     room_value_list (vl, vl -> nrofvs << 1);
	  vl -> vals [vl -> nrofvs] = v;
	  vl -> nrofvs++;
	};

#define attach_value(val) (val) -> ref_count++
private void detach_value (value val)
	{ if (val == value_nil) return;
	  val -> ref_count--;
	  if (val -> ref_count) return;
	  if (val -> type == tuple_type)
	     { value_list old = val -> v.tuple;
	       int ix;
	       if (old == value_list_nil) return;
	       for (ix = 0; ix < old -> nrofvs; ix++)
		  detach_value (old -> vals[ix]);
	       free_value_list (old);
	     };
	  val -> v.string = (char *) free_values;
	  free_values = val;
	};

private void dump_value (value val)
	{ if (val == value_nil) eprint_log ("nil");
	  else
	     switch (val -> type)
		{ case string_type: output_string (stderr, val -> v.string);
				    break;
		  case integer_type: eprint_log ("%d", val -> v.number);
				     break;
		  case tuple_type:
		     { value_list vl = val -> v.tuple;
		       eprint_log ("<");
		       if (vl != value_list_nil)
			  { int ix;
			    for (ix = 0; ix < vl -> nrofvs; ix++)
			       { if (ix != 0) eprint_log (" * ");
				 dump_value (vl -> vals[ix]);
			       };
			  };
		       eprint_log (">");
		     };
		};
	};

/*
   for the meta rule 'nil' a special value must be entered in the
   meta definition
*/
private void enter_special_nil_value ()
	{ meta_rule nil_def = lookup_meta_rule ("nil");
	  if (nil_def == meta_rule_nil)
	     panic ("somebody erased nil from stddefs.eag");
	  nil_def -> mvalue = new_value (tuple_type);
	  nil_def -> mvalue -> v.tuple = new_value_list ();
	};

private void init_meta ()
	{ meta_errors = 0;
	  free_values = value_nil;
	  enter_special_nil_value ();
	};

/*
   Analyze the meta grammar
*/
private void analyze_affix (affix a, int *type, int *kind,
			    value *mvalue, int *empty)
	{ switch (a -> tag)
	     { case tag_affix_nonterminal:
		  { meta_rule def = a -> u.var -> def;
		    *type = def -> type;
		    *kind = def -> kind;
		    *empty = def -> empty;
		    if (def -> kind == single_meta_value)
		       { *mvalue = def -> mvalue;
			 attach_value (def -> mvalue);
		       }
		    else *mvalue = value_nil;
		    return;
		  };
	       case tag_affix_terminal:
		  { value new = new_value (string_type);
		    new -> v.string = a -> u.string;
		    *type = string_type;
		    *kind = single_meta_value;
		    if (strlen (a -> u.string) == 0) *empty = h_mayproduceempty;
		    else *empty = h_neverproducesempty;
		    *mvalue = new;
		    return;
		  };
	       case tag_affix_number:
		  { value new = new_value (integer_type);
		    new -> v.number = a -> u.number;
		    *type = integer_type;
		    *kind = single_meta_value;
		    if (a -> u.number <= 0) *empty = h_mayproduceempty;
		    else *empty = h_neverproducesempty;
		    *mvalue = new;
		    return;
		  };
	       case tag_affix_set:
		  { set aset = a -> u.aset;
		    if ((aset -> kind & star) || (!(aset -> kind & non) &&
			(strlen (aset -> string) == 0)))
		       *empty = h_mayproduceempty;
		    else *empty = h_neverproducesempty;
		    *type = string_type;
		    *kind = recognizer_meta_value;
		    *mvalue = value_nil;
		    return;
		  };
	     };
	}; 

private void check_concat_types (meta_rule rule, int i, int j,
				 int type, int *ctype, int *kind)
	{ if ((type == any_type) || (type == tuple_type))
	     { meta_error (
"in metarule %s, alt %d:\nconcatenatation of metarules of any or tuple type",
		  	rule -> meta_nonterminal, i + 1);
	       *ctype = error_type;
	       *kind = error_meta_value;
	       return;
	     };
	  if (j == 0) *ctype = type;
	  if (*ctype == undefined_type) *ctype = type;
	  else if (type == undefined_type) return;
	  else if (type == error_type) *ctype = error_type;
	  else if (type == *ctype) return;
	  else
	     { meta_error
		     ("in metarule %s, alt %d:\ntype mismatch in concatenation",
		     rule -> meta_nonterminal, i + 1);
	       *ctype = error_type;
	       *kind = error_meta_value;
	     };
	};

/*
   A bit tricky since it assumes that
   undefined < single < multiple < recognizer < error
*/
private void check_kinds (int j, int kind, int *ckind)
	{ if (j == 0) *ckind = kind;
	  else if (*ckind == undefined_meta_value) return;
	  else if (kind == undefined_meta_value) *ckind = undefined_meta_value;
	  else if (kind == error_meta_value) *ckind = error_meta_value;
	  else if (*ckind < kind) *ckind = kind;
	};

private void check_concat_empties (int j, int empty, int *cempty)
	{ if (j == 0) *cempty = empty;
	  else if ((*cempty == h_mayproduceempty) &&
		   (empty == h_mayproduceempty)) return;
	  else if (*cempty == h_neverproducesempty) return;
	  else if (empty == h_neverproducesempty)
             *cempty = h_neverproducesempty;
	  else *cempty = h_undefined;
	};

private char strstore[MAXSTORE];
private void analyze_concatenation (meta_rule rule, int i, affix_list affs,
				    int *type, int *kind,
				    value *mvalue, int *empty)
	{ int j;
	  char *dptr = strstore;
	  char *sptr;
	  int itotal = 0;
	  value new;
	  for (j=0; j < affs -> nrofas; j++)
	     { int type2, kind2, empty2;
	       value tvalue;
	       analyze_affix (affs -> as[j], &type2, &kind2, &tvalue, &empty2);
	       check_concat_types (rule, i, j, type2, type, &kind2);
	       check_kinds (j, kind2, kind);
	       check_concat_empties (j, empty2, empty);
	       if (*kind == single_meta_value)
		  { if (*type == string_type)
		       for (sptr = tvalue -> v.string; *sptr; sptr++, dptr++)
			  *dptr = *sptr;
		    else if (*type == integer_type)
		       itotal += tvalue -> v.number;
		  };
	       detach_value (tvalue);
	     };
	  if (*kind == single_meta_value)
	     { if (*type == string_type)
		  { *dptr = '\0';
		    new = new_value (string_type);
		    new -> v.string = addto_names (strstore);
		    *mvalue = new;
		  }
	       else if (*type == integer_type)
		  { new = new_value (integer_type);
		    new -> v.number = itotal;
		    *mvalue = new;
		  };
	     }
	  else *mvalue = value_nil;
	};

/*
   In case of composition with only one production build the value
*/
private void analyze_composition (affix_list affs, int *type, int *kind,
				  value *mvalue, int *empty)
	{ int j;
	  value cvalue = new_value (tuple_type);
	  value_list vl = new_value_list ();
	  value nvalue;
	  cvalue -> v.tuple = vl;
	  for (j=0; j < affs -> nrofas; j++)
	     { int type2, kind2, empty2;
	       analyze_affix (affs -> as[j], &type2, &kind2, &nvalue, &empty2);
	       app_value_list (vl, nvalue);
	       check_kinds (j, kind2, kind);
	     };
	  if (*kind == single_meta_value) *mvalue = cvalue;
	  else
	     { detach_value (cvalue);
	       *mvalue = value_nil;
	     };
	  *type = tuple_type;
	  *empty = h_neverproducesempty;
	};

private void analyze_alt (meta_rule rule, int i, expr e,
			  int *type, int *kind, value *mvalue, int *empty)
	{ if (e == expr_nil)
	     { value new = new_value (string_type);
	       new -> v.string = "";
	       *type = string_type;
	       *kind = single_meta_value;
	       *mvalue = new;
	       *empty = h_mayproduceempty;
	     }
	  else
	     switch (e -> tag)
		{ case tag_single:
		     analyze_affix (e -> u.single, type, kind, mvalue, empty);
		     break;
		  case tag_compos:
		     analyze_composition (e -> u.compos,
				type, kind, mvalue, empty);
		     break;
		  case tag_concat:
		     analyze_concatenation (rule, i, e -> u.concat,
				type, kind, mvalue, empty);
		     break;
		};
	};

private void check_alt_types (meta_rule rule, int i, int type, int *ctype)
	{ if ((i == 0) || (*ctype == undefined_type)) *ctype = type;
	  else if (type == undefined_type) return;
	  else if (type == error_type) *ctype = error_type;
	  else if (type == *ctype) return;
	  else
	     { meta_error
			("in metarule %s:\ntype mismatch between alternatives",
			rule -> meta_nonterminal);
	       *ctype = error_type;
	     };
	};

private void check_alt_kinds (int i, int kind, int *ckind)
	{ if (i == 0) *ckind = kind;
	  else if (*ckind == undefined_meta_value) return;
	  else if (kind == undefined_meta_value) *ckind = undefined_meta_value;
	  else if (kind == error_meta_value) *ckind = error_meta_value;
	  else if ((kind == single_meta_value) && (*ckind == single_meta_value))
			*ckind = multiple_meta_value;
	  else if (*ckind < kind) *ckind = kind;
	};

private void check_alt_empties (int i, int empty, int *cempty)
	{ if (i == 0) *cempty = empty;
	  else if ((*cempty == h_neverproducesempty) &&
		   (empty == h_neverproducesempty)) return;
	  else if (*cempty == h_mayproduceempty) return;
	  else if (empty == h_mayproduceempty) *cempty = h_mayproduceempty;
	  else *cempty = h_undefined;
	};

private void analyze_alts (meta_rule rule, meta_alt_list alts,
			   int *type, int *kind, value *mvalue, int *empty)
	{ int i;
	  if (alts == meta_alt_list_nil)
	     { value new = new_value (string_type);
	       new -> v.string = "";
	       *type = string_type;
	       *kind = single_meta_value;
	       *mvalue = new;
	       *empty = h_mayproduceempty;
	       return;
	     };
	  if (alts -> nrofas == 1)
	     { analyze_alt (rule, 0, alts -> as[0] -> e,
			type, kind, mvalue, empty);
	       return;
	     };
	  for (i=0; i < alts -> nrofas; i++)
	     { int type2, kind2, empty2;
	       value tvalue;
	       analyze_alt (rule, i, alts -> as[i] -> e,
			&type2, &kind2, &tvalue, &empty2);
	       check_alt_types (rule, i, type2, type);
	       check_alt_kinds (i, kind2, kind);
	       check_alt_empties (i, empty2, empty);
	       detach_value (tvalue);
	     };
	};

private int change;
private void update_rule_type (meta_rule rule, int type)
	{ if (type == undefined_type) return;
	  else if (rule -> type == undefined_type)
	     { rule -> type = type;
	       change = 1;
	     }
	  else if (rule -> type == error_type) return;
	  else if (type == rule -> type) return;
	  else if (type == error_type)
	     { rule -> type = error_type;
	       change = 1;
	     }
	  else 
	     { meta_error ("typing inconsistency in metarule %s",
			rule -> meta_nonterminal);
	       rule -> type = error_type;
	       change = 1;
	     };
	};

private void update_rule_kind (meta_rule rule, int kind)
	{ if (kind == undefined_meta_value) return;
	  else if (rule -> kind == undefined_meta_value)
	     { rule -> kind = kind;
	       change = 1;
	     }
	  else if (rule -> kind == error_meta_value) return;
	  else if (kind == rule -> kind) return;
	  else if (kind == error_meta_value)
	     { rule -> kind = error_meta_value;
	       change = 1;
	     }
	  else 
	     { meta_error ("kind inconsistency in metarule %s",
				rule -> meta_nonterminal);
	       rule -> kind = error_meta_value;
	       change = 1;
	     };
	};

private void update_rule_value (meta_rule rule, value mvalue)
	{ if (rule -> kind != single_meta_value) return;
	  if (rule -> mvalue != value_nil) return;
	  rule -> mvalue = mvalue;
	  attach_value (mvalue);
	  change = 1;
	};

private void update_rule_empty (meta_rule rule, int empty)
	{ if (rule -> empty == h_undefined)
	     { rule -> empty = empty;
	       if (empty != h_undefined) change = 1;
	     }
	  else if (rule -> empty != empty)
	     panic ("empty inconsistency in metarule %s",
			rule -> meta_nonterminal);
	};

private void analyze_meta_rule (meta_rule rule)
	{ int type = undefined_type;
	  int kind = undefined_meta_value;
	  value mvalue = value_nil;
	  int empty = h_undefined;
	  if (rule -> ext) return;
	  analyze_alts (rule, rule -> meta_alts, &type, &kind, &mvalue, &empty);
	  update_rule_type (rule, type);
	  update_rule_kind (rule, kind);
	  update_rule_value (rule, mvalue);
	  update_rule_empty (rule, empty);
	  detach_value (mvalue);
	};

private void analyze_meta_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     analyze_meta_rule (all_meta_rules[i]);
	};

private void finish_meta_rule_analysis (meta_rule rule)
	{ if (rule -> kind == undefined_meta_value)
	     rule -> kind = recognizer_meta_value;
	  if (rule -> empty == h_undefined)
	     rule -> empty = h_neverproducesempty;
	  if (rule -> type == undefined_type)
	     meta_error ("metarule %s could not be typed",
			rule -> meta_nonterminal);
	};

private void finish_meta_grammar_analysis ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     finish_meta_rule_analysis (all_meta_rules[i]);
	};

private void analyze_metagrammar_typing ()
	{ int nr_passes = 0;
	  warning ("analyzing meta grammar...");
	  do
	     { change = 0;
	       nr_passes++;
	       analyze_meta_rules ();
	     }
	  while (change);
	  finish_meta_grammar_analysis ();
	  hint ("needed %d pass%s for meta grammar analysis",
		nr_passes, (nr_passes == 1)?"":"es");
	};

/*
   Analyze the leftrecursiveness of meta rules
*/
private char *mcalls_left;
private char *mcalls_left_closure;
private void allocate_space_for_mcalls ()
	{ int i,j;
	  mcalls_left = (char *) ckcalloc
			(nr_of_meta_rules * nr_of_meta_rules, sizeof (char));
	  for (i=0; i < nr_of_meta_rules; i++)
	     for (j=0; j < nr_of_meta_rules; j++)
		mcalls_left[i * nr_of_meta_rules + j] = 0;
	};

private int detect_lcall_in_affix (meta_rule rule, affix a)
	{ switch (a -> tag)
	     { case tag_affix_nonterminal:
		  { meta_rule def = a -> u.var -> def;
		    int rulenumber = rule -> number;
		    int calleenumber = def -> number;
		    mcalls_left
			[rulenumber * nr_of_meta_rules + calleenumber] = 1;
		    return (def -> empty == h_mayproduceempty);
		  };
	       case tag_affix_terminal: return (strlen (a -> u.string) == 0);
	       case tag_affix_number: return (a -> u.number <= 0);
	       case tag_affix_set:
		  { set aset = a -> u.aset;
		    return ((aset -> kind & star) ||
		     (!(aset -> kind & non) && (strlen (aset -> string) == 0)));
		  };
	       default: return (0);
	     };
	};

private void detect_lcalls_in_concat (meta_rule rule, affix_list affs)
	{ int i = 0;
	  int empty;
	  do
	     { empty = detect_lcall_in_affix (rule, affs -> as[i]);
	       i++;
	     }
	  while (empty && (i < affs -> nrofas));
	};

private void detect_lcalls_in_meta_alt (meta_rule rule, expr e)
	{ if (e == expr_nil) return;
	  switch (e -> tag)
	     { case tag_single:
		  (void) detect_lcall_in_affix (rule, e -> u.single);
		  break;
	       case tag_concat:
		  detect_lcalls_in_concat (rule, e -> u.concat);
	          break;
	       default:
		  internal_error ("detect lcalls in meta alt");
	     };
	};

private void detect_lcalls_in_meta_alts (meta_rule rule, meta_alt_list alts)
	{ int i;
	  if (alts == meta_alt_list_nil) return;
	  for (i=0; i < alts -> nrofas; i++)
	     detect_lcalls_in_meta_alt (rule, alts -> as[i] -> e);
	};

private void detect_lcalls_in_metarule (meta_rule rule)
	{ if (rule -> ext) return;
	  if (rule -> type == undefined_type) return;
	  if (rule -> type == tuple_type) return;
	  detect_lcalls_in_meta_alts (rule, rule -> meta_alts);
	};

private void detect_lcalls_in_metarules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     detect_lcalls_in_metarule (all_meta_rules[i]);
	};

private void take_lcalls_closure ()
	{ mcalls_left_closure = warshall (nr_of_meta_rules, mcalls_left);
	};

private void complain_on_leftrecursive_meta_rules ()
	{ int i;
	  for (i = 0; i < nr_of_meta_rules; i++)
	     if (mcalls_left_closure [i * nr_of_meta_rules + i])
		meta_error ("meta rule %s is left recursive",
			    all_meta_rules[i] -> meta_nonterminal);
	};

private void analyze_metagrammar_leftrecursiveness ()
	{ warning ("detecting left recursive meta rules...");
	  allocate_space_for_mcalls ();
	  detect_lcalls_in_metarules ();
	  take_lcalls_closure ();
	  complain_on_leftrecursive_meta_rules ();
	};

/*
   dump rule analysis
*/
private char *string_from_kind (int kind)
	{ switch (kind)
	     { case undefined_meta_value: return ("undefined");
	       case single_meta_value: return ("single");
	       case multiple_meta_value: return ("multiple");
	       case recognizer_meta_value: return ("recognizer");
	       default: return ("error");
	     };
	};

private void dump_rule_analysis (meta_rule rule)
	{ eprint_log ("%s :: %s, %s, ", rule -> meta_nonterminal,
		string_from_type (rule -> type),
		string_from_kind (rule -> kind));
	  if (rule -> empty == h_mayproduceempty)
	     eprint_log ("may produce empty");
	  else eprint_log ("never produces empty");
	  if (rule -> kind == single_meta_value)
	     { eprint_log (", value = ");
	       dump_value (rule -> mvalue);
	     };
	  eprint_log (".\n");
	};

private void try_dump_metagrammar_analysis ()
	{ int i;
	  if (!full_verbose) return;
	  wlog ("Meta rules have been classified as follows:");
	  for (i = 0; i < nr_of_meta_rules; i++)
	     dump_rule_analysis (all_meta_rules[i]);
	};

public void analyze_metagrammar ()
	{ init_meta ();
	  analyze_metagrammar_typing ();
	  analyze_metagrammar_leftrecursiveness ();
	  if (meta_errors)
	     panic ("%d error%s w%s found in meta grammar analysis",
		    meta_errors, (meta_errors == 1)?"":"s",
		    (meta_errors == 1)?"as":"ere");
	  try_dump_metagrammar_analysis ();
	};
